<?php
namespace MaxAddonsPro;

use MaxAddonsPro\Classes\Helper;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class MaxQuery {

	public function __construct() {
		add_filter( 'bricks/setup/control_options', [ $this, 'add_maxquery_control_options' ], 10, 1 );
		add_filter( "bricks/elements/container/controls", [ $this, 'add_maxquery_controls' ], 10, 1 );
		add_filter( "bricks/elements/div/controls", [ $this, 'add_maxquery_controls' ], 10, 1 );
		add_filter( "bricks/elements/block/controls", [ $this, 'add_maxquery_controls' ], 10, 1 );
		add_filter( 'bricks/query/run', [ $this, 'run_max_query' ], 10, 2 );

		if ( Helper::is_woocommerce_active() ) {
			add_filter( 'bricks/query/run', [ $this, 'run_max_woo_query' ], 10, 2 );
		}

		add_filter( 'bricks/query/loop_object', [ $this, 'set_loop_object' ], 10, 3);
	}

	/**
	 * Adds the max query to the Query Loop builder
	 *
	 * @param array $control_options
	 * @return array
	 */
	public function add_maxquery_control_options( $control_options ) {
		$control_options['queryTypes']['maxQuery'] = esc_html__( 'Max Query', 'max-addons' );

		if ( Helper::is_woocommerce_active() ) {
			$control_options['queryTypes']['maxWooQuery'] = esc_html__( 'Max Woo Query', 'max-addons' );
		}

		return $control_options;
	}


	public function add_maxquery_controls( $controls ) {
		$controls = $this->maxquery_controls( $controls );

		if ( Helper::is_woocommerce_active() ) {
			$controls = $this->maxwooquery_controls( $controls );
		}

		return $controls;
	}

	public function maxquery_controls( $controls ) {
		$max_query_options = [
			'adjacent'    => esc_html__('Adjacent posts', 'max-addons'),
			'related'     => esc_html__('Related posts', 'max-addons'),
		];

		if ( class_exists( 'ACF' ) ) {
			$max_query_options['acf-gallery'] = esc_html__('ACF gallery', 'max-addons');
		}

		$maxQueryControls['maxQueryType'] = [
			'tab'      => 'content',
			'label'    => esc_html__( 'Max query type', 'max-addons' ),
			'type'     => 'select',
			'options'  => $max_query_options,
			'inline'   => true,
			'rerender' => true,
			'required' => array(
				[ 'query.objectType', '=', 'maxQuery' ],
				[ 'hasLoop', '!=', false ]
			),
		];

		// Related Posts
		$maxQueryControls['maxRelatedPostType'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Post type', 'max-addons' ),
			'type'        => 'select',
			'options'     => bricks_is_builder() ? \Bricks\Helpers::get_registered_post_types() : [],
			'inline'      => true,
			'clearable'   => true,
			'placeholder' => esc_html__( 'Default', 'max-addons' ),
			'required'    => [
				[ 'maxQueryType', '=', 'related' ],
				[ 'query.objectType', '=', 'maxQuery' ],
				[ 'hasLoop', '!=', false ]
			],
		];

		$maxQueryControls['maxRelatedPostsCount'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Max. related posts', 'max-addons' ),
			'type'        => 'number',
			'min'         => 1,
			'max'         => 4,
			'placeholder' => 3,
			'required'    => [
				[ 'maxQueryType', '=', 'related' ],
				[ 'query.objectType', '=', 'maxQuery' ],
				[ 'hasLoop', '!=', false ]
			],
		];

		$maxQueryControls['maxRelatedPostsOrder'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Order', 'max-addons' ),
			'type'        => 'select',
			'options'     => \Bricks\Setup::$control_options['queryOrder'],
			'inline'      => true,
			'placeholder' => esc_html__( 'Descending', 'max-addons' ),
			'required'    => [
				[ 'maxQueryType', '=', 'related' ],
				[ 'query.objectType', '=', 'maxQuery' ],
				[ 'hasLoop', '!=', false ]
			],
		];

		$maxQueryControls['maxRelatedPostsOrderby'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Order by', 'max-addons' ),
			'type'        => 'select',
			'options'     => \Bricks\Setup::$control_options['queryOrderBy'],
			'inline'      => true,
			'placeholder' => esc_html__( 'Random', 'max-addons' ),
			'required'    => [
				[ 'maxQueryType', '=', 'related' ],
				[ 'query.objectType', '=', 'maxQuery' ],
				[ 'hasLoop', '!=', false ]
			],
		];

		$maxQueryControls['maxRelatedPostsTaxonomies'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Common taxonomies', 'max-addons' ),
			'type'        => 'select',
			'options'     => \Bricks\Setup::$control_options['taxonomies'],
			'multiple'    => true,
			'default'     => [
				'category',
				'post_tag'
			],
			'description' => esc_html__( 'Taxonomies related posts must have in common.', 'max-addons' ),
			'required'    => [
				[ 'maxQueryType', '=', 'related' ],
				[ 'query.objectType', '=', 'maxQuery' ],
				[ 'hasLoop', '!=', false ]
			],
		];

		// Adjacent Posts
		$maxQueryControls['maxAdjacentPost'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Post type', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'prev'  => esc_html__( 'Previous', 'max-addons' ),
				'next'  => esc_html__( 'Next', 'max-addons' ),
			],
			'inline'      => true,
			'placeholder' => esc_html__( 'Previous', 'max-addons' ),
			'rerender'    => true,
			'multiple'    => false,
			'required'    => [
				[ 'maxQueryType', '=', 'adjacent' ],
				[ 'query.objectType', '=', 'maxQuery' ],
				[ 'hasLoop', '!=', false ]
			],
		];

		$maxQueryControls['maxAdjacentPostInSameTerm'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'In same term', 'max-addons' ),
			'type'        => 'checkbox',
			'description' => esc_html__( 'Posts should be in a same taxonomy term.', 'max-addons' ),
			'required'    => [
				[ 'maxQueryType', '=', 'adjacent' ],
				[ 'query.objectType', '=', 'maxQuery' ],
				[ 'hasLoop', '!=', false ]
			],
		];

		$maxQueryControls['maxAdjacentPostTaxonomy'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Taxonomy', 'max-addons' ),
			'type'        => 'select',
			'options'     => \Bricks\Setup::$control_options['taxonomies'],
			'placeholder' => 'category',
			'required'    => [
				[ 'maxQueryType', '=', 'adjacent' ],
				[ 'query.objectType', '=', 'maxQuery' ],
				[ 'hasLoop', '!=', false ],
				[ 'maxAdjacentPostInSameTerm', '=', true ]
			],
		];

		$maxQueryControls['maxAdjacentPostExcludedTerms'] = [
			'tab'      => 'content',
			'label'    => esc_html__( 'Excluded Terms', 'max-addons' ),
			'type'     => 'select',
			'options'  => bricks_is_builder() ? \Bricks\Helpers::get_terms_options() : [],
			'multiple' => true,
			'required' => [
				[ 'maxQueryType', '=', 'adjacent' ],
				[ 'query.objectType', '=', 'maxQuery' ],
				[ 'hasLoop', '!=', false ],
			],
		];

		// ACF Gallery
		$maxQueryControls['maxAcfGalleryField'] = [
			'tab'            => 'content',
			'label'          => esc_html__( 'ACF Gallery Field Name', 'max-addons' ),
			'type'           => 'text',
			'hasDynamicData' => false,
			'required'       => [
				[ 'maxQueryType', '=', 'acf-gallery' ],
				[ 'query.objectType', '=', 'maxQuery' ],
				[ 'hasLoop', '!=', false ]
			],
		];

		$query_key_index = absint( array_search( 'query', array_keys( $controls ) ) );
		$new_controls    = array_slice( $controls, 0, $query_key_index + 1, true ) + $maxQueryControls + array_slice( $controls, $query_key_index + 1, null, true );

		return $new_controls;
	}

	public function maxwooquery_controls( $controls ) {
		$maxWooQueryControls['maxWooQueryType'] = [
			'tab'      => 'content',
			'label'    => esc_html__( 'Max woo query type', 'max-addons' ),
			'type'     => 'select',
			'options'  => [
				'best-selling'    => esc_html__( 'Best selling products', 'max-addons' ),
				'featured'        => esc_html__( 'Featured products', 'max-addons' ),
				'sale'            => esc_html__( 'On sale products', 'max-addons' ),
				'top_rated'       => esc_html__( 'Top rated products', 'max-addons' ),
				'downloadable'    => esc_html__( 'Downloadable products', 'max-addons' ),
				'cross-sells'     => esc_html__( 'Product cross-sells', 'max-addons' ),
				'upsells'         => esc_html__( 'Product upsells', 'max-addons' ),
				'product-gallery' => esc_html__( 'Product gallery', 'max-addons' ),
				'ordered_items'   => esc_html__( 'Ordered items', 'max-addons' ),
			],
			'inline'   => true,
			'rerender' => true,
			'required' => array(
				[ 'query.objectType', '=', 'maxWooQuery' ],
				[ 'hasLoop', '!=', false ]
			),
		];

		$maxWooQueryControls['maxProductsCount'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Max. products', 'max-addons' ),
			'type'        => 'number',
			'min'         => 1,
			'max'         => 4,
			'placeholder' => 4,
			'required'    => array(
				[ 'query.objectType', '=', 'maxWooQuery' ],
				[ 'hasLoop', '!=', false ]
			),
		];

		$maxWooQueryControls['maxProductCategories'] = [
			'tab'      => 'content',
			'label'    => esc_html__( 'Product categories', 'max-addons' ),
			'type'     => 'select',
			'options'  => \Bricks\Woocommerce::$product_categories,
			'multiple' => true,
			'required' => array(
				[ 'query.objectType', '=', 'maxWooQuery' ],
				[ 'maxWooQueryType', '=', [ 'best-selling', 'featured', 'sale', 'top_rated', 'downloadable' ] ],
				[ 'hasLoop', '!=', false ]
			),
		];

		$maxWooQueryControls['maxProductsOrderby'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Order by', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'price'      => esc_html__( 'Price', 'max-addons' ),
				'popularity' => esc_html__( 'Popularity', 'max-addons' ),
				'rating'     => esc_html__( 'Rating', 'max-addons' ),
				'name'       => esc_html__( 'Name', 'max-addons' ),
				'rand'       => esc_html__( 'Random', 'max-addons' ),
				'date'       => esc_html__( 'Published date', 'max-addons' ),
				'modified'   => esc_html__( 'Modified date', 'max-addons' ),
				'menu_order' => esc_html__( 'Menu order', 'max-addons' ),
				'id'         => esc_html__( 'Product ID', 'max-addons' ),
			],
			'inline'      => true,
			'placeholder' => esc_html__( 'Date', 'max-addons' ),
			'required'    => array(
				[ 'query.objectType', '=', 'maxWooQuery' ],
				[ 'maxWooQueryType', '=', [ 'featured', 'sale', 'cross-sells', 'upsells', 'downloadable' ] ],
				[ 'hasLoop', '!=', false ]
			),
		];

		$maxWooQueryControls['maxProductsOrder'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Order', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'ASC'  => esc_html__( 'Ascending', 'max-addons' ),
				'DESC' => esc_html__( 'Descending', 'max-addons' ),
			],
			'inline'      => true,
			'placeholder' => esc_html__( 'Descending', 'max-addons' ),
			'required'    => array(
				[ 'query.objectType', '=', 'maxWooQuery' ],
				[ 'maxWooQueryType', '=', [ 'featured', 'sale', 'cross-sells', 'upsells', 'downloadable' ] ],
				[ 'hasLoop', '!=', false ]
			),
		];

		$maxWooQueryControls['maxHideOutOfStock'] = [
			'tab'      => 'content',
			'label'    => esc_html__( 'Hide out of stock', 'max-addons' ),
			'type'     => 'checkbox',
			'required' => array(
				[ 'query.objectType', '=', 'maxWooQuery' ],
				[ 'maxWooQueryType', '!=', [ 'product-gallery' ] ],
				[ 'hasLoop', '!=', false ]
			),
		];

		$maxWooQueryControls['maxShowHidden'] = [
			'tab'      => 'content',
			'label'    => esc_html__( 'Show hidden', 'max-addons' ),
			'type'     => 'checkbox',
			'required' => array(
				[ 'query.objectType', '=', 'maxWooQuery' ],
				[ 'maxWooQueryType', '!=', [ 'product-gallery' ] ],
				[ 'hasLoop', '!=', false ]
			),
		];

		$query_key_index = absint( array_search( 'query', array_keys( $controls ) ) );
		$new_controls    = array_slice( $controls, 0, $query_key_index + 1, true ) + $maxWooQueryControls + array_slice( $controls, $query_key_index + 1, null, true );

		return $new_controls;
	}

	public function run_max_query( $results, $query ) {
		if ( $query->object_type !== 'maxQuery' ) {
			return $results;
		}

		$settings = $query->settings;
		$max_query = isset( $settings['maxQueryType'] ) ? $settings['maxQueryType'] : false;

		if ( ! $max_query ) {
			return $results;
		}

		if ( 'related' === $max_query ) {
			global $post;

			$post_id = $post->ID;

			$args = [
				'posts_per_page' => isset( $settings['maxRelatedPostsCount'] ) ? $settings['maxRelatedPostsCount'] : 3,
				'post__not_in'   => [ $post_id ],
				'post_status'    => 'publish',
				'no_found_rows'  => true, // No pagination
				'orderby'        => isset( $settings['maxRelatedPostsOrderby'] ) ? $settings['maxRelatedPostsOrderby'] : 'rand',
				'order'          => isset( $settings['maxRelatedPostsOrder'] ) ? $settings['maxRelatedPostsOrder'] : 'DESC',
			];

			if ( ! empty( $settings['maxRelatedPostType'] ) ) {
				$args['post_type'] = $settings['maxRelatedPostType'];
			}

			$taxonomies = ! empty( $settings['maxRelatedPostsTaxonomies'] ) ? $settings['maxRelatedPostsTaxonomies'] : [];

			foreach ( $taxonomies as $taxonomy ) {
				$terms_ids = wp_get_post_terms(
					$post_id,
					$taxonomy,
					[ 'fields' => 'ids' ]
				);

				if ( ! empty( $terms_ids ) ) {
					$args['tax_query'][] = [
						'taxonomy' => $taxonomy,
						'field'    => 'term_id',
						'terms'    => $terms_ids,
					];
				}
			}

			if ( count( $taxonomies ) > 1 && isset( $args['tax_query'] ) ) {
				$args['tax_query']['relation'] = 'OR';
			}

			$related_posts_query = new \WP_Query( $args );

			return $related_posts_query->posts;
		}

		if ( 'adjacent' === $max_query ) {
			$adjacent_post_type = isset( $settings['maxAdjacentPost'] ) ? $settings['maxAdjacentPost'] : 'previous';
			$in_same_term = isset( $settings['maxAdjacentPostInSameTerm'] );
			$taxonomy = $in_same_term && isset( $settings['maxAdjacentPostTaxonomy'] ) ? $settings['maxAdjacentPostTaxonomy'] : 'category';
			$excluded_terms = isset( $settings['maxAdjacentPostExcludedTerms'] ) ? $settings['maxAdjacentPostExcludedTerms'] : '';

			if ( 'prev' === $adjacent_post_type && ! empty( get_previous_post( $in_same_term, $excluded_terms, $taxonomy ) ) ) {
				return [ get_previous_post( $in_same_term, $excluded_terms, $taxonomy ) ];
			}

			if ( 'next' === $adjacent_post_type && !empty( get_next_post( $in_same_term, $excluded_terms, $taxonomy ) ) ) {
				return [ get_next_post( $in_same_term, $excluded_terms, $taxonomy ) ];
			}
		}

		if ( 'acf-gallery' === $max_query && class_exists( 'ACF' ) ) {
			global $post;

			$preview_id = \Bricks\Database::$page_data['preview_or_post_id'];
			$post_id    = $post->ID;

			$acf_gallery_field = isset( $settings['maxAcfGalleryField'] ) ? $settings['maxAcfGalleryField'] : '';

			if ( $acf_gallery_field ) {
				$acf_gallery_image_ids = get_field( $acf_gallery_field, $post_id, false );

				if ( ! empty( $acf_gallery_image_ids ) ) {
					$acf_query_args['post_type']      = 'attachment';
					$acf_query_args['post_status']    = 'inherit';
					$acf_query_args['post__in']       = $acf_gallery_image_ids;
					$acf_query_args['posts_per_page'] = -1;

					$acf_gallery_query   = new \WP_Query( $acf_query_args );
					$acf_gallery_results = $acf_gallery_query->posts;
					wp_reset_query();

					return $acf_gallery_results;
				} else {
					return [];
				}
			} else {
				return [];
			}
		}
	}

	public function run_max_woo_query( $results, $query ) {
		if ( $query->object_type !== 'maxWooQuery' ) {
			return $results;
		}

		$settings = $query->settings;

		global $post;

		$args = [
			'post_type'      => 'product',
			'post_status'    => 'publish',
			'posts_per_page' => isset( $settings['maxProductsCount'] ) ? $settings['maxProductsCount'] : 4,
			'no_found_rows'  => true, // No pagination
		];

		$max_woo_query = isset( $settings['maxWooQueryType'] ) ? $settings['maxWooQueryType'] : '';

		$product_visibility_term_ids = wc_get_product_visibility_term_ids();

		if ( 'featured' === $max_woo_query || 'sale' === $max_woo_query || 'upsells' === $max_woo_query || 'cross-sells' === $max_woo_query || 'downloadable' === $max_woo_query ) {
			$orderby = ! empty( $settings['maxProductsOrderby'] ) ? $settings['maxProductsOrderby'] : 'date';
			$order   = ! empty( $settings['maxProductsOrder'] ) ? $settings['maxProductsOrder'] : 'DESC';

			if ( 'price' === $orderby || 'popularity' === $orderby || 'rating' === $orderby ) {
				if ( 'price' === $orderby ) {
					$args['meta_key'] = '_price';
				} elseif ( 'popularity' === $orderby ) {
					$args['meta_key'] = 'total_sales';
				} elseif ( 'rating' === $orderby ) {
					$args['meta_key'] = '_wc_average_rating';
				}

				$args['orderby'] = 'meta_value_num';
				$args['order']   = $order;
			} else {
				$ordering_args = WC()->query->get_catalog_ordering_args( $orderby, $order );

				$args['orderby'] = $ordering_args['orderby'];
				$args['order']   = $ordering_args['order'];
			}
		}

		if ( 'upsells' === $max_woo_query || 'cross-sells' === $max_woo_query ) {
			$preview_id = \Bricks\Database::$page_data['preview_or_post_id'];
			$product    = '';

			if ( is_singular( 'product' ) ) {
				$product = wc_get_product( ( ! empty( $preview_id ) ? $preview_id : $post->ID ) );
			}

			if ( empty( $product ) ) {
				return [];
			}
		}

		if ( 'best-selling' === $max_woo_query ) {
			$args['meta_key']   = 'total_sales';
			$args['order']      = 'DESC';
			$args['orderby']    = 'meta_value_num';
			$args['meta_query'] = WC()->query->get_meta_query();
			$args['tax_query']  = WC()->query->get_tax_query();
		}

		if ( 'featured' === $max_woo_query ) {
			$args['tax_query'][] = array(
				'taxonomy' => 'product_visibility',
				'field'    => 'term_taxonomy_id',
				'terms'    => $product_visibility_term_ids['featured'],
			);
		}

		if ( 'sale' === $max_woo_query ) {
			$args['post__in'] = array_merge( array( 0 ), wc_get_product_ids_on_sale() );
		}

		if ( 'top_rated' === $max_woo_query ) {
			$args['meta_key']   = '_wc_average_rating';
			$args['orderby']    = 'meta_value_num';
			$args['meta_query'] = WC()->query->get_meta_query();
			$args['tax_query']  = WC()->query->get_tax_query();
		}

		if ( 'best-selling' === $max_woo_query || 'featured' === $max_woo_query || 'sale' === $max_woo_query || 'downloadable' === $max_woo_query ) {
			if ( ! empty( $settings['maxProductCategories'] ) ) {
				$args['tax_query'][] = [
					'taxonomy' => 'product_cat',
					'field'    => 'term_id',
					'terms'    => $settings['maxProductCategories'],
				];
			}
		}

		if ( 'upsells' === $max_woo_query ) {
			$upsells = $product->get_upsell_ids();

			if ( ! empty( $upsells ) ) {
				$args['post__in'] = $upsells;
			} else {
				return [];
			}
		}

		if ( 'cross-sells' === $max_woo_query ) {
			$cross_sells = $product->get_cross_sell_ids();

			if ( ! empty( $cross_sells ) ) {
				$args['post__in'] = $cross_sells;
			} else {
				return [];
			}
		}

		if ( 'downloadable' === $max_woo_query ) {
			$args['meta_query'][] = [
				'key'     => '_downloadable',
				'value'   => 'yes',
				'compare' => '=',
			];
		}

		if ( 'product-gallery' === $max_woo_query ) {
			$preview_id = \Bricks\Database::$page_data['preview_or_post_id'];
			$product    = '';

			if ( is_singular( 'product' ) ) {
				$product = wc_get_product( ( ! empty( $preview_id ) ? $preview_id : $post->ID ) );
			}

			if ( empty( $product ) ) {
				return [];
			}

			$gallery_image_ids = $product->get_gallery_image_ids();

			if ( ! empty( $gallery_image_ids ) ) {
				$args['post_type']   = 'attachment';
				$args['post_status'] = 'inherit';
				$args['post__in']    = $gallery_image_ids;
			} else {
				return [];
			}
		}

		if ( 'product-gallery' !== $max_woo_query ) {
			// Exclude Out of Stock
			if ( isset( $settings['maxHideOutOfStock'] ) ) {
				$product_visibility_not_in[] = $product_visibility_term_ids['outofstock'];
			}

			if ( ! empty( $product_visibility_not_in ) ) {
				$args['tax_query'][] = [
					'taxonomy' => 'product_visibility',
					'field'    => 'term_taxonomy_id',
					'terms'    => $product_visibility_not_in,
					'operator' => 'NOT IN',
				];
			}

			if ( ! isset( $settings['maxShowHidden'] ) ) {
				$args['tax_query'][] = array(
					'taxonomy' => 'product_visibility',
					'field'    => 'term_taxonomy_id',
					'terms'    => is_search() ? $product_visibility_term_ids['exclude-from-search'] : $product_visibility_term_ids['exclude-from-catalog'],
					'operator' => 'NOT IN',
				);

				$args['post_parent'] = 0;
			}
		}

		if ( 'ordered_items' === $max_woo_query ) {
			$order       = $this->get_order();
			$order_items = ( $order ) ? $order->get_items( apply_filters( 'woocommerce_purchase_order_item_types', 'line_item' ) ) : [];

			if ( empty( $order_items ) ) {
				return $results;
			}

			return $order_items;
		}

		$woo_products_query = new \WP_Query( $args );

		return $woo_products_query->posts;
	}

	/**
	 * Sets the loop object (to WP_Post) in each query loop iteration
	 *
	 * @param array  $loop_object
	 * @param string $loop_key
	 * @param Query  $query
	 * @return array
	 */
	public function set_loop_object( $loop_object, $loop_key, $query ) {
		if ( $query->object_type !== 'maxQuery' && $query->object_type !== 'maxWooQuery' ) {
			return $loop_object;
		}

		global $post;

		$post = get_post( $loop_object );

		setup_postdata( $post );
	
		return $loop_object;	
	}

	/**
	 * Get current order.
	 *
	 * @return \WC_Order|null
	 */
	public function get_order() {
		$order_id  = 0;
		$order     = false;
		$order_key = false;

		// Order pay
		if ( ! empty( get_query_var( 'order-pay' ) ) ) {
			$order_id  = absint( get_query_var( 'order-pay' ) );
			$order_key = isset( $_GET['key'] ) ? wc_clean( wp_unslash( $_GET['key'] ) ) : '';
		}

		// Order received
		elseif ( ! empty( get_query_var( 'order-received' ) ) ) {
			$order_id = absint( get_query_var( 'order-received' ) );

			$order_id  = apply_filters( 'woocommerce_thankyou_order_id', $order_id );
			$order_key = apply_filters( 'woocommerce_thankyou_order_key', empty( $_GET['key'] ) ? '' : wc_clean( wp_unslash( $_GET['key'] ) ) );
		}

		// View order (my-account) (@since 1.9.6)
		elseif ( ! empty( get_query_var( 'view-order' ) ) ) {
			$order_id = absint( get_query_var( 'view-order' ) );
		}

		if ( $order_id > 0 ) {
			$order = wc_get_order( $order_id );

			// 'view-order' endpoint already checks the order key, so we don't need to check it again (@since 1.9.6)
			if ( ! is_wc_endpoint_url( 'view-order' ) && ( ! $order || ! hash_equals( $order->get_order_key(), $order_key ) ) ) {
				$order = false;
			}
		}

		return $order;
	}
}
