<?php
namespace MaxAddonsPro\Features;

use Bricks\Elements as Bricks_Elements;

/**
 * Handles logic for floating effects.
 *
 * @package MaxAddons
 * @since 1.3.0
 */

/**
 * Exit if accessed directly.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * MAB_Wrapper_Link.
 */
class MAB_Wrapper_Link extends MAB_Feature_Base {

	/**
	 * Initializing extensions.
	 *
	 * @since 1.3.0
	 */
	public static function init() {
		if ( ! self::is_enabled() ) {
			return;
		}

		$elements = Bricks_Elements::$elements;

		foreach( $elements as $element_key => $element ) {
			add_filter( "bricks/elements/$element_key/controls", array( __CLASS__, 'mab_wrapper_link_controls' ) );
		}

		add_filter( "bricks/element/render_attributes", array( __CLASS__, 'mab_wrapper_link_render_attr' ), 10, 3 );
	}

	/**
	 * Enqueue scrips and styles
	 *
	 * @since 1.3.5
	 */
	public static function enqueue_scripts() {
		wp_enqueue_style(
			'mab-wrapper-link',
			MAB_PRO_URL . 'assets/css/wrapper-link.css',
			array(),
			MAB_PRO_VER,
			'all'
		);
	}

	/**
	 * Is enabled.
	 *
	 * Check if maintenance mode is enabled or not.
	 *
	 * @since 1.3.0
	 *
	 * @return boolean true or false
	 */
	public static function is_enabled() {
		$enabled_features = get_option('max_bricks_features');

		if ( 'disabled' != $enabled_features ) {
			if ( is_array( $enabled_features ) && in_array( 'wrapper-link', $enabled_features ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Adds Max Addons group
	 *
	 * @since 1.3.0
	 *
	 * @return array
	 */
	public static function mab_wrapper_link_controls( $controls ) {
		$controls['maxEnableWrapperLinkSeparator'] = [
			'tab'   => 'content',
			'group' => 'mabFeatures',
			'type'  => 'separator',
		];

		$controls['maxEnableWrapperLink'] = [
			'tab'   => 'content',
			'group' => 'mabFeatures',
			'label' => esc_html__( 'Wrapper Link', 'max-addons' ),
			'type'  => 'checkbox',
		];

		$controls['maxWrapperLink'] = [
			'tab'      => 'content',
			'group'    => 'mabFeatures',
			'label'    => esc_html__( 'Link type', 'max-addons' ),
			'type'     => 'link',
			'required' => [ 'maxEnableWrapperLink', '!=', '' ],
		];
    
        return $controls;
	}

	/**
	 * Set link attributes
	 *
	 * Helper to set attributes for control wrapper link
	 *
	 * @since 1.3.0
	 *
	 * @param array $attributes
	 * @param string $link_settings Element control type 'link' settings.
	 */
	public static function set_wrapper_link_attributes( $attributes, $link_settings ) {
		$wrapper_link_options = array();

		if ( isset( $link_settings['type'] ) ) {
			// Trigger popup via link type
			$link_type = strpos( $link_settings['type'], 'lightbox' ) !== false ? 'lightbox' : $link_settings['type'];

			if ( $link_type === 'lightbox' ) {
				$wrapper_link_options['data-link'] = $link_type;
			}

			if ( $link_settings['type'] === 'internal' && isset( $link_settings['postId'] ) ) {
				$permalink = get_the_permalink( $link_settings['postId'] );

				$wrapper_link_options['url'] = $permalink;
			}

			if ( $link_settings['type'] === 'external' && isset( $link_settings['url'] ) ) {
				$wrapper_link_options['url'] = \Bricks\Integrations\Dynamic_Data\Providers::render_content( $link_settings['url'], get_the_ID(), 'link' );
			}

			if ( $link_settings['type'] === 'lightboxImage' && isset( $link_settings['lightboxImage'] ) ) {
				if ( ! empty( $link_settings['lightboxImage']['useDynamicData'] ) ) {
					$image_size = isset( $link_settings['lightboxImage']['size'] ) ? $link_settings['lightboxImage']['size'] : BRICKS_DEFAULT_IMAGE_SIZE;

					$images = Integrations\Dynamic_Data\Providers::render_tag( $link_settings['lightboxImage']['useDynamicData'], get_the_ID(), 'image', [ 'size' => $image_size ] );

					if ( ! empty( $images[0] ) ) {
						$lightbox_image_url = is_numeric( $images[0] ) ? wp_get_attachment_image_url( $images[0], $image_size ) : $images[0];
					} else {
						$lightbox_image_url = '';
					}
				} else {
					$lightbox_image_url = isset( $link_settings['lightboxImage']['url'] ) ? $link_settings['lightboxImage']['url'] : '';
				}

				$wrapper_link_options['data-pswp-src'] = $lightbox_image_url;
			}

			if ( $link_settings['type'] === 'lightboxVideo' && isset( $link_settings['lightboxVideo'] ) ) {
				$wrapper_link_options['data-pswp-video-url'] = $link_settings['lightboxVideo'];
			}

			if ( $link_settings['type'] === 'meta' && ! empty( $link_settings['useDynamicData'] ) ) {
				$wrapper_link_options['url'] = \Bricks\Integrations\Dynamic_Data\Providers::render_content( $link_settings['useDynamicData'], get_the_ID(), 'link' );
			}

			if ( $link_settings['type'] === 'media' && isset( $link_settings['mediaData']['id'] ) ) {
				$wrapper_link_options['url'] = wp_get_attachment_url( $link_settings['mediaData']['id'] );
			}

			if ( isset( $link_settings['rel'] ) ) {
				$wrapper_link_options['rel'] = $link_settings['rel'];
			}

			if ( isset( $link_settings['newTab'] ) ) {
				$wrapper_link_options['target'] = '_blank';
			}

			if ( isset( $link_settings['ariaLabel'] ) ) {
				$wrapper_link_options['aria-label'] = esc_attr( $link_settings['ariaLabel'] );
			}

			if ( isset( $link_settings['title'] ) ) {
				$wrapper_link_options['title'] = esc_attr( $link_settings['title'] );
			}
		}

		return $wrapper_link_options;
	}

	public static function mab_wrapper_link_render_attr( $attributes, $key, $element ) {
		$settings = $element->settings;

		if ( bricks_is_frontend() ) {
			if ( isset( $settings['maxEnableWrapperLink'] ) && true === $settings['maxEnableWrapperLink'] ) {

				wp_enqueue_script( 'mab-wrapper-link' );

				$attributes[ '_root' ]['class'][] = 'max-wrapper-link';

				if ( isset( $settings['maxWrapperLink'] ) ) {
					$wrapper_link_options = self::set_wrapper_link_attributes( $attributes, $settings['maxWrapperLink'] );

					if ( isset( $wrapper_link_options['data-link'] ) ) {
						$attributes[ '_root']['data-link'] = $wrapper_link_options['data-link'];

						if ( isset( $wrapper_link_options['data-pswp-video-url'] ) ) {
							$attributes[ '_root']['data-pswp-video-url'] = $wrapper_link_options['data-pswp-video-url'];
						}

						if ( isset( $wrapper_link_options['data-pswp-src'] ) ) {
							$attributes[ '_root']['data-pswp-src'] = $wrapper_link_options['data-pswp-src'];
						}
					} else {
						$attributes[ '_root' ]['data-wrapper-link'] = wp_json_encode( $wrapper_link_options );
					}
				}

			}
        }

		return $attributes;
	}

}

// Initialize the class.
MAB_Wrapper_Link::init();
