<?php
namespace MaxAddonsPro\Features;

use Bricks\Elements as Bricks_Elements;

/**
 * Handles logic for tooltip.
 *
 * @package MaxAddons
 * @since 1.2.0
 */

/**
 * Exit if accessed directly.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * MAB_Tooltip.
 */
class MAB_Tooltip extends MAB_Feature_Base {

	/**
	 * Initializing features.
	 *
	 * @since 1.2.0
	 */
	public static function init() {
		if ( ! self::is_enabled() ) {
			return;
		}

		$elements = Bricks_Elements::$elements;

		foreach( $elements as $element_key => $element ) {
			add_filter( "bricks/elements/$element_key/controls", array( __CLASS__, 'mab_tooltip_controls' ) );
		}

		add_filter( "bricks/element/set_root_attributes", array( __CLASS__, 'mab_tooltip_render_attr' ), 10, 2 );
	}

	/**
	 * Is enabled.
	 *
	 * Check if feature is enabled or not.
	 *
	 * @since 1.2.0
	 *
	 * @return boolean true or false
	 */
	public static function is_enabled() {
		$enabled_features = get_option('max_bricks_features');

		if ( 'disabled' != $enabled_features ) {
			if ( is_array( $enabled_features ) && in_array( 'tooltip', $enabled_features ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Adds tooltip controls
	 *
	 * @since 1.2.0
	 *
	 * @return array
	 */
	public static function mab_tooltip_controls( $controls ) {
		$controls['maxEnableTooltip'] = [
			'tab'   => 'style',
			'group' => 'mabFeatures',
			'label' => esc_html__( 'Tooltip', 'max-addons' ),
			'type'  => 'checkbox',
		];

		$controls['maxTooltipText'] = [
			'tab'            => 'style',
			'group'          => 'mabFeatures',
			'label'          => esc_html__( 'Tooltip Content', 'max-addons' ),
			'type'           => 'textarea',
			'hasDynamicData' => 'text',
			'required'       => [ 'maxEnableTooltip', '!=', '' ],
		];

		$controls['maxTooltipTrigger'] = [
			'tab'         => 'style',
			'group'       => 'mabFeatures',
			'label'       => esc_html__( 'Trigger', 'max-addons' ),
			'placeholder' => esc_html__( 'Hover', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'hover' => __( 'Hover', 'max-addons' ),
				'click' => __( 'Click', 'max-addons' ),
			],
			'inline'      => true,
			'clearable'   => true,
			'required'    => [ 'maxEnableTooltip', '!=', '' ],
		];

		$controls['maxTooltipPosition'] = [
			'tab'         => 'style',
			'group'       => 'mabFeatures',
			'label'       => esc_html__( 'Position', 'max-addons' ),
			'placeholder' => esc_html__( 'Top', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'top'    => __( 'Top', 'max-addons' ),
				'bottom' => __( 'Bottom', 'max-addons' ),
				'left'   => __( 'Left', 'max-addons' ),
				'right'  => __( 'Right', 'max-addons' ),
			],
			'inline'      => true,
			'clearable'   => true,
			'required'    => [ 'maxEnableTooltip', '!=', '' ],
		];

		$controls['maxTooltipArrow'] = [
			'tab'         => 'style',
			'group'       => 'mabFeatures',
			'label'       => esc_html__( 'Show Arrow', 'max-addons' ),
			'placeholder' => esc_html__( 'Yes', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'yes' => __( 'Yes', 'max-addons' ),
				'no'  => __( 'No', 'max-addons' ),
			],
			'inline'      => true,
			'clearable'   => true,
			'required'    => [ 'maxEnableTooltip', '!=', '' ],
		];

		$controls['maxTooltipBackgroundColor'] = [
			'tab'      => 'style',
			'group'    => 'mabFeatures',
			'label'    => esc_html__( 'Background Color', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property'  => 'background-color',
					'selector'  => '.tippy-box',
					'important' => true
				],
				[
					'property'  => 'color',
					'selector'  => '.tippy-arrow',
					'important' => true
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'maxEnableTooltip', '!=', '' ],
		];

		$controls['maxTooltipTypography'] = [
			'tab'      => 'style',
			'group'    => 'mabFeatures',
			'type'     => 'typography',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.tippy-box',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'maxEnableTooltip', '!=', '' ],
		];

		$controls['maxTooltipPadding'] = [
			'tab'      => 'style',
			'group'    => 'mabFeatures',
			'type'     => 'spacing',
			'label'    => esc_html__( 'Padding', 'max-addons' ),
			'css'      => [
				[
					'property' => 'padding',
					'selector' => '.tippy-content',
				],
			],
			'required' => [ 'maxEnableTooltip', '!=', '' ],
		];
    
        return $controls;
	}

	public static function mab_tooltip_render_attr( $attributes, $element ) {
		$settings = $element->settings;

		if ( bricks_is_frontend() ) {
			if ( isset( $settings['maxEnableTooltip'] ) && true === $settings['maxEnableTooltip'] ) {

				wp_enqueue_style( 'mab-tooltip' );
				wp_enqueue_script( 'mab-tooltip' );

				$arrow = '';

				if ( isset( $settings['maxTooltipArrow'] ) ) {
					if ( 'no' == $settings['maxTooltipArrow'] ) {
						$arrow = 'no';
					}
				}

				$trigger = ! empty( $settings['maxTooltipTrigger'] ) ? ( ( 'click' === $settings['maxTooltipTrigger'] ) ? 'click' : '' ) : '';
				$position = ! empty( $settings['maxTooltipPosition'] ) ? $settings['maxTooltipPosition'] : '';

				$tooltip_settings = array(
					'content'  => ! empty( $settings['maxTooltipText'] ) ? $settings['maxTooltipText'] : '',
				);

				if ( $trigger ) {
					$tooltip_settings['trigger'] = $trigger;
				}

				if ( $position ) {
					$tooltip_settings['position'] = $position;
				}

				if ( $arrow ) {
					$tooltip_settings['arrow'] = false;
				}

				$attributes['class'][] = 'max-tooltip';

				if ( $element->name === 'icon' ) {
					$attributes['data-max-tooltip'] = $settings['maxTooltipText'];

					if ( $trigger ) {
						$attributes['data-max-tooltip-trigger'] = $trigger;
					}

					if ( $position ) {
						$attributes['data-max-tooltip-position'] = $position;
					}

					if ( $arrow ) {
						$attributes['data-max-tooltip-arrow'] = false;
					}
				} else {
					$attributes['data-max-tooltip'] = wp_json_encode( $tooltip_settings );
				}
			}
        }

		return $attributes;
	}

}

// Initialize the class.
MAB_Tooltip::init();
