<?php
namespace MaxAddonsPro\Elements;

use MaxAddonsPro\Classes\Helper;
use Automattic\WooCommerce\Checkout\Helpers\ReserveStock;
use Automattic\WooCommerce\StoreApi\Utilities\DraftOrderTrait;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class Woo_Stock_Progress_Bar_Element extends \Bricks\Element {
	use DraftOrderTrait;

	public $category     = 'max-addons-woo-elements';
	public $name         = 'max-woo-stock-progress-bar';
	public $icon         = 'ti-line-double max-element';
	public $css_selector = '.bar';
	public $scripts      = [ 'mabWooStockProgressBar' ];

	public function get_label() {
		return esc_html__( 'Stock Progress Bar', 'max-addons' );
	}

	public function get_keywords() {
		return [ 'woo' ];
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		wp_enqueue_style( 'mab-woo-stock-progress-bar' );
		wp_enqueue_script( 'mab-woo-stock-progress-bar' );
	}

	public function set_control_groups() {
		$this->control_groups['settings'] = [
			'title' => esc_html__( 'Settings', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['style'] = [
			'title' => esc_html__( 'Style', 'max-addons' ),
			'tab'   => 'content',
		];
	}

	public function set_controls() {
		$this->controls['_padding']['css'][0]['selector'] = '';

		$this->controls['showSoldCount'] = [
			'tab'     => 'content',
			'label'   => esc_html__( 'Show sold count', 'max-addons' ),
			'type'    => 'checkbox',
			'default' => true,
		];

		$this->controls['soldText'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Sold count text', 'max-addons' ),
			'type'        => 'text',
			'placeholder' => esc_html__( 'Custom text', 'max-addons' ),
			'required'    => [ 'showSoldCount', '=', true ],
		];

		$this->controls['soldTextInfo'] = [
			'type'     => 'info',
			'content'  => esc_html__( 'Use {woo_max_product_sales_count} dynamic tag to show product sales count', 'max-addons' ),
			'required' => [ 'showSoldCount', '=', true ],
		];

		$this->controls['showAvailableCount'] = [
			'tab'     => 'content',
			'label'   => esc_html__( 'Show available count', 'max-addons' ),
			'type'    => 'checkbox',
			'default' => true,
		];

		$this->controls['availableText'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Available count text', 'max-addons' ),
			'type'        => 'text',
			'placeholder' => esc_html__( 'Custom text', 'max-addons' ),
			'required'    => [ 'showAvailableCount', '=', true ],
		];

		$this->controls['availableTextInfo'] = [
			'type'     => 'info',
			'content'  => esc_html__( 'Use {woo_max_product_stock_quantity} dynamic tag to show available product stock count', 'max-addons' ),
			'required' => [ 'showAvailableCount', '=', true ],
		];

		// Style
		$this->controls['height'] = [
			'tab'         => 'content',
			'group'       => 'style',
			'label'       => esc_html__( 'Height', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'css'         => [
				[
					'property' => 'height',
				]
			],
			'placeholder' => 8,
		];

		$this->controls['barColor'] = [
			'tab'   => 'content',
			'group' => 'style',
			'label' => esc_html__( 'Bar color', 'max-addons' ),
			'type'  => 'color',
			'css'   => [
				[
					'property' => 'background-color',
					'selector' => '.bar span',
				],
			],
		];

		$this->controls['barBackgroundColor'] = [
			'tab'   => 'content',
			'group' => 'style',
			'label' => esc_html__( 'Bar background color', 'max-addons' ),
			'type'  => 'color',
			'css'   => [
				[
					'property' => 'background-color',
					'selector' => '.bar',
				],
			],
		];

		$this->controls['barBorder'] = [
			'tab'   => 'content',
			'group' => 'style',
			'label' => esc_html__( 'Bar border', 'max-addons' ),
			'type'  => 'border',
			'css'   => [
				[
					'property' => 'border',
					'selector' => '.bar',
				],
			],
		];

		$this->controls['labelTypography'] = [
			'tab'      => 'content',
			'group'    => 'style',
			'label'    => esc_html__( 'Label typography', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.label',
				],
			],
			'required' => [ 'showSoldCount', '=', true ],
		];

		$this->controls['availableTypography'] = [
			'tab'      => 'content',
			'group'    => 'style',
			'label'    => esc_html__( 'Available typography', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-available',
				],
			],
			'required' => [ 'showAvailableCount', '=', true ],
		];

		// Settings
		$this->controls['isLowStock'] = [
			'tab'         => 'content',
			'group'       => 'settings',
			'label'       => esc_html__( 'Show only if low stock', 'max-addons' ),
			'description' => esc_html__( 'Show progres bar only if stock is low', 'max-addons' ),
			'type'        => 'checkbox',
		];

		$this->controls['hideIfSoldOut'] = [
			'tab'         => 'content',
			'group'       => 'settings',
			'label'       => esc_html__( 'Hide if out of stock', 'max-addons' ),
			'description' => esc_html__( 'Hide progres bar if product is sold out', 'max-addons' ),
			'type'        => 'checkbox',
		];
	}

	public function is_low_stock( $product ) {
		$low_stock_amount = $product->get_low_stock_amount();
		$total_stock      = $product->get_stock_quantity();
		$is_low_stock     = $low_stock_amount && $total_stock <= $low_stock_amount;

		return $is_low_stock;
	}

	/**
	 * Returns the remaining stock for a product if it has stock.
	 *
	 * @param \WC_Product $product Product instance.
	 * @return integer|null
	 */
	public function get_stock_quantity( $product ) {
		if ( ! $product->is_type('variable') ) {
			return $product->get_stock_quantity();
		}

		$stock_quantity = 0;

		if ( $product->is_type('variable') ) {
			foreach ( $product->get_visible_children() as $variation_id ) {
				$variation = wc_get_product( $variation_id );
				$stock_quantity += $variation->get_stock_quantity();
			}
		}

		return $stock_quantity;
	}

	public function render() {
		$settings = $this->settings;

		if ( ! Helper::is_woocommerce_active() ) {
			return;
		}

		global $product;

		$product = wc_get_product();

		if ( empty( $product ) ) {
			return;
		};

		if ( ! $product->managing_stock() ) {
			return; // Don't show the progress bar if stock isn't being managed
		}

		if ( isset( $settings['isLowStock'] ) && ! $this->is_low_stock( $product ) ) {
			return;
		}

		if ( isset( $settings['hideIfSoldOut'] ) && ! $product->is_in_stock() ) {
			return;
		}

		echo "<div {$this->render_attributes( '_root' )}>";
			$available_stock = $this->get_stock_quantity( $product );
			$total_sales     = get_post_meta( $product->get_id(), 'total_sales', true );
			$total_stock     = ( $total_sales ? $total_sales : 0 ) + $available_stock;

			$percentage = ( $total_sales > 0 ) ? round( $total_sales / $total_stock * 100 ) : 0;
			$percentage = $percentage ? $percentage : 0;

			$this->set_attribute( "bar-inner", 'data-width', "{$percentage}%" );

			echo '<div class="bar-wrapper">';
				echo '<label>';
					if ( isset( $settings['showSoldCount'] ) ) {
						$sold_text = isset( $settings['soldText'] ) ? $settings['soldText'] : esc_html__( 'Ordered', 'max-addons' ) . ': ' . intval( $total_sales );

						echo '<span class="label">';
						echo esc_html( $sold_text );
						echo '</span>';
					}

					if ( isset( $settings['showAvailableCount'] ) ) {
						$available_text = isset( $settings['availableText'] ) ? $settings['availableText'] : esc_html__( 'Available', 'max-addons' ) . ': ' . intval( $available_stock );

						echo '<span class="mab-available">';
						echo esc_html( $available_text );
						echo '</span>';
					}
				echo '</label>';

				echo '<div class="mab-stock-progress-bar bar">';
					echo '<div class="bar"><span ' . $this->render_attributes( "bar-inner" ) . '></span></div>';
				echo '</div>';
			echo '</div>';
		echo '</div>';
	}
}
