<?php
namespace MaxAddonsPro\Elements;

use MaxAddonsPro\Classes\Helper;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Woo_Product_Reviews_Element extends \Bricks\Element {
	// Element properties
	public $category     = 'max-addons-woo-elements'; // Use predefined element category 'general'
	public $name         = 'max-product-reviews'; // Make sure to prefix your elements
	public $icon         = 'ti-medall max-element'; // Themify icon font class
	public $css_selector = '.mab-review'; // Default CSS selector
	public $scripts      = []; // Script(s) run when element is rendered on frontend or updated in builder

	// Return localized element label
	public function get_label() {
		return esc_html__( 'Product Reviews', 'max-addons' );
	}

	// Set builder control groups
	public function set_control_groups() {
		$this->control_groups['layout'] = [
			'title' => esc_html__( 'Layout', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['reviewerDetails'] = [
			'title' => esc_html__( 'Review Details', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['rating'] = [
			'title'    => esc_html__( 'Rating', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'starRating', '!=', '' ],
		];

		$this->control_groups['image'] = [
			'title'    => esc_html__( 'Image', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'reviewerImage', '!=', '' ],
		];
	}

	public function set_controls() {

		$this->set_layout_controls();
		$this->set_reviewer_details_controls();
		$this->set_image_controls();
		$this->set_star_rating_controls();
		$this->set_review_text_controls();
	}

	// Set layout controls
	public function set_layout_controls() {

		$this->controls['reviewsSource'] = [
			'tab'         => 'content',
			'group'       => 'layout',
			'label'       => __( 'Reviews Source', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'all'    => __( 'All Products', 'max-addons' ),
				'select' => __( 'Selected Product', 'max-addons' ),
				'loop'   => __( 'Query Loop', 'max-addons' ),
			],
			'inline'      => false,
			'clearable'   => true,
			'default'     => '',
			'inline'      => true,
			'placeholder' => __( 'All Products', 'max-addons' ),
		];

		$this->controls['selectProduct'] = [
			'tab'         => 'content',
			'group'       => 'layout',
			'label'       => esc_html__( 'Select Product', 'max-addons' ),
			'type'        => 'select',
			'optionsAjax' => [
				'action'   => 'bricks_get_posts',
				'postType' => 'product',
			],
			'multiple'    => false,
			'searchable'  => true,
			'placeholder' => esc_html__( 'Select Product', 'max-addons' ),
			'required'    => [ 'reviewsSource', '=', 'select' ],
		];

		$this->controls['reviewsCount'] = [
			'tab'      => 'content',
			'group'    => 'layout',
			'label'    => esc_html__( 'Number of Reviews', 'max-addons' ),
			'type'     => 'number',
			'min'      => 1,
			'step'     => 1,
			'inline'   => true,
			'required' => [ 'reviewsSource', '!=', 'select' ],
		];

		$this->controls['columns'] = [
			'tab'         => 'content',
			'group'       => 'layout',
			'label'       => esc_html__( 'Columns', 'max-addons' ),
			'type'        => 'number',
			'min'         => 1,
			'css'         => [
				[
					'property' => '--columns',
					'selector' => '',
				],
			],
			'placeholder' => 3,
		];

		$this->controls['spacing'] = [
			'tab'     => 'content',
			'group'   => 'layout',
			'label'   => esc_html__( 'Spacing', 'max-addons' ),
			'type'    => 'number',
			'units'   => true,
			'css'     => [
				[
					'property' => '--gutter',
					'selector' => '',
				],
			],
			'default' => 30,
		];
	}

	// Set reviewer details controls
	public function set_reviewer_details_controls() {

		$this->controls['headingReviewerImage'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Reviewer Image', 'max-addons' ),
			'type'      => 'separator',
		];

		$this->controls['reviewerImage'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Show Image', 'max-addons' ),
			'type'      => 'checkbox',
			'default'   => true,
			'inline'    => true,
		];

		$this->controls['imageAlign'] = [
			'tab'         => 'content',
			'group'       => 'reviewerDetails',
			'label'       => __( 'Image Position', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'top'      => __( 'Above Name', 'max-addons' ),
				'left'     => __( 'Left of Name', 'max-addons' ),
				'all_left' => __( 'Left of all content', 'max-addons' ),
			],
			'inline'      => false,
			'clearable'   => true,
			'default'     => 'left',
			'placeholder' => __( 'Left of Name', 'max-addons' ),
			'required'    => [ 'reviewerImage', '!=', '' ],
		];

		$this->controls['textAlign'] = [
			'tab'     => 'content',
			'group'   => 'reviewerDetails',
			'label'   => esc_html__( 'Text align', 'max-addons' ),
			'type'    => 'text-align',
			'exclude' => 'justify',
			'inline'  => true,
			'css'     => [
				[
				'property' => 'text-align',
				'selector' => '.mab-review',
				],
			],
			'required' => [ 'imageAlign', '=', 'top' ],
		  ];

		$this->controls['headingreviewerName'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Reviewer Name', 'max-addons' ),
			'type'      => 'separator',
		];

		$this->controls['reviewerName'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Show Name', 'max-addons' ),
			'type'      => 'checkbox',
			'default'   => true,
			'inline'    => true,
		];

		$this->controls['reviewerNameLink'] = [
			'tab'      => 'content',
			'group'    => 'reviewerDetails',
			'label'    => __( 'Link Name', 'max-addons' ),
			'type'     => 'checkbox',
			'default'  => true,
			'inline'   => true,
			'required' => [ 'reviewerName', '!=', '' ],
		];

		$this->controls['headingStarRating'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Star Rating', 'max-addons' ),
			'type'      => 'separator',
		];

		$this->controls['starRating'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Show Star Rating', 'max-addons' ),
			'type'      => 'checkbox',
			'default'   => true,
			'inline'    => true,
		];

		$this->controls['headingReviewerRole'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Reviewer Role', 'max-addons' ),
			'type'      => 'separator',
		];

		$this->controls['reviewerRole'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Show Reviewer Role', 'max-addons' ),
			'type'      => 'checkbox',
			'default'   => true,
			'inline'    => true,
		];

		$this->controls['headingReviewDate'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Review Date', 'max-addons' ),
			'type'      => 'separator',
		];

		$this->controls['reviewDate'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Show Review Date', 'max-addons' ),
			'type'      => 'checkbox',
			'default'   => true,
			'inline'    => true,
		];

		$this->controls['headingReviewText'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Review Text', 'max-addons' ),
			'type'      => 'separator',
		];

		$this->controls['reviewContent'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Show Review Text', 'max-addons' ),
			'type'      => 'checkbox',
			'default'   => true,
			'inline'    => true,
		];
	}

	// Set image controls
	public function set_image_controls() {

		$this->controls['imageSize'] = [
			'tab'         => 'content',
			'group'       => 'image',
			'label'       => esc_html__( 'Image size', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'css'         => [
				[
					'property' => 'width',
					'selector' => '.mab-review-image',
				],
				[
					'property' => 'height',
					'selector' => '.mab-review-image',
				],
			],
			'placeholder' => 60,
		];

		$this->controls['imageBorder'] = [
			'tab'   => 'content',
			'group' => 'image',
			'label' => esc_html__( 'Image border', 'max-addons' ),
			'type'  => 'border',
			'css'   => [
				[
					'property' => 'border',
					'selector' => '.mab-review-image',
				],
			],
		];

		$this->controls['imageBoxShadow'] = [
			'tab'   => 'content',
			'group' => 'image',
			'label' => esc_html__( 'Image box shadow', 'max-addons' ),
			'type'  => 'box-shadow',
			'css'   => [
				[
					'property' => 'box-shadow',
					'selector' => '.mab-review-image',
				],
			],
		];
	}

	// Set rating controls
	public function set_star_rating_controls() {

		$this->controls['ratingIcon'] = [
			'tab'      => 'content',
			'group'    => 'rating',
			'label'    => esc_html__( 'Icon', 'max-addons' ),
			'type'     => 'icon',
			'default'  => [
				'library' => 'themify',
				'icon'    => 'ti-star',
			],
			'required' => [ 'starRating', '!=', '' ],
		];

		$this->controls['ratingIconMarkedColor'] = [
			'tab'      => 'content',
			'group'    => 'rating',
			'label'    => esc_html__( 'Marked Icon color', 'max-addons' ),
			'type'     => 'color',
			'inline'   => true,
			'default'  => array(
				'hex' => '#ffc107',
			),
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.mab-icon-marked',
				],
				[
					'property' => 'fill',
					'selector' => '.mab-icon-marked svg',
				],
			],
			'required' => [
				[ 'starRating', '!=', '' ],
				[ 'ratingIcon.icon', '!=', '' ],
			],
		];

		$this->controls['ratingIconUnmarkedColor'] = [
			'tab'      => 'content',
			'group'    => 'rating',
			'label'    => esc_html__( 'Unmarked Icon color', 'max-addons' ),
			'type'     => 'color',
			'inline'   => true,
			'default'  => array(
				'hex' => '#e0e0e0',
			),
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.mab-icon-unmarked',
				],
				[
					'property' => 'fill',
					'selector' => '.mab-icon-unmarked svg',
				],
			],
			'required' => [
				[ 'starRating', '!=', '' ],
				[ 'ratingIcon.icon', '!=', '' ],
			],
		];

		$this->controls['ratingIconSize'] = [
			'tab'      => 'content',
			'group'    => 'rating',
			'label'    => esc_html__( 'Icon Size', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'font-size',
					'selector' => '.mab-icon',
				],
			],
			'required' => [
				[ 'starRating', '!=', '' ],
				[ 'ratingIcon.icon', '!=', '' ],
			],
		];

		$this->controls['ratingIconGap'] = [
			'tab'      => 'content',
			'group'    => 'rating',
			'label'    => esc_html__( 'Icon Spacing', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'margin-inline-end',
					'selector' => '.mab-icon',
				],
			],
			'required' => [
				[ 'starRating', '!=', '' ],
				[ 'ratingIcon.icon', '!=', '' ],
			],
		];
	}

	// Set review text controls
	public function set_review_text_controls() {

		unset( $this->controls['_typography'] );

		$this->controls['reviewNameTypography'] = [
			'tab'      => 'style',
			'group'    => '_typography',
			'label'    => esc_html__( 'Name', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-reviewer-name, .mab-reviewer-name a',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'reviewerName', '!=', '' ],
		];

		$this->controls['reviewRoleTypography'] = [
			'tab'      => 'style',
			'group'    => '_typography',
			'label'    => esc_html__( 'Role', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-review-time',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'reviewerRole', '!=', '' ],
		];

		$this->controls['reviewTextTypography'] = [
			'tab'      => 'style',
			'group'    => '_typography',
			'label'    => esc_html__( 'Review Text', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-review-content',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'reviewContent', '!=', '' ],
		];
	}

	public function get_rating_scale(): int {
		$settings = $this->settings;

		$rating_scale = 5;

		return intval( $rating_scale );
	}

	public function get_rating_value( $rating ): float {
		$settings = $this->settings;
		$initial_value = $this->get_rating_scale();
		$rating_value = $rating;

		if ( '' === $rating_value ) {
			$rating_value = $initial_value;
		}

		$rating_value = floatval( $rating_value );

		return round( $rating_value, 2 );
	}

	public function get_icon_marked_width( $rating, $icon_index ): string {
		$rating_value = $this->get_rating_value( $rating );

		$width = '0%';

		if ( $rating_value >= $icon_index ) {
			$width = '100%';
		} elseif ( intval( ceil( $rating_value ) ) === $icon_index ) {
			$width = ( $rating_value - ( $icon_index - 1 ) ) * 100 . '%';
		}

		return $width;
	}

	public function get_icon_markup( $index, $rating ): string {
		$settings     = $this->settings;
		$rating_icon  = $settings['ratingIcon'];
		$rating_scale = $this->get_rating_scale();

		ob_start();

		for ( $i = 1; $i <= $rating_scale; $i++ ) {
			$marked_icon_key = 'icon_marked_' . intval( $index );

			$this->set_attribute( $marked_icon_key, 'class', 'mab-icon-marked' );

			$icon_marked_width = $this->get_icon_marked_width( $rating, $i );

			if ( '100%' !== $icon_marked_width ) {
				$this->set_attribute( 'icon_marked_' . intval( $index ), 'style', '--mab-rating-icon-marked-width: ' . esc_attr( $icon_marked_width ) . ';' );
			}
			?>
			<div class="mab-icon">
				<div <?php echo wp_kses_post( $this->render_attributes( $marked_icon_key ) ); ?>>
					<?php echo self::render_icon( $rating_icon, [ 'aria-hidden' => 'true' ] );; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
				</div>
				<div class="mab-icon-unmarked">
					<?php echo self::render_icon( $rating_icon, [ 'aria-hidden' => 'true' ] );; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
				</div>
			</div>
			<?php
		}

		return ob_get_clean();
	}

	public function render() {
		$settings = $this->settings;

		if ( ! Helper::is_woocommerce_active() ) {
			return;
		}

		$reviews = array();

		$args = array(
			'status' => 'approve',
			'type'   => 'review'
		);

		$reviews_source = isset( $settings['reviewsSource'] ) ? $settings['reviewsSource'] : 'all';
		$reviews_count  = isset( $settings['reviewsCount'] ) ? $settings['reviewsCount'] : '';

		if ( $reviews_source === 'all' ) {

			if ( $reviews_count ) {
				$args['number'] = $reviews_count;
			}

		} elseif ( $reviews_source === 'select' ) {

			$args['post_id'] = isset( $settings['selectProduct'] ) ? $settings['selectProduct'] : '';

		} elseif ( $reviews_source === 'loop' || \Bricks\Query::is_looping() ) {
			global $product;
	
			$product = wc_get_product();
	
			if ( empty( $product ) ) {
				return;
			}

			$product_id = $product->get_id();

			$args['post_id'] = $product_id;

			if ( $reviews_count ) {
				$args['number'] = $reviews_count;
			}
		}

		// The Query
		$comments_query = new \WP_Comment_Query;
		$comments = $comments_query->query( $args );

		if ( ! $comments ) {
			return;
		}

		foreach ( $comments as $comment ) {
			$user_id   = get_comment( $comment->comment_ID )->user_id;
            $user_info = get_userdata( $user_id );
			$rating    = intval( get_comment_meta( $comment->comment_ID, 'rating', true ) );

			$reviews[] = array(
				'image'  => isset( $settings['reviewerImage'] ) ? get_avatar( $comment, '150' ) : '',
				'name'   => isset( $settings['reviewerName'] ) ? get_comment_author( $comment ) : '',
				'role'   => isset( $settings['reviewerRole'] ) ? ( ! empty( $user_info->roles ) ? implode( ', ', $user_info->roles ) : '' ) : '',
				'rating' => isset( $settings['starRating'] ) ? $rating : '',
				'review' => isset( $settings['reviewContent'] ) ? $comment->comment_content : '',
			);
		}

		// print_r( $reviews );

		$this->set_attribute( '_root', 'class', 'bricks-layout-wrapper mab-reviews-wrapper mab-review-image-left' );
		$this->set_attribute( '_root', 'data-layout', 'grid' );
		$this->set_attribute( '_root', 'data-review-skin', 'classic' );

		echo "<ul {$this->render_attributes( '_root' )}>";
			foreach ( $reviews as $index => $review ) {
				echo '<li class="bricks-layout-item mab-review">';
					echo '<div class="mab-review-header">';
					if ( $review['image'] ) {
						echo '<div class="mab-review-image">';
						echo $review['image'];
						echo '</div>';
					}
					echo '<div class="mab-review-details">';
					if ( $review['name'] ) {
						echo '<div class="name">';
						echo $review['name'];
						echo '</div>';
					}

					if ( $review['role'] ) {
						echo '<div class="role">';
						echo $review['role'];
						echo '</div>';
					}

					if ( $review['rating'] ) {
						echo '<div class="mab-rating">';
						echo $this->get_icon_markup( $index, $review['rating'] );
						echo '</div>';
					}
					echo '</div>';
					echo '</div>';

					if ( $review['review'] ) {
						$this->set_attribute( "review-$index", 'class', [ 'mab-review-content' ] );

						echo "<div {$this->render_attributes( "review-$index" )}>{$review['review']}</div>";
					}
				echo '</li>';
			}
		echo '</ul>';
	}
}
