<?php
namespace MaxAddonsPro\Elements;

use MaxAddonsPro\Classes\Helper;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Woo_Product_Gallery_Element extends \Bricks\Element {
	// Element properties
	public $category     = 'max-addons-woo-elements'; // Use predefined element category 'general'
	public $name         = 'max-woo-product-gallery'; // Make sure to prefix your elements
	public $icon         = 'ti-gallery max-element'; // Themify icon font class
	public $css_selector = ''; // Default CSS selector
	public $scripts      = [ 'mabWooProductGallery' ]; // Script(s) run when element is rendered on frontend or updated in builder

	// Return localized element label
	public function get_label() {
		return esc_html__( 'Product Gallery', 'max-addons' );
	}

	public function get_keywords() {
		return [ 'woo' ];
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		$layout = ! empty( $this->settings['layout'] ) ? $this->settings['layout'] : 'grid';

		wp_enqueue_style( 'mab-woo-product-gallery' );

		if ( isset( $this->settings['lightbox'] ) ) {
			wp_enqueue_script( 'bricks-photoswipe' );
			wp_enqueue_script( 'bricks-photoswipe-lightbox' );
			wp_enqueue_style( 'bricks-photoswipe' );
		}

		if ( 'carousel' == $layout || 'thumb-slider' == $layout ) {
			wp_enqueue_script( 'bricks-splide' );
			wp_enqueue_style( 'bricks-splide' );
			wp_enqueue_script( 'mab-woo-product-gallery' );
		}
	}

	public function set_control_groups() {
		$this->control_groups['options'] = [
			'title' => esc_html__( 'Options', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['thumbnails'] = [
			'title' => esc_html__( 'Thumbnails', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['arrows'] = [
			'title'    => esc_html__( 'Arrows', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'layout', '=', [ 'carousel', 'thumb-slider' ] ],
		];

		$this->control_groups['pagination'] = [
			'title'    => esc_html__( 'Pagination', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'layout', '=', [ 'carousel', 'thumb-slider' ] ],
		];
	}

	public function set_controls() {
		$this->controls['layout'] = [
			'label'       => esc_html__( 'Layout', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'grid'     => esc_html__( 'Grid', 'max-addons' ),
				'carousel' => esc_html__( 'Carousel', 'max-addons' ),
				'thumb-slider' => esc_html__( 'Thumbnail Slider', 'max-addons' ),
			],
			'inline'      => true,
			'clearable'   => false,
			'default'     => 'grid',
			'placeholder' => '',
		];

		$this->controls['productImageSize'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Product', 'max-addons' ) . ': ' . esc_html__( 'Image size', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['imageSizes'],
			'placeholder' => 'woocommerce_single',
		];

		$this->controls['thumbnailImageSize'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Thumbnail', 'max-addons' ) . ': ' . esc_html__( 'Image size', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['imageSizes'],
			'placeholder' => 'woocommerce_gallery_thumbnail',
			'required'    => [ 'layout', '=', 'thumb-slider' ],
		];

		$this->controls['lightboxImageSize'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Lightbox', 'max-addons' ) . ': ' . esc_html__( 'Image size', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['imageSizes'],
			'placeholder' => 'full',
			'required'    => [ 'lightbox', '!=', '' ],
		];

		$this->controls['lightbox'] = [
			'tab'   => 'content',
			'label' => esc_html__( 'Link to lightbox', 'max-addons' ),
			'type'  => 'checkbox',
		];

		$this->set_layout_controls();
		$this->set_thumbs_controls();
		$this->set_navigation_controls();
		$this->set_pagination_controls();
	}

	// Set layout controls
	public function set_layout_controls() {
		$this->controls['columns'] = [
			'tab'      => 'content',
			'group'    => 'options',
			'label'    => esc_html__( 'Columns', 'max-addons' ),
			'type'     => 'number',
			'min'      => 1,
			'max'      => 6,
			'css'      => [
				[
					'property' => '--columns',
					'selector' => '.mab-woo-product-gallery',
				],
			],
			'required' => [ 'layout', '=', 'grid' ],
		];

		$this->controls['spacing'] = [
			'tab'         => 'content',
			'group'       => 'options',
			'label'       => esc_html__( 'Spacing', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'css'         => [
				[
					'property' => '--gutter',
					'selector' => '.mab-woo-product-gallery',
				],
			],
			'placeholder' => 0,
			'required'    => [ 'layout', '=', 'grid' ],
		];

		$this->controls['type'] = [
			'group'       => 'options',
			'label'       => esc_html__( 'Type', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'loop'  => esc_html__( 'Loop', 'max-addons' ),
				'slide' => esc_html__( 'Slide', 'max-addons' ),
				'fade'  => esc_html__( 'Fade', 'max-addons' ),
			],
			'inline'      => true,
			'placeholder' => esc_html__( 'Loop', 'max-addons' ),
			'fullAccess'  => true,
			'required'    => [ 'layout', '=', [ 'carousel', 'thumb-slider' ] ],
		];

		$this->controls['direction'] = [
			'group'       => 'options',
			'label'       => esc_html__( 'Direction', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'ltr' => esc_html__( 'Left to right', 'max-addons' ),
				'rtl' => esc_html__( 'Right to left', 'max-addons' ),
				'ttb' => esc_html__( 'Vertical', 'max-addons' ),
			],
			'inline'      => true,
			'placeholder' => esc_html__( 'Left to right', 'max-addons' ),
			'breakpoints' => true,
			'fullAccess'  => true,
			'required'    => [ 'layout', '=', [ 'carousel', 'thumb-slider' ] ],
		];

		$this->controls['height'] = [
			'group'       => 'options',
			'label'       => esc_html__( 'Height', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'placeholder' => '50vh',
			'breakpoints' => true,
			'fullAccess'  => true,
			'required'    => [
				[ 'layout', '=', [ 'carousel', 'thumb-slider' ] ],
				[ 'direction', '=', 'ttb' ]
			],
		];

		$this->controls['gap'] = [
			'group'       => 'options',
			'label'       => esc_html__( 'Spacing', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'placeholder' => 0,
			'breakpoints' => true,
			'fullAccess'  => true,
			'required'    => [
				[ 'layout', '=', 'carousel' ],
				[ 'type', '!=', 'fade' ],
			],
		];

		$this->controls['start'] = [
			'group'       => 'options',
			'label'       => esc_html__( 'Start index', 'max-addons' ),
			'type'        => 'number',
			'placeholder' => 0,
			'fullAccess'  => true,
			'required'    => [ 'layout', '=', [ 'carousel', 'thumb-slider' ] ],
		];

		$this->controls['perPage'] = [
			'group'       => 'options',
			'label'       => esc_html__( 'Items to show', 'max-addons' ),
			'type'        => 'number',
			'placeholder' => 1,
			'breakpoints' => true,
			'fullAccess'  => true,
			'required'    => [
				[ 'layout', '=', 'carousel' ],
				[ 'type', '!=', 'fade' ],
			],
		];

		$this->controls['perMove'] = [
			'group'       => 'options',
			'label'       => esc_html__( 'Items to scroll', 'max-addons' ),
			'type'        => 'number',
			'placeholder' => 1,
			'breakpoints' => true,
			'fullAccess'  => true,
			'required'    => [
				[ 'layout', '=', 'carousel' ],
				[ 'type', '!=', 'fade' ],
			],
		];

		$this->controls['speed'] = [
			'group'       => 'options',
			'label'       => esc_html__( 'Speed in ms', 'max-addons' ),
			'type'        => 'number',
			'placeholder' => 400,
			'breakpoints' => true,
			'fullAccess'  => true,
			'required'    => [ 'layout', '=', [ 'carousel', 'thumb-slider' ] ],
		];

		$this->controls['focus'] = [
			'group'       => 'options',
			'info'        => esc_html__( 'Number', 'max-addons' ),
			'label'       => esc_html__( 'Focus', 'max-addons' ),
			'type'        => 'number',
			'breakpoints' => true,
			'desc'        => sprintf(
				'<a href="https://splidejs.com/guides/options/#focus" target="_blank">%s</a>',
				esc_html__( 'Determines which slide should be active if the carousel has multiple slides in a page.', 'max-addons' ),
			),
			'fullAccess'  => true,
			'required'    => [
				[ 'layout', '=', 'carousel' ],
				[ 'type', '!=', 'fade' ],
				[ 'perPage', '>=', 2 ],
			],
		];

		// AUTOPLAY

		$this->controls['autoplaySeparator'] = [
			'group'      => 'options',
			'label'      => esc_html__( 'Autoplay', 'max-addons' ),
			'type'       => 'separator',
			'fullAccess' => true,
			'required'    => [ 'layout', '=', [ 'carousel', 'thumb-slider' ] ],
		];

		$this->controls['autoplay'] = [
			'group'      => 'options',
			'label'      => esc_html__( 'Autoplay', 'max-addons' ),
			'type'       => 'checkbox',
			'fullAccess' => true,
			'required'    => [ 'layout', '=', [ 'carousel', 'thumb-slider' ] ],
		];

		$this->controls['pauseOnHover'] = [
			'group'      => 'options',
			'label'      => esc_html__( 'Pause on hover', 'max-addons' ),
			'type'       => 'checkbox',
			'inline'     => true,
			'required'   => [
				[ 'layout', '=', [ 'carousel', 'thumb-slider' ] ],
				[ 'autoplay', '!=', '' ],
			],
			'fullAccess' => true,
		];

		$this->controls['pauseOnFocus'] = [
			'group'      => 'options',
			'label'      => esc_html__( 'Pause on focus', 'max-addons' ),
			'type'       => 'checkbox',
			'inline'     => true,
			'required'   => [
				[ 'layout', '=', [ 'carousel', 'thumb-slider' ] ],
				[ 'autoplay', '!=', '' ],
			],
			'fullAccess' => true,
		];

		$this->controls['interval'] = [
			'group'       => 'options',
			'label'       => esc_html__( 'Interval in ms', 'max-addons' ),
			'type'        => 'number',
			'placeholder' => 3000,
			'required'    => [
				[ 'layout', '=', [ 'carousel', 'thumb-slider' ] ],
				[ 'autoplay', '!=', '' ],
			],
			'fullAccess'  => true,
		];

		// REWIND: If 'type' != 'loop'

		$this->controls['rewindSeparator'] = [
			'group'      => 'options',
			'label'      => esc_html__( 'Rewind', 'max-addons' ),
			'type'       => 'separator',
			'required'   => [
				[ 'layout', '=', 'carousel' ],
				[ 'type', '!=', [ '', 'loop' ] ],
			],
			'fullAccess' => true,
		];

		$this->controls['rewind'] = [
			'group'      => 'options',
			'label'      => esc_html__( 'Rewind', 'max-addons' ),
			'type'       => 'checkbox',
			'inline'     => true,
			'required'   => [
				[ 'layout', '=', 'carousel' ],
				[ 'type', '!=', [ '', 'loop' ] ],
			],
			'fullAccess' => true,
		];

		$this->controls['rewindByDrag'] = [
			'group'      => 'options',
			'label'      => esc_html__( 'Rewind by drag', 'max-addons' ),
			'type'       => 'checkbox',
			'inline'     => true,
			'required'   => [
				[ 'layout', '=', 'carousel' ],
				[ 'type', '!=', [ '', 'loop' ] ],
				[ 'rewind', '!=', '' ]
			],
			'fullAccess' => true,
		];

		$this->controls['rewindSpeed'] = [
			'group'      => 'options',
			'label'      => esc_html__( 'Speed in ms', 'max-addons' ),
			'type'       => 'number',
			'inline'     => true,
			'required'   => [
				[ 'layout', '=', 'carousel' ],
				[ 'type', '!=', [ '', 'loop' ] ],
				[ 'rewind', '!=', '' ]
			],
			'fullAccess' => true,
		];
	}

	// Set thumbnails controls
	public function set_thumbs_controls() {
		$this->controls['thumbnailPosition'] = [
			'tab'         => 'content',
			'group'       => 'thumbnails',
			'label'       => esc_html__( 'Position', 'max-addons' ),
			'type'        => 'select',
			'inline'      => true,
			'options'     => [
				'top'    => esc_html__( 'Top', 'max-addons' ),
				'right'  => esc_html__( 'Right', 'max-addons' ),
				'bottom' => esc_html__( 'Bottom', 'max-addons' ),
				'left'   => esc_html__( 'Left', 'max-addons' ),
			],
			'placeholder' => esc_html__( 'Bottom', 'max-addons' ),
			'required'    => [ 'layout', '=', 'thumb-slider' ],
		];

		$this->controls['itemWidth'] = [
			'tab'         => 'content',
			'group'       => 'thumbnails',
			'label'       => esc_html__( 'Item width', 'max-addons' ) . ' (px)',
			'type'        => 'number',
			'units'       => true,
			'css'         => [
				[
					'selector' => '&[data-pos="right"] .mab-woo-product-gallery-slider-thumbs',
					'property' => 'width',
				],
				[
					'selector' => '&[data-pos="left"] .mab-woo-product-gallery-slider-thumbs',
					'property' => 'width',
				],
			],
			'placeholder' => '100px',
			'rerender'    => true,
			'required'    => [
				[ 'layout', '=', 'thumb-slider' ],
				[ 'thumbnailPosition', '=', [ 'left', 'right' ] ]
			],
		];

		$this->controls['thumbsGap'] = [
			'tab'         => 'content',
			'group'       => 'thumbnails',
			'label'       => esc_html__( 'Spacing', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'placeholder' => 0,
			'breakpoints' => true,
			'fullAccess'  => true,
			'required'    => [ 'layout', '=', 'thumb-slider' ],
		];

		$this->controls['thumbsPerPage'] = [
			'tab'         => 'content',
			'group'       => 'thumbnails',
			'label'       => esc_html__( 'Items to show', 'max-addons' ),
			'type'        => 'number',
			'placeholder' => 3,
			'breakpoints' => true,
			'fullAccess'  => true,
			'required'    => [ 'layout', '=', 'thumb-slider' ],
		];

		$this->controls['thumbsPerMove'] = [
			'tab'         => 'content',
			'group'       => 'thumbnails',
			'label'       => esc_html__( 'Items to scroll', 'max-addons' ),
			'type'        => 'number',
			'placeholder' => 1,
			'breakpoints' => true,
			'fullAccess'  => true,
			'required'    => [ 'layout', '=', 'thumb-slider' ],
		];

		$this->controls['thumbnailOpacity'] = [
			'tab'         => 'content',
			'group'       => 'thumbnails',
			'label'       => esc_html__( 'Opacity', 'max-addons' ),
			'type'        => 'number',
			'step'        => 0.1,
			'css'         => [
				[
					'selector' => '.woocommerce-product-gallery .mab-woo-product-gallery-slider-thumbs .splide__slide:not(.is-active)',
					'property' => 'opacity',
				],
			],
			'placeholder' => '0.3',
			'required'    => [ 'layout', '=', 'thumb-slider' ],
		];

		$this->controls['thumbnailActiveOpacity'] = [
			'tab'      => 'content',
			'group'    => 'thumbnails',
			'label'    => esc_html__( 'Opacity', 'max-addons' ) . ' (' . esc_html__( 'Active', 'max-addons' ) . ')',
			'type'     => 'number',
			'step'     => 0.1,
			'css'      => [
				[
					'selector' => '.woocommerce-product-gallery .mab-woo-product-gallery-slider-thumbs .splide__slide.is-active',
					'property' => 'opacity',
				],
			],
			'required' => [ 'layout', '=', 'thumb-slider' ],
		];

		$this->controls['thumbnailBorder'] = [
			'tab'      => 'content',
			'group'    => 'thumbnails',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'selector' => '.mab-woo-product-gallery .bricks-layout-item',
					'property' => 'border',
				],
				[
					'selector' => '.mab-woo-product-gallery-carousel .splide__slide',
					'property' => 'border',
				],
				[
					'selector' => '.mab-woo-product-gallery-slider-thumbs .splide__slide',
					'property' => 'border',
				],
			],
		];

		$this->controls['thumbnailActiveBorder'] = [
			'tab'      => 'content',
			'group'    => 'thumbnails',
			'label'    => esc_html__( 'Border', 'max-addons' ) . ' (' . esc_html__( 'Active', 'max-addons' ) . ')',
			'type'     => 'border',
			'css'      => [
				[
					'selector' => '.mab-woo-product-gallery-carousel .splide__slide.is-active',
					'property' => 'border',
				],
				[
					'selector' => '.mab-woo-product-gallery-slider-thumbs .splide__slide.is-active',
					'property' => 'border',
				],
			],
			'required' => [ 'layout', '=', [ 'carousel', 'thumb-slider' ] ],
		];
	}

	// Set navigation controls
	public function set_navigation_controls() {

		$this->controls['arrows'] = [
			'group'      => 'arrows',
			'label'      => esc_html__( 'Show', 'max-addons' ),
			'type'       => 'checkbox',
			'inline'     => true,
			'rerender'   => true,
			'fullAccess' => true,
		];

		$this->controls['arrowHeight'] = [
			'group'       => 'arrows',
			'label'       => esc_html__( 'Height', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'css'         => [
				[
					'property' => 'height',
					'selector' => '.splide__arrow',
				],
			],
			'placeholder' => 50,
			'required'    => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowWidth'] = [
			'group'       => 'arrows',
			'label'       => esc_html__( 'Width', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'css'         => [
				[
					'property' => 'width',
					'selector' => '.splide__arrow',
				],
			],
			'placeholder' => 50,
			'required'    => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowBackground'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Background', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.splide__arrow',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowBorder'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.splide__arrow',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowColor'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Color', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.splide__arrow',
				],
				[
					'property' => 'fill',
					'selector' => '.splide__arrow svg',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowSize'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Size', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'font-size',
					'selector' => '.splide__arrow',
				],
				[
					'property' => 'height',
					'selector' => '.splide__arrow svg',
				],
				[
					'property' => 'width',
					'selector' => '.splide__arrow svg',
				],
				[
					'property' => 'min-height',
					'selector' => '.splide__arrow',
				],
				[
					'property' => 'min-width',
					'selector' => '.splide__arrow',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		// text-shadow (@since 1.8.5)
		$this->controls['arrowTextShadow'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Text shadow', 'max-addons' ),
			'type'     => 'text-shadow',
			'css'      => [
				[
					'property' => 'text-shadow',
					'selector' => '.splide__arrow',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		// Custom arrows typography
		$this->controls['arrowTypography'] = [
			'deprecated' => true, // @since 1.8.5 (use 'arrowTextShadow' setting above)
			'group'      => 'arrows',
			'label'      => esc_html__( 'Typography', 'max-addons' ),
			'type'       => 'typography',
			'css'        => [
				[
					'property' => 'font',
					'selector' => '.splide__arrow',
				],
			],
			'exclude'    => [
				'font-family',
				'font-weight',
				'font-style',
				'text-align',
				'letter-spacing',
				'line-height',
				'text-decoration',
				'text-transform',
			],
			'required'   => [
				[ 'arrows', '!=', '' ],
				[ 'prevArrow.icon', '!=', '' ],
				[ 'nextArrow.icon', '!=', '' ],
			],
		];

		$this->controls['disabledArrowSep'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Disabled', 'max-addons' ),
			'type'     => 'separator',
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowDisabledBackground'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Background', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.splide__arrow:disabled',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowDisabledBorder'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.splide__arrow:disabled',
				],
			],
			'required' => [ 'arrows', '!=', '' ],

		];

		$this->controls['arrowDisabledColor'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Color', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.splide__arrow:disabled',
				],
				[
					'property' => 'fill',
					'selector' => '.splide__arrow:disabled svg',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowDisabledOpacity'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Opacity', 'max-addons' ),
			'type'     => 'number',
			'inline'   => true,
			'min'      => 0,
			'max'      => 1,
			'step'     => 0.1,
			'css'      => [
				[
					'property' => 'opacity',
					'selector' => '.splide__arrow:disabled',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		// PREV ARROW

		$this->controls['prevArrowSeparator'] = [
			'group'      => 'arrows',
			'label'      => esc_html__( 'Prev arrow', 'max-addons' ),
			'type'       => 'separator',
			'required'   => [ 'arrows', '!=', '' ],
			'fullAccess' => true,
		];

		$this->controls['prevArrow'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Prev arrow', 'max-addons' ),
			'type'     => 'icon',
			'rerender' => true,
			'css'      => [
				[
					'selector' => '.splide__arrow--prev > *',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['prevArrowTop'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Top', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'top',
					'selector' => '.splide__arrow--prev',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['prevArrowRight'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Right', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'right',
					'selector' => '.splide__arrow--prev',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['prevArrowBottom'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Bottom', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'bottom',
					'selector' => '.splide__arrow--prev',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['prevArrowLeft'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Left', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'left',
					'selector' => '.splide__arrow--prev',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['prevArrowTransform'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Transform', 'max-addons' ),
			'type'     => 'transform',
			'css'      => [
				[
					'property' => 'transform',
					'selector' => '.splide__arrow--prev',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['prevArrowTransformInfo'] = [
			'group'    => 'arrows',
			'content'  => esc_html__( 'Please make sure to set the "Scale X" value inside the transform setting above to "-1".', 'max-addons' ),
			'type'     => 'info',
			'required' => [
				[ 'arrows', '!=', '' ],
				[ 'prevArrow', '=', '' ],
				[ 'prevArrowTransform', '!=', '' ],
			],
		];

		// NEXT ARROW

		$this->controls['nextArrowSeparator'] = [
			'group'      => 'arrows',
			'label'      => esc_html__( 'Next arrow', 'max-addons' ),
			'type'       => 'separator',
			'required'   => [ 'arrows', '!=', '' ],
			'fullAccess' => true,
		];

		$this->controls['nextArrow'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Next arrow', 'max-addons' ),
			'type'     => 'icon',
			'rerender' => true,
			'css'      => [
				[
					'selector' => '.splide__arrow--next > *',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['nextArrowTop'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Top', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'top',
					'selector' => '.splide__arrow--next',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['nextArrowRight'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Right', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'right',
					'selector' => '.splide__arrow--next',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['nextArrowBottom'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Bottom', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'bottom',
					'selector' => '.splide__arrow--next',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['nextArrowLeft'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Left', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'left',
					'selector' => '.splide__arrow--next',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['nextArrowTransform'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Transform', 'max-addons' ),
			'type'     => 'transform',
			'css'      => [
				[
					'property' => 'transform',
					'selector' => '.splide__arrow--next',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];
		
	}

	// Set pagination controls
	public function set_pagination_controls() {

		$this->controls['pagination'] = [
			'group'      => 'pagination',
			'label'      => esc_html__( 'Show', 'max-addons' ),
			'type'       => 'checkbox',
			'inline'     => true,
			'rerender'   => true,
			'default'    => true,
			'fullAccess' => true,
		];

		$this->controls['paginationSpacing'] = [
			'group'       => 'pagination',
			'label'       => esc_html__( 'Margin', 'max-addons' ),
			'type'        => 'spacing',
			'css'         => [
				[
					'property' => 'margin',
					'selector' => '.splide__pagination .splide__pagination__page',
				],
			],
			'placeholder' => [
				'top'    => '5px',
				'right'  => '5px',
				'bottom' => '5px',
				'left'   => '5px',
			],
			'required'    => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationHeight'] = [
			'group'       => 'pagination',
			'label'       => esc_html__( 'Height', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'units'       => [
				'px' => [
					'min' => 1,
					'max' => 100,
				],
			],
			'css'         => [
				[
					'property' => 'height',
					'selector' => '.splide__pagination .splide__pagination__page',
				],
			],
			'placeholder' => '10px',
			'required'    => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationWidth'] = [
			'group'       => 'pagination',
			'label'       => esc_html__( 'Width', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'units'       => [
				'px' => [
					'min' => 1,
					'max' => 100,
				],
			],
			'css'         => [
				[
					'property' => 'width',
					'selector' => '.splide__pagination .splide__pagination__page',
				],
			],
			'placeholder' => '10px',
			'required'    => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationColor'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Color', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.splide__pagination .splide__pagination__page',
				],
				[
					'property' => 'background-color',
					'selector' => '.splide__pagination .splide__pagination__page',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationBorder'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.splide__pagination .splide__pagination__page',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];

		// ACTIVE

		$this->controls['paginationActiveSeparator'] = [
			'group'      => 'pagination',
			'label'      => esc_html__( 'Active', 'max-addons' ),
			'type'       => 'separator',
			'required'   => [ 'pagination', '!=', '' ],
			'fullAccess' => true,
		];

		$this->controls['paginationHeightActive'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Height', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'height',
					'selector' => '.splide__pagination .splide__pagination__page.is-active',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationWidthActive'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Width', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'width',
					'selector' => '.splide__pagination .splide__pagination__page.is-active',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationColorActive'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Color', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.splide__pagination .splide__pagination__page.is-active',
				],
				[
					'property' => 'background-color',
					'selector' => '.splide__pagination .splide__pagination__page.is-active',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationBorderActive'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.splide__pagination .splide__pagination__page.is-active',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];

		// POSITION

		$this->controls['paginationPositionSeparator'] = [
			'group'      => 'pagination',
			'label'      => esc_html__( 'Position', 'max-addons' ),
			'type'       => 'separator',
			'required'   => [ 'pagination', '!=', '' ],
			'fullAccess' => true,
		];

		$this->controls['paginationTop'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Top', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'top',
					'selector' => '.splide__pagination',
				],
				[
					'property' => 'bottom',
					'value'    => 'auto',
					'selector' => '.splide__pagination',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationRight'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Right', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'right',
					'selector' => '.splide__pagination',
				],
				[
					'property' => 'left',
					'value'    => 'auto',
					'selector' => '.splide__pagination',
				],
				[
					'property' => 'transform',
					'selector' => '.splide__pagination',
					'value'    => 'translateX(0)',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationBottom'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Bottom', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'bottom',
					'selector' => '.splide__pagination',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationLeft'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Left', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'left',
					'selector' => '.splide__pagination',
				],
				[
					'property' => 'right',
					'value'    => 'auto',
					'selector' => '.splide__pagination',
				],
				[
					'property' => 'transform',
					'selector' => '.splide__pagination',
					'value'    => 'translateX(0)',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];
	}

	public function carousel_settings() {
		$settings = $this->settings;

		$type = ! empty( $settings['type'] ) ? $settings['type'] : 'loop';

		// Spacing requires a unit
		$gap = ! empty( $settings['gap'] ) ? $settings['gap'] : 0;

		// Add default unit
		if ( is_numeric( $gap ) ) {
			$gap = "{$gap}px";
		}

		$direction = ! empty( $settings['direction'] ) ? $settings['direction'] : ( is_rtl() ? 'rtl' : 'ltr' );

		// Arrows
		$arrows = isset( $settings['arrows'] );

		if ( $arrows ) {
			// Custom arrows set OR use default splide SVG arrows if no custom arrows set
			$arrows = ( ! empty( $settings['prevArrow'] ) && ! empty( $settings['nextArrow'] ) ) || ( empty( $settings['prevArrow'] ) && empty( $settings['nextArrow'] ) );
		}

		$splide_options = [
			'type'         => $type,
			'direction'    => ! empty( $settings['direction'] ) ? $settings['direction'] : ( is_rtl() ? 'rtl' : 'ltr' ),
			'keyboard'     => ! empty( $settings['keyboard'] ) ? $settings['keyboard'] : 'global', // 'focused', false
			'height'       => ( 'ttb' === $direction ) ? ( ! empty( $settings['height'] ) ? $settings['height'] : '50vh' ) : '',
			'gap'          => $gap,
			'start'        => ! empty( $settings['start'] ) ? $settings['start'] : 0,
			'perPage'      => ! empty( $settings['perPage'] ) && $type !== 'fade' ? $settings['perPage'] : 1,
			'perMove'      => ! empty( $settings['perMove'] ) && $type !== 'fade' ? $settings['perMove'] : 1,
			'speed'        => ! empty( $settings['speed'] ) ? $settings['speed'] : 400,
			'interval'     => ! empty( $settings['interval'] ) ? $settings['interval'] : 3000,
			'autoHeight'   => isset( $settings['autoHeight'] ),
			'autoplay'     => isset( $settings['autoplay'] ),
			'pauseOnHover' => isset( $settings['pauseOnHover'] ),
			'pauseOnFocus' => isset( $settings['pauseOnFocus'] ),
			'arrows'       => $arrows,
			'pagination'   => isset( $settings['pagination'] ),
		];

		if ( isset( $settings['focus'] ) ) {
			$splide_options['focus'] = $settings['focus'];
		}

		// Auto height enabled: Set height to "auto"
		if ( isset( $settings['autoHeight'] ) ) {
			$splide_options['height'] = 'auto';
		}

		if ( isset( $settings['rewind'] ) && $type !== 'loop' ) {
			$splide_options['rewind'] = $settings['rewind'];

			if ( ! empty( $settings['rewindSpeed'] ) ) {
				$splide_options['rewindSpeed'] = $settings['rewindSpeed'];
			}

			if ( isset( $settings['rewindByDrag'] ) ) {
				$splide_options['rewindByDrag'] = $settings['rewindByDrag'];
			}
		}

		// STEP: Add settings per breakpoints to splide options
		$breakpoints = [];

		foreach ( \Bricks\Breakpoints::$breakpoints as $breakpoint ) {
			foreach ( array_keys( $splide_options ) as $option ) {
				$setting_key      = "$option:{$breakpoint['key']}";
				$breakpoint_width = ! empty( $breakpoint['width'] ) ? $breakpoint['width'] : false;
				$setting_value    = isset( $settings[ $setting_key ] ) ? $settings[ $setting_key ] : false;

				// Spacing requires a unit
				if ( $option === 'gap' ) {
					// Add default unit
					if ( is_numeric( $setting_value ) ) {
						$setting_value = "{$setting_value}px";
					}
				}

				if ( $breakpoint_width && $setting_value !== false ) {
					$breakpoints[ $breakpoint_width ][ $option ] = $setting_value;
				}
			}
		}

		if ( count( $breakpoints ) ) {
			$splide_options['breakpoints'] = $breakpoints;
		}

		if ( is_array( $splide_options ) ) {
			$splide_options = wp_json_encode( $splide_options );
		}

		// Remove line breaks
		$splide_options = str_replace( [ "\r", "\n", ' ' ], '', $splide_options );

		$this->set_attribute( '_root', 'data-splide', trim( $splide_options ) );
	}

	public function slider_settings() {
		$settings = $this->settings;

		$type = ! empty( $settings['type'] ) ? $settings['type'] : 'loop';
		$thumbnail_position = ! empty( $settings['thumbnailPosition'] ) ? $settings['thumbnailPosition'] : '';

		// Arrows
		$arrows = isset( $settings['arrows'] );

		if ( $arrows ) {
			// Custom arrows set OR use default splide SVG arrows if no custom arrows set
			$arrows = ( ! empty( $settings['prevArrow'] ) && ! empty( $settings['nextArrow'] ) ) || ( empty( $settings['prevArrow'] ) && empty( $settings['nextArrow'] ) );
		}

		$splide_options = [
			'type'         => $type,
			'direction'    => ! empty( $settings['direction'] ) ? $settings['direction'] : ( is_rtl() ? 'rtl' : 'ltr' ),
			'height'       => ( 'left' === $thumbnail_position || 'right' === $thumbnail_position ) ? ( ! empty( $settings['height'] ) ? $settings['height'] : '300px' ) : '',
			'start'        => ! empty( $settings['start'] ) ? $settings['start'] : 0,
			'speed'        => ! empty( $settings['speed'] ) ? $settings['speed'] : 400,
			'interval'     => ! empty( $settings['interval'] ) ? $settings['interval'] : 3000,
			'autoplay'     => isset( $settings['autoplay'] ),
			'pauseOnHover' => isset( $settings['pauseOnHover'] ),
			'pauseOnFocus' => isset( $settings['pauseOnFocus'] ),
			'rewind'       => true,
			'arrows'       => $arrows,
			'pagination'   => isset( $settings['pagination'] ),
		];

		if ( is_array( $splide_options ) ) {
			$splide_options = wp_json_encode( $splide_options );
		}

		// Remove line breaks
		$splide_options = str_replace( [ "\r", "\n", ' ' ], '', $splide_options );

		$this->set_attribute( '_root', 'data-splide', trim( $splide_options ) );
	}

	public function slider_thumbs_settings() {
		$settings = $this->settings;

		$type = 'loop';

		// Spacing requires a unit
		$gap = ! empty( $settings['thumbsGap'] ) ? $settings['thumbsGap'] : 0;

		// Add default unit
		if ( is_numeric( $gap ) ) {
			$gap = "{$gap}px";
		}

		$thumbnail_position = ! empty( $settings['thumbnailPosition'] ) ? $settings['thumbnailPosition'] : '';
		$direction = ( 'left' === $thumbnail_position || 'right' === $thumbnail_position ) ? 'ttb' : 'ltr';

		$splide_options = [
			'rewind'       => true,
			'direction'    => $direction,
			'height'       => ( 'left' === $thumbnail_position || 'right' === $thumbnail_position ) ? ( ! empty( $settings['height'] ) ? $settings['height'] : '300px' ) : '',
			'gap'          => $gap,
			'start'        => ! empty( $settings['start'] ) ? $settings['start'] : 0,
			'perPage'      => ! empty( $settings['thumbsPerPage'] ) && $type !== 'fade' ? $settings['thumbsPerPage'] : 3,
			'perMove'      => ! empty( $settings['thumbsPerMove'] ) && $type !== 'fade' ? $settings['thumbsPerMove'] : 1,
			'speed'        => ! empty( $settings['speed'] ) ? $settings['speed'] : 400,
			/* 'interval'     => ! empty( $settings['interval'] ) ? $settings['interval'] : 3000,
			'autoplay'     => isset( $settings['autoplay'] ),
			'pauseOnHover' => isset( $settings['pauseOnHover'] ),
			'pauseOnFocus' => isset( $settings['pauseOnFocus'] ), */
			'arrows'       => false,
			'pagination'   => false,
			'isNavigation' => true,
		];

		if ( is_array( $splide_options ) ) {
			$splide_options = wp_json_encode( $splide_options );
		}

		// Remove line breaks
		$splide_options = str_replace( [ "\r", "\n", ' ' ], '', $splide_options );

		$this->set_attribute( 'slider_nav', 'data-splide', trim( $splide_options ) );
	}

	public function render_gallery_image( $index, $image_id, $image_size ) {
		$settings = $this->settings;

		$lightbox      = isset( $settings['lightbox'] );
		$gallery_image = wp_get_attachment_image( $image_id, $image_size );

		if ( $lightbox ) {
			$lightbox_image_size = ! empty( $settings['lightboxImageSize'] ) ? $settings['lightboxImageSize'] : 'full';
			$lightbox_image_src  = wp_get_attachment_image_src( $image_id, $lightbox_image_size );
			
			$this->set_attribute( "a-$index", 'data-pswp-src', $lightbox_image_src[0] );
			$this->set_attribute( "a-$index", 'data-pswp-width', $lightbox_image_src[1] );
			$this->set_attribute( "a-$index", 'data-pswp-height', $lightbox_image_src[2] );
			$this->set_attribute( "a-$index", 'data-pswp-id', $this->id );
			
			$gallery_image = "<a {$this->render_attributes( "a-$index" )}>" . $gallery_image . '</a>';
		}

		return $gallery_image;
	}

	/**
	 * Render arrows (use custom HTML solution as splideJS only accepts SVG path via 'arrowPath')
	 */
	public function render_arrows() {
		$prev_arrow = ! empty( $this->settings['prevArrow'] ) ? self::render_icon( $this->settings['prevArrow'] ) : false;
		$next_arrow = ! empty( $this->settings['nextArrow'] ) ? self::render_icon( $this->settings['nextArrow'] ) : false;

		if ( ! $prev_arrow && ! $next_arrow ) {
			return;
		}

		$output = '<div class="splide__arrows custom">';

		if ( $prev_arrow ) {
			$output .= '<button class="splide__arrow splide__arrow--prev" type="button">' . $prev_arrow . '</button>';
		}

		if ( $next_arrow ) {
			$output .= '<button class="splide__arrow splide__arrow--next" type="button">' . $next_arrow . '</button>';
		}

		$output .= '</div>';

		return $output;
	}

	public function render() {
		$settings = $this->settings;

		if ( ! Helper::is_woocommerce_active() ) {
			return;
		}

		global $product;

		$product = wc_get_product();

		if ( empty( $product ) ) {
			return;
		};

		$product_img_id    = $product->get_image_id();
		$gallery_image_ids = wc_get_product()->get_gallery_image_ids();

		$product_image_size   = ! empty( $settings['productImageSize'] ) ? $settings['productImageSize'] : 'woocommerce_single';
		$thumbnail_image_size = ! empty( $settings['thumbnailImageSize'] ) ? $settings['thumbnailImageSize'] : 'woocommerce_thumbnail';

		if ( $product_img_id ) {
			$product_img = wp_get_attachment_image( $product_img_id, $product_image_size );
		}

		$classes = array(
			'mab-woo-product-gallery',
			'bricks-layout-wrapper',
		);

		$this->set_attribute( 'gallery', 'class', $classes );
		$this->set_attribute( 'gallery', 'data-layout', 'grid' );

		array_unshift( $gallery_image_ids , $product_img_id );
		// $gallery_image_ids = array_merge( array($product_img_id), $gallery_image_ids );

		$lightbox = isset( $settings['lightbox'] );
		$layout = isset( $settings['layout'] ) ? $settings['layout'] : 'grid';

		if ( $lightbox ) {
			$this->set_attribute( '_root', 'class', 'bricks-lightbox brxe-carousel' );
		}

		if ( 'carousel' == $layout ) {
			$this->carousel_settings();
		} else if( 'thumb-slider' == $layout ) {
			$this->slider_settings();

			if ( ! empty( $settings['thumbnailPosition'] ) ) {
				$this->set_attribute( '_root', 'data-pos', esc_attr( $settings['thumbnailPosition'] ) );
			}
		}
		?>
		<div <?php echo wp_kses_post( $this->render_attributes( '_root' ) ); ?>>
			<div class="woocommerce-product-gallery" >
				<?php if ( 'grid' == $layout ) { ?>
					<ul <?php echo wp_kses_post( $this->render_attributes( 'gallery' ) ); ?>>
						<?php
							if ( $gallery_image_ids ) {
								if ( is_array( $gallery_image_ids ) ) {
									foreach ( $gallery_image_ids as $index => $image_id ) {
										$this->set_attribute( "item-{$index}", 'class', 'bricks-layout-item' );
										?>
										<li <?php echo wp_kses_post( $this->render_attributes( "item-{$index}" ) ); ?>>
											<?php echo $this->render_gallery_image( $index, $image_id, $product_image_size ); ?>
										</li>
										<?php
									}
								}
							}
						?>
					</ul>
				<?php } else if ( 'carousel' == $layout ) { ?>
					<div class="mab-woo-product-gallery-carousel splide" >
						<div class="splide__track">
							<div class="splide__list">
								<?php
									if ( $gallery_image_ids ) {
										if ( is_array( $gallery_image_ids ) ) {
											foreach ( $gallery_image_ids as $index => $image_id ) {
												?>
												<div class="splide__slide">
													<?php echo $this->render_gallery_image( $index, $image_id, $product_image_size ); ?>
												</div>
												<?php
											}
										}
									}
								?>
							</div>
						</div>
						<?php $this->render_arrows(); ?>
					</div>
				<?php } else if ( 'thumb-slider' == $layout ) { ?>
					<div class="mab-woo-product-gallery-slider splide" >
						<div class="splide__track">
							<div class="splide__list">
								<?php
									if ( $gallery_image_ids ) {
										if ( is_array( $gallery_image_ids ) ) {
											foreach ( $gallery_image_ids as $index => $image_id ) {
												?>
												<div class="splide__slide">
													<?php echo $this->render_gallery_image( $index, $image_id, $product_image_size ); ?>
												</div>
												<?php
											}
										}
									}
								?>
							</div>
						</div>
					</div>

					<?php
						$this->set_attribute( 'slider_nav', 'class', [ 'mab-woo-product-gallery-slider-thumbs', 'splide' ] );

						$this->slider_thumbs_settings();
					?>
					<div <?php echo wp_kses_post( $this->render_attributes( 'slider_nav' ) ); ?>>
						<div class="splide__track">
							<div class="splide__list">
								<?php
									if ( $gallery_image_ids ) {
										if ( is_array( $gallery_image_ids ) ) {
											foreach ( $gallery_image_ids as $index => $image_id ) {
												?>
												<div class="splide__slide">
													<?php echo wp_get_attachment_image( $image_id, $thumbnail_image_size ); ?>
												</div>
												<?php
											}
										}
									}
								?>
							</div>
						</div>
					</div>
				<?php } ?>
			</div>
		</div>
		<?php
	}
}
