<?php
namespace MaxAddonsPro\Elements;

use MaxAddonsPro\Classes\Helper;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Woo_Product_Badge_Element extends \Bricks\Element {
	// Element properties
	public $category     = 'max-addons-woo-elements'; // Use predefined element category 'general'
	public $name         = 'max-product-badge'; // Make sure to prefix your elements
	public $icon         = 'ti-medall max-element'; // Themify icon font class
	public $css_selector = ''; // Default CSS selector
	public $scripts      = []; // Script(s) run when element is rendered on frontend or updated in builder

	// Return localized element label
	public function get_label() {
		return esc_html__( 'Product Badges', 'max-addons' );
	}

	public function get_keywords() {
		return [ 'woo', 'sale', 'featured', 'best selling', 'top rated' ];
	}

	public function set_controls() {

		$this->controls['badgeType'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Badge Type', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'sale'         => esc_html__( 'Sale', 'max-addons' ),
				'featured'     => esc_html__( 'Featured', 'max-addons' ),
				'top-rated'    => esc_html__( 'Top Rated', 'max-addons' ),
				'best-selling' => esc_html__( 'Best Selling', 'max-addons' ),
			],
			'inline'      => true,
			'clearable'   => true,
			'default'     => 'sale',
			'placeholder' => esc_html__( 'Sale', 'max-addons' ),
		];

		$this->controls['saleBadgeCustomText'] = [
			'tab'            => 'content',
			'label'          => esc_html__( 'Custom Text', 'max-addons' ),
			'type'           => 'text',
			'default'        => '',
			'hasDynamicData' => 'text',
			'inline'         => true,
			'required'       => [ 'badgeType', '=', 'sale' ],
		];

		$this->controls['featuredBadgeCustomText'] = [
			'tab'            => 'content',
			'label'          => esc_html__( 'Custom Text', 'max-addons' ),
			'type'           => 'text',
			'default'        => '',
			'hasDynamicData' => 'text',
			'inline'         => true,
			'required'       => [ 'badgeType', '=', 'featured' ],
		];

		$this->controls['topRatedBadgeCustomText'] = [
			'tab'            => 'content',
			'label'          => esc_html__( 'Custom Text', 'max-addons' ),
			'type'           => 'text',
			'default'        => '',
			'hasDynamicData' => 'text',
			'inline'         => true,
			'required'       => [ 'badgeType', '=', 'top-rated' ],
		];

		$this->controls['ratingsCount'] = [
			'tab'      => 'content',
			'label'    => esc_html__( 'Rating', 'max-addons' ),
			'description' => __( 'Show badge according to count of total rating greater than rating.', 'max-addons' ),
			'type'     => 'number',
			'min'      => 1,
			'max'      => 10,
			'inline'   => true,
			'small'    => true,
			'required' => [ 'badgeType', '=', 'top-rated' ],
		];

		$this->controls['bestSellingBadgeCustomText'] = [
			'tab'            => 'content',
			'label'          => esc_html__( 'Custom Text', 'max-addons' ),
			'type'           => 'text',
			'default'        => '',
			'hasDynamicData' => 'text',
			'inline'         => true,
			'required'       => [ 'badgeType', '=', 'best-selling' ],
		];

		$this->controls['salesCount'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Number of Sales', 'max-addons' ),
			'description' => __( 'Minimum number of sales.', 'max-addons' ),
			'type'        => 'number',
			'min'         => 1,
			'inline'      => true,
			'small'       => true,
			'required'    => [ 'badgeType', '=', 'best-selling' ],
		];

		$this->controls['badgeStyle'] = [
			'tab'         => 'content',
			'label'       => esc_html__( 'Style', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['styles'],
			'inline'      => true,
			'reset'       => true,
			'default'     => 'primary',
			'placeholder' => esc_html__( 'None', 'max-addons' ),
		];

		$this->controls['badgeCircle'] = [
			'tab'   => 'content',
			'label' => esc_html__( 'Circle', 'max-addons' ),
			'type'  => 'checkbox',
			'reset' => true,
		];

		$this->controls['badgeOutline'] = [
			'tab'   => 'content',
			'label' => esc_html__( 'Outline', 'max-addons' ),
			'type'  => 'checkbox',
			'reset' => true,
		];
	}

	/**
	 * Get Top Rated Product for Badge.
	 *
	 * @since 1.1.0
	 * @access public
	 */
	protected function is_top_rated_product( $product_id ) {
		$settings = $this->settings;
		$rating = isset( $settings['ratingsCount'] ) ? $settings['ratingsCount'] : '';

		if ( empty( $rating ) ) {
			return false;
		}

		$total_rating = get_post_meta( $product_id, '_wc_average_rating', true );

		if ( ! $total_rating || empty( $total_rating ) ) {
			return false;
		}

		return $total_rating >= $rating;
	}

	/**
	 * Get Best Selling Product for Badge.
	 *
	 * @since 1.1.0
	 * @access public
	 */
	protected function is_best_selling_product( $product_id ) {
		$settings = $this->settings;
		$sales_count = isset( $settings['salesCount'] ) ? $settings['salesCount'] : '';

		if ( empty( $sales_count ) ) {
			return false;
		}

		$total_sales = get_post_meta( $product_id, 'total_sales', true );

		if ( ! $total_sales || empty( $total_sales ) ) {
			return false;
		}

		return $total_sales >= $sales_count;
	}

	/**
	 * Render sale badge output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @access protected
	 */
	protected function render_sale_badge( $product ) {
		$settings = $this->settings;

		if ( 'variable' === $product->get_type() ) {
			$regular_price = $product->get_variation_regular_price();
			$sale_price = $product->get_variation_sale_price();
		} else {
			$regular_price = $product->get_regular_price();
			$sale_price = $product->get_sale_price();
		}

		$badge_text = isset( $settings['saleBadgeCustomText'] ) ? $settings['saleBadgeCustomText'] : __( 'Sale', 'max-addons' );

		if ( $sale_price ) {
			$percent_sale = round( ( ( ( $regular_price - $sale_price ) / $regular_price ) * 100 ), 0 );
			$badge_text   = $badge_text ? $badge_text : '-[value]%';
			$badge_text   = str_replace( '[value]', $percent_sale, $badge_text );
		}

		if ( 'grouped' === $product->get_type() ) {
			if ( $badge_text ) {
				$badge_text = isset( $settings['saleBadgeCustomText'] ) ? $settings['saleBadgeCustomText'] : '';
			}
		}

		if ( $product->is_on_sale() ) {

			return wp_kses_post( $badge_text );

		}
	}

	/**
	 * Render featured badge output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @access protected
	 */
	protected function render_featured_badge( $product ) {
		$settings = $this->settings;

		$badge_text = isset( $settings['featuredBadgeCustomText'] ) ? $settings['featuredBadgeCustomText'] : __( 'Featured', 'max-addons' );

		if ( $product->is_featured() ) {

			return wp_kses_post( $badge_text );

		}
	}

	/**
	 * Render top rated badge output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @access protected
	 */
	protected function render_top_rated_badge( $product ) {
		$settings = $this->settings;

		$badge_text = isset( $settings['topRatedBadgeCustomText'] ) ? $settings['topRatedBadgeCustomText'] : __( 'Top Rated', 'max-addons' );

		if ( $this->is_top_rated_product( $product->get_id() ) ) {

			return wp_kses_post( $badge_text );

		}
	}

	/**
	 * Render best selling badge output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @access protected
	 */
	protected function render_best_selling_badge( $product ) {
		$settings = $this->settings;

		$badge_text = isset( $settings['bestSellingBadgeCustomText'] ) ? $settings['bestSellingBadgeCustomText'] : __( 'Best Selling', 'max-addons' );

		if ( $this->is_best_selling_product( $product->get_id() ) ) {

			return wp_kses_post( $badge_text );

		}
	}

	public function render() {
		$settings = $this->settings;

		if ( ! Helper::is_woocommerce_active() ) {
			return;
		}

		global $product;

		$product = wc_get_product();

		if ( empty( $product ) ) {
			return;
		}

		$badge_type = isset( $settings['badgeType'] ) ? $settings['badgeType'] : 'sale';

		if ( 'featured' === $badge_type ) {
			$badge_text = $this->render_featured_badge( $product );
		} elseif ( 'top-rated' === $badge_type ) {
			$badge_text = $this->render_top_rated_badge( $product );
		} elseif ( 'best-selling' === $badge_type ) {
			$badge_text = $this->render_best_selling_badge( $product );
		} else {
			$badge_text = $this->render_sale_badge( $product );
		}

		if ( '' == $badge_text ) {
			return;
		}

		$button_classes[] = 'bricks-button mab-product-badge';
		$button_classes[] = 'mab-' . $badge_type . '-badge';

		if ( isset( $settings['size'] ) ) {
			$button_classes[] = $settings['size'];
		}

		if ( isset( $settings['badgeStyle'] ) ) {
			// Outline
			if ( isset( $settings['badgeOutline'] ) ) {
				$button_classes[] = 'outline';
				$button_classes[] = 'bricks-color-' . $settings['badgeStyle'];
			} else {
				// Fill (default)
				$button_classes[] = 'bricks-background-' . $settings['badgeStyle'];
			}
		}

		// Button circle
		if ( isset( $settings['badgeCircle'] ) ) {
			$button_classes[] = 'circle';
		}

		$this->set_attribute( '_root', 'class', $button_classes );
		?>
		<div <?php echo wp_kses_post( $this->render_attributes( '_root' ) ); ?>>
			<?php echo wp_kses_post( $badge_text ); ?>
		</div>
		<?php
	}
}
