<?php
namespace MaxAddonsPro\Elements;

use MaxAddonsPro\Classes\Helper;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Woo_Add_To_Cart_Element extends \Bricks\Element {
	// Element properties
	public $category     = 'max-addons-woo-elements'; // Use predefined element category 'general'
	public $name         = 'max-woo-add-to-cart'; // Make sure to prefix your elements
	public $icon         = 'ti-shopping-cart max-element'; // Themify icon font class
	public $css_selector = ''; // Default CSS selector
	public $scripts      = []; // Script(s) run when element is rendered on frontend or updated in builder

	// Return localized element label
	public function get_label() {
		return esc_html__( 'Add to Cart', 'max-addons' );
	}

	public function get_keywords() {
		return [ 'woo' ];
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		wp_enqueue_script( 'mab-add-to-cart' );
		wp_enqueue_style( 'mab-woocommerce' );
	}

	// Set builder control groups
	public function set_control_groups() {
		$this->control_groups['product'] = [
			'title' => esc_html__( 'Product', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['variations'] = [
			'title'    => esc_html__( 'Variations', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'dynamicProduct', '!=', '' ],
		];

		$this->control_groups['quantity'] = [
			'title' => esc_html__( 'Quantity', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['button'] = [
			'title' => esc_html__( 'Button', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['viewCart'] = [
			'title' => esc_html__( 'View Cart', 'max-addons' ),
			'tab'   => 'content',
			'required' => [ 'dynamicProduct', '=', '' ],
		];
	}

	public function set_controls() {
		$this->set_product_controls();
		$this->set_variations_controls();
		$this->set_quantity_controls();
		$this->set_button_controls();
		$this->set_view_cart_controls();
	}

	// Set product controls
	public function set_product_controls() {
		$this->controls['dynamicProduct'] = [
			'tab'         => 'content',
			'group'       => 'product',
			'label'       => esc_html__( 'Dynamic Product', 'max-addons' ),
			'description' => __( 'Enable this option to use Add to Cart button in loop or on single product page.', 'max-addons' ),
			'type'        => 'checkbox',
			'reset'       => true,
		];

		$this->controls['productId'] = [
			'tab'         => 'content',
			'type'        => 'select',
			'group'       => 'product',
			'label'       => esc_html__( 'Select Product', 'max-addons' ),
			'optionsAjax' => [
				'action'   => 'bricks_get_posts',
				'postType' => 'product',
			],
			'multiple'    => false,
			'searchable'  => true,
			'required'    => [ 'dynamicProduct', '=', '' ],
		];

		$this->controls['showQuantity'] = [
			'tab'         => 'content',
			'group'       => 'product',
			'label'       => esc_html__( 'Show Quantity', 'max-addons' ),
			'type'        => 'checkbox',
			'reset'       => true,
			'required'    => [ 'dynamicProduct', '=', '' ],
		];

		$this->controls['quantity'] = [
			'tab'         => 'content',
			'group'       => 'product',
			'label'       => esc_html__( 'Quantity', 'max-addons' ),
			'type'        => 'number',
			'min'         => 1,
			'placeholder' => 1,
			'required'    => [
				[ 'dynamicProduct', '=', '' ],
				[ 'showQuantity', '=', '' ],
			],
		];

		$this->controls['enableRedirect'] = [
			'tab'         => 'content',
			'group'       => 'product',
			'label'       => esc_html__( 'Auto Redirect', 'max-addons' ),
			'description' => __( 'Enable this option to redirect to cart page after the product gets added to cart', 'max-addons' ),
			'type'        => 'checkbox',
			'reset'       => true,
			'required'    => [ 'dynamicProduct', '=', '' ],
		];
	}

	// Set variations controls
	public function set_variations_controls() {
		$this->controls['variationsTypography'] = [
			'tab'   => 'content',
			'group' => 'variations',
			'label' => esc_html__( 'Typography', 'max-addons' ),
			'type'  => 'typography',
			'css'   => [
				[
					'property' => 'font',
					'selector' => 'table.variations label',
				],
			],
			'required' => [ 'dynamicProduct', '!=', '' ],
		];

		$this->controls['variationsBackgroundColor'] = [
			'tab'   => 'content',
			'group' => 'variations',
			'label' => esc_html__( 'Background color', 'max-addons' ),
			'type'  => 'color',
			'css'   => [
				[
					'property' => 'background-color',
					'selector' => 'table.variations tr',
				],
			],
			'required' => [ 'dynamicProduct', '!=', '' ],
		];

		$this->controls['variationsBorder'] = [
			'tab'   => 'content',
			'group' => 'variations',
			'label' => esc_html__( 'Border', 'max-addons' ),
			'type'  => 'border',
			'css'   => [
				[
					'property' => 'border',
					'selector' => '.cart .variations tr',
				]
			],
			'required' => [ 'dynamicProduct', '!=', '' ],
		];

		$this->controls['variationsMargin'] = [
			'tab'         => 'content',
			'group'       => 'variations',
			'label'       => esc_html__( 'Margin', 'max-addons' ),
			'type'        => 'spacing',
			'css'         => [
				[
					'selector' => '.cart table.variations',
					'property' => 'margin',
				],
			],
			'placeholder' => [
				'bottom' => 30,
			],
			'required' => [ 'dynamicProduct', '!=', '' ],
		];

		$this->controls['variationsPadding'] = [
			'tab'         => 'content',
			'group'       => 'variations',
			'label'       => esc_html__( 'Padding', 'max-addons' ),
			'type'        => 'spacing',
			'css'         => [
				[
					'selector' => '.cart table.variations td',
					'property' => 'padding',
				],
			],
			'placeholder' => [
				'top'    => 15,
				'bottom' => 15,
			],
			'required' => [ 'dynamicProduct', '!=', '' ],
		];
	}

	// Set quantity controls
	public function set_quantity_controls() {
		$this->controls['quantityWidth'] = [
			'tab'   => 'content',
			'group' => 'quantity',
			'type'  => 'number',
			'units' => true,
			'label' => esc_html__( 'Width', 'max-addons' ),
			'css'   => [
				[
					'selector' => '.quantity, .mab-add-to-cart-qty-ajax',
					'property' => 'width',
				],
			],
		];

		$this->controls['quantityBackground'] = [
			'tab'   => 'content',
			'group' => 'quantity',
			'type'  => 'color',
			'label' => esc_html__( 'Background', 'max-addons' ),
			'css'   => [
				[
					'selector' => '.quantity, .mab-add-to-cart-qty-ajax',
					'property' => 'background-color',
				],
			],
		];

		$this->controls['quantityBorder'] = [
			'tab'   => 'content',
			'group' => 'quantity',
			'type'  => 'border',
			'label' => esc_html__( 'Border', 'max-addons' ),
			'css'   => [
				[
					'selector' => '.quantity, .mab-add-to-cart-qty-ajax',
					'property' => 'border',
				],
				[
					'selector' => '.minus',
					'property' => 'border',
				],
				[
					'selector' => '.plus',
					'property' => 'border',
				],
			],
		];
	}

	// Set button controls
	public function set_button_controls() {
		$this->controls['buttonText'] = [
			'tab'         => 'content',
			'group'       => 'button',
			'label'       => esc_html__( 'Text', 'max-addons' ),
			'type'        => 'text',
			'inline'      => true,
			'default'     => esc_html__( 'Add to cart', 'max-addons' ),
			'placeholder' => esc_html__( 'Add to cart', 'max-addons' ),
			// 'hidden'      => true,
		];

		$this->controls['buttonStyle'] = [
			'tab'         => 'content',
			'group'       => 'button',
			'label'       => esc_html__( 'Style', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['styles'],
			'inline'      => true,
			'reset'       => true,
			'default'     => 'primary',
			'placeholder' => esc_html__( 'None', 'max-addons' ),
		];

		$this->controls['buttonOutline'] = [
			'tab'   => 'content',
			'group' => 'button',
			'label' => esc_html__( 'Outline', 'max-addons' ),
			'type'  => 'checkbox',
			'reset' => true,
		];

		$this->controls['buttonPadding'] = [
			'tab'   => 'content',
			'group' => 'button',
			'label' => esc_html__( 'Padding', 'max-addons' ),
			'type'  => 'spacing',
			'css'   => [
				[
					'selector' => '.mab-add-to-cart-button',
					'property' => 'padding',
				],
			],
		];

		$this->controls['buttonWidth'] = [
			'tab'   => 'content',
			'group' => 'button',
			'label' => esc_html__( 'Width', 'max-addons' ),
			'type'  => 'number',
			'units' => true,
			'css'   => [
				[
					'selector' => '.mab-add-to-cart-button',
					'property' => 'min-width',
				],
			],
		];

		$this->controls['buttonBackgroundColor'] = [
			'tab'   => 'content',
			'group' => 'button',
			'label' => esc_html__( 'Background color', 'max-addons' ),
			'type'  => 'color',
			'css'   => [
				[
					'selector' => '.mab-add-to-cart-button',
					'property' => 'background-color',
				],
			],
		];

		$this->controls['buttonBorder'] = [
			'tab'   => 'content',
			'group' => 'button',
			'label' => esc_html__( 'Border', 'max-addons' ),
			'type'  => 'border',
			'css'   => [
				[
					'property' => 'border',
					'selector' => '.mab-add-to-cart-button',
				],
			],
		];

		$this->controls['buttonTypography'] = [
			'tab'   => 'content',
			'group' => 'button',
			'label' => esc_html__( 'Typography', 'max-addons' ),
			'type'  => 'typography',
			'css'   => [
				[
					'selector' => '.mab-add-to-cart-button',
					'property' => 'font',
				],
			],
		];

		// Icon
		$this->controls['iconSeparator'] = [
			'tab'   => 'content',
			'group' => 'button',
			'label' => esc_html__( 'Icon', 'max-addons' ),
			'type'  => 'separator',
		];

		$this->controls['icon'] = [
			'tab'   => 'content',
			'group' => 'button',
			'label' => esc_html__( 'Icon', 'max-addons' ),
			'type'  => 'icon',
		];

		$this->controls['iconTypography'] = [
			'tab'      => 'content',
			'group'    => 'button',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => 'i',
				],
			],
			'exclude'  => [
				'font-family',
				'font-weight',
				'font-style',
				'text-align',
				'text-decoration',
				'text-transform',
				'line-height',
				'letter-spacing',
			],
			'required' => [ 'icon.icon', '!=', '' ],
		];

		$this->controls['iconPosition'] = [
			'tab'         => 'content',
			'group'       => 'button',
			'label'       => esc_html__( 'Position', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['iconPosition'],
			'inline'      => true,
			'placeholder' => esc_html__( 'Right', 'max-addons' ),
			'required'    => [ 'icon', '!=', '' ],
		];
	}

	// Set view cart controls
	public function set_view_cart_controls() {
		$this->controls['viewCartBackgroundColor'] = [
			'tab'   => 'content',
			'group' => 'viewCart',
			'label' => esc_html__( 'Background color', 'max-addons' ),
			'type'  => 'color',
			'css'   => [
				[
					'selector' => '.added_to_cart',
					'property' => 'background-color',
				],
			],
			'required' => [ 'dynamicProduct', '=', '' ],
		];

		$this->controls['viewCartBorder'] = [
			'tab'   => 'content',
			'group' => 'viewCart',
			'label' => esc_html__( 'Border', 'max-addons' ),
			'type'  => 'border',
			'css'   => [
				[
					'property' => 'border',
					'selector' => '.added_to_cart',
				],
			],
			'required' => [ 'dynamicProduct', '=', '' ],
		];

		$this->controls['viewCartTypography'] = [
			'tab'   => 'content',
			'group' => 'viewCart',
			'label' => esc_html__( 'Typography', 'max-addons' ),
			'type'  => 'typography',
			'css'   => [
				[
					'selector' => '.added_to_cart',
					'property' => 'font',
				],
			],
			'required' => [ 'dynamicProduct', '=', '' ],
		];
	}

	public function get_button_classes( $return_type = 'array' ) {
		$settings = $this->settings;

		$button_classes = array(
			'bricks-button',
			'mab-add-to-cart-button',
		);

		if ( ! empty( $settings['buttonStyle'] ) ) {
			// Outline
			if ( isset( $settings['buttonOutline'] ) ) {
				$button_classes[] = 'outline';
				$button_classes[] = "bricks-color-{$settings['buttonStyle']}";
			}

			// Fill (= default)
			else {
				$button_classes[] = "bricks-background-{$settings['buttonStyle']}";
			}
		}

		if ( 'string' === $return_type ) {
			$string = implode( ' ', $button_classes );

			return $string;
		}

		return $button_classes;
	}

	/**
	 * Add custom text and/or icon to the button
	 *
	 * @param string     $text
	 * @param WC_Product $product
	 * @return void
	 */
	public function add_to_cart_text() {
		$settings = $this->settings;

		$text = ! empty( $settings['buttonText'] ) ? $settings['buttonText'] : $text;

		$icon          = ! empty( $settings['icon'] ) ? self::render_icon( $settings['icon'], [ 'icon' ] ) : false;
		$icon_position = isset( $settings['iconPosition'] ) ? $settings['iconPosition'] : 'right';

		$output = '';

		if ( $icon && $icon_position === 'left' ) {
			$output .= $icon;
		}

		$output .= "<span>$text</span>";

		if ( $icon && $icon_position === 'right' ) {
			$output .= $icon;
		}

		return $output;
	}

	public function avoid_esc_html( $safe_text, $text ) {
		return $text;
	}

	private function render_form_button( $product ) {
		$settings = $this->settings;

		if ( ! $product && current_user_can( 'manage_options' ) ) {
			return;
		}

		add_filter( 'woocommerce_get_stock_html', '__return_empty_string' );
		add_filter( 'woocommerce_product_single_add_to_cart_text', [ $this, 'add_to_cart_text' ], 10, 2 );
		add_filter( 'esc_html', [ $this, 'avoid_esc_html' ], 10, 2 );

		$button_classes = $this->get_button_classes('string');

		$this->set_attribute( '_root', 'class', array(
			'mab-woo-add-to-cart',
			'mab-woo-product-' . esc_attr( wc_get_product()->get_type() ),
		) );
		echo "<div {$this->render_attributes( '_root' )}>";
		ob_start();
		woocommerce_template_single_add_to_cart();
		$form = ob_get_clean();
		$form = str_replace( 'single_add_to_cart_button', 'single_add_to_cart_button ' . $button_classes, $form );
		echo $form;
		echo '</div>';

		remove_filter( 'woocommerce_get_stock_html', '__return_empty_string' );
		remove_filter( 'woocommerce_product_single_add_to_cart_text', [ $this, 'add_to_cart_text' ], 10, 2 );
		remove_filter( 'esc_html', [ $this, 'avoid_esc_html' ], 10, 2 );
	}

	/**
	 * @param \WC_Product $product
	 */
	private function render_ajax_button( $product, $show_quantity = false ) {
		$settings = $this->settings;
		$atc_html = '';

		if ( $product ) {
			$product_id   = $product->get_id();
			$product_type = $product->get_type();

			$button_classes = $this->get_button_classes();

			$button_classes[] = 'product_type_' . $product_type;
			$button_classes[] = $product->is_purchasable() && $product->is_in_stock() ? 'add_to_cart_button' : '';
			$button_classes[] = $product->supports( 'ajax_add_to_cart' ) ? 'ajax_add_to_cart' : '';

			if ( isset( $settings['enableRedirect'] ) ) {
				$button_classes[] = 'mab-redirect';
			}

			$this->set_attribute( 'button', 'rel', 'nofollow' );
			$this->set_attribute( 'button', 'href', $product->add_to_cart_url() );
			$this->set_attribute( 'button', 'data-quantity', ( isset( $settings['quantity'] ) ? $settings['quantity'] : 1 ) );
			$this->set_attribute( 'button', 'data-product_id', $product_id );
			$this->set_attribute( 'button', 'class', $button_classes );

			$this->set_attribute( '_root', 'class', 'mab-woo-add-to-cart' );
			?>
			<div <?php echo wp_kses_post( $this->render_attributes( '_root' ) ); ?>>
				<?php
				if ( true === $show_quantity ) {
					$total_get_stock_quantity = 0;
					$min_value                = 0;
					if ( $product->is_purchasable() && $product->is_in_stock() ) {
						if ( $product->get_stock_quantity() ) {
							$total_get_stock_quantity = ( ! empty( $product->get_stock_quantity() ) ) ? ( $product->get_stock_quantity() ) : 1;
						} else {
							$total_get_stock_quantity = 0;
						}
						$min_value = 1;
					}

					$this->set_attribute( 'input-number', 'type', 'number' );
					$this->set_attribute( 'input-number', 'value', $min_value );
					$this->set_attribute( 'input-number', 'min', $min_value );
					$this->set_attribute( 'input-number', 'class', 'mab-add-to-cart-qty-ajax' );

					if ( ! empty( $total_get_stock_quantity ) ) {
						$this->set_attribute( 'input-number', 'max', $total_get_stock_quantity );
					}

					printf( '<input %s >', $this->render_attributes( 'input-number' ) );
				}
				?>
				<a <?php echo wp_kses_post( $this->render_attributes( 'button' ) ); ?>>
					<?php echo $this->add_to_cart_text(); ?>
				</a>
			</div>
			<?php
		}
	}

	public function render() {
		$settings = $this->settings;

		if ( isset( $settings['dynamicProduct'] ) ) {
			global $product;

			$product = wc_get_product();

			if ( empty( $product ) ) {
				return;
			}

			$this->render_form_button( $product );
		} else {
			$product = false;

			if ( isset( $settings['productId'] ) ) {
				$product_data = get_post( $settings['productId'] );
			}

			$product = ! empty( $product_data ) && in_array( $product_data->post_type, array( 'product', 'product_variation' ) ) ? wc_setup_product_data( $product_data ) : false;

			if ( $product ) {
				if ( isset( $settings['showQuantity'] ) ) {
					$this->render_ajax_button( $product, true );
				} else {
					$this->render_ajax_button( $product );
				}
			} elseif ( current_user_can( 'manage_options' ) ) {
				$this->set_attribute( '_root', 'class', 'mab-woo-add-to-cart' );

				$button_classes = $this->get_button_classes();
				$this->set_attribute( 'button', 'class', $button_classes );
				?>
				<div <?php echo wp_kses_post( $this->render_attributes( '_root' ) ); ?>>
					<a <?php echo wp_kses_post( $this->render_attributes( 'button' ) ); ?>>
						<?php esc_attr_e( 'Please select the product', 'max-addons' ); ?>
					</a>
				</div>
				<?php
			}
		}
	}
}
