<?php
namespace MaxAddonsPro\Elements;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Video_Gallery_Element extends \Bricks\Element {
	// Element properties
	public $category     = 'max-addons-elements'; // Use predefined element category 'general'
	public $name         = 'max-video-gallery'; // Make sure to prefix your elements
	public $icon         = 'ti-video-clapper max-element'; // Themify icon font class
	public $css_selector = ''; // Default CSS selector
	public $loop_index   = 0;
	public $scripts      = [ 'mabVideoGallery', 'bricksSwiper', 'bricksIsotope' ]; // Script(s) run when element is rendered on frontend or updated in builder
	public $tag          = 'ul';

	public function get_label() {
		return esc_html__( 'Video Gallery', 'max-addons' );
	}

	public function get_keywords() {
		return [ 'youtube', 'vimeo', 'query', 'posts' ];
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		$layout = ! empty( $this->settings['layout'] ) ? $this->settings['layout'] : 'grid';

		wp_enqueue_style( 'mab-video-gallery' );

		if ( $layout === 'carousel' ) {
			wp_enqueue_script( 'bricks-swiper' );
		}

		if ( $this->is_filters_enabled() ) {
			wp_enqueue_script( 'imagesloaded' );
			wp_enqueue_script( 'bricks-isotope' );
			wp_enqueue_style( 'bricks-isotope' );
		}

		wp_enqueue_script( 'mab-video-gallery' );
	}

	public function set_control_groups() {

		$this->control_groups['items'] = [
			'title' => esc_html__( 'Videos', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['settings'] = [
			'title' => esc_html__( 'Layout', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['gallerySettings'] = [
			'title' => esc_html__( 'Settings', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['filterTabs'] = [
			'title'    => esc_html__( 'Filters', 'max-addons' ),
			'tab'      => 'content',
			'required' => [
				[ 'layout', '=', 'grid' ],
				[ 'source', '!=', [ 'yt-playlist', 'yt-channel' ] ],
			],
		];

		$this->control_groups['icon'] = [
			'title' => esc_html__( 'Play Icon', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['content'] = [
			'title' => esc_html__( 'Content', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['arrows'] = [
			'title'    => esc_html__( 'Arrows', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'layout', '=', 'carousel' ],
		];

		$this->control_groups['dots'] = [
			'title'    => esc_html__( 'Dots', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'layout', '=', 'carousel' ],
		];

	}

	public function set_controls() {
		$this->set_items_controls();

		$this->set_layout_controls();

		$this->set_settings_controls();

		$this->set_filters_controls();

		$this->set_icon_controls();

		$this->set_content_controls();

		// Slider Controls
		// Group: Settings

		if ( method_exists( '\Bricks\Setup', "get_carousel_controls" ) ) {
			$carousel_controls = \Bricks\Setup::get_carousel_controls();
		} else {
			$carousel_controls = self::get_swiper_controls();
		}

		$this->controls['slidesToShow']               = $carousel_controls['slidesToShow'];
		$this->controls['slidesToShow']['required']   = [ 'layout', '=', 'carousel' ];
		$this->controls['slidesToScroll']             = $carousel_controls['slidesToScroll'];
		$this->controls['slidesToScroll']['required'] = [ 'layout', '=', 'carousel' ];
		$this->controls['gutter']                     = $carousel_controls['gutter'];
		$this->controls['gutter']['required']         = [ 'layout', '=', 'carousel' ];
		$this->controls['infinite']                   = $carousel_controls['infinite'];
		$this->controls['infinite']['required']       = [ 'layout', '=', 'carousel' ];

		$this->controls['autoplay'] = [
			'tab'      => 'content',
			'group'    => 'settings',
			'label'    => esc_html__( 'Autoplay', 'max-addons' ),
			'type'     => 'checkbox',
			'required' => [ 'layout', '=', 'carousel' ],
		];

		$this->controls['pauseOnHover'] = [
			'tab'      => 'content',
			'group'    => 'settings',
			'label'    => esc_html__( 'Pause on hover', 'max-addons' ),
			'type'     => 'checkbox',
			'required' => [
				[ 'layout', '=', 'carousel' ],
				[ 'autoplay', '!=', '' ]
			],
			'inline'   => true,
		];

		$this->controls['autoplaySpeed'] = [
			'tab'         => 'content',
			'group'       => 'settings',
			'label'       => esc_html__( 'Autoplay speed in ms', 'max-addons' ),
			'type'        => 'number',
			'required'    => [
				[ 'layout', '=', 'carousel' ],
				[ 'autoplay', '!=', '' ]
			],
			'placeholder' => 3000,
		];

		$this->controls['speed']             = $carousel_controls['speed'];
		$this->controls['speed']['required'] = [ 'layout', '=', 'carousel' ];

		// Arrows

		$this->controls['arrows']          = $carousel_controls['arrows'];
		$this->controls['arrowHeight']     = $carousel_controls['arrowHeight'];
		$this->controls['arrowWidth']      = $carousel_controls['arrowWidth'];
		$this->controls['arrowBackground'] = $carousel_controls['arrowBackground'];
		$this->controls['arrowBorder']     = $carousel_controls['arrowBorder'];
		$this->controls['arrowTypography'] = $carousel_controls['arrowTypography'];

		$this->controls['prevArrowSeparator'] = $carousel_controls['prevArrowSeparator'];
		$this->controls['prevArrow']          = $carousel_controls['prevArrow'];
		$this->controls['prevArrowTop']       = $carousel_controls['prevArrowTop'];
		$this->controls['prevArrowRight']     = $carousel_controls['prevArrowRight'];
		$this->controls['prevArrowBottom']    = $carousel_controls['prevArrowBottom'];
		$this->controls['prevArrowLeft']      = $carousel_controls['prevArrowLeft'];

		$this->controls['nextArrowSeparator'] = $carousel_controls['nextArrowSeparator'];
		$this->controls['nextArrow']          = $carousel_controls['nextArrow'];
		$this->controls['nextArrowTop']       = $carousel_controls['nextArrowTop'];
		$this->controls['nextArrowRight']     = $carousel_controls['nextArrowRight'];
		$this->controls['nextArrowBottom']    = $carousel_controls['nextArrowBottom'];
		$this->controls['nextArrowLeft']      = $carousel_controls['nextArrowLeft'];

		// Dots

		$this->controls['dots']            = $carousel_controls['dots'];
		$this->controls['dotsDynamic']     = $carousel_controls['dotsDynamic'];
		$this->controls['dotsVertical']    = $carousel_controls['dotsVertical'];
		$this->controls['dotsHeight']      = $carousel_controls['dotsHeight'];
		$this->controls['dotsWidth']       = $carousel_controls['dotsWidth'];
		$this->controls['dotsTop']         = $carousel_controls['dotsTop'];
		$this->controls['dotsRight']       = $carousel_controls['dotsRight'];
		$this->controls['dotsBottom']      = $carousel_controls['dotsBottom'];
		$this->controls['dotsLeft']        = $carousel_controls['dotsLeft'];
		$this->controls['dotsBorder']      = $carousel_controls['dotsBorder'];
		$this->controls['dotsSpacing']     = $carousel_controls['dotsSpacing'];
		$this->controls['dotsColor']       = $carousel_controls['dotsColor'];
		$this->controls['dotsActiveColor'] = $carousel_controls['dotsActiveColor'];
	}

	// Set items controls
	public function set_items_controls() {

		$this->controls['source'] = [
			'tab'         => 'content',
			'group'       => 'items',
			'label'       => esc_html__( 'Source', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'custom'      => esc_html__( 'Custom', 'max-addons' ),
				'yt-playlist' => esc_html__( 'YouTube Playlist', 'max-addons' ),
				'yt-channel'  => esc_html__( 'YouTube Channel', 'max-addons' ),
			],
			'default'     => 'custom',
			'placeholder' => esc_html__( 'Custom', 'max-addons' ),
			'clearable'   => true,
			'inline'      => true,
		];

		// $this->controls = array_replace_recursive( $this->controls, $this->get_loop_builder_controls('items') );

		$this->controls['hasLoop'] = [
			'tab'      => 'content',
			'group'    => 'items',
			'label'    => esc_html__( 'Use query loop', 'max-addons' ),
			'type'     => 'checkbox',
			'required' => [ 'source', '!=', [ 'yt-playlist', 'yt-channel' ] ],
		];

		$this->controls['query'] = [
			'tab'      => 'content',
			'group'    => 'items',
			'label'    => esc_html__( 'Query', 'max-addons' ),
			'type'     => 'query',
			'popup'    => true,
			'inline'   => true,
			'required' => [
				[ 'source', '!=', [ 'yt-playlist', 'yt-channel' ] ],
				[ 'hasLoop', '!=', '' ]
			],
		];

		$this->controls['ytPlaylistId'] = [
			'tab'            => 'content',
			'group'          => 'items',
			'label'          => esc_html__( 'YouTube playlist ID', 'max-addons' ),
			'type'           => 'text',
			'hasDynamicData' => 'text',
			'default'        => '',
			'required'       => [ 'source', '=', 'yt-playlist' ],
		];

		$this->controls['ytChannelId'] = [
			'tab'            => 'content',
			'group'          => 'items',
			'label'          => esc_html__( 'YouTube channel ID', 'max-addons' ),
			'type'           => 'text',
			'hasDynamicData' => 'text',
			'default'        => '',
			'required'       => [ 'source', '=', 'yt-channel' ],
		];

		$this->controls['ytVideosCount'] = [
			'tab'         => 'content',
			'group'       => 'items',
			'label'       => esc_html__( 'Number of videos', 'max-addons' ),
			'description' => esc_html__( 'Specify number of videos to show on the page', 'max-addons' ),
			'type'        => 'number',
			'inline'      => true,
			'small'       => true,
			'default'     => 9,
			'required'    => [ 'source', '=', 'yt-channel' ],
		];

		$this->controls['showTitle'] = [
			'tab'      => 'content',
			'group'    => 'items',
			'label'    => esc_html__( 'Show video title', 'max-addons' ),
			'type'     => 'checkbox',
			'required' => [ 'source', '=', [ 'yt-channel', 'yt-playlist' ] ],
		];

		$this->controls['showDescription'] = [
			'tab'      => 'content',
			'group'    => 'items',
			'label'    => esc_html__( 'Show video description', 'max-addons' ),
			'type'     => 'checkbox',
			'required' => [ 'source', '=', [ 'yt-channel', 'yt-playlist' ] ],
		];

		$this->controls['videosRefreshTime'] = [
			'tab'         => 'content',
			'group'       => 'items',
			'label'       => esc_html__( 'Refresh videos after', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'hour'  => __( 'Hour', 'max-addons' ),
				'day'   => __( 'Day', 'max-addons' ),
				'week'  => __( 'Week', 'max-addons' ),
				'month' => __( 'Month', 'max-addons' ),
				'year'  => __( 'Year', 'max-addons' ),
			],
			'default'     => 'day',
			'placeholder' => esc_html__( 'Day', 'max-addons' ),
			'inline'      => true,
			'clearable'   => false,
			'required'    => [ 'source', '=', [ 'yt-channel', 'yt-playlist' ] ],
		];

		$this->controls['items'] = [
			'tab'           => 'content',
			'group'         => 'items',
			'placeholder'   => esc_html__( 'Video', 'max-addons' ),
			'type'          => 'repeater',
			'checkLoop'     => true,
			'titleProperty' => 'videoTitle',
			'fields'        => [
				'videoType'            => [
					'label'     => esc_html__( 'Type', 'max-addons' ),
					'type'      => 'select',
					'options'   => [
						'youtube' => esc_html__( 'YouTube', 'max-addons' ),
						'vimeo'   => esc_html__( 'Vimeo', 'max-addons' ),
						'media'   => esc_html__( 'Media', 'max-addons' ),
						'file'    => esc_html__( 'File URL', 'max-addons' ),
						'meta'    => esc_html__( 'Dynamic Data', 'max-addons' ),
					],
					'default'   => 'youtube',
					'inline'    => true,
					'clearable' => false,
				],

				'youTubeUrl'           => [
					'label'          => esc_html__( 'YouTube video URL', 'max-addons' ),
					'type'           => 'text',
					'hasDynamicData' => 'text',
					'default'        => 'https://www.youtube.com/watch?v=5DGo0AYOJ7s',
					'required'       => [ 'videoType', '=', 'youtube' ],
				],

				'vimeoUrl'             => [
					'label'          => esc_html__( 'Vimeo video URL', 'max-addons' ),
					'type'           => 'text',
					'hasDynamicData' => 'text',
					'default'        => 'https://vimeo.com/235215203',
					'required'       => [ 'videoType', '=', 'vimeo' ],
				],

				'mediaUrl'             => [
					'label'    => esc_html__( 'Media', 'max-addons' ),
					'type'     => 'video',
					'required' => [ 'videoType', '=', 'media' ],
				],

				'fileUrl'              => [
					'label'    => esc_html__( 'Video file URL', 'max-addons' ),
					'type'     => 'text',
					'required' => [ 'videoType', '=', 'file' ],
				],

				'useDynamicData'       => [
					'label'          => '',
					'type'           => 'text',
					'placeholder'    => esc_html__( 'Select dynamic data', 'max-addons' ),
					'hasDynamicData' => 'link',
					'required'       => [ 'videoType', '=', 'meta' ],
				],

				'start'                => [
					'label'       => esc_html__( 'Start Time', 'max-addons' ),
					'description' => esc_html__( 'Specify start time (in seconds)', 'max-addons' ),
					'type'        => 'number',
					'inline'      => true,
					'small'       => true,
					'required'    => [ 'videoType', '=', [ 'media', 'file', 'meta' ] ],
				],

				'end'                  => [
					'label'       => esc_html__( 'End Time', 'max-addons' ),
					'description' => esc_html__( 'Specify end time (in seconds)', 'max-addons' ),
					'type'        => 'number',
					'inline'      => true,
					'small'       => true,
					'required'    => [ 'videoType', '=', [ 'media', 'file', 'meta' ] ],
				],

				'filterLabel'          => [
					'label' => esc_html__( 'Filter label', 'max-addons' ),
					'type'  => 'text',
				],

				'videoTitle'           => [
					'label'          => esc_html__( 'Title', 'max-addons' ),
					'type'           => 'text',
					'hasDynamicData' => 'text',
				],

				'videoDescription'     => [
					'label'          => esc_html__( 'Description', 'max-addons' ),
					'type'           => 'textarea',
					'hasDynamicData' => 'text',
				],

				'thumbnailSize'        => [
					'label'     => esc_html__( 'Thumbnail Size', 'max-addons' ),
					'type'      => 'select',
					'options'   => [
						'maxresdefault' => __( 'Maximum Resolution', 'max-addons' ),
						'hqdefault'     => __( 'High Quality', 'max-addons' ),
						'mqdefault'     => __( 'Medium Quality', 'max-addons' ),
						'sddefault'     => __( 'Standard Quality', 'max-addons' ),
					],
					'default'   => 'maxresdefault',
					'inline'    => true,
					'clearable' => false,
					'required'  => [ 'videoType', '=', 'youtube' ],
				],

				'customThumbnail'      => [
					'label' => esc_html__( 'Custom Thumbnail', 'max-addons' ),
					'type'  => 'checkbox',
				],

				'customThumbnailImage' => [
					'label'    => esc_html__( 'Image', 'max-addons' ),
					'type'     => 'image',
					'required' => [ 'customThumbnail', '!=', '' ],
				],
			],
			'default'       => [
				[
					'videoType'   => 'youtube',
					'youTubeUrl'  => 'https://www.youtube.com/watch?v=5DGo0AYOJ7s',
					'filterLabel' => esc_html__( 'YouTube', 'max-addons' ),
				],
				[
					'videoType'   => 'youtube',
					'youTubeUrl'  => 'https://www.youtube.com/watch?v=Mmwv94WKmnI',
					'filterLabel' => esc_html__( 'YouTube', 'max-addons' ),
				],
				[
					'videoType'   => 'vimeo',
					'vimeoUrl'    => 'https://vimeo.com/23169366',
					'filterLabel' => esc_html__( 'Vimeo', 'max-addons' ),
				],
			],
			'required' => [ 'source', '!=', [ 'yt-playlist', 'yt-channel' ] ],
		];
	}

	// Set layout controls
	public function set_layout_controls() {

		$this->controls['layout'] = [
			'tab'       => 'content',
			'group'     => 'settings',
			'label'     => esc_html__( 'Layout', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'grid'     => esc_html__( 'Grid', 'max-addons' ),
				'carousel' => esc_html__( 'Carousel', 'max-addons' ),
			],
			'default'   => 'grid',
			'clearable' => false,
			'inline'    => true,
		];

		$this->controls['columns'] = [
			'tab'         => 'content',
			'group'       => 'settings',
			'label'       => esc_html__( 'Videos per row', 'max-addons' ),
			'type'        => 'number',
			'min'         => 1,
			'css'         => [
				[
					'property' => '--columns',
					'selector' => '.mab-video-gallery',
				],
			],
			'rerender'    => true,
			'inline'      => true,
			'small'       => true,
			'placeholder' => 3,
			'required'    => [ 'layout', '=', 'grid' ],
		];

		$this->controls['gridSpacing'] = [
			'tab'      => 'content',
			'group'    => 'settings',
			'label'    => esc_html__( 'Spacing', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => '--gutter',
					'selector' => '.mab-video-gallery',
				],
			],
			'required' => [ 'layout', '=', 'grid' ],
		];
	}

	// Set settings controls
	public function set_settings_controls() {

		$this->controls['preload'] = [
			'tab'         => 'content',
			'group'       => 'gallerySettings',
			'label'       => esc_html__( 'Preload', 'max-addons' ),
			'description' => esc_html__( 'Use in case of self hosted video', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'auto'     => __( 'Auto', 'max-addons' ),
				'metadata' => __( 'Meta Data', 'max-addons' ),
				'none'     => __( 'None', 'max-addons' ),
			],
			'default'     => 'auto',
			'inline'      => true,
		];

		$this->controls['order'] = [
			'tab'     => 'content',
			'group'   => 'gallerySettings',
			'label'   => esc_html__( 'Order', 'max-addons' ),
			'type'    => 'select',
			'options' => [
				'default' => __( 'Default', 'max-addons' ),
				'random'  => __( 'Random', 'max-addons' ),
			],
			'default' => '',
			'inline'  => true,
		];

		$this->controls['clickAction'] = [
			'tab'       => 'content',
			'group'     => 'gallerySettings',
			'label'     => esc_html__( 'Click action', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'inline'   => __( 'Play Inline', 'max-addons' ),
				'lightbox' => __( 'Play in Lightbox', 'max-addons' ),
			],
			'default'   => 'inline',
			'inline'    => true,
			'clearable' => false,
		];
	}

	// Set filters controls
	public function set_filters_controls() {
		$this->controls['filtersEnable'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Enable Filters', 'max-addons' ),
			'type'     => 'checkbox',
			'required' => [
				[ 'layout', '=', 'grid' ],
				[ 'source', '!=', [ 'yt-playlist', 'yt-channel' ] ],
			],
		];

		$this->controls['allFilterLabel'] = [
			'tab'            => 'content',
			'group'          => 'filterTabs',
			'label'          => esc_html__( '"All" Filter Label', 'max-addons' ),
			'default'        => esc_html__( 'All', 'max-addons' ),
			'type'           => 'text',
			'hasDynamicData' => 'text',
			'inline'         => true,
			'required'       => [ 'filtersEnable', '!=', '' ],
		];

		$this->controls['filtersOrder'] = [
			'tab'         => 'content',
			'group'       => 'filterTabs',
			'label'       => esc_html__( 'Order', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'asc'  => esc_html__( 'Ascending', 'max-addons' ),
				'desc' => esc_html__( 'Descending', 'max-addons' ),
			],
			'default'     => '',
			'placeholder' => esc_html__( 'Default', 'max-addons' ),
			'inline'      => true,
			'required'    => [ 'filtersEnable', '!=', '' ],
		];

		$this->controls['filtersTypography'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.bricks-isotope-filters',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'filtersEnable', '!=', '' ],
		];

		$this->controls['filtersTypographyActive'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Typography active', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.bricks-isotope-filters .active',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'filtersEnable', '!=', '' ],
		];

		$this->controls['filtersBorder'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.bricks-isotope-filters li',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'filtersEnable', '!=', '' ],
		];

		$this->controls['filtersBackground'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Background', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.bricks-isotope-filters li',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'filtersEnable', '!=', '' ],
		];

		$this->controls['filtersBackgroundActive'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Background active', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.bricks-isotope-filters .active',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'filtersEnable', '!=', '' ],
		];

		$this->controls['filtersSpacing'] = [
			'tab'         => 'content',
			'group'       => 'filterTabs',
			'label'       => esc_html__( 'Bottom Spacing', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'css'         => [
				[
					'property' => 'margin-bottom',
					'selector' => '.bricks-isotope-filters',
				]
			],
			'placeholder' => 20,
			'required' => [ 'filtersEnable', '!=', '' ],
		];

		$this->controls['filtersMargin'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Margin', 'max-addons' ),
			'type'     => 'spacing',
			'css'      => [
				[
					'property' => 'margin',
					'selector' => '.bricks-isotope-filters li',
				],
			],
			'required' => [ 'filtersEnable', '!=', '' ],
		];

		$this->controls['filtersPadding'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Padding', 'max-addons' ),
			'type'     => 'spacing',
			'css'      => [
				[
					'property' => 'padding',
					'selector' => '.bricks-isotope-filters li',
				],
			],
			'required' => [ 'filtersEnable', '!=', '' ],
		];
	}

	// Set icon controls
	public function set_icon_controls() {
		$this->controls['playIconType'] = [
			'tab'     => 'content',
			'group'   => 'icon',
			'label'   => esc_html__( 'Icon type', 'max-addons' ),
			'type'    => 'select',
			'options' => [
				'icon'  => esc_html__( 'Icon', 'max-addons' ),
				'image' => esc_html__( 'Image', 'max-addons' ),
			],
			'default' => 'icon',
			'inline'  => true,
		];

		$this->controls['playIcon'] = [
			'tab'      => 'content',
			'group'    => 'icon',
			'label'    => esc_html__( 'Icon', 'max-addons' ),
			'type'     => 'icon',
			'default'  => [
				'library' => 'themify',
				'icon'    => 'ti-control-play',
			],
			'required' => [ 'playIconType', '=', [ 'icon' ] ],
		];

		$this->controls['iconTypography'] = [
			'tab'      => 'content',
			'group'    => 'icon',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-video-play-icon',
				],
			],
			'exclude'  => [
				'font-family',
				'font-weight',
				'font-style',
				'text-align',
				'text-decoration',
				'text-transform',
				'line-height',
				'letter-spacing',
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'playIconType', '=', [ 'icon' ] ],
		];

		$this->controls['playIconImage'] = [
			'tab'      => 'content',
			'group'    => 'icon',
			'type'     => 'image',
			'label'    => esc_html__( 'Image', 'max-addons' ),
			'required' => [ 'playIconType', '=', [ 'image' ] ],
		];

		$this->controls['iconImageHeight'] = [
			'tab'      => 'content',
			'group'    => 'icon',
			'label'    => esc_html__( 'Height', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'height',
					'selector' => '.mab-play-icon-image img',
				],
			],
			'required' => [ 'playIconType', '=', [ 'image' ] ],
		];

		$this->controls['iconImageWidth'] = [
			'tab'      => 'content',
			'group'    => 'icon',
			'label'    => esc_html__( 'Width', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'width',
					'selector' => '.mab-play-icon-image img',
				],
			],
			'required' => [ 'playIconType', '=', [ 'image' ] ],
		];
	}

	// Set content controls
	public function set_content_controls() {

		$this->controls['contentPadding'] = [
			'tab'   => 'content',
			'group' => 'content',
			'label' => esc_html__( 'Padding', 'max-addons' ),
			'type'  => 'spacing',
			'css'   => [
				[
					'property' => 'padding',
					'selector' => '.mab-video-content',
				],
			],
		];

		$this->controls['contentAlign'] = [
			'tab'    => 'content',
			'group'  => 'content',
			'label'  => esc_html__( 'Text align', 'max-addons' ),
			'type'   => 'text-align',
			'css'    => [
				[
					'property' => 'text-align',
					'selector' => '.mab-video-content',
				],
			],
			'inline' => true,
		];

		$this->controls['contentBackground'] = [
			'tab'     => 'content',
			'group'   => 'content',
			'label'   => esc_html__( 'Background color', 'max-addons' ),
			'type'    => 'background',
			'css'     => [
				[
					'property' => 'background',
					'selector' => '.mab-video-content',
				],
			],
			'exclude' => [
				'parallax',
				'videoUrl',
				'videoScale',
			],
			'inline'  => true,
			'small'   => true,
		];

		$this->controls['videoTitleTag'] = [
			'tab'         => 'content',
			'group'       => 'content',
			'label'       => esc_html__( 'Title tag', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'h2'  => esc_html__( 'Heading 2 (h2)', 'max-addons' ),
				'h3'  => esc_html__( 'Heading 3 (h3)', 'max-addons' ),
				'h4'  => esc_html__( 'Heading 4 (h4)', 'max-addons' ),
				'h5'  => esc_html__( 'Heading 5 (h5)', 'max-addons' ),
				'h6'  => esc_html__( 'Heading 6 (h6)', 'max-addons' ),
				'p'   => esc_html__( 'Paragraph (p)', 'max-addons' ),
				'div' => 'div',
			],
			'inline'      => true,
			'placeholder' => 'h4',
		];

		$this->controls['titleTypography'] = [
			'tab'    => 'content',
			'group'  => 'content',
			'label'  => esc_html__( 'Title typography', 'max-addons' ),
			'type'   => 'typography',
			'css'    => [
				[
					'property' => 'font',
					'selector' => '.mab-video-title',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['titleSpacing'] = [
			'tab'   => 'content',
			'group' => 'content',
			'label' => esc_html__( 'Title spacing', 'max-addons' ),
			'type'  => 'number',
			'units' => true,
			'css'   => [
				[
					'property' => 'margin-bottom',
					'selector' => '.mab-video-title',
				],
			],
		];

		$this->controls['descriptionTypography'] = [
			'tab'    => 'content',
			'group'  => 'content',
			'label'  => esc_html__( 'Description typography', 'max-addons' ),
			'type'   => 'typography',
			'css'    => [
				[
					'property' => 'font',
					'selector' => '.mab-video-description',
				],
			],
			'inline' => true,
			'small'  => true,
		];
	}

	public function slider_settings() {
		$settings = $this->settings;

		$slidesToShow = isset( $settings['slidesToShow'] ) ? intval( $settings['slidesToShow'] ) : 3;

		$options = [
			'speed'          => isset( $settings['speed'] ) ? intval( $settings['speed'] ) : 300,
			'autoHeight'     => true, // isset( $settings['adaptiveHeight'] ),
			'spaceBetween'   => isset( $settings['gutter'] ) ? intval( $settings['gutter'] ) : 0,
			'slidesPerView'  => $slidesToShow,
			'slidesPerGroup' => isset( $settings['slidesToScroll'] ) ? intval( $settings['slidesToScroll'] ) : 1,
			'loop'           => isset( $settings['infinite'] ),
		];

		if ( isset( $settings['autoplay'] ) ) {
			$options['autoplay'] = [
				'delay'                => isset( $settings['autoplaySpeed'] ) ? intval( $settings['autoplaySpeed'] ) : 3000,
				'disableOnInteraction' => true,
			];

			// Custom as SwiperJS does not provide this option
			if ( isset( $settings['pauseOnHover'] ) ) {
				$options['pauseOnHover'] = true;
			}
		}

		// Arrow navigation
		if ( isset( $settings['arrows'] ) ) {
			$options['navigation'] = [
				'prevEl' => '.bricks-swiper-button-prev',
				'nextEl' => '.bricks-swiper-button-next',
			];
		}

		// Dots
		if ( isset( $settings['dots'] ) ) {
			$options['pagination'] = [
				'el'   => '.swiper-pagination',
				'type' => 'bullets',
			];

			if ( isset( $settings['dotsDynamic'] ) ) {
				$options['dynamicBullets'] = true;
			}
		}

		$breakpoints = [];

		// Breakpoint values compare to ">="
		if ( isset( $settings['responsive'] ) ) {
			// Custom breakpoints
			foreach ( $settings['responsive'] as $responsive_option ) {
				if ( isset( $responsive_option['breakpoint'] ) ) {
					$breakpoints[ $responsive_option['breakpoint'] ] = [
						'slidesPerView'  => isset( $responsive_option['slidesToShow'] ) ? intval( $responsive_option['slidesToShow'] ) : 1,
						'slidesPerGroup' => isset( $responsive_option['slidesToScroll'] ) ? intval( $responsive_option['slidesToScroll'] ) : 1,
					];
				}
			}
		} else {
			// Default breakpoints
			$breakpoints[992] = [
				'slidesPerView'  => $options['slidesPerView'],
				'slidesPerGroup' => $options['slidesPerGroup'],
			];

			$breakpoints[768] = [
				'slidesPerView'  => $slidesToShow > 1 ? 2 : 1,
				'slidesPerGroup' => 1,
			];

			$breakpoints[1] = [
				'slidesPerView'  => 1,
				'slidesPerGroup' => 1,
			];
		}

		$options['breakpoints'] = $breakpoints;

		$this->set_attribute( 'gallery', 'data-script-args', wp_json_encode( $options ) );
	}

	/**
	 * Clean string - Removes spaces and special chars.
	 *
	 * @since 1.0.0
	 * @param  String $string String to be cleaned.
	 * @return array Google Map languages List.
	 */
	public function clean( $string ) {

		// Replaces all spaces with hyphens.
		$string = str_replace( ' ', '-', $string );

		// Removes special chars.
		$string = preg_replace( '/[^A-Za-z0-9\-]/', '', $string );

		// Turn into lower case characters.
		return strtolower( $string );
	}

	/**
	 * Render filter keys array.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function get_filter_keys( $item ) {
		if ( isset( $item['filterLabel'] ) ) {
			$filters = explode( ',', $item['filterLabel'] );
			$filters = array_map( 'trim', $filters );
		} else {
			$filters = [];
		}

		$filters_array = [];

		foreach ( $filters as $key => $value ) {
			$filters_array[ $this->clean( $value ) ] = $value;
		}

		return $filters_array;
	}

	/**
	 * Get Filter values array.
	 *
	 * Returns the Filter array of objects.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function get_filter_values() {
		$settings = $this->settings;

		$filters = array();

		if ( isset( $settings['items'] ) && ! empty( $settings['items'] ) ) {

			foreach ( $settings['items'] as $key => $value ) {

				$filter_keys = $this->get_filter_keys( $value );

				if ( ! empty( $filter_keys ) ) {

					$filters = array_unique( array_merge( $filters, $filter_keys ) );
				}
			}
		}

		return $filters;
	}

	/**
	 * Render Video Gallery Filters
	 *
	 * @since 1.0.0
	 * @return void
	 */
	protected function render_filters() {
		$settings = $this->settings;

		if ( isset( $settings['layout'] ) && 'grid' === $settings['layout'] ) {
			if ( $this->is_filters_enabled() ) {
				$all_text = ( isset( $settings['allFilterLabel'] ) ) ? $settings['allFilterLabel'] : esc_html__( 'All', 'max-addons' );
				$gallery  = $this->get_filter_values();
				$order    = isset( $settings['filtersOrder'] ) ? $settings['filtersOrder'] : '';

				if ( $order === 'asc' ) {
					asort( $gallery );
				} elseif ( $order === 'desc' ) {
					arsort( $gallery );
				}
				?>
				<ul class="mab-gallery-filters bricks-isotope-filters">
					<li class="mab-gallery-filter active" data-filter="*" data-gallery-index="all">
						<?php echo esc_attr( $all_text ); ?>
					</li>
					<?php
					foreach ( $gallery as $index => $item ) {
						$filter_label = $item;
						if ( $item ) { ?>
						<li class="mab-gallery-filter" data-filter="<?php echo '.' . esc_attr( $index ); ?>" data-gallery-index="<?php echo esc_attr( $index ); ?>">
							<?php echo esc_attr( $filter_label ); ?>
						</li>
						<?php } ?>
					<?php } ?>
						</ul>
				<?php
			}
		}
	}

	/**
	 * Provider match masks.
	 *
	 * Holds a list of supported providers with their URL structure in a regex format.
	 *
	 */
	private static $provider_match_masks = [
		'youtube'     => '/^.*(?:youtu\.be\/|youtube(?:-nocookie)?\.com\/(?:(?:watch)?\?(?:.*&)?vi?=|(?:embed|v|vi|user)\/))([^\?&\"\'>]+)/',
		'vimeo'       => '/^.*vimeo\.com\/(?:[a-z]*\/)*([‌​0-9]{6,11})[?]?.*/',
		'dailymotion' => '/^.*dailymotion.com\/(?:video|hub)\/([^_]+)[^#]*(#video=([^_&]+))?/',
	];

	/**
	 * Embed patterns.
	 *
	 * Holds a list of supported providers with their embed patters.
	 *
	 */
	private static $embed_patterns = [
		'youtube'     => 'https://www.youtube{NO_COOKIE}.com/embed/{VIDEO_ID}?feature=oembed',
		'vimeo'       => 'https://player.vimeo.com/video/{VIDEO_ID}#t={TIME}',
		'dailymotion' => 'https://dailymotion.com/embed/video/{VIDEO_ID}',
	];

	/**
	 * Get video properties.
	 *
	 * Retrieve the video properties for a given video URL.
	 *
	 */
	public static function get_video_properties( $video_url ) {
		foreach ( self::$provider_match_masks as $provider => $match_mask ) {
			preg_match( $match_mask, $video_url, $matches );

			if ( $matches ) {
				return [
					'provider' => $provider,
					'video_id' => $matches[1],
				];
			}
		}

		return null;
	}

	/**
	 * Get embed URL.
	 *
	 * Retrieve the embed URL for a given video.
	 *
	 */
	public static function get_embed_url( $video_url, array $embed_url_params = [], array $options = [] ) {
		$video_properties = self::get_video_properties( $video_url );

		if ( ! $video_properties ) {
			return null;
		}

		$embed_pattern = self::$embed_patterns[ $video_properties['provider'] ];

		$replacements = [
			'{VIDEO_ID}' => $video_properties['video_id'],
		];

		if ( 'youtube' === $video_properties['provider'] ) {
			$replacements['{NO_COOKIE}'] = ! empty( $options['privacy'] ) ? '-nocookie' : '';
		} elseif ( 'vimeo' === $video_properties['provider'] ) {
			$time_text = '';

			if ( ! empty( $options['start'] ) ) {
				$time_text = date( 'H\hi\ms\s', $options['start'] ); // PHPCS:Ignore WordPress.DateTime.RestrictedFunctions.date_date
			}

			$replacements['{TIME}'] = $time_text;
		}

		$embed_pattern = str_replace( array_keys( $replacements ), $replacements, $embed_pattern );

		return add_query_arg( $embed_url_params, $embed_pattern );
	}

	/**
	 * @param bool $from_media
	 *
	 * @return string
	 * @since 1.0.0
	 * @access private
	 */
	protected function get_hosted_video_url( $item ) {
		if ( isset( $item['fileUrl'] ) || isset( $item['mediaUrl'] ) ) {
			if ( isset( $item['fileUrl'] ) ) {
				$video_url = $item['fileUrl'];
			} else {
				$video_url = $item['mediaUrl']['url'];
			}
		}

		if ( empty( $video_url ) ) {
			return '';
		}

		if ( isset( $item['start'] ) || isset( $item['end'] ) ) {
			$video_url .= '#t=';
		}

		if ( isset( $item['start'] ) ) {
			$video_url .= $item['start'];
		}

		if ( isset( $item['end'] ) ) {
			$video_url .= ',' . $item['end'];
		}

		return $video_url;
	}

	/**
	 * @since 1.0.0
	 * @access private
	 */
	protected function get_hosted_params( $item ) {
		$settings = $this->get_settings_for_display();

		$video_params = [];

		$video_params['controls'] = '';

		$video_params['controlsList'] = 'nodownload';

		if ( $settings['mute'] ) {
			$video_params['muted'] = 'muted';
		}

		if ( 'yes' === $item['custom_thumbnail'] ) {
			if ( $item['custom_image']['url'] ) {
				$video_params['poster'] = $item['custom_image']['url'];
			}
		}

		return $video_params;
	}

	/**
	 * Get the image by control key
	 *
	 * Similar to get_normalized_image_settings() in the image element.
	 *
	 * Might be a fix image, a dynamic data tag or external URL.
	 *
	 * @since 1.6.1
	 *
	 * @return array
	 */
	public function get_video_image_by_key( $settings, $control_key = '' ) {
		if ( empty( $control_key ) ) {
			return [];
		}

		$image = isset( $settings[ $control_key ] ) ? $settings[ $control_key ] : false;

		if ( ! $image ) {
			return [];
		}

		// STEP: Set image size
		$image['size'] = isset( $image['size'] ) && ! empty( $image['size'] ) ? $image['size'] : BRICKS_DEFAULT_IMAGE_SIZE;

		// STEP: Image ID or URL from dynamic data
		if ( ! empty( $image['useDynamicData'] ) ) {
			$dynamic_image = $this->render_dynamic_data_tag( $image['useDynamicData'], 'image', [ 'size' => $image['size'] ] );

			if ( ! empty( $dynamic_image[0] ) ) {
				if ( is_numeric( $dynamic_image[0] ) ) {
					// Use the image ID to populate and set $dynamic_image['url']
					$image['id']  = $dynamic_image[0];
					$image['url'] = wp_get_attachment_image_url( $image['id'], $image['size'] );
				} else {
					$image['url'] = $dynamic_image[0];
				}
			} else {
				return [];
			}
		}

		// Set image ID
		$image['id'] = empty( $image['id'] ) ? 0 : $image['id'];

		// Set image URL
		if ( ! isset( $image['url'] ) ) {
			$image['url'] = ! empty( $image['id'] ) ? wp_get_attachment_image_url( $image['id'], $image['size'] ) : false;
		}

		return $image;
	}

	/**
	 * Returns Video Thumbnail.
	 *
	 * @param  array $item       Video.
	 * @param  mixed $thumb_size Thumbnail size.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function get_video_thumbnail( $item, $thumb_size = '' ) {
		$thumb_url    = '';
		$custom_thumb = '';
		$video_id     = $this->get_video_id( $item );

		if ( isset( $item['customThumbnail'] ) ) {
			$video_thumb = $this->get_video_image_by_key( $item, 'customThumbnailImage' );

			if ( ! empty( $video_thumb['url'] ) ) {
				$custom_thumb = $video_thumb['url'];
			}
		}

		if ( 'media' === $item['videoType'] || 'file' === $item['videoType'] ) {
			if ( $custom_thumb ) {
				$thumb_url = $custom_thumb;
			}
		}

		if ( $custom_thumb ) {
			$thumb_url = $custom_thumb;
		} elseif ( 'youtube' === $item['videoType'] ) {
			if ( $video_id ) {
				$thumb_url = 'https://i.ytimg.com/vi/' . $video_id . '/' . $thumb_size . '.jpg';
			}
		} elseif ( 'vimeo' === $item['videoType'] ) {
			if ( $video_id ) {
				$response = wp_remote_get( "https://vimeo.com/api/v2/video/$video_id.php" );

				if ( is_wp_error( $response ) ) {
					return;
				}
				$vimeo = maybe_unserialize( $response['body'] );
				$thumb_url = $vimeo[0]['thumbnail_large'];
			}
		}

		return $thumb_url;
	}

	/**
	 * Returns Video ID.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function get_video_id( $item ) {
		$video_id = '';

		if ( 'youtube' === $item['videoType'] ) {
			$url = $this->render_dynamic_data( $item['youTubeUrl'] );

			if ( preg_match( '#(?<=v=|v\/|vi=|vi\/|youtu.be\/)[a-zA-Z0-9_-]{11}#', $url, $matches ) ) {
				$video_id = $matches[0];
			}
		} elseif ( 'vimeo' === $item['videoType'] ) {
			$url = $this->render_dynamic_data( $item['vimeoUrl'] );

			$video_id = preg_replace( '/[^\/]+[^0-9]|(\/)/', '', rtrim( $url, '/' ) );

		}

		return $video_id;
	}

	/**
	 * Get embed params.
	 *
	 * Retrieve video widget embed parameters.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Video embed parameters.
	 */
	public function get_embed_params( $item ) {
		$settings = $this->settings;

		$params = array();

		$params_dictionary = array();

		if ( 'youtube' === $item['videoType'] ) {

			$params_dictionary = array(
				'mute',
			);

			$params['autoplay'] = 1;

			$params['wmode'] = 'opaque';
		} elseif ( 'vimeo' === $item['videoType'] ) {

			$params_dictionary = array(
				'mute' => 'muted',
			);

			$params['autopause'] = '0';
			$params['autoplay']  = '1';
		}

		foreach ( $params_dictionary as $key => $param_name ) {
			$setting_name = $param_name;

			if ( is_string( $key ) ) {
				$setting_name = $key;
			}

			$setting_value = isset( $settings[ $setting_name ] ) ? '1' : '0';

			$params[ $param_name ] = $setting_value;
		}

		return $params;
	}

	public function is_filters_enabled() {
		$settings = $this->settings;

		if (
			isset( $settings['layout'] ) && 'grid' === $settings['layout']
			&& isset( $settings['filtersEnable'] )
			&& isset( $settings['source'] )
			&& 'yt-playlist' !== $settings['source']
			&& 'yt-channel' !== $settings['source']
			) {
			return true;
		}

		return false;
	}

	/**
	 * Helper function to parse the settings when videoType = meta
	 *
	 * @return array
	 */
	public function get_normalized_video_settings( $item = [] ) {
		if ( empty( $item['videoType'] ) ) {
			return $item;
		}

		if ( $item['videoType'] === 'youtube' ) {

			if ( ! empty( $item['youTubeUrl'] ) ) {
				$item['youTubeUrl'] = $this->render_dynamic_data( $item['youTubeUrl'] );
			}

			/* if ( ! empty( $item['iframeTitle'] ) ) {
				$item['iframeTitle'] = $this->render_dynamic_data( $item['iframeTitle'] );
			} */

			return $item;
		}

		if ( $item['videoType'] === 'vimeo' ) {

			if ( ! empty( $item['vimeoUrl'] ) ) {
				$item['vimeoUrl'] = $this->render_dynamic_data( $item['vimeoUrl'] );
			}

			/* if ( ! empty( $item['iframeTitle'] ) ) {
				$item['iframeTitle'] = $this->render_dynamic_data( $item['iframeTitle'] );
			}

			if ( ! empty( $item['vimeoHash'] ) ) {
				$item['vimeoHash'] = $this->render_dynamic_data( $item['vimeoHash'] );
			} */

			return $item;
		}

		// Check 'file' and 'meta' videoType for dynamic data
		$dynamic_data = false;

		if ( $item['videoType'] === 'file' && ! empty( $item['fileUrl'] ) && strpos( $item['fileUrl'], '{' ) === 0 ) {
			$dynamic_data = $item['fileUrl'];
		}

		if ( $item['videoType'] === 'meta' && ! empty( $item['useDynamicData'] ) ) {
			$dynamic_data = $item['useDynamicData'];
		}

		if ( ! $dynamic_data ) {
			return $item;
		}

		$meta_video_url = $this->render_dynamic_data_tag( $dynamic_data, 'link' );

		if ( empty( $meta_video_url ) ) {
			return $item;
		}

		$video_properties = self::get_video_properties( $meta_video_url );

		$video_type = ( $video_properties ) ? $video_properties['provider'] : 'file';

		// Is YouTube video
		if ( 'youtube' === $video_type ) {
			$item['youTubeUrl'] = $meta_video_url;
			$item['videoType'] = 'youtube';
		}

		// Is Vimeo video
		elseif ( 'vimeo' === $video_type ) {
			$item['vimeoUrl']   = $meta_video_url;
			$item['videoType'] = 'vimeo';

		} else {
			// Url of a video file (either hosted or external)
			$item['fileUrl']   = $meta_video_url;
			$item['videoType'] = 'file';
		}

		// Later the settings are used to control the video and the custom field should not be present
		unset( $item['useDynamicData'] );

		return $item;
	}

	/**
	 * Render custom content output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @access public
	 */
	public function render_repeater_item( $item ) {
		$settings = $this->settings;
		$index    = $this->loop_index;
		$output   = '';
		$item     = $this->get_normalized_video_settings( $item );

		$layout   = isset( $settings['layout'] ) ? $settings['layout'] : 'grid';
		$item_tag = 'li';

		if ( 'carousel' == $layout ) {
			$item_tag = 'div';
		}

		$video_container_key = 'video_container_' . $index;
		$item_key = 'item_wrap_video_' . $index;
		$this->set_attribute( $video_container_key, 'class', 'mab-video-container' );

		if ( 'carousel' === $layout ) {
			$this->set_attribute( $item_key, 'class', 'swiper-slide' );
		} else {
			$tags = $this->get_filter_keys( $item );

			$this->set_attribute( $item_key, 'class', 'bricks-layout-item' );

			if ( $this->is_filters_enabled() ) {
				$this->set_attribute( $item_key, 'class', array_keys( $tags ) );
				$this->set_attribute( $item_key, 'data-category', array_keys( $tags ) );
			}
		}
		
		$output .= "<{$item_tag} {$this->render_attributes( $item_key )}>";
			$output .= '<div class="mab-grid-item mab-video-gallery-item">';
				$output .= '<div class="mab-video-player-wrapper">';
					$video_url_src = '';
					$thumb_size    = '';

					if ( 'youtube' === $item['videoType'] ) {
						$video_url_src = isset( $item['youTubeUrl'] ) ? $item['youTubeUrl'] : '';
						$thumb_size    = isset( $item['thumbnailSize'] ) ? $item['thumbnailSize'] : 'hqdefault';
					} elseif ( 'vimeo' === $item['videoType'] ) {
						$video_url_src = isset( $item['vimeoUrl'] ) ? $item['vimeoUrl'] : '';
					}

					$video_play_key = 'play_video_' . $index;
					$this->set_attribute( $video_play_key, 'class', 'mab-video-play' );

					if ( isset( $settings['clickAction'] ) && 'inline' === $settings['clickAction'] ) {
						$embed_params = $this->get_embed_params( $item );

						if ( 'media' === $item['videoType'] || 'file' === $item['videoType'] ) {
							$video_url = $this->get_hosted_video_url( $item );
						} else {
							$video_url = $this->get_embed_url( $video_url_src, $embed_params, array() );
						}

						$this->set_attribute( $video_play_key, 'data-src', $video_url );

						$this->set_attribute( $video_play_key, 'href', $video_url );
					} else {
						if ( 'media' === $item['videoType'] || 'file' === $item['videoType'] ) {
							$video_url = $this->get_hosted_video_url( $item );
						} else {
							$video_url = $video_url_src;
						}

						// Lightbox: Photoswipe required width & height
						$lightbox_width  = \Bricks\Theme_Styles::$active_settings['general']['lightboxWidth'] ?? 1280;
						$lightbox_height = \Bricks\Theme_Styles::$active_settings['general']['lightboxHeight'] ?? 720;

						$this->set_attribute( $video_play_key, 'class', 'bricks-lightbox' );
						$this->set_attribute( $video_play_key, 'data-pswp-width', $lightbox_width );
						$this->set_attribute( $video_play_key, 'data-pswp-height', $lightbox_height );
						$this->set_attribute( $video_play_key, 'data-pswp-video-url', $video_url );
					}

					$output .= "<div {$this->render_attributes( $video_container_key )}>";
						$output .= "<a {$this->render_attributes( $video_play_key )}>";
							$filter_label = isset( $item['filterLabel'] ) ? $item['filterLabel'] : '';
							$output .= '<div class="mab-video-player">';
								if ( 'media' === $item['videoType'] || 'file' === $item['videoType'] ) {
									$video_thumb = $this->get_video_thumbnail( $item, $thumb_size );
									$video_url = $this->get_hosted_video_url( $item );

									if ( $video_thumb ) {
										$output .= "<img class='mab-video-thumb' src='" . esc_url( $video_thumb ) . "' alt='" . esc_attr( $filter_label ) . "'>";
									} else {
										$output .= '<video class="mab-hosted-video" src="' . esc_url( $video_url ) . '" preload="' . esc_attr( $settings['preload'] ) . '"></video>';
									}
								} else {
									$video_thumb = $this->get_video_thumbnail( $item, $thumb_size );

									if ( $video_thumb ) {
										$output .= "<img class='mab-video-thumb' src='" . esc_url( $video_thumb ) . "' alt='" . esc_attr( $filter_label ) . "'>";
									}
								}

								// Icon
								if ( isset( $settings['playIconType'] ) && $settings['playIconType'] == 'icon' ) {
									$icon_html = isset( $settings['playIcon'] ) ? self::render_icon( $settings['playIcon'] ) : false;
									if ( $icon_html ) {
										$output .= "<span class='mab-video-play-icon'>";
										$output .= $icon_html;
										$output .= "</span>";
									}
								}
								// Image
								elseif ( isset( $settings['playIconType'] ) && $settings['playIconType'] == 'image' ) {
									$play_icon_image = $this->get_video_image_by_key( $settings, 'playIconImage' );

									if ( ! empty( $play_icon_image['url'] ) ) {
										$this->set_attribute( 'play-icon', 'class', 'mab-video-play-icon' );

										$output .= "<span {$this->render_attributes('play-icon')}>";
										$output .= "<img src='" . esc_url( $play_icon_image['url'] ) . "'>";
										$output .= "</span>";
									}
								}
							$output .= "</div>";
						$output .= "</a>";
					$output .= "</div>";
				$output .= "</div>";

				if ( isset( $item['videoTitle'] ) || isset( $item['videoDescription'] ) ) {
					$title_tag = isset( $settings['videoTitleTag'] ) ? esc_attr( $settings['videoTitleTag'] ) : 'h4';
					$output .= '<div class="mab-video-content">';
						if ( isset( $item['videoTitle'] ) ) {
							$output .= '<' . esc_attr( $title_tag ) . ' class="mab-video-title">';
							$output .= wp_kses_post( $item['videoTitle'] );
							$output .= '</' . esc_attr( $title_tag ) . '>';
						}

						if ( isset( $item['videoDescription'] ) ) {
							$output .= "<div class='mab-video-description'>";
							$output .= wp_kses_post( $item['videoDescription'] );
							$output .= "</div>";
						}
					$output .= "</div>";
				}

			$output .= "</div>";
		$output .= "</{$item_tag}>";

		$this->loop_index++;

		return $output;
	}

	/**
	 * Render Videos
	 *
	 * @access protected
	 */
	protected function render_videos() {
		$settings  = $this->settings;
		$is_editor = ! $this->is_frontend;
		$gallery   = array();
		$output    = '';
		$source    = isset( $settings['source'] ) ? $settings['source'] : 'custom';

		if ( 'yt-playlist' === $source || 'yt-channel' === $source ) {
			$items = $this->get_youtube_videos();

			if ( is_wp_error( $items ) ) {
				$error_message = $items->get_error_message();
	
				if ( $is_editor ) {
					return $this->render_element_placeholder(
						[
							'title' => $error_message,
						]
					);
				}
				return;
			}

			unset( $settings['hasLoop'] );
		} else {
			$items = $settings['items'];
		}

		if ( isset( $settings['order'] ) && 'random' === $settings['order'] ) {

			$keys = array_keys( $items );
			shuffle( $keys );

			foreach ( $keys as $key ) {
				$gallery[ $key ] = $items[ $key ];
			}
		} else {
			$gallery = $items;
		}

		if ( isset( $settings['hasLoop'] ) ) {
			$query = new \Bricks\Query( [
				'id'       => $this->id,
				'settings' => $settings,
			] );

			$item = $gallery[0];
			$output .= $query->render( [ $this, 'render_repeater_item' ], compact( 'item' ) );

			// We need to destroy the Query to explicitly remove it from the global store
			$query->destroy();
			unset( $query );
		} else {
			foreach ( $gallery as $index => $item ) {
				if ( ! empty( $item ) ) {
					$output .= self::render_repeater_item( $item );
				}
			}
		}

		echo $output;

		if ( $this->is_filters_enabled() ) {
			echo '<li class="bricks-isotope-sizer"></li>';
			echo '<li class="bricks-gutter-sizer"></li>';
		}
	}

	/**
	 * Gets expire time of transient.
	 *
	 * @since 1.9.0
	 * @param array $settings The settings array.
	 * @return the reviews transient expire time.
	 * @access public
	 */
	public function get_transient_expire( $settings ) {
		$expire_value = isset( $settings['videosRefreshTime'] ) ? $settings['videosRefreshTime'] : 'day';
		$expire_time  = 24 * HOUR_IN_SECONDS;

		if ( 'hour' === $expire_value ) {
			$expire_time = 60 * MINUTE_IN_SECONDS;
		} elseif ( 'week' === $expire_value ) {
			$expire_time = 7 * DAY_IN_SECONDS;
		} elseif ( 'month' === $expire_value ) {
			$expire_time = 30 * DAY_IN_SECONDS;
		} elseif ( 'year' === $expire_value ) {
			$expire_time = 365 * DAY_IN_SECONDS;
		}

		return $expire_time;
	}

	/**
	 * Get YouTube API from Max Addons options.
	 *
	 * @since 1.9.0
	 * @return string
	 */
	public function get_youtube_api() {
		return \MaxAddonsPro\Classes\Admin_Settings::get_option( 'mab_youtube_api_key' );
	}

	public function get_youtube_api_data() {
		$settings = $this->settings;

		$api_key = $this->get_youtube_api();
		$source  = ( 'yt-channel' === $settings['source'] ) ? 'channel' : 'playlist';

		if ( empty( $api_key ) ) {
			return new \WP_Error( 'missing_api_key', sprintf( __( 'To display videos from YouTube %s, you need to setup API key.', 'max-addons' ), $source ) );
		}

		if ( 'channel' === $source ) {
			$channel_id = ! empty( $settings['ytChannelId'] ) ? $this->render_dynamic_data( $settings['ytChannelId'] ) : '';
	
			if ( empty( $channel_id ) ) {
				return new \WP_Error( 'missing_channel_id', __( 'To display videos from YouTube channel, you need to provide valid Channel ID.', 'max-addons' ) );
			}
		} else {
			$playlist_id = ! empty( $settings['ytPlaylistId'] ) ? $this->render_dynamic_data( $settings['ytPlaylistId'] ) : '';

			if ( empty( $playlist_id ) ) {
				return new \WP_Error( 'missing_plylist_id', __( 'To display videos from YouTube playlist, you need to provide valid Playlist ID.', 'max-addons' ) );
			}
		}

		$api_args = array(
			'method'      => 'GET',
			'timeout'     => 60,
			'httpversion' => '1.0',
			'sslverify'   => false,
		);

		if ( 'yt-channel' === $settings['source'] ) {
			$url = add_query_arg(
				array(
					'part'       => 'snippet,id',
					'order'      => 'date',
					'maxResults' => 50,
					'key'        => $api_key,
					'channelId'  => $channel_id,
				),
				'https://www.googleapis.com/youtube/v3/search'
			);
		} else {
			$url = add_query_arg(
				array(
					'part'       => 'snippet',
					'maxResults' => 50,
					'key'        => $api_key,
					'playlistId' => $playlist_id,
				),
				'https://www.googleapis.com/youtube/v3/playlistItems'
			);
		}

		$response = wp_remote_post(
			esc_url_raw( $url ),
			$api_args
		);

		if ( ! is_wp_error( $response ) ) {
			$body = json_decode( wp_remote_retrieve_body( $response ) );
			if ( isset( $body->error_message ) && ! empty( $body->error_message ) ) {
				$status = isset( $body->status ) ? $body->status : $source . '_api_error';
				return new \WP_Error( $status, $body->error_message );
			}
		}

		return $response;
	}

	public function get_youtube_videos_data() {
		$settings = $this->settings;

		$response = array(
			'data'	=> array(),
			'error' => false,
		);

		$transient_name = '';

		if ( 'yt-channel' === $settings['source'] ) {
			if ( isset( $settings['ytChannelId'] ) ) {
				$transient_name = 'mab_videos_' . $this->render_dynamic_data( $settings['ytChannelId'] );
			}
		} else {
			if ( isset( $settings['ytPlaylistId'] ) ) {
				$transient_name = 'mab_videos_' . $this->render_dynamic_data( $settings['ytPlaylistId'] );
			}
		}

		$response['data'] = get_transient( $transient_name );

		if ( empty( $response['data'] ) ) {
			$api_data = $this->get_youtube_api_data();

			if ( is_wp_error( $api_data ) ) {
					
				$response['error'] = $api_data;

			} else {
				if ( 200 === wp_remote_retrieve_response_code( $api_data ) ) {
					$data = json_decode( wp_remote_retrieve_body( $api_data ), false );

					if ( $data->pageInfo->totalResults ) {
						$response['data'] = array(
							'video' => $data->items
						);

						set_transient( $transient_name, $response['data'], $this->get_transient_expire( $settings ) );

						$response['error'] = false;
					} else {
						$response['error'] = __( 'This playlist doesn\'t have any videos.', 'max-addons' );
					}
				}
			}
		}

		return $response;
	}

	public function get_youtube_videos() {
		$settings = $this->settings;

		$videos_data = $this->get_youtube_videos_data();
		
		if ( is_wp_error( $videos_data['error'] ) ) {
			return $videos_data['error'];
		}

		if ( empty( $videos_data['data'] ) ) {
			return;
		}

		$parsed_videos = array();
		$videos        = $videos_data['data']['video'];
		$videos_count  = isset( $settings['ytVideosCount'] ) ? $settings['ytVideosCount'] : 9;
		$videos        = array_slice( $videos, 0, $videos_count );

		foreach ( $videos as $video ) {
			$_video = array();

			if ( 'yt-channel' === $settings['source'] ) {
				$video_id = isset( $video->id->videoId ) ? $video->id->videoId : '';
			} elseif ( 'yt-playlist' === $settings['source'] ) {
				$video_id = $video->snippet->resourceId->videoId;
			}

			if ( $video_id ) {
				$_video['source']           = 'youtube';
				$_video['videoType']        = 'youtube';
				$_video['videoId']          = $video_id;

				if ( isset( $settings['showTitle'] ) ) {
					$_video['videoTitle']       = $video->snippet->title;
				}

				if ( isset( $settings['showDescription'] ) ) {
					$_video['videoDescription'] = $video->snippet->description;
				}
				$youtube_url                = 'https://www.youtube.com/watch?v=' . $video_id;
				$_video['youTubeUrl']       = $this->render_dynamic_data( $youtube_url );
			}
			
			array_push( $parsed_videos, $_video );
		}

		return $parsed_videos;
	}

	public function render() {
		$settings = $this->settings;

		// Element placeholder
		if ( ! isset( $settings['items'] ) || empty( $settings['items'] ) ) {
			return $this->render_element_placeholder( [ 'title' => esc_html__( 'No video added.', 'max-addons' ) ] );
		}

		$layout       = isset( $settings['layout'] ) ? $settings['layout'] : 'grid';
		$click_action = isset( $settings['clickAction'] ) ? $settings['clickAction'] : 'inline';

		$classes = array(
			'mab-video-gallery',
		);

		$this->set_attribute( '_root', 'class', 'mab-video-gallery-' . $layout );

		if ( 'grid' === $layout ) {
			$classes[] = 'bricks-layout-wrapper';

			if ( $this->is_filters_enabled() ) {
				$classes[] = 'isotope';
			}
		}

		if ( 'carousel' === $layout ) {
			$this->tag = 'div';

			$this->slider_settings();

			$classes[] = 'bricks-swiper-container';
		}

		$this->set_attribute( 'gallery', 'class', $classes );

		$this->set_attribute( 'gallery', 'data-action', $click_action );
		$this->set_attribute( 'gallery', 'data-layout', $layout );
		?>
		<div <?php echo wp_kses_post( $this->render_attributes( '_root' ) ); ?>>
			<?php $this->render_filters(); ?>
			<<?php echo $this->tag; ?> <?php echo wp_kses_post( $this->render_attributes( 'gallery' ) ); ?>>
				<?php if ( 'carousel' === $layout ) { ?>
					<div class="swiper-wrapper">
						<?php $this->render_videos(); ?>
					</div>
				<?php } else {
					$this->render_videos();
				} ?>
				<?php
				if ( 'carousel' === $layout ) {
					if ( isset( $settings['dots'] ) ) {
						echo '<div class="swiper-pagination"></div>';
					}

					if ( isset( $settings['arrows'] ) && isset( $settings['prevArrow']['icon'] ) ) {
						echo '<div class="swiper-button bricks-swiper-button-prev ' . esc_attr( $settings['prevArrow']['icon'] ) . '"></div>';
					}

					if ( isset( $settings['arrows'] ) && isset( $settings['nextArrow']['icon'] ) ) {
						echo '<div class="swiper-button bricks-swiper-button-next ' . esc_attr( $settings['nextArrow']['icon'] ) . '"></div>';
					}
				}
				?>
			</<?php echo $this->tag; ?>>
		</div>
		<?php
	}
}
