<?php
namespace MaxAddonsPro\Elements;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Timeline_Element extends \Bricks\Element {
	// Element properties
	public $category     = 'max-addons-elements'; // Use predefined element category 'general'
	public $name         = 'max-timeline'; // Make sure to prefix your elements
	public $icon         = 'ti-view-list-alt max-element'; // Themify icon font class
	public $css_selector = ''; // Default CSS selector
	public $loop_index   = 0;
	public $scripts      = [ 'mabTimeline' ]; // Script(s) run when element is rendered on frontend or updated in builder

	public function get_label() {
		return esc_html__( 'Timeline', 'max-addons' );
	}

	public function get_keywords() {
		return [ 'query', 'posts' ];
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		$layout = ! empty( $this->settings['layout'] ) ? $this->settings['layout'] : 'vertical';

		if ( $layout === 'horizontal' ) {
			wp_enqueue_script( 'bricks-swiper' );
		}

		wp_enqueue_style( 'mab-timeline' );
		wp_enqueue_script( 'mab-timeline' );
	}

	// Set builder control groups
	public function set_control_groups() {
		$this->control_groups['layoutStyle'] = [
			'title' => esc_html__( 'Layout', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['generalSettings'] = [
			'title'    => esc_html__( 'Settings', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'layout', '=', 'horizontal' ],
		];

		$this->control_groups['cards'] = [
			'title' => esc_html__( 'Cards', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['marker'] = [
			'title' => esc_html__( 'Marker', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['date'] = [
			'title' => esc_html__( 'Date', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['connector'] = [
			'title' => esc_html__( 'Connector', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['arrows'] = [
			'title'    => esc_html__( 'Arrows', 'max-addons' ),
			'tab'      => 'content',
			'required' => [
				[ 'layout', '=', 'horizontal' ],
				[ 'arrows', '=', true ],
			],
		];

		$this->control_groups['dots'] = [
			'title'    => esc_html__( 'Dots', 'max-addons' ),
			'tab'      => 'content',
			'required' => [
				[ 'layout', '=', 'horizontal' ],
				[ 'dots', '=', true ],
			],
		];

	}

	public function set_controls() {

		$this->set_timeline_controls();
		$this->set_layout_style_controls();
		$this->set_settings_controls();
		$this->set_cards_style_controls();
		$this->set_marker_style_controls();
		$this->set_dates_style_controls();
		$this->set_connector_style_controls();
		$this->set_arrow_style_controls();

		// Slider Controls
		// Group: Settings

		if ( method_exists( '\Bricks\Setup', "get_carousel_controls" ) ) {
			$carousel_controls = \Bricks\Setup::get_carousel_controls();
		} else {
			$carousel_controls = self::get_swiper_controls();
		}

		// Arrows

		$this->controls['arrowHeight']     = $carousel_controls['arrowHeight'];
		$this->controls['arrowWidth']      = $carousel_controls['arrowWidth'];
		$this->controls['arrowBackground'] = $carousel_controls['arrowBackground'];
		$this->controls['arrowBorder']     = $carousel_controls['arrowBorder'];
		$this->controls['arrowTypography'] = $carousel_controls['arrowTypography'];

		$this->controls['prevArrowSeparator'] = $carousel_controls['prevArrowSeparator'];
		$this->controls['prevArrow']          = $carousel_controls['prevArrow'];
		$this->controls['prevArrowTop']       = $carousel_controls['prevArrowTop'];
		$this->controls['prevArrowRight']     = $carousel_controls['prevArrowRight'];
		$this->controls['prevArrowBottom']    = $carousel_controls['prevArrowBottom'];
		$this->controls['prevArrowLeft']      = $carousel_controls['prevArrowLeft'];

		$this->controls['nextArrowSeparator'] = $carousel_controls['nextArrowSeparator'];
		$this->controls['nextArrow']          = $carousel_controls['nextArrow'];
		$this->controls['nextArrowTop']       = $carousel_controls['nextArrowTop'];
		$this->controls['nextArrowRight']     = $carousel_controls['nextArrowRight'];
		$this->controls['nextArrowBottom']    = $carousel_controls['nextArrowBottom'];
		$this->controls['nextArrowLeft']      = $carousel_controls['nextArrowLeft'];

		// Dots
		$this->controls['dotsHeight']      = $carousel_controls['dotsHeight'];
		$this->controls['dotsWidth']       = $carousel_controls['dotsWidth'];
		$this->controls['dotsTop']         = $carousel_controls['dotsTop'];
		$this->controls['dotsRight']       = $carousel_controls['dotsRight'];
		$this->controls['dotsBottom']      = $carousel_controls['dotsBottom'];
		$this->controls['dotsLeft']        = $carousel_controls['dotsLeft'];
		$this->controls['dotsBorder']      = $carousel_controls['dotsBorder'];
		$this->controls['dotsSpacing']     = $carousel_controls['dotsSpacing'];
		$this->controls['dotsColor']       = $carousel_controls['dotsColor'];
		$this->controls['dotsActiveColor'] = $carousel_controls['dotsActiveColor'];
	}

	// Timeline
	public function set_timeline_controls() {
		$this->controls['items'] = [
			'tab'           => 'content',
			'label'         => esc_html__( 'Timeline Items', 'max-addons' ),
			'type'          => 'repeater',
			'checkLoop'     => true,
			'placeholder'   => esc_html__( 'Item', 'max-addons' ),
			'titleProperty' => 'timelineTitle',
			'fields'        => [
				'timelineDate'           => [
					'label'          => esc_html__( 'Date', 'max-addons' ),
					'type'           => 'text',
					'hasDynamicData' => 'text',
				],

				'timelineTitle'          => [
					'label'          => esc_html__( 'Title', 'max-addons' ),
					'type'           => 'text',
					'hasDynamicData' => 'text',
				],

				'timelineContent'        => [
					'label' => esc_html__( 'Content', 'max-addons' ),
					'type'  => 'editor',
				],

				'timelineItemLink'       => [
					'label'   => esc_html__( 'Link', 'max-addons' ),
					'type'    => 'link',
					'exclude' => [
						'lightboxImage',
						'lightboxVideo',
					],
				],

				'timelineImage'          => [
					'label' => esc_html__( 'Show Image', 'max-addons' ),
					'type'  => 'checkbox',
				],

				'image'                  => [
					'label'    => esc_html__( 'Image', 'max-addons' ),
					'type'     => 'image',
					'required' => [ 'timelineImage', '!=', '' ],
				],

				'customStyle'            => [
					'label' => esc_html__( 'Custom Style', 'max-addons' ),
					'type'  => 'checkbox',
				],

				'singleMarkerColor'      => [
					'type'     => 'color',
					'label'    => esc_html__( 'Marker Color', 'max-addons' ),
					'css'      => [
						[
							'property' => 'color',
							'selector' => '.mab-timeline-marker',
						],
						[
							'property' => 'fill',
							'selector' => '.mab-timeline-marker svg',
						],
					],
					'inline'   => true,
					'small'    => true,
					'required' => [ 'customStyle', '=', true ],
				],

				'singleMarkerBgColor'    => [
					'type'     => 'color',
					'label'    => esc_html__( 'Marker Background Color', 'max-addons' ),
					'css'      => [
						[
							'property' => 'background-color',
							'selector' => '.mab-timeline-marker',
						],
					],
					'inline'   => true,
					'small'    => true,
					'required' => [ 'customStyle', '=', true ],
				],

				'singleCardBgColor'      => [
					'type'     => 'color',
					'label'    => esc_html__( 'Card Background Color', 'max-addons' ),
					'css'      => [
						[
							'property' => 'background-color',
							'selector' => '.mab-timeline-card',
						],
						[
							'property' => 'color',
							'selector' => '.mab-timeline-arrow',
						],
					],
					'inline'   => true,
					'small'    => true,
					'required' => [ 'customStyle', '=', true ],
				],

				'singleTitleColor'       => [
					'type'     => 'color',
					'label'    => esc_html__( 'Title Color', 'max-addons' ),
					'css'      => [
						[
							'property' => 'color',
							'selector' => '.mab-timeline-card-title',
						],
					],
					'inline'   => true,
					'small'    => true,
					'required' => [ 'customStyle', '=', true ],
				],

				'singleContentColor'     => [
					'type'     => 'color',
					'label'    => esc_html__( 'Content Color', 'max-addons' ),
					'css'      => [
						[
							'property' => 'color',
							'selector' => '.mab-timeline-card-content',
						],
					],
					'inline'   => true,
					'small'    => true,
					'required' => [ 'customStyle', '=', true ],
				],
			],
			'default'       => [
				[
					'timelineDate'     => esc_html__( '1 August 2021', 'max-addons' ),
					'timelineTitle'    => esc_html__( 'Timeline Item 1', 'max-addons' ),
					'timelineContent'  => esc_html__( 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo. ', 'max-addons' ),
					'markerIconSingle' => [
						'library' => 'themify',
						'icon'    => 'ti-calendar',
					],
				],
				[
					'timelineDate'     => esc_html__( '1 September 2021', 'max-addons' ),
					'timelineTitle'    => esc_html__( 'Timeline Item 2', 'max-addons' ),
					'timelineContent'  => esc_html__( 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo. ', 'max-addons' ),
					'markerIconSingle' => [
						'library' => 'themify',
						'icon'    => 'ti-calendar',
					],
				],
				[
					'timelineDate'     => esc_html__( '1 October 2021', 'max-addons' ),
					'timelineTitle'    => esc_html__( 'Timeline Item 3', 'max-addons' ),
					'timelineContent'  => esc_html__( 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo. ', 'max-addons' ),
					'markerIconSingle' => [
						'library' => 'themify',
						'icon'    => 'ti-calendar',
					],
				],
				[
					'timelineDate'     => esc_html__( '1 November 2021', 'max-addons' ),
					'timelineTitle'    => esc_html__( 'Timeline Item 4', 'max-addons' ),
					'timelineContent'  => esc_html__( 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo. ', 'max-addons' ),
					'markerIconSingle' => [
						'library' => 'themify',
						'icon'    => 'ti-calendar',
					],
				],
			],
		];

		$this->controls = array_replace_recursive( $this->controls, $this->get_loop_builder_controls() );
	}

	// Layout Style
	public function set_layout_style_controls() {
		$this->controls['layout'] = [
			'tab'       => 'content',
			'group'     => 'layoutStyle',
			'label'     => esc_html__( 'Layout', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'horizontal' => esc_html__( 'Horizontal', 'max-addons' ),
				'vertical'   => esc_html__( 'Vertical', 'max-addons' ),
			],
			'inline'    => true,
			'clearable' => false,
			'default'   => 'vertical',
		];

		$this->controls['direction'] = [
			'tab'       => 'content',
			'group'     => 'layoutStyle',
			'label'     => esc_html__( 'Direction', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'left'   => esc_html__( 'Left', 'max-addons' ),
				'center' => esc_html__( 'Center', 'max-addons' ),
				'right'  => esc_html__( 'Right', 'max-addons' ),
			],
			'default'   => 'center',
			'inline'    => true,
			'clearable' => false,
			'required'  => [ 'layout', '=', 'vertical' ],
		];

		$this->controls['cardsArrowsAlignment'] = [
			'tab'       => 'content',
			'group'     => 'layoutStyle',
			'label'     => esc_html__( 'Arrows Alignment', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'top'    => esc_html__( 'Top', 'max-addons' ),
				'middle' => esc_html__( 'Middle', 'max-addons' ),
				'bottom' => esc_html__( 'Bottom', 'max-addons' ),
			],
			'default'   => 'top',
			'inline'    => true,
			'clearable' => false,
			'required'  => [ 'layout', '=', 'vertical' ],
		];

		$this->controls['horizontalSpacing'] = [
			'tab'   => 'content',
			'group' => 'layoutStyle',
			'label' => esc_html__( 'Horizontal Spacing', 'max-addons' ),
			'type'  => 'number',
			'units' => true,
			'css'   => [
				[
					'property' => 'margin-right',
					'selector' => '.mab-timeline-vertical.mab-timeline-left .mab-timeline-marker-wrapper, .mab-timeline-vertical.mab-timeline-center .mab-timeline-marker-wrapper',
				],
				[
					'property' => 'margin-left',
					'selector' => '.mab-timeline-vertical.mab-timeline-right .mab-timeline-marker-wrapper, .mab-timeline-vertical.mab-timeline-center .mab-timeline-marker-wrapper',
				],
				[
					'property' => 'margin-top',
					'selector' => '.mab-timeline-horizontal',
				],
				[
					'property' => 'margin-bottom',
					'selector' => '.mab-timeline-navigation .mab-timeline-card-date-wrapper',
				],
			],
		];

		$this->controls['verticalSpacing'] = [
			'tab'   => 'content',
			'group' => 'layoutStyle',
			'label' => esc_html__( 'Vertical Spacing', 'max-addons' ),
			'type'  => 'number',
			'units' => true,
			'css'   => [
				[
					'property' => 'margin-bottom',
					'selector' => '.mab-timeline-vertical .mab-timeline-item:not(:last-child)',
				],
				[
					'property' => 'padding-left',
					'selector' => '.mab-timeline-horizontal .mab-timeline-item',
				],
				[
					'property' => 'padding-right',
					'selector' => '.mab-timeline-horizontal .mab-timeline-item',
				],
				[
					'property' => 'margin-left',
					'selector' => '.mab-timeline-horizontal .mab-timeline-item',
					'invert'   => true,
				],
				[
					'property' => 'margin-right',
					'selector' => '.mab-timeline-horizontal .swiper-wrapper',
					'invert'   => true,
				],
			],
		];
	}

	// Set items controls
	public function set_settings_controls() {
		$this->controls['columns'] = [
			'tab'      => 'content',
			'group'    => 'generalSettings',
			'label'    => esc_html__( 'Columns', 'max-addons' ),
			'type'     => 'number',
			'min'      => 1,
			'max'      => 8,
			'step'     => 1,
			'inline'   => true,
			'default'  => 3,
			'required' => [ 'layout', '=', 'horizontal' ],
		];

		$this->controls['speed'] = [
			'tab'      => 'content',
			'group'    => 'generalSettings',
			'label'    => esc_html__( 'Slider Speed', 'max-addons' ),
			'type'     => 'number',
			'inline'   => true,
			'default'  => 600,
			'required' => [ 'layout', '=', [ 'horizontal' ] ],
		];

		$this->controls['arrows'] = [
			'tab'      => 'content',
			'group'    => 'generalSettings',
			'label'    => esc_html__( 'Arrows', 'max-addons' ),
			'type'     => 'checkbox',
			'reset'    => true,
			'default'  => true,
			'required' => [ 'layout', '=', [ 'horizontal' ] ],
		];

		$this->controls['dots'] = [
			'tab'      => 'content',
			'group'    => 'generalSettings',
			'label'    => esc_html__( 'Dots', 'max-addons' ),
			'type'     => 'checkbox',
			'reset'    => true,
			'required' => [ 'layout', '=', [ 'horizontal' ] ],
		];

		$this->controls['infinite'] = [
			'tab'      => 'content',
			'group'    => 'generalSettings',
			'label'    => esc_html__( 'Infinite Loop', 'max-addons' ),
			'type'     => 'checkbox',
			'reset'    => true,
			'default'  => true,
			'required' => [ 'layout', '=', [ 'horizontal' ] ],
		];

		$this->controls['centerMode'] = [
			'tab'      => 'content',
			'group'    => 'generalSettings',
			'label'    => esc_html__( 'Center Mode', 'max-addons' ),
			'type'     => 'checkbox',
			'reset'    => true,
			'required' => [
				[ 'layout', '=', [ 'horizontal' ] ],
				[ 'infinite', '=', true ],
			],
		];

		$this->controls['autoplay'] = [
			'tab'      => 'content',
			'group'    => 'generalSettings',
			'label'    => esc_html__( 'Autoplay', 'max-addons' ),
			'type'     => 'checkbox',
			'reset'    => true,
			'default'  => true,
			'required' => [ 'layout', '=', [ 'horizontal' ] ],
		];

		$this->controls['autoplaySpeed'] = [
			'tab'      => 'content',
			'group'    => 'generalSettings',
			'label'    => esc_html__( 'Autoplay Speed', 'max-addons' ),
			'type'     => 'number',
			'inline'   => true,
			'default'  => 3000,
			'required' => [
				[ 'layout', '=', [ 'horizontal' ] ],
				[ 'autoplay', '=', true ],
			],
		];

		$this->controls['pauseOnHover'] = [
			'tab'      => 'content',
			'group'    => 'generalSettings',
			'label'    => esc_html__( 'Pause on Hover', 'max-addons' ),
			'type'     => 'checkbox',
			'reset'    => true,
			'default'  => true,
			'required' => [
				[ 'layout', '=', [ 'horizontal' ] ],
				[ 'autoplay', '=', true ],
			],
		];
	}

	// Cards Style
	public function set_cards_style_controls() {

		$this->controls['cardArrow'] = [
			'tab'     => 'content',
			'group'   => 'cards',
			'label'   => esc_html__( 'Card Arrow', 'max-addons' ),
			'type'    => 'checkbox',
			'reset'   => true,
			'default' => true,
		];

		$this->controls['animateCards'] = [
			'tab'      => 'content',
			'group'    => 'cards',
			'label'    => esc_html__( 'Animate Cards', 'max-addons' ),
			'type'     => 'checkbox',
			'reset'    => true,
			'default'  => true,
			'required' => [ 'layout', '=', [ 'vertical' ] ],
		];

		$this->controls['cardsPadding'] = [
			'tab'   => 'content',
			'group' => 'cards',
			'label' => esc_html__( 'Cards Padding', 'max-addons' ),
			'type'  => 'spacing',
			'css'   => [
				[
					'property' => 'padding',
					'selector' => '.mab-timeline .mab-timeline-card',
				],
			],
		];

		$this->controls['cardsTextAlign'] = [
			'tab'     => 'content',
			'group'   => 'cards',
			'label'   => esc_html__( 'Text Align', 'max-addons' ),
			'type'    => 'text-align',
			'default' => 'left',
			'css'     => [
				[
					'property' => 'text-align',
					'selector' => '.mab-timeline-card',
				],
			],
			'inline'  => true,
		];

		$this->controls['cardsBackground'] = [
			'tab'    => 'content',
			'group'  => 'cards',
			'type'   => 'color',
			'label'  => esc_html__( 'Background', 'max-addons' ),
			'css'    => [
				[
					'property' => 'background-color',
					'selector' => '.mab-timeline .mab-timeline-card',
				],
				[
					'property' => 'color',
					'selector' => '.mab-timeline .mab-timeline-arrow',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['cardsBorder'] = [
			'tab'    => 'content',
			'group'  => 'cards',
			'label'  => esc_html__( 'Border', 'max-addons' ),
			'type'   => 'border',
			'css'    => [
				[
					'property' => 'border',
					'selector' => '.mab-timeline .mab-timeline-card',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['cardBoxShadow'] = [
			'tab'    => 'content',
			'group'  => 'cards',
			'label'  => esc_html__( 'Box Shadow', 'max-addons' ),
			'type'   => 'box-shadow',
			'css'    => [
				[
					'property' => 'box-shadow',
					'selector' => '.mab-timeline-card',
				],
			],
			'inline' => true,
		];

		$this->controls['imageSpacing'] = [
			'tab'     => 'content',
			'group'   => 'cards',
			'label'   => esc_html__( 'Image Spacing', 'max-addons' ),
			'type'    => 'number',
			'units'   => true,
			'default' => '20px',
			'css'     => [
				[
					'property' => 'margin-bottom',
					'selector' => '.mab-timeline-card-image',
				],
			],
		];

		$this->controls['titleSeparator'] = [
			'tab'   => 'content',
			'group' => 'cards',
			'type'  => 'separator',
			'label' => esc_html__( 'Title', 'max-addons' ),
		];

		$this->controls['titleHtmlTag'] = [
			'tab'         => 'content',
			'group'       => 'cards',
			'label'       => esc_html__( 'Title HTML Tag', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'h1'   => esc_html__( 'Heading 1 (h1)', 'max-addons' ),
				'h2'   => esc_html__( 'Heading 2 (h2)', 'max-addons' ),
				'h3'   => esc_html__( 'Heading 3 (h3)', 'max-addons' ),
				'h4'   => esc_html__( 'Heading 4 (h4)', 'max-addons' ),
				'h5'   => esc_html__( 'Heading 5 (h5)', 'max-addons' ),
				'h6'   => esc_html__( 'Heading 6 (h6)', 'max-addons' ),
				'div'  => esc_html__( 'Div (div)', 'max-addons' ),
				'span' => esc_html__( 'Span (span)', 'max-addons' ),
				'p'    => esc_html__( 'Paragraph (p)', 'max-addons' ),
			],
			'inline'      => true,
			'clearable'   => false,
			'pasteStyles' => false,
			'default'     => 'h4',
		];

		$this->controls['titlePadding'] = [
			'tab'   => 'content',
			'group' => 'cards',
			'label' => esc_html__( 'Padding', 'max-addons' ),
			'type'  => 'spacing',
			'css'   => [
				[
					'property' => 'padding',
					'selector' => '.mab-timeline .mab-timeline-card-title-wrap',
				],
			],
		];

		$this->controls['titleTypography'] = [
			'tab'    => 'content',
			'group'  => 'cards',
			'label'  => esc_html__( 'Typography', 'max-addons' ),
			'type'   => 'typography',
			'css'    => [
				[
					'property' => 'font',
					'selector' => '.mab-timeline-card-title',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['titleBackground'] = [
			'tab'    => 'content',
			'group'  => 'cards',
			'type'   => 'color',
			'label'  => esc_html__( 'Background', 'max-addons' ),
			'css'    => [
				[
					'property' => 'background-color',
					'selector' => '.mab-timeline .mab-timeline-card-title-wrap',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['titleBorder'] = [
			'tab'    => 'content',
			'group'  => 'cards',
			'label'  => esc_html__( 'Border', 'max-addons' ),
			'type'   => 'border',
			'css'    => [
				[
					'property' => 'border',
					'selector' => '.mab-timeline .mab-timeline-card-title-wrap',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['titleSpacing'] = [
			'tab'   => 'content',
			'group' => 'cards',
			'label' => esc_html__( 'Spacing', 'max-addons' ),
			'type'  => 'number',
			'units' => true,
			'css'   => [
				[
					'property' => 'margin-bottom',
					'selector' => '.mab-timeline-card-title-wrap',
				],
			],
		];

		$this->controls['contentSeparator'] = [
			'tab'   => 'content',
			'group' => 'cards',
			'type'  => 'separator',
			'label' => esc_html__( 'Content', 'max-addons' ),
		];

		$this->controls['contentTypography'] = [
			'tab'    => 'content',
			'group'  => 'cards',
			'label'  => esc_html__( 'Typography', 'max-addons' ),
			'type'   => 'typography',
			'css'    => [
				[
					'property' => 'font',
					'selector' => '.mab-timeline-card-content',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['cardsContentPadding'] = [
			'tab'   => 'content',
			'group' => 'cards',
			'label' => esc_html__( 'Padding', 'max-addons' ),
			'type'  => 'spacing',
			'css'   => [
				[
					'property' => 'padding',
					'selector' => '.mab-timeline .mab-timeline-card-content',
				],
			],
		];

		$this->controls['focusColorSeparator'] = [
			'tab'   => 'content',
			'group' => 'cards',
			'type'  => 'separator',
			'label' => esc_html__( 'Focused Card', 'max-addons' ),
		];

		$this->controls['cardsFocusBackground'] = [
			'tab'    => 'content',
			'group'  => 'cards',
			'type'   => 'color',
			'label'  => esc_html__( 'Background', 'max-addons' ),
			'css'    => [
				[
					'property' => 'background-color',
					'selector' => '.mab-timeline .mab-timeline-item-active .mab-timeline-card, .mab-timeline .swiper-slide-active .mab-timeline-card',
				],
				[
					'property' => 'color',
					'selector' => '.mab-timeline .mab-timeline-item-active .mab-timeline-arrow, .mab-timeline .swiper-slide-active .mab-timeline-arrow',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['cardsFocusBorder'] = [
			'tab'    => 'content',
			'group'  => 'cards',
			'label'  => esc_html__( 'Border', 'max-addons' ),
			'type'   => 'border',
			'css'    => [
				[
					'property' => 'border',
					'selector' => '.mab-timeline .mab-timeline-item-active .mab-timeline-card, .mab-timeline .swiper-slide-active .mab-timeline-card',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['titleFocusBackground'] = [
			'tab'    => 'content',
			'group'  => 'cards',
			'type'   => 'color',
			'label'  => esc_html__( 'Title Background', 'max-addons' ),
			'css'    => [
				[
					'property' => 'background-color',
					'selector' => '.mab-timeline .mab-timeline-item-active .mab-timeline-card-title-wrap, .mab-timeline .swiper-slide-active .mab-timeline-card-title-wrap',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['titleFocusTypography'] = [
			'tab'    => 'content',
			'group'  => 'cards',
			'label'  => esc_html__( 'Title Typography', 'max-addons' ),
			'type'   => 'typography',
			'css'    => [
				[
					'property' => 'font',
					'selector' => '.mab-timeline .mab-timeline-item-active .mab-timeline-card-title, .mab-timeline .swiper-slide-active .mab-timeline-card-title',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['titleFocusBorder'] = [
			'tab'    => 'content',
			'group'  => 'cards',
			'label'  => esc_html__( 'Title Border', 'max-addons' ),
			'type'   => 'border',
			'css'    => [
				[
					'property' => 'border',
					'selector' => '.mab-timeline .mab-timeline-item-active .mab-timeline-card-title-wrap, .mab-timeline .swiper-slide-active .mab-timeline-card-title-wrap',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['contentFocusTypography'] = [
			'tab'    => 'content',
			'group'  => 'cards',
			'label'  => esc_html__( 'Content Typography', 'max-addons' ),
			'type'   => 'typography',
			'css'    => [
				[
					'property' => 'font',
					'selector' => '.mab-timeline .mab-timeline-item-active .mab-timeline-card-content, .mab-timeline .swiper-slide-active .mab-timeline-card-content',
				],
			],
			'inline' => true,
			'small'  => true,
		];
	}

	// Marker Style
	public function set_marker_style_controls() {
		$this->controls['markerType'] = [
			'tab'     => 'content',
			'group'   => 'marker',
			'label'   => esc_html__( 'Marker Type', 'max-addons' ),
			'type'    => 'select',
			'options' => [
				'icon'   => esc_html__( 'Icon', 'max-addons' ),
				'image'  => esc_html__( 'Image', 'max-addons' ),
				'number' => esc_html__( 'Numbers', 'max-addons' ),
				'letter' => esc_html__( 'Letters', 'max-addons' ),
			],
			'reset'   => true,
			'inline'  => true,
			'default' => 'icon',
		];

		$this->controls['markerIcon'] = [
			'tab'      => 'content',
			'group'    => 'marker',
			'label'    => esc_html__( 'Icon', 'max-addons' ),
			'type'     => 'icon',
			'default'  => [
				'library' => 'themify',
				'icon'    => 'ti-calendar',
			],
			'required' => [ 'markerType', '=', [ 'icon' ] ],
		];

		$this->controls['markerImage'] = [
			'tab'      => 'content',
			'group'    => 'marker',
			'type'     => 'image',
			'label'    => esc_html__( 'Image', 'max-addons' ),
			'required' => [ 'markerType', '=', [ 'image' ] ],
		];

		$this->controls['markerSize'] = [
			'tab'      => 'content',
			'group'    => 'marker',
			'label'    => esc_html__( 'Marker Size', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'font-size',
					'selector' => '.mab-timeline-marker',
				],
				[
					'property' => 'width',
					'selector' => '.mab-timeline-marker img',
				],
			],
			'required' => [ 'markerType', '!=', [ 'none' ] ],
		];

		$this->controls['markerBoxSize'] = [
			'tab'   => 'content',
			'group' => 'marker',
			'label' => esc_html__( 'Marker Box Size', 'max-addons' ),
			'type'  => 'number',
			'units' => true,
			'css'   => [
				[
					'property' => 'bottom',
					'selector' => '.mab-timeline-navigation:before, .mab-timeline-navigation .mab-slider-arrow',
				],
				[
					'property' => 'width',
					'selector' => '.mab-timeline-marker, .mab-timeline-connector-wrap',
				],
				[
					'property' => 'height',
					'selector' => '.mab-timeline-marker',
				],
			],
		];

		$this->controls['markerTextTypography'] = [
			'tab'      => 'content',
			'group'    => 'marker',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-timeline-marker',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'markerType', '!=', [ 'image' ] ],
		];

		$this->controls['markerBackgroundColor'] = [
			'tab'    => 'content',
			'group'  => 'marker',
			'type'   => 'color',
			'label'  => esc_html__( 'Background', 'max-addons' ),
			'css'    => [
				[
					'property' => 'background-color',
					'selector' => '.mab-timeline-marker',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['markerBorder'] = [
			'tab'    => 'content',
			'group'  => 'marker',
			'label'  => esc_html__( 'Border', 'max-addons' ),
			'type'   => 'border',
			'css'    => [
				[
					'property' => 'border',
					'selector' => '.mab-timeline-marker',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['markerBoxShadow'] = [
			'tab'    => 'content',
			'group'  => 'marker',
			'label'  => esc_html__( 'Box Shadow', 'max-addons' ),
			'type'   => 'box-shadow',
			'css'    => [
				[
					'property' => 'box-shadow',
					'selector' => '.mab-timeline-marker',
				],
			],
			'inline' => true,
		];

		$this->controls['markerFocusColorSeparator'] = [
			'tab'   => 'content',
			'group' => 'marker',
			'type'  => 'separator',
			'label' => esc_html__( 'Focused Marker Color', 'max-addons' ),
		];

		$this->controls['markerFocusColor'] = [
			'tab'    => 'content',
			'group'  => 'marker',
			'type'   => 'color',
			'label'  => esc_html__( 'Color', 'max-addons' ),
			'css'    => [
				[
					'property' => 'color',
					'selector' => '.mab-timeline-item-active .mab-timeline-marker, .swiper-slide-active .mab-timeline-marker',
				],
				[
					'property' => 'fill',
					'selector' => '.mab-timeline-item-active .mab-timeline-marker svg, .swiper-slide-active .mab-timeline-marker svg',
				],
			],
			'inline' => true,
			'small'  => true,
			'required' => [ 'markerType', '!=', [ 'image' ] ],
		];

		$this->controls['markerFocusBackground'] = [
			'tab'    => 'content',
			'group'  => 'marker',
			'type'   => 'color',
			'label'  => esc_html__( 'Background', 'max-addons' ),
			'css'    => [
				[
					'property' => 'background-color',
					'selector' => '.mab-timeline-item-active .mab-timeline-marker, .swiper-slide-active .mab-timeline-marker',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['markerFocusBorder'] = [
			'tab'    => 'content',
			'group'  => 'marker',
			'label'  => esc_html__( 'Border', 'max-addons' ),
			'type'   => 'border',
			'css'    => [
				[
					'property' => 'border',
					'selector' => '.mab-timeline-item-active .mab-timeline-marker, .swiper-slide-active .mab-timeline-marker',
				],
			],
			'inline' => true,
			'small'  => true,
		];
	}

	// Date Style
	public function set_dates_style_controls() {

		$this->controls['datePadding'] = [
			'tab'   => 'content',
			'group' => 'date',
			'label' => esc_html__( 'Padding', 'max-addons' ),
			'type'  => 'spacing',
			'css'   => [
				[
					'property' => 'padding',
					'selector' => '.mab-timeline-card-date',
				],
			],
		];

		$this->controls['dateTypography'] = [
			'tab'    => 'content',
			'group'  => 'date',
			'label'  => esc_html__( 'Typography', 'max-addons' ),
			'type'   => 'typography',
			'css'    => [
				[
					'property' => 'font',
					'selector' => '.mab-timeline-card-date',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['dateBackgroundColor'] = [
			'tab'    => 'content',
			'group'  => 'date',
			'type'   => 'color',
			'label'  => esc_html__( 'Background', 'max-addons' ),
			'css'    => [
				[
					'property' => 'background-color',
					'selector' => '.mab-timeline-card-date',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['dateBorder'] = [
			'tab'    => 'content',
			'group'  => 'date',
			'label'  => esc_html__( 'Border', 'max-addons' ),
			'type'   => 'border',
			'css'    => [
				[
					'property' => 'border',
					'selector' => '.mab-timeline-card-date',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['dateBoxShadow'] = [
			'tab'    => 'content',
			'group'  => 'date',
			'label'  => esc_html__( 'Box Shadow', 'max-addons' ),
			'type'   => 'box-shadow',
			'css'    => [
				[
					'property' => 'box-shadow',
					'selector' => '.mab-timeline-card-date',
				],
			],
			'inline' => true,
		];

		$this->controls['dateFocusColorSeparator'] = [
			'tab'   => 'content',
			'group' => 'date',
			'type'  => 'separator',
			'label' => esc_html__( 'Focused Date', 'max-addons' ),
		];

		$this->controls['dateFocusTypography'] = [
			'tab'    => 'content',
			'group'  => 'date',
			'label'  => esc_html__( 'Typography', 'max-addons' ),
			'type'   => 'typography',
			'css'    => [
				[
					'property' => 'font',
					'selector' => '.mab-timeline-item-active .mab-timeline-card-date, .swiper-slide-active .mab-timeline-card-date',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['dateFocusBackgroundColor'] = [
			'tab'    => 'content',
			'group'  => 'date',
			'type'   => 'color',
			'label'  => esc_html__( 'Background', 'max-addons' ),
			'css'    => [
				[
					'property' => 'background-color',
					'selector' => '.mab-timeline-item-active .mab-timeline-card-date, .swiper-slide-active .mab-timeline-card-date',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['dateFocusBorder'] = [
			'tab'    => 'content',
			'group'  => 'date',
			'label'  => esc_html__( 'Border', 'max-addons' ),
			'type'   => 'border',
			'css'    => [
				[
					'property' => 'border',
					'selector' => '.mab-timeline-item-active .mab-timeline-card-date, .swiper-slide-active .mab-timeline-card-date',
				],
			],
			'inline' => true,
			'small'  => true,
		];
	}

	// Connector Style
	public function set_connector_style_controls() {
		$this->controls['connector_thickness'] = [
			'tab'   => 'content',
			'group' => 'connector',
			'label' => esc_html__( 'Thickness', 'max-addons' ),
			'type'  => 'number',
			'units' => true,
			'css'   => [
				[
					'property' => 'width',
					'selector' => '.mab-timeline-vertical .mab-timeline-connector',
				],
				[
					'property' => 'height',
					'selector' => '.mab-timeline-navigation:before',
				],
			],
		];

		$this->controls['connectorBackgroundGradient'] = [
			'tab'   => 'content',
			'group' => 'connector',
			'label' => esc_html__( 'Background', 'max-addons' ),
			'type'  => 'gradient',
			'css'   => [
				[
					'property' => 'background-image',
					'selector' => '.mab-timeline-connector, .mab-timeline-navigation:before',
				],
			],
		];

		$this->controls['connectorProgressBackgroundGradient'] = [
			'tab'      => 'content',
			'group'    => 'connector',
			'label'    => esc_html__( 'Progress Background', 'max-addons' ),
			'type'     => 'gradient',
			'css'      => [
				[
					'property' => 'background-image',
					'selector' => '.mab-timeline-connector-inner',
				],
			],
			'required' => [ 'layout', '=', [ 'vertical' ] ],
		];
	}

	// Set arrow style controls
	public function set_arrow_style_controls() {

		$this->controls['arrowHeight'] = [
			'tab'         => 'content',
			'group'       => 'arrowsStyle',
			'label'       => esc_html__( 'Height in px', 'max-addons' ),
			'type'        => 'number',
			'unit'        => 'px',
			'css'         => [
				[
					'property' => 'height',
					'selector' => '.mab-slider-arrow',
				],
				[
					'property' => 'line-height',
					'selector' => '.mab-slider-arrow',
				],
			],
			'inline'      => true,
			'small'       => true,
			'placeholder' => 40,
			'required'    => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowWidth'] = [
			'tab'         => 'content',
			'group'       => 'arrowsStyle',
			'label'       => esc_html__( 'Width in px', 'max-addons' ),
			'type'        => 'number',
			'unit'        => 'px',
			'css'         => [
				[
					'property' => 'width',
					'selector' => '.mab-slider-arrow',
				],
			],
			'inline'      => true,
			'small'       => true,
			'placeholder' => 40,
			'required'    => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowBackground'] = [
			'tab'      => 'content',
			'group'    => 'arrowsStyle',
			'label'    => esc_html__( 'Background', 'max-addons' ),
			'type'     => 'color',
			'inline'   => true,
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.mab-slider-arrow',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowBorder'] = [
			'tab'      => 'content',
			'group'    => 'arrowsStyle',
			'type'     => 'border',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.mab-slider-arrow',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowTypography'] = [
			'tab'      => 'content',
			'group'    => 'arrowsStyle',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-slider-arrow',
				],
			],
			'exclude'  => [
				'font-family',
				'font-weight',
				'font-style',
				'text-align',
				'letter-spacing',
				'line-height',
				'text-transform',
				'text-decoration',
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['prevArrow'] = [
			'tab'      => 'content',
			'group'    => 'arrowsStyle',
			'label'    => esc_html__( 'Prev arrow', 'max-addons' ),
			'type'     => 'icon',
			'default'  => [
				'library' => 'themify',
				'icon'    => 'ti-angle-left',
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['nextArrow'] = [
			'tab'      => 'content',
			'group'    => 'arrowsStyle',
			'label'    => esc_html__( 'Next arrow', 'max-addons' ),
			'type'     => 'icon',
			'default'  => [
				'library' => 'themify',
				'icon'    => 'ti-angle-right',
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowSpacing'] = [
			'tab'      => 'content',
			'group'    => 'arrowsStyle',
			'label'    => esc_html__( 'Spacing', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'margin-right',
					'selector' => '.abx-prev',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];
	}

	public function slider_settings() {
		$settings = $this->settings;

		$slidesToShow = isset( $settings['columns'] ) ? intval( $settings['columns'] ) : 3;

		$options = [
			'speed'          => isset( $settings['speed'] ) ? intval( $settings['speed'] ) : 600,
			'autoHeight'     => true,
			'effect'         => 'slide',
			'navigation'     => false,
			'spaceBetween'   => 0,
			'initialSlide'   => 0,
			'slidesPerView'  => $slidesToShow,
			'loop'           => isset( $settings['infinite'] ),
			'centeredSlides' => isset( $settings['centerMode'] ),
		];

		if ( isset( $settings['autoplay'] ) ) {
			$options['autoplay'] = [
				'delay'                => isset( $settings['autoplaySpeed'] ) ? intval( $settings['autoplaySpeed'] ) : 3000,
				'disableOnInteraction' => true,
			];

			// Custom as SwiperJS does not provide this option
			if ( isset( $settings['pauseOnHover'] ) ) {
				$options['pauseOnHover'] = isset( $settings['pauseOnHover'] );
			}
		}

		// Dots
		if ( isset( $settings['dots'] ) ) {
			$options['pagination'] = [
				'el'   => '.swiper-pagination',
				'type' => 'bullets',
			];
		}

		$breakpoints = [];

		// Breakpoint values compare to ">="
		if ( isset( $settings['responsive'] ) ) {
			// Custom breakpoints
			foreach ( $settings['responsive'] as $responsive_option ) {
				if ( isset( $responsive_option['breakpoint'] ) ) {
					$breakpoints[ $responsive_option['breakpoint'] ] = [
						'slidesPerView'  => isset( $responsive_option['slidesToShow'] ) ? intval( $responsive_option['slidesToShow'] ) : 1,
					];
				}
			}
		} else {
			// Default breakpoints
			$breakpoints[992] = [
				'slidesPerView'  => $options['slidesPerView'],
			];

			$breakpoints[768] = [
				'slidesPerView'  => $slidesToShow > 1 ? 2 : 1,
			];

			$breakpoints[1] = [
				'slidesPerView'  => 1,
			];
		}

		$options['breakpoints'] = $breakpoints;

		return $options;
	}

	public function slider_nav_settings() {
		$settings = $this->settings;

		$slidesToShow = isset( $settings['columns'] ) ? intval( $settings['columns'] ) : 3;

		$options = [
			'speed'          => isset( $settings['speed'] ) ? intval( $settings['speed'] ) : 600,
			'autoHeight'     => true,
			'effect'         => 'slide',
			'spaceBetween'   => 0,
			'initialSlide'   => 0,
			'slidesPerView'  => $slidesToShow,
			'loop'           => isset( $settings['infinite'] ),
			'centeredSlides' => isset( $settings['centerMode'] ),
			'pagination'     => false,
		];

		// Arrow navigation
		if ( isset( $settings['arrows'] ) ) {
			$options['navigation'] = [
				'prevEl' => '.mab-arrow-prev',
				'nextEl' => '.mab-arrow-next',
			];
		}

		$breakpoints = [];

		// Breakpoint values compare to ">="
		if ( isset( $settings['responsive'] ) ) {
			// Custom breakpoints
			foreach ( $settings['responsive'] as $responsive_option ) {
				if ( isset( $responsive_option['breakpoint'] ) ) {
					$breakpoints[ $responsive_option['breakpoint'] ] = [
						'slidesPerView'  => isset( $responsive_option['slidesToShow'] ) ? intval( $responsive_option['slidesToShow'] ) : 1,
					];
				}
			}
		} else {
			// Default breakpoints
			$breakpoints[992] = [
				'slidesPerView'  => $options['slidesPerView'],
			];

			$breakpoints[768] = [
				'slidesPerView'  => $slidesToShow > 1 ? 2 : 1,
			];

			$breakpoints[1] = [
				'slidesPerView'  => 1,
			];
		}

		$options['breakpoints'] = $breakpoints;

		return $options;
	}

	/**
	 * Render timeline widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @access protected
	 */
	public function render() {
		$settings = $this->settings;

		$timeline_classes = array();

		/**
		 * Render timeline HTML
		 */

		$timeline = empty( $settings['items'] ) ? false : $settings['items'];

		if ( ! $timeline ) {
			return $this->render_element_placeholder(
				[
					'title' => esc_html__( 'No timeline item added.', 'max-addons' ),
				]
			);
		}

		$title_tag = ! empty( $settings['titleHtmlTag'] ) ? $settings['titleHtmlTag'] : 'h4';

		$timeline_classes[] = 'mab-timeline';
		$layout = isset( $settings['layout'] ) ? $settings['layout'] : 'vertical';
		// Layout.
		if ( isset( $settings['layout'] ) ) {
			$timeline_classes[] = 'mab-timeline-' . $settings['layout'];
		}

		// Direction.
		if ( isset( $settings['direction'] ) ) {
			$timeline_classes[] = 'mab-timeline-' . $settings['direction'];
		}

		if ( isset( $settings['direction_tablet'] ) ) {
			$timeline_classes[] = 'mab-timeline-tablet-' . $settings['direction_tablet'];
		}

		if ( isset( $settings['direction_mobile'] ) ) {
			$timeline_classes[] = 'mab-timeline-mobile-' . $settings['direction_mobile'];
		}

		$timeline_classes[] = 'mab-timeline-dates';

		if ( isset( $settings['cardsArrowsAlignment'] ) ) {
			$timeline_classes[] = 'mab-timeline-arrows-' . $settings['cardsArrowsAlignment'];
		}

		if ( isset( $settings['layout'] ) && $settings['layout'] === 'horizontal' ) {
			$timeline_classes[] = 'swiper';
		}

		$this->set_attribute( 'timeline', 'class', $timeline_classes );

		$this->set_attribute( 'timeline', 'data-timeline-layout', $layout );

		$this->set_attribute( '_root', 'class', 'mab-timeline-wrapper' );
		$this->set_attribute( '_root', 'data-element-id', $this->id );
		$this->set_attribute( 'timeline-items', 'class', 'mab-timeline-items' );

		if ( 'horizontal' === $layout ) {
			$this->set_attribute( 'timeline-items', 'class', 'swiper-wrapper' );
			//slider settings
			$this->set_attribute( 'timeline', 'data-options', wp_json_encode( $this->slider_settings() ) );
		}

		if ( 'horizontal' === $layout && is_rtl() ) {
			$this->set_attribute( '_root', 'data-rtl', 'yes' );
		}

		$this->set_attribute( 'post-categories', 'class', 'mab-post-categories' );
		?>
		<div <?php echo wp_kses_post( $this->render_attributes( '_root' ) ); ?>>
			<?php echo $this->render_horizontal_timeline_nav(); ?>

			<div <?php echo wp_kses_post( $this->render_attributes( 'timeline' ) ); ?>>
				<?php if ( 'vertical' === $layout ) { ?>
					<div class="mab-timeline-connector-wrap">
						<div class="mab-timeline-connector">
							<div class="mab-timeline-connector-inner">
							</div>
						</div>
					</div>
				<?php } ?>
				<div <?php echo wp_kses_post( $this->render_attributes( 'timeline-items' ) ); ?>>
					<?php
					$output = '';

					// Query Loop
					if ( isset( $settings['hasLoop'] ) ) {
						$query = new \Bricks\Query( [
							'id'       => $this->id,
							'settings' => $settings,
						] );

						$timeline_item = $timeline[0];

						$output .= $query->render( [ $this, 'render_repeater_item' ], compact( 'timeline_item', 'title_tag' ) );

						// We need to destroy the Query to explicitly remove it from the global store
						$query->destroy();
						unset( $query );
					} else {
						foreach ( $timeline as $index => $item ) {
							$output .= self::render_repeater_item( $item, $title_tag );
						}
					}

					echo $output;
					?>
				</div>
				<?php
				if ( 'horizontal' === $layout ) {
					if ( isset( $settings['dots'] ) ) {
						echo '<div class="swiper-pagination"></div>';
					}
				} ?>
			</div>
		</div>
		<?php
	}

	/**
	 * Render vertical timeline output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @access protected
	 */
	protected function render_horizontal_timeline_nav() {
		$settings = $this->settings;

		$timeline = empty( $settings['items'] ) ? false : $settings['items'];

		$this->set_attribute( 'navigation', 'class', 'mab-timeline-navigation' );

		$output = '';

		if ( isset( $settings['layout'] ) && $settings['layout'] === 'horizontal' ) {
			$this->set_attribute( 'navigation', 'class', 'swiper' );
			//slider settings
			$this->set_attribute( 'navigation', 'data-options', wp_json_encode( $this->slider_nav_settings() ) );

			$output .= '<div ' . wp_kses_post( $this->render_attributes( 'navigation' ) ) . '>';
				$output .= '<div class="swiper-wrapper">';

					// Query Loop
					if ( isset( $settings['hasLoop'] ) ) {
						$query = new \Bricks\Query( [
							'id'       => $this->id,
							'settings' => $settings,
						] );

						$timeline_item = $timeline[0];
						$date = isset( $item['timelineDate'] ) ? $item['timelineDate'] : '';

						$output .= $query->render( [ $this, 'render_horizontal_connector_marker' ], compact( 'timeline_item' ) );

						// We need to destroy the Query to explicitly remove it from the global store
						$query->destroy();
						unset( $query );
					} else {
						foreach ( $timeline as $index => $accordion ) {

							$output .= self::render_horizontal_connector_marker( $accordion );

						}
					}
				$output .= '</div>';

				if ( isset( $settings['arrows'] ) && isset( $settings['prevArrow']['icon'] ) ) {
					$output .= '<div class="swiper-button mab-slider-arrow mab-arrow-prev ' . esc_attr( $settings['prevArrow']['icon'] ) . '"></div>';
				}

				if ( isset( $settings['arrows'] ) && isset( $settings['nextArrow']['icon'] ) ) {
					$output .= '<div class="swiper-button mab-slider-arrow mab-arrow-next ' . esc_attr( $settings['nextArrow']['icon'] ) . '"></div>';
				}
			$output .= '</div>';

			return $output;
		}
	}

	/**
	 * Render custom content output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @param  int    $number  Item count.
	 * @param  string $date    Item Date.
	 * @param  array  $item    Item.
	 *
	 * @access public
	 */
	public function render_horizontal_connector_marker( $item ) {
		$output = '<div class="mab-timeline-nav-item repeater-item swiper-slide">';
			$date = isset( $item['timelineDate'] ) ? $item['timelineDate'] : '';

			$output .= self::render_connector_marker( $item, $date );

		$output .= '</div>';

		return $output;
	}

	/**
	 * Render custom content output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @param  int    $number  Item count.
	 * @param  string $date    Item Date.
	 * @param  array  $item    Item.
	 *
	 * @access public
	 */
	public function render_connector_marker( $item, $date = '', $echo = false ) {
		$settings = $this->settings;
		$index    = $this->loop_index;
		$number   = $index + 1;

		$output = '';

		$output .= '<div class="mab-timeline-marker-wrapper">';
			if ( 'horizontal' === $settings['layout'] ) {
				$output .= '<div class="mab-timeline-card-date-wrapper">';
				$output .= '<div class="mab-timeline-card-date">';
				$output .= esc_html( $date );
				$output .= '</div>';
				$output .= '</div>';
			}

			$output .= '<div class="mab-timeline-marker">';
				if ( isset( $settings['markerType'] ) ) {
					if ( 'icon' === $settings['markerType'] ) {

						$output .= '<span class="mab-icon">';
							$icon_html = isset( $settings['markerIcon'] ) ? self::render_icon( $settings['markerIcon'] ) : false;
							$output .= $icon_html;
						$output .= '</span>';

					} elseif ( 'image' === $settings['markerType'] ) {
						if ( isset( $settings['markerImage'] ) && ! empty( $settings['markerImage'] ) ) {

							if ( isset( $settings['markerImage']['useDynamicData']['name'] ) ) {
								$images = $this->render_dynamic_data_tag( $settings['markerImage']['useDynamicData']['name'], 'markerImage' );
								$size   = isset( $settings['markerImage']['size'] ) ? $settings['markerImage']['size'] : BRICKS_DEFAULT_IMAGE_SIZE;
								$url    = $images ? wp_get_attachment_image_url( $images[0], $size ) : $settings['markerImage']['url'];
							} else {
								$url = $settings['markerImage']['url'];
							}
							if ( $this->lazy_load() ) {
								$output .= '<img class="image css-filter bricks-lazy-hidden" src="' . esc_url( $url ) . '">';
							} else {
								$output .= '<img class="image css-filter" src="' . esc_url( $url ) . '">';
							}
						}
					} elseif ( 'number' === $settings['markerType'] ) {

						$output .= wp_kses_post( $number );

					} elseif ( 'letter' === $settings['markerType'] ) {
						$alphabets = range( 'A', 'Z' );

						$alphabets = array_combine( range( 1, count( $alphabets ) ), $alphabets );

						$output .= wp_kses_post( $alphabets[ $number ] );
					}
				}
			$output .= '</div>';
		$output .= '</div>';

		if ( isset( $settings['layout'] ) && $settings['layout'] === 'vertical' ) {
			$output .= '<div class="mab-timeline-card-date-wrapper">';
				$output .= '<div class="mab-timeline-card-date">';
					$output .= wp_kses_post( $date );
				$output .= '</div>';
			$output .= '</div>';
		}

		if ( true == $echo ) {
			echo $output;
		} else {
			return $output;
		}
	}

	/**
	 * Render custom content output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @access protected
	 */
	public function render_repeater_item( $item, $title_tag ) {
		$settings = $this->settings;
		$index    = $this->loop_index;
		$output   = '';

		$item_key    = "item-{$index}";
		$title_key   = "timeline-title-{$index}";
		$content_key = "timeline-content-{$index}";

		$this->set_attribute(
			$item_key,
			'class',
			array(
				'mab-timeline-item',
				'repeater-item',
			)
		);

		if ( isset( $settings['layout'] ) && $settings['layout'] === 'horizontal' ) {
			$this->set_attribute( $item_key, 'class', 'swiper-slide' );
		}

		if ( isset( $settings['animateCards'] ) ) {
			$this->set_attribute( $item_key, 'class', 'mab-timeline-item-hidden' );
		}

		$this->set_attribute( $title_key, 'class', 'mab-timeline-card-title' );

		$this->set_attribute( $content_key, 'class', 'mab-timeline-card-content' );

		if ( isset( $item['timelineItemLink'] ) ) {
			$link_key = "link-{$index}";

			$this->set_link_attributes( $link_key, $item['timelineItemLink'] );
		}

		$output .= '<div ' . $this->render_attributes( $item_key ) . '>';
			$output .= '<div class="mab-timeline-card-wrapper">';
				if ( isset( $item['timelineItemLink'] ) ) {
					$output .= '<a ' . wp_kses_post( $this->render_attributes( $link_key ) ) . '>';
				}
				$output .= '<div class="mab-timeline-card">';
					if ( isset( $item['timelineImage'] ) && ! empty( $item['image'] ) ) {
						$output .= '<div class="mab-timeline-card-image">';
							if ( isset( $item['image'] ) && ! empty( $item['image'] ) ) {
								if ( isset( $item['image']['useDynamicData']['name'] ) ) {
									$images = $this->render_dynamic_data_tag( $item['image']['useDynamicData']['name'], 'image' );
									$size   = isset( $item['image']['size'] ) ? $item['image']['size'] : BRICKS_DEFAULT_IMAGE_SIZE;
									$url    = $images ? wp_get_attachment_image_url( $images[0], $size ) : $item['image']['url'];
								} else {
									$url = $item['image']['url'];
								}
								if ( $this->lazy_load() ) {
									$output .= '<img class="image css-filter bricks-lazy-hidden" src="' . esc_url( $url ) . '">';
								} else {
									$output .= '<img class="image css-filter" src="' . esc_url( $url ) . '">';
								}
							}
						$output .= '</div>';
					}

					if ( isset( $item['timelineTitle'] ) ) {
						$output .= '<div class="mab-timeline-card-title-wrap">';
							if ( isset( $item['timelineTitle'] ) ) {
								if ( isset( $settings['layout'] ) && $settings['layout'] === 'vertical' ) {
									$output .= '<div class="mab-timeline-card-date">';
									$output .= wp_kses_post( $item['timelineDate'] );
									$output .= '</div>';
								}

								$output .= '<' . esc_html( $title_tag ) . ' ' .  wp_kses_post( $this->render_attributes( $title_key ) ) . '>';
								$output .= wp_kses_post( $item['timelineTitle'] );
								$output .= '</' . esc_html( $title_tag ) . '>';
							}
						$output .= '</div>';
					}

					$content = ! empty( $item['timelineContent'] ) ? $item['timelineContent'] : false;

					if ( $content ) {
						$content = $this->render_dynamic_data( $content );

						if ( method_exists( '\Bricks\Helpers', 'parse_editor_content' ) ) {
							$content = \Bricks\Helpers::parse_editor_content( $content );
						} else {
							$content = do_shortcode( $content );
						}

						$output .= '<div ' . wp_kses_post( $this->render_attributes( $content_key ) ) . '>';
						$output .= wp_kses_post( $content ); //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
						$output .= '</div>';
					}
					if ( isset( $settings['cardArrow'] ) ) {
						$output .= '<div class="mab-timeline-arrow"></div>';
					}
				$output .= '</div>';
				if ( isset( $item['timelineItemLink'] ) ) {
					$output .= '</a>';
				}
				$output .= '</div>';

			if ( isset( $settings['layout'] ) && $settings['layout'] === 'vertical' ) {
				$date = isset( $item['timelineDate'] ) ? $item['timelineDate'] : '';
				$output .= wp_kses_post( $this->render_connector_marker( $item, $date, false ) );
			}
		$output .= '</div>';

		$this->loop_index++;

		return $output;
	}

	/**
	 * Get post date.
	 *
	 * Returns the post date.
	 *
	 * @since 1.0.0
	 * @param array $settings object.
	 * @access public
	 */
	public function mab_get_date( $settings ) {
		$date_type = isset( $settings['dateFormat'] ) ? $settings['dateFormat'] : 'publish';
		$date_format = isset( $settings['dateFormatSelect'] ) ? $settings['dateFormatSelect'] : '';
		$date_custom_format = isset( $settings['postDateFormat'] ) ? $settings['postDateFormat'] : get_option( $date_type );
		$date        = '';

		if ( 'custom' === $date_format && $date_custom_format ) {
			$date_format = $date_custom_format;
		}

		if ( 'ago' === $date_type ) {
			$date = sprintf( _x( '%s ago', '%s = human-readable time difference', 'max-addons' ), human_time_diff( get_the_time( 'U' ), current_time( 'timestamp' ) ) );
		} elseif ( 'modified' === $date_type ) {
			$date = get_the_modified_date( $date_format, get_the_ID() );
		} elseif ( 'key' === $date_type ) {
			$date_meta_key = $settings['postDateKey'];
			if ( $date_meta_key ) {
				$date = get_post_meta( get_the_ID(), $date_meta_key, 'true' );
			}
		} else {
			$date = get_the_date( $date_format );
		}

		if ( $date == '' ) {
			$date = get_the_date( $date_format );
		}

		return apply_filters( 'mab_timeline_date_format', $date, get_option( 'date_format' ), '', '' );
	}
}
