<?php
namespace MaxAddonsPro\Elements;

use MaxAddonsPro\Classes\Helper;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Thumbnail_Slider_Element extends \Bricks\Element {
	// Element properties
	public $category     = 'max-addons-elements'; // Use predefined element category 'general'
	public $name         = 'max-thumbnail-slider'; // Make sure to prefix your elements
	public $icon         = 'ti-gallery max-element'; // Themify icon font class
	public $css_selector = ''; // Default CSS selector
	public $scripts      = [ 'mabThumbnailSlider' ]; // Script(s) run when element is rendered on frontend or updated in builder

	// Return localized element label
	public function get_label() {
		return esc_html__( 'Thumbnail Slider', 'max-addons' );
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		wp_enqueue_style( 'mab-thumbnail-slider' );

		if ( isset( $this->settings['lightbox'] ) ) {
			wp_enqueue_script( 'bricks-photoswipe' );
			wp_enqueue_script( 'bricks-photoswipe-lightbox' );
			wp_enqueue_style( 'bricks-photoswipe' );
		}

		wp_enqueue_script( 'bricks-splide' );
		wp_enqueue_style( 'bricks-splide' );
		wp_enqueue_script( 'mab-thumbnail-slider' );
	}

	public function set_control_groups() {
		$this->control_groups['gallery'] = [
			'title' => esc_html__( 'Gallery', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['layout'] = [
			'title' => esc_html__( 'Layout', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['preview'] = [
			'title' => esc_html__( 'Preview', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['thumbnails'] = [
			'title' => esc_html__( 'Thumbnails', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['arrows'] = [
			'title' => esc_html__( 'Arrows', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['pagination'] = [
			'title' => esc_html__( 'Pagination', 'max-addons' ),
			'tab'   => 'content',
		];
	}

	public function set_controls() {
		$this->set_gallery_controls();
		$this->set_layout_controls();
		$this->set_preview_controls();
		$this->set_thumbs_controls();
		$this->set_navigation_controls();
		$this->set_pagination_controls();
	}

	// Set gallery controls
	public function set_gallery_controls() {
		$this->controls['source'] = [
			'tab'         => 'content',
			'group'       => 'gallery',
			'type'        => 'select',
			'label'       => esc_html__( 'Source', 'max-addons' ),
			'options'     => [
				'media' => esc_html__( 'Media', 'max-addons' ),
				'posts' => esc_html__( 'Posts', 'max-addons' ),
			],
			'inline'      => true,
			'placeholder' => esc_html__( 'Media', 'max-addons' ),
		];

		$this->controls['items'] = [
			'tab'      => 'content',
			'group'    => 'gallery',
			'type'     => 'image-gallery',
			'rerender' => true,
			'label'    => esc_html__( 'Images', 'max-addons' ),
			'exclude'  => [
				'size',
			],
			'required' => [ 'source', '!=', 'posts' ],
		];

		$this->controls['query'] = [
			'tab'      => 'content',
			'group'    => 'gallery',
			'label'    => esc_html__( 'Query', 'max-addons' ),
			'type'     => 'query',
			'popup'    => true,
			'inline'   => true,
			'required' => [ 'source', '=', 'posts' ],
			'exclude'  => [
				'objectType',
				'infinite_scroll_separator',
				'infinite_scroll',
				'infinite_scroll_margin',
			],
		];

		$this->controls['previewImageSize'] = [
			'tab'         => 'content',
			'group'       => 'gallery',
			'label'       => esc_html__( 'Preview', 'max-addons' ) . ': ' . esc_html__( 'Image size', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['imageSizes'],
			'placeholder' => 'full',
		];

		$this->controls['thumbnailImageSize'] = [
			'tab'         => 'content',
			'group'       => 'gallery',
			'label'       => esc_html__( 'Thumbnail', 'max-addons' ) . ': ' . esc_html__( 'Image size', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['imageSizes'],
			'placeholder' => 'thumbnail',
		];
	}

	// Set layout controls
	public function set_layout_controls() {
		$this->controls['thumbnailPosition'] = [
			'tab'         => 'content',
			'group'       => 'layout',
			'label'       => esc_html__( 'Thumbnails position', 'max-addons' ),
			'type'        => 'select',
			'inline'      => true,
			'options'     => [
				'top'    => esc_html__( 'Top', 'max-addons' ),
				'right'  => esc_html__( 'Right', 'max-addons' ),
				'bottom' => esc_html__( 'Bottom', 'max-addons' ),
				'left'   => esc_html__( 'Left', 'max-addons' ),
			],
			'placeholder' => esc_html__( 'Bottom', 'max-addons' ),
		];

		// Get all breakpoints except base (@since 1.8.0)
		$breakpoints        = \Bricks\Breakpoints::$breakpoints;
		$breakpoint_options = [];

		foreach ( $breakpoints as $index => $breakpoint ) {
			if ( ! isset( $breakpoint['base'] ) ) {
				$breakpoint_options[ $breakpoint['key'] ] = $breakpoint['label'] . ' (<= ' . $breakpoint['width'] . 'px)';
			}
		}

		$breakpoint_options['none']  = esc_html__( 'None', 'max-addons' );

		$this->controls['stackOn'] = [
			'tab'         => 'content',
			'group'       => 'layout',
			'label'       => esc_html__( 'Stack on', 'max-addons' ),
			'type'        => 'select',
			'options'     => $breakpoint_options,
			'placeholder' => esc_html__( 'None', 'max-addons' ),
			'clearable'   => true,
			'pasteStyles' => false,
			'default'     => 'mobile_landscape',
			'required'    => [ 'thumbnailPosition', '=', [ 'left', 'right' ] ]
		];

		$this->controls['thumbnailsWidth'] = [
			'tab'         => 'content',
			'group'       => 'layout',
			'label'       => esc_html__( 'Thumbnails width', 'max-addons' ) . ' (px)',
			'type'        => 'number',
			'units'       => true,
			'css'         => [
				[
					'selector' => '.mab-thumbnail-slider-right .mab-thumbnail-slider-thumbs',
					'property' => 'width',
				],
				[
					'selector' => '.mab-thumbnail-slider-left .mab-thumbnail-slider-thumbs',
					'property' => 'width',
				],
			],
			'placeholder' => '100px',
			'rerender'    => true,
			'required'    => [ 'thumbnailPosition', '=', [ 'left', 'right' ] ]
		];

		$this->controls['thumbsPreviewGap'] = [
			'tab'         => 'content',
			'group'       => 'layout',
			'label'       => esc_html__( 'Spacing', 'max-addons' ),
			'info'        => esc_html__( 'Spacing between thumbnails and preview', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'css'         => [
				[
					'selector' => '.mab-thumbnail-slider',
					'property' => 'gap',
				]
			],
			'placeholder' => 20,
			'rerender'    => true,
		];

		$this->controls['thumbsAlignVertical'] = [
			'tab'      => 'content',
			'group'    => 'layout',
			'label'    => esc_html__( 'Vertical align', 'max-addons' ),
			'type'     => 'align-items',
			'exclude'  => 'stretch',
			'css'      => [
				[
					'property' => 'align-items',
					'selector' => '.mab-thumbnail-slider-left, .mab-thumbnail-slider-right',
				],
			],
			'inline'   => true,
			'required' => [
				[ 'thumbnailPosition', '=', [ 'left', 'right' ] ],
				[ 'thumbsCarousel', '=', '' ]
			]
		];
	}

	// Set preview controls
	public function set_preview_controls() {
		$this->controls['type'] = [
			'group'       => 'preview',
			'label'       => esc_html__( 'Type', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'loop'  => esc_html__( 'Loop', 'max-addons' ),
				'slide' => esc_html__( 'Slide', 'max-addons' ),
				'fade'  => esc_html__( 'Fade', 'max-addons' ),
			],
			'inline'      => true,
			'placeholder' => esc_html__( 'Loop', 'max-addons' ),
			'fullAccess'  => true,
		];

		$this->controls['direction'] = [
			'group'       => 'preview',
			'label'       => esc_html__( 'Direction', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'ltr' => esc_html__( 'Left to right', 'max-addons' ),
				'rtl' => esc_html__( 'Right to left', 'max-addons' ),
				'ttb' => esc_html__( 'Vertical', 'max-addons' ),
			],
			'inline'      => true,
			'placeholder' => esc_html__( 'Left to right', 'max-addons' ),
			'breakpoints' => true,
			'fullAccess'  => true,
		];

		$this->controls['height'] = [
			'group'       => 'preview',
			'label'       => esc_html__( 'Height', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'placeholder' => '',
			'breakpoints' => true,
			'fullAccess'  => true,
		];

		$this->controls['start'] = [
			'group'       => 'preview',
			'label'       => esc_html__( 'Start index', 'max-addons' ),
			'type'        => 'number',
			'placeholder' => 0,
			'fullAccess'  => true,
		];

		$this->controls['speed'] = [
			'group'       => 'preview',
			'label'       => esc_html__( 'Speed in ms', 'max-addons' ),
			'type'        => 'number',
			'placeholder' => 400,
			'breakpoints' => true,
			'fullAccess'  => true,
		];

		// AUTOPLAY

		/* $this->controls['autoplaySeparator'] = [
			'group'      => 'preview',
			'label'      => esc_html__( 'Autoplay', 'max-addons' ),
			'type'       => 'separator',
			'fullAccess' => true,
		]; */

		$this->controls['autoplay'] = [
			'group'      => 'preview',
			'label'      => esc_html__( 'Autoplay', 'max-addons' ),
			'type'       => 'checkbox',
			'fullAccess' => true,
		];

		$this->controls['pauseOnHover'] = [
			'group'      => 'preview',
			'label'      => esc_html__( 'Pause on hover', 'max-addons' ),
			'type'       => 'checkbox',
			'inline'     => true,
			'required'   => [ 'autoplay', '!=', '' ],
			'fullAccess' => true,
		];

		$this->controls['pauseOnFocus'] = [
			'group'      => 'preview',
			'label'      => esc_html__( 'Pause on focus', 'max-addons' ),
			'type'       => 'checkbox',
			'inline'     => true,
			'required'   => [ 'autoplay', '!=', '' ],
			'fullAccess' => true,
		];

		$this->controls['interval'] = [
			'group'       => 'preview',
			'label'       => esc_html__( 'Interval in ms', 'max-addons' ),
			'type'        => 'number',
			'placeholder' => 3000,
			'required'    => [ 'autoplay', '!=', '' ],
			'fullAccess'  => true,
		];

		$this->controls['linkToSeparator'] = [
			'tab'   => 'content',
			'group' => 'preview',
			'type'  => 'separator',
			'label' => esc_html__( 'Link To', 'max-addons' ),
		];

		$this->controls['linkTo'] = [
			'tab'         => 'content',
			'group'       => 'preview',
			'label'       => esc_html__( 'Link to', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'lightbox'   => esc_html__( 'Lightbox', 'max-addons' ),
				'attachment' => esc_html__( 'Attachment Page', 'max-addons' ),
				'media'      => esc_html__( 'Media File', 'max-addons' ),
				'custom'     => esc_html__( 'Custom URL', 'max-addons' ),
			],
			'inline'      => true,
			'placeholder' => esc_html__( 'None', 'max-addons' ),
		];

		$this->controls['lightboxImageSize'] = [
			'tab'         => 'content',
			'group'       => 'preview',
			'label'       => esc_html__( 'Lightbox', 'max-addons' ) . ': ' . esc_html__( 'Image size', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['imageSizes'],
			'placeholder' => 'full',
			'required'    => [ 'linkTo', '=', 'lightbox' ],
		];

		$this->controls['lightboxAnimationType'] = [
			'tab'         => 'content',
			'group'       => 'preview',
			'label'       => esc_html__( 'Lightbox animation type', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['lightboxAnimationTypes'],
			'placeholder' => esc_html__( 'Zoom', 'max-addons' ),
			'required'    => [ 'linkTo', '=', 'lightbox' ],
		];

		$this->controls['linkCustom'] = [
			'tab'         => 'content',
			'group'   	  => 'preview',
			'label'       => esc_html__( 'Custom links', 'max-addons' ),
			'type'        => 'repeater',
			'fields'      => [
				'link' => [
					'label'   => esc_html__( 'Link', 'max-addons' ),
					'type'    => 'link',
					'exclude' => [
						'lightboxImage',
						'lightboxVideo',
					],
				],
			],
			'placeholder' => esc_html__( 'Custom link', 'max-addons' ),
			'required'    => [ 'linkTo', '=', 'custom' ],
		];

		$this->controls['fieldsSeparator'] = [
			'tab'   => 'content',
			'group' => 'preview',
			'type'  => 'separator',
			'label' => esc_html__( 'Fields', 'max-addons' ),
		];

		$this->controls['fields'] = [
			'tab'           => 'content',
			'group'         => 'preview',
			'placeholder'   => esc_html__( 'Field', 'max-addons' ),
			'type'          => 'repeater',
			'selector'      => 'fieldId',
			'titleProperty' => 'dynamicData',
			'fields'        => [

				'dynamicData'       => [
					'label' => esc_html__( 'Dynamic data', 'max-addons' ),
					'type'  => 'text',
				],

				'tag'               => [
					'label'       => esc_html__( 'HTML tag', 'max-addons' ),
					'type'        => 'select',
					'options'     => [
						'div' => 'div',
						'p'   => 'p',
						'h1'  => 'h1',
						'h2'  => 'h2',
						'h3'  => 'h3',
						'h4'  => 'h4',
						'h5'  => 'h5',
						'h6'  => 'h6',
					],
					'inline'      => true,
					'placeholder' => 'div',
				],

				'dynamicMargin'     => [
					'label' => esc_html__( 'Margin', 'max-addons' ),
					'type'  => 'spacing',
					'css'   => [
						[
							'property' => 'margin',
						],
					],
				],

				'dynamicPadding'    => [
					'label' => esc_html__( 'Padding', 'max-addons' ),
					'type'  => 'spacing',
					'css'   => [
						[
							'property' => 'padding',
						],
					],
				],

				'dynamicBackground' => [
					'label' => esc_html__( 'Background color', 'max-addons' ),
					'type'  => 'color',
					'css'   => [
						[
							'property' => 'background-color',
						],
					],
				],

				'dynamicBorder'     => [
					'label' => esc_html__( 'Border', 'max-addons' ),
					'type'  => 'border',
					'css'   => [
						[
							'property' => 'border',
						],
					],
				],

				'dynamicTypography' => [
					'label' => esc_html__( 'Typography', 'max-addons' ),
					'type'  => 'typography',
					'css'   => [
						[
							'property' => 'font',
						],
					],
				],
			],
		];

		$this->controls['contentSeparator'] = [
			'tab'      => 'content',
			'group'    => 'preview',
			'label'    => esc_html__( 'Content', 'max-addons' ),
			'type'     => 'separator',
			'required' => [ 'fields', '!=', '' ],
		];

		$this->controls['contentWidth'] = [
			'tab'      => 'content',
			'group'    => 'preview',
			'label'    => esc_html__( 'Content width', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'width',
					'selector' => '.mab-thumbnail-slider-content',
				]
			],
			'required' => [ 'fields', '!=', '' ],
		];

		$this->controls['contentBackgroundColor'] = [
			'tab'      => 'content',
			'group'    => 'preview',
			'label'    => esc_html__( 'Content background', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.mab-thumbnail-slider-content',
				]
			],
			'required' => [ 'fields', '!=', '' ],
		];

		$this->controls['contentTypography'] = [
			'tab'   => 'style',
			'group' => 'preview',
			'label' => esc_html__( 'Content typography', 'max-addons' ),
			'type'  => 'typography',
			'css'   => [
				[
					'property' => 'font',
					'selector' => '.mab-thumbnail-slider-content',
				],
			],
			'required' => [ 'fields', '!=', '' ],
		];

		$this->controls['contentMargin'] = [
			'tab'      => 'content',
			'group'    => 'preview',
			'label'    => esc_html__( 'Content margin', 'max-addons' ),
			'type'     => 'spacing',
			'css'      => [
				[
					'property' => 'margin',
					'selector' => '.mab-thumbnail-slider-content',
				],
			],
			'required' => [ 'fields', '!=', '' ],
		];

		$this->controls['contentPadding'] = [
			'tab'         => 'content',
			'group'       => 'preview',
			'label'       => esc_html__( 'Content padding', 'max-addons' ),
			'type'        => 'spacing',
			'css'         => [
				[
					'property' => 'padding',
					'selector' => '.mab-thumbnail-slider-content',
				],
			],
			'placeholder' => [
				'top'    => 30,
				'right'  => 60,
				'bottom' => 30,
				'left'   => 60,
			],
			'required'    => [ 'fields', '!=', '' ],
		];

		$this->controls['contentAlignHorizontal'] = [
			'tab'      => 'content',
			'group'    => 'preview',
			'label'    => esc_html__( 'Content align horizontal', 'max-addons' ),
			'type'     => 'justify-content',
			'exclude'  => 'space',
			'css'      => [
				[
					'property' => 'justify-content',
					'selector' => '.mab-thumbnail-slider-content-wrap',
				],
			],
			'required' => [ 'fields', '!=', '' ],
		];

		$this->controls['contentAlignVertical'] = [
			'tab'      => 'content',
			'group'    => 'preview',
			'label'    => esc_html__( 'Content align vertical', 'max-addons' ),
			'type'     => 'align-items',
			'exclude'  => 'stretch',
			'css'      => [
				[
					'property' => 'align-items',
					'selector' => '.mab-thumbnail-slider-content-wrap',
				],
			],
			'required' => [ 'fields', '!=', '' ],
		];

		$this->controls['contentTextAlign'] = [
			'tab'      => 'content',
			'group'    => 'preview',
			'type'     => 'text-align',
			'label'    => esc_html__( 'Content text align', 'max-addons' ),
			'css'      => [
				[
					'property' => 'text-align',
					'selector' => '.mab-thumbnail-slider-content',
				],
			],
			'required' => [ 'fields', '!=', '' ],
		];
	}

	// Set thumbnails controls
	public function set_thumbs_controls() {
		$this->controls['thumbsCarousel'] = [
			'group'      => 'thumbnails',
			'label'      => esc_html__( 'Carousel', 'max-addons' ),
			'type'       => 'checkbox',
			'defauly'    => true,
			'fullAccess' => true,
		];

		$this->controls['thumbsColumns'] = [
			'tab'      => 'content',
			'group'    => 'thumbnails',
			'label'    => esc_html__( 'Columns', 'max-addons' ),
			'type'     => 'number',
			'min'      => 1,
			'max'      => 6,
			'css'      => [
				[
					'property' => '--columns',
					'selector' => '.mab-thumbnail-slider-thumbs',
				],
			],
			'required' => [ 'thumbsCarousel', '=', '' ],
		];

		$this->controls['thumbsGap'] = [
			'tab'         => 'content',
			'group'       => 'thumbnails',
			'label'       => esc_html__( 'Spacing', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'css'         => [
				[
					'property' => '--gutter',
					'selector' => '.mab-thumbnail-slider-thumbs',
				],
			],
			'placeholder' => 0,
			'breakpoints' => true,
			'fullAccess'  => true,
			'required'    => [ 'thumbsCarousel', '=', '' ],
		];

		$this->controls['gap'] = [
			'tab'         => 'content',
			'group'       => 'thumbnails',
			'label'       => esc_html__( 'Spacing', 'max-addons' ) . ' (px)',
			'type'        => 'number',
			'placeholder' => 0,
			'breakpoints' => true,
			'fullAccess'  => true,
			'required'    => [ 'thumbsCarousel', '!=', '' ],
		];

		$this->controls['perPage'] = [
			'tab'         => 'content',
			'group'       => 'thumbnails',
			'label'       => esc_html__( 'Items to show', 'max-addons' ),
			'type'        => 'number',
			'placeholder' => 3,
			'breakpoints' => true,
			'fullAccess'  => true,
			'required'    => [ 'thumbsCarousel', '!=', '' ],
		];

		$this->controls['perMove'] = [
			'tab'         => 'content',
			'group'       => 'thumbnails',
			'label'       => esc_html__( 'Items to scroll', 'max-addons' ),
			'type'        => 'number',
			'placeholder' => 1,
			'breakpoints' => true,
			'fullAccess'  => true,
			'required'    => [ 'thumbsCarousel', '!=', '' ],
		];

		$this->controls['thumbnailOpacity'] = [
			'tab'         => 'content',
			'group'       => 'thumbnails',
			'label'       => esc_html__( 'Opacity', 'max-addons' ),
			'type'        => 'number',
			'step'        => 0.1,
			'css'         => [
				[
					'selector' => '.mab-thumbnail-slider-thumbs .bricks-layout-item:not(.is-active)',
					'property' => 'opacity',
				],
			],
			'placeholder' => '0.3',
		];

		$this->controls['thumbnailActiveOpacity'] = [
			'tab'   => 'content',
			'group' => 'thumbnails',
			'label' => esc_html__( 'Opacity', 'max-addons' ) . ' (' . esc_html__( 'Active', 'max-addons' ) . ')',
			'type'  => 'number',
			'step'  => 0.1,
			'css'   => [
				[
					'selector' => '.mab-thumbnail-slider-thumbs .bricks-layout-item.is-active',
					'property' => 'opacity',
				],
			],
		];

		$this->controls['thumbnailBorder'] = [
			'tab'      => 'content',
			'group'    => 'thumbnails',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'selector' => '.mab-thumbnail-slider-thumbs .bricks-layout-item',
					'property' => 'border',
				],
			],
		];

		$this->controls['thumbnailActiveBorder'] = [
			'tab'   => 'content',
			'group' => 'thumbnails',
			'label' => esc_html__( 'Border', 'max-addons' ) . ' (' . esc_html__( 'Active', 'max-addons' ) . ')',
			'type'  => 'border',
			'css'   => [
				[
					'selector' => '.mab-thumbnail-slider-thumbs .bricks-layout-item.is-active',
					'property' => 'border',
				],
			],
		];
	}

	// Set navigation controls
	public function set_navigation_controls() {

		$this->controls['arrows'] = [
			'group'      => 'arrows',
			'label'      => esc_html__( 'Show', 'max-addons' ),
			'type'       => 'checkbox',
			'inline'     => true,
			'rerender'   => true,
			'fullAccess' => true,
		];

		$this->controls['arrowHeight'] = [
			'group'       => 'arrows',
			'label'       => esc_html__( 'Height', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'css'         => [
				[
					'property' => 'height',
					'selector' => '.splide__arrow',
				],
			],
			'placeholder' => 50,
			'required'    => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowWidth'] = [
			'group'       => 'arrows',
			'label'       => esc_html__( 'Width', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'css'         => [
				[
					'property' => 'width',
					'selector' => '.splide__arrow',
				],
			],
			'placeholder' => 50,
			'required'    => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowBackground'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Background', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.splide__arrow',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowBorder'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.splide__arrow',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowColor'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Color', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.splide__arrow',
				],
				[
					'property' => 'fill',
					'selector' => '.splide__arrow svg',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowSize'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Size', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'font-size',
					'selector' => '.splide__arrow',
				],
				[
					'property' => 'height',
					'selector' => '.splide__arrow svg',
				],
				[
					'property' => 'width',
					'selector' => '.splide__arrow svg',
				],
				[
					'property' => 'min-height',
					'selector' => '.splide__arrow',
				],
				[
					'property' => 'min-width',
					'selector' => '.splide__arrow',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		// text-shadow (@since 1.8.5)
		$this->controls['arrowTextShadow'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Text shadow', 'max-addons' ),
			'type'     => 'text-shadow',
			'css'      => [
				[
					'property' => 'text-shadow',
					'selector' => '.splide__arrow',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		// Custom arrows typography
		$this->controls['arrowTypography'] = [
			'deprecated' => true, // @since 1.8.5 (use 'arrowTextShadow' setting above)
			'group'      => 'arrows',
			'label'      => esc_html__( 'Typography', 'max-addons' ),
			'type'       => 'typography',
			'css'        => [
				[
					'property' => 'font',
					'selector' => '.splide__arrow',
				],
			],
			'exclude'    => [
				'font-family',
				'font-weight',
				'font-style',
				'text-align',
				'letter-spacing',
				'line-height',
				'text-decoration',
				'text-transform',
			],
			'required'   => [
				[ 'arrows', '!=', '' ],
				[ 'prevArrow.icon', '!=', '' ],
				[ 'nextArrow.icon', '!=', '' ],
			],
		];

		$this->controls['disabledArrowSep'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Disabled', 'max-addons' ),
			'type'     => 'separator',
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowDisabledBackground'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Background', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.splide__arrow:disabled',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowDisabledBorder'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.splide__arrow:disabled',
				],
			],
			'required' => [ 'arrows', '!=', '' ],

		];

		$this->controls['arrowDisabledColor'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Color', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.splide__arrow:disabled',
				],
				[
					'property' => 'fill',
					'selector' => '.splide__arrow:disabled svg',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['arrowDisabledOpacity'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Opacity', 'max-addons' ),
			'type'     => 'number',
			'inline'   => true,
			'min'      => 0,
			'max'      => 1,
			'step'     => 0.1,
			'css'      => [
				[
					'property' => 'opacity',
					'selector' => '.splide__arrow:disabled',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		// PREV ARROW

		$this->controls['prevArrowSeparator'] = [
			'group'      => 'arrows',
			'label'      => esc_html__( 'Prev arrow', 'max-addons' ),
			'type'       => 'separator',
			'required'   => [ 'arrows', '!=', '' ],
			'fullAccess' => true,
		];

		$this->controls['prevArrow'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Prev arrow', 'max-addons' ),
			'type'     => 'icon',
			'rerender' => true,
			'css'      => [
				[
					'selector' => '.splide__arrow--prev > *',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['prevArrowTop'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Top', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'top',
					'selector' => '.splide__arrow--prev',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['prevArrowRight'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Right', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'right',
					'selector' => '.splide__arrow--prev',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['prevArrowBottom'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Bottom', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'bottom',
					'selector' => '.splide__arrow--prev',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['prevArrowLeft'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Left', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'left',
					'selector' => '.splide__arrow--prev',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['prevArrowTransform'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Transform', 'max-addons' ),
			'type'     => 'transform',
			'css'      => [
				[
					'property' => 'transform',
					'selector' => '.splide__arrow--prev',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['prevArrowTransformInfo'] = [
			'group'    => 'arrows',
			'content'  => esc_html__( 'Please make sure to set the "Scale X" value inside the transform setting above to "-1".', 'max-addons' ),
			'type'     => 'info',
			'required' => [
				[ 'arrows', '!=', '' ],
				[ 'prevArrow', '=', '' ],
				[ 'prevArrowTransform', '!=', '' ],
			],
		];

		// NEXT ARROW

		$this->controls['nextArrowSeparator'] = [
			'group'      => 'arrows',
			'label'      => esc_html__( 'Next arrow', 'max-addons' ),
			'type'       => 'separator',
			'required'   => [ 'arrows', '!=', '' ],
			'fullAccess' => true,
		];

		$this->controls['nextArrow'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Next arrow', 'max-addons' ),
			'type'     => 'icon',
			'rerender' => true,
			'css'      => [
				[
					'selector' => '.splide__arrow--next > *',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['nextArrowTop'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Top', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'top',
					'selector' => '.splide__arrow--next',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['nextArrowRight'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Right', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'right',
					'selector' => '.splide__arrow--next',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['nextArrowBottom'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Bottom', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'bottom',
					'selector' => '.splide__arrow--next',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['nextArrowLeft'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Left', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'left',
					'selector' => '.splide__arrow--next',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];

		$this->controls['nextArrowTransform'] = [
			'group'    => 'arrows',
			'label'    => esc_html__( 'Transform', 'max-addons' ),
			'type'     => 'transform',
			'css'      => [
				[
					'property' => 'transform',
					'selector' => '.splide__arrow--next',
				],
			],
			'required' => [ 'arrows', '!=', '' ],
		];
		
	}

	// Set pagination controls
	public function set_pagination_controls() {

		$this->controls['pagination'] = [
			'group'      => 'pagination',
			'label'      => esc_html__( 'Show', 'max-addons' ),
			'type'       => 'checkbox',
			'inline'     => true,
			'rerender'   => true,
			'default'    => true,
			'fullAccess' => true,
		];

		$this->controls['paginationSpacing'] = [
			'group'       => 'pagination',
			'label'       => esc_html__( 'Margin', 'max-addons' ),
			'type'        => 'spacing',
			'css'         => [
				[
					'property' => 'margin',
					'selector' => '.splide__pagination .splide__pagination__page',
				],
			],
			'placeholder' => [
				'top'    => '5px',
				'right'  => '5px',
				'bottom' => '5px',
				'left'   => '5px',
			],
			'required'    => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationHeight'] = [
			'group'       => 'pagination',
			'label'       => esc_html__( 'Height', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'units'       => [
				'px' => [
					'min' => 1,
					'max' => 100,
				],
			],
			'css'         => [
				[
					'property' => 'height',
					'selector' => '.splide__pagination .splide__pagination__page',
				],
			],
			'placeholder' => '10px',
			'required'    => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationWidth'] = [
			'group'       => 'pagination',
			'label'       => esc_html__( 'Width', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'units'       => [
				'px' => [
					'min' => 1,
					'max' => 100,
				],
			],
			'css'         => [
				[
					'property' => 'width',
					'selector' => '.splide__pagination .splide__pagination__page',
				],
			],
			'placeholder' => '10px',
			'required'    => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationColor'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Color', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.splide__pagination .splide__pagination__page',
				],
				[
					'property' => 'background-color',
					'selector' => '.splide__pagination .splide__pagination__page',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationBorder'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.splide__pagination .splide__pagination__page',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];

		// ACTIVE

		$this->controls['paginationActiveSeparator'] = [
			'group'      => 'pagination',
			'label'      => esc_html__( 'Active', 'max-addons' ),
			'type'       => 'separator',
			'required'   => [ 'pagination', '!=', '' ],
			'fullAccess' => true,
		];

		$this->controls['paginationHeightActive'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Height', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'height',
					'selector' => '.splide__pagination .splide__pagination__page.is-active',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationWidthActive'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Width', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'width',
					'selector' => '.splide__pagination .splide__pagination__page.is-active',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationColorActive'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Color', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.splide__pagination .splide__pagination__page.is-active',
				],
				[
					'property' => 'background-color',
					'selector' => '.splide__pagination .splide__pagination__page.is-active',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationBorderActive'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.splide__pagination .splide__pagination__page.is-active',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];

		// POSITION

		$this->controls['paginationPositionSeparator'] = [
			'group'      => 'pagination',
			'label'      => esc_html__( 'Position', 'max-addons' ),
			'type'       => 'separator',
			'required'   => [ 'pagination', '!=', '' ],
			'fullAccess' => true,
		];

		$this->controls['paginationTop'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Top', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'top',
					'selector' => '.splide__pagination',
				],
				[
					'property' => 'bottom',
					'value'    => 'auto',
					'selector' => '.splide__pagination',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationRight'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Right', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'right',
					'selector' => '.splide__pagination',
				],
				[
					'property' => 'left',
					'value'    => 'auto',
					'selector' => '.splide__pagination',
				],
				[
					'property' => 'transform',
					'selector' => '.splide__pagination',
					'value'    => 'translateX(0)',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationBottom'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Bottom', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'bottom',
					'selector' => '.splide__pagination',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['paginationLeft'] = [
			'group'    => 'pagination',
			'label'    => esc_html__( 'Left', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'left',
					'selector' => '.splide__pagination',
				],
				[
					'property' => 'right',
					'value'    => 'auto',
					'selector' => '.splide__pagination',
				],
				[
					'property' => 'transform',
					'selector' => '.splide__pagination',
					'value'    => 'translateX(0)',
				],
			],
			'required' => [ 'pagination', '!=', '' ],
		];
	}

	public function slider_settings() {
		$settings = $this->settings;

		$type = ! empty( $settings['type'] ) ? $settings['type'] : 'loop';
		$thumbnail_position = ! empty( $settings['thumbnailPosition'] ) ? $settings['thumbnailPosition'] : '';

		// Arrows
		$arrows = isset( $settings['arrows'] );

		if ( $arrows ) {
			// Custom arrows set OR use default splide SVG arrows if no custom arrows set
			$arrows = ( ! empty( $settings['prevArrow'] ) && ! empty( $settings['nextArrow'] ) ) || ( empty( $settings['prevArrow'] ) && empty( $settings['nextArrow'] ) );
		}

		$direction = ! empty( $settings['direction'] ) ? $settings['direction'] : ( is_rtl() ? 'rtl' : 'ltr' );
		$height    = ! empty( $settings['height'] ) ? $settings['height'] : '';

		if ( 'ttb' === $direction ) {
			$height = ! empty( $settings['height'] ) ? $settings['height'] : '350px';
		}

		$splide_options = [
			'type'         => $type,
			'direction'    => $direction,
			'start'        => ! empty( $settings['start'] ) ? $settings['start'] : 0,
			'speed'        => ! empty( $settings['speed'] ) ? $settings['speed'] : 400,
			'interval'     => ! empty( $settings['interval'] ) ? $settings['interval'] : 3000,
			'autoplay'     => isset( $settings['autoplay'] ),
			'pauseOnHover' => isset( $settings['pauseOnHover'] ),
			'pauseOnFocus' => isset( $settings['pauseOnFocus'] ),
			'rewind'       => true,
			'arrows'       => $arrows,
			'pagination'   => isset( $settings['pagination'] ),
		];

		if ( $height ) {
			$splide_options['height'] = $height;
		}

		if ( is_array( $splide_options ) ) {
			$splide_options = wp_json_encode( $splide_options );
		}

		// Remove line breaks
		$splide_options = str_replace( [ "\r", "\n", ' ' ], '', $splide_options );

		$this->set_attribute( '_root', 'data-splide', trim( $splide_options ) );
	}

	public function slider_thumbs_settings() {
		$settings = $this->settings;

		$type = 'loop';

		// Spacing requires a unit
		$gap = ! empty( $settings['gap'] ) ? $settings['gap'] : 0;

		// Add default unit
		if ( is_numeric( $gap ) ) {
			$gap = "{$gap}px";
		}

		$thumbnail_position = ! empty( $settings['thumbnailPosition'] ) ? $settings['thumbnailPosition'] : '';
		$direction = ( 'left' === $thumbnail_position || 'right' === $thumbnail_position ) ? 'ttb' : 'ltr';

		$splide_options = [
			'rewind'       => true,
			'direction'    => $direction,
			'height'       => ( 'left' === $thumbnail_position || 'right' === $thumbnail_position ) ? ( ! empty( $settings['height'] ) ? $settings['height'] : '350px' ) : '',
			'gap'          => $gap,
			'start'        => ! empty( $settings['start'] ) ? $settings['start'] : 0,
			'perPage'      => ! empty( $settings['perPage'] ) && $type !== 'fade' ? $settings['perPage'] : 3,
			'perMove'      => ! empty( $settings['perMove'] ) && $type !== 'fade' ? $settings['perMove'] : 1,
			'speed'        => ! empty( $settings['speed'] ) ? $settings['speed'] : 400,
			/* 'interval'     => ! empty( $settings['interval'] ) ? $settings['interval'] : 3000,
			'autoplay'     => isset( $settings['autoplay'] ),
			'pauseOnHover' => isset( $settings['pauseOnHover'] ),
			'pauseOnFocus' => isset( $settings['pauseOnFocus'] ), */
			'arrows'       => false,
			'pagination'   => false,
			'isNavigation' => true,
		];

		// STEP: Add settings per breakpoints to splide options
		$breakpoints = [];

		foreach ( \Bricks\Breakpoints::$breakpoints as $breakpoint ) {
			foreach ( array_keys( $splide_options ) as $option ) {
				$setting_key      = "$option:{$breakpoint['key']}";
				$breakpoint_width = ! empty( $breakpoint['width'] ) ? $breakpoint['width'] : false;
				$setting_value    = isset( $settings[ $setting_key ] ) ? $settings[ $setting_key ] : false;

				// Spacing requires a unit
				if ( $option === 'gap' ) {
					// Add default unit
					if ( is_numeric( $setting_value ) ) {
						$setting_value = "{$setting_value}px";
					}
				}

				if ( $breakpoint_width && $setting_value !== false ) {
					$breakpoints[ $breakpoint_width ][ $option ] = $setting_value;
				}
			}
		}

		if ( count( $breakpoints ) ) {
			$splide_options['breakpoints'] = $breakpoints;
		}

		if ( is_array( $splide_options ) ) {
			$splide_options = wp_json_encode( $splide_options );
		}

		if ( is_array( $splide_options ) ) {
			$splide_options = wp_json_encode( $splide_options );
		}

		// Remove line breaks
		$splide_options = str_replace( [ "\r", "\n", ' ' ], '', $splide_options );

		$this->set_attribute( 'slider_nav', 'data-splide', trim( $splide_options ) );
	}

	/**
	 * Render arrows (use custom HTML solution as splideJS only accepts SVG path via 'arrowPath')
	 */
	public function render_arrows() {
		$prev_arrow = ! empty( $this->settings['prevArrow'] ) ? self::render_icon( $this->settings['prevArrow'] ) : false;
		$next_arrow = ! empty( $this->settings['nextArrow'] ) ? self::render_icon( $this->settings['nextArrow'] ) : false;

		if ( ! $prev_arrow && ! $next_arrow ) {
			return;
		}

		$output = '<div class="splide__arrows custom">';

		if ( $prev_arrow ) {
			$output .= '<button class="splide__arrow splide__arrow--prev" type="button">' . $prev_arrow . '</button>';
		}

		if ( $next_arrow ) {
			$output .= '<button class="splide__arrow splide__arrow--next" type="button">' . $next_arrow . '</button>';
		}

		$output .= '</div>';

		return $output;
	}

	public function render_preview( $source, $carousel_posts ) {
		$settings             = $this->settings;
		$preview_image_size   = ! empty( $settings['previewImageSize'] ) ? $settings['previewImageSize'] : 'full';
		$thumbnail_image_size = ! empty( $settings['thumbnailImageSize'] ) ? $settings['thumbnailImageSize'] : 'thumbnail';
		?>
		<div class="mab-thumbnail-slider-preview splide">
			<div class="splide__track">
				<div class="splide__list">
					<?php
						foreach ( $carousel_posts as $index => $item ) {
							$image_id  = false;
							$has_image = false;

							$preview_slide_key = 'preview-slide-' . $index;

							if ( $source === 'media' ) {
								$image_id = $item->ID ?? false;
							}

							if ( $source === 'posts' && has_post_thumbnail( $item->ID ) ) {
								$image_id = get_post_thumbnail_id( $item->ID ) ?? false;
							}

							if ( $image_id ) {
								$has_image = true;
							}

							if ( isset( $settings['linkTo'] ) ) {
								$this->set_attribute( $preview_slide_key, 'class', 'mab-thumbnail-slider-link' );

								if ( $settings['linkTo'] == 'lightbox' ) {
									/* $this->set_attribute( $preview_slide_key, 'class', 'bricks-lightbox' );

									if ( ! empty( $settings['lightboxAnimationType'] ) ) {
										$this->set_attribute( $preview_slide_key, 'data-animation-type', esc_attr( $settings['lightboxAnimationType'] ) );
									} */

									if ( $has_image ) {
										$lightbox_image_size = ! empty( $settings['lightboxImageSize'] ) ? $settings['lightboxImageSize'] : 'full';
										$lightbox_image      = wp_get_attachment_image_src( $image_id, $lightbox_image_size );

										$this->set_attribute( $preview_slide_key, 'data-pswp-src', $lightbox_image[0] );
										$this->set_attribute( $preview_slide_key, 'data-pswp-width', $lightbox_image[1] );
										$this->set_attribute( $preview_slide_key, 'data-pswp-height', $lightbox_image[2] );
										// $this->set_attribute( $preview_slide_key, 'data-bricks-lightbox-index', $count );
										$this->set_attribute( $preview_slide_key, 'data-pswp-id', $this->id );
									}

								} elseif ( $settings['linkTo'] === 'custom' && isset( $settings['linkCustom'][ $index ]['link'] ) ) {

									$this->set_link_attributes( $preview_slide_key, $settings['linkCustom'][ $index ]['link'] );

								} elseif ( $settings['linkTo'] === 'attachment' ) {

									if ( $has_image ) {
										$link = get_attachment_link( $image_id );

										$this->set_attribute( $preview_slide_key, 'href', $link );
									}

								} elseif ( $settings['linkTo'] === 'media' ) {

									if ( $has_image ) {
										$link = wp_get_attachment_image_url( $image_id, 'full' );

										$this->set_attribute( $preview_slide_key, 'href', $link );
									}
								}
							}

							if ( $has_image ) { ?>
								<div class="splide__slide repeater-item">
									<?php if ( isset( $settings['linkTo'] ) ) { ?>
									<a <?php echo wp_kses_post( $this->render_attributes( $preview_slide_key ) ); ?>></a>
									<?php } ?>
									<?php
									echo wp_get_attachment_image( $image_id, $preview_image_size );

									if ( ! empty( $settings['fields'] ) && is_array( $settings['fields'] ) ) {
										if ( count( $settings['fields'] ) ) {
											echo '<div class="mab-thumbnail-slider-content-wrap">';
												echo '<div class="mab-thumbnail-slider-content">';
												echo \Bricks\Frontend::get_content_wrapper( $settings, $settings['fields'], $item );
												echo '</div>';
											echo '</div>';
										}
									}
									?>
								</div>
								<?php
							}
						}
					?>
				</div>
			</div>
			<?php echo $this->render_arrows(); ?>
		</div>
		<?php
	}

	public function render_thumbnails( $source, $carousel_posts ) {
		$settings   = $this->settings;
		$image_size = ! empty( $settings['thumbnailImageSize'] ) ? $settings['thumbnailImageSize'] : 'thumbnail';

		$is_carousel = isset( $settings['thumbsCarousel'] );

		if ( $is_carousel ) {
			$classes = array(
				'mab-thumbnail-slider-thumbs',
				'splide',
			);

			$layout = 'carousel';
		} else {
			$classes = array(
				'mab-thumbnail-slider-thumbs',
				'bricks-layout-wrapper',
			);

			$layout = 'grid';
		}

		$this->set_attribute( 'slider_nav', 'class', $classes );
		$this->set_attribute( 'slider_nav', 'data-layout', $layout );

		$this->slider_thumbs_settings();
		?>
		<div class="mab-thumbnail-slider-thumbs-wrap">
			<div <?php echo wp_kses_post( $this->render_attributes( 'slider_nav' ) ); ?>><?php
			if ( $is_carousel ) { ?>
				<div class="splide__track">
					<div class="splide__list"><?php
			}
						foreach ( $carousel_posts as $index => $item ) {
							$image_id  = false;
							$has_image = false;
							if ( $source === 'media' ) {
								$image_id = $item->ID ?? false;
							}

							if ( $source === 'posts' && has_post_thumbnail( $item->ID ) ) {
								$image_id = get_post_thumbnail_id( $item->ID ) ?? false;
							}

							if ( $image_id ) {
								$has_image = true;
							}

							$this->set_attribute( "item-{$index}", 'class', 'bricks-layout-item' );

							if ( $is_carousel ) {
								$this->set_attribute( "item-{$index}", 'class', 'splide__slide' );
							}

							if ( $has_image ) {
								?>
								<div <?php echo wp_kses_post( $this->render_attributes( "item-{$index}" ) ); ?>>
									<?php echo wp_get_attachment_image( $image_id, $image_size ); ?>
								</div>
								<?php
							}
						}
				if ( $is_carousel ) { ?>
					</div>
				</div><?php
				} ?>
			</div>
		</div>
		<?php
	}

	public function render() {
		$settings  = $this->settings;
		$image_id  = false;
		$has_image = false;

		$source = $settings['source'] ?? 'media';

		// TYPE: MEDIA
		if ( $source === 'media' ) {
			// NOTE: $this->element['settings'] is not updated by bricks/element/settings filter. Best if we update in base.php (#86bwkh7y2; @since 1.9.4)
			$this->element['settings'] = $this->settings;
			// Dynamic data already checked inside this helper function (@since 1.9.3)
			$query_settings = Helper::populate_query_vars_for_element( $this->element, $this->post_id );

			if ( ! empty( $query_settings ) ) {
				// Set lang to empty string if Polylang is active to fetch all images even if they are not translated (@since 1.9.4)
				if ( \Bricks\Integrations\Polylang\Polylang::$is_active ) {
					$query_settings['lang'] = '';
				}

				// Add query_settings to element_settings under query key
				$this->element['settings']['query'] = $query_settings;

				$carousel_query = new \Bricks\Query( $this->element );

				// Destroy query to explicitly remove it from the global store
				$carousel_query->destroy();

				$carousel_query = $carousel_query ? $carousel_query->query_result : false;
			}

			// Element placeholder
			else {
				return $this->render_element_placeholder( [ 'title' => esc_html__( 'No image selected.', 'max-addons' ) ] );
			}
		}

		// TYPE: POSTS
		elseif ( $source === 'posts' ) {
			$carousel_query = new \Bricks\Query(
				[
					'id'       => $this->id,
					'settings' => $settings,
				]
			);

			if ( $carousel_query->count === 0 ) {
				// No results: Empty by default (@since 1.4)
				$no_results_content = $carousel_query->get_no_results_content();

				if ( ! $no_results_content ) {
					return $this->render_element_placeholder( [ 'title' => esc_html__( 'No results', 'max-addons' ) ] );
				}
			}

			// Destroy query to explicitly remove it from the global store
			$carousel_query->destroy();

			$carousel_query = $carousel_query ? $carousel_query->query_result : false;
		}

		$carousel_posts = $carousel_query ? $carousel_query->get_posts() : [];

		$preview_image_size   = ! empty( $settings['previewImageSize'] ) ? $settings['previewImageSize'] : 'full';
		$thumbnail_image_size = ! empty( $settings['thumbnailImageSize'] ) ? $settings['thumbnailImageSize'] : 'thumbnail';

		$this->slider_settings();

		$thumbnail_position = ! empty( $settings['thumbnailPosition'] ) ? $settings['thumbnailPosition'] : '';
		
		$this->set_attribute( 'thumbnail-slider', 'class', [ 'mab-thumbnail-slider', 'mab-thumbnail-slider-' . $thumbnail_position ] );

		if ( $thumbnail_position ) {
			$this->set_attribute( '_root', 'data-pos', esc_attr( $settings['thumbnailPosition'] ) );
		}

		if ( ( 'left' === $thumbnail_position || 'right' === $thumbnail_position ) && isset( $settings['stackOn'] ) ) {
			$breakpoint = \Bricks\Breakpoints::get_breakpoint_by( 'key', $settings['stackOn'] );

			$this->set_attribute( '_root', 'data-stack-on', intval( $breakpoint['width'] ) );
		}

		if ( isset( $settings['linkTo'] ) && $settings['linkTo'] == 'lightbox' ) {
			$this->set_attribute( '_root', 'class', 'bricks-lightbox brxe-carousel' );

			if ( ! empty( $settings['lightboxAnimationType'] ) ) {
				$this->set_attribute( '_root', 'data-animation-type', esc_attr( $settings['lightboxAnimationType'] ) );
			}
		}
		?>
		<div <?php echo wp_kses_post( $this->render_attributes( '_root' ) ); ?>>
			<div <?php echo wp_kses_post( $this->render_attributes( 'thumbnail-slider' ) ); ?>>
				<?php $this->render_preview( $source, $carousel_posts ); ?>
				<?php $this->render_thumbnails( $source, $carousel_posts ); ?>
			</div>
		</div>
		<?php
	}
}
