<?php
namespace MaxAddonsPro\Elements;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Table_Element extends \Bricks\Element {
	// Element properties
	public $category     = 'max-addons-elements'; // Use predefined element category 'general'
	public $name         = 'max-table'; // Make sure to prefix your elements
	public $icon         = 'ti-widgetized max-element'; // Themify icon font class
	public $css_selector = ''; // Default CSS selector
	public $header_loop_index = 0;
	public $loop_index   = 0;
	public $deprecated   = true;
	public $scripts      = [ 'mabTable' ]; // Script(s) run when element is rendered on frontend or updated in builder

	// Return localized element label
	public function get_label() {
		return esc_html__( 'Table', 'max-addons' );
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		$table_type = isset( $this->settings['tableType'] ) ? $this->settings['tableType'] : '';
		$sortable = isset( $this->settings['sortable'] ) ? $this->settings['sortable'] : '';
		$scrollable = isset( $this->settings['scrollable'] ) ? $this->settings['scrollable'] : '';

		if ( 'responsive' === $table_type || $sortable || $scrollable ) {
			wp_enqueue_style( 'tablesaw' );
			wp_enqueue_style( 'mab-table' );
			wp_enqueue_script( 'tablesaw' );
			wp_enqueue_script( 'tablesaw-init' );
			wp_enqueue_script( 'mab-frontend' );
		}
	}

	// Set builder control groups
	public function set_control_groups() {
		$this->control_groups['general'] = [ // Unique group identifier (lowercase, no spaces)
			'title' => esc_html__( 'General', 'max-addons' ), // Localized control group title
			'tab'   => 'content', // Set to either "content" or "style"
		];

		$this->control_groups['header'] = [
			'title'    => esc_html__( 'Header', 'max-addons' ),
			'tab'      => 'content',
			'required' => [
				[ 'source', '!=', 'query' ],
				[ 'hideTableHeader', '=', '' ]
			],
		];

		$this->control_groups['body'] = [
			'title'    => esc_html__( 'Body', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'source', '!=', 'query' ],
		];

		$this->control_groups['footer'] = [
			'title'    => esc_html__( 'Footer', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'source', '!=', 'query' ],
		];

		$this->control_groups['dynamicContent'] = [
			'title' => esc_html__( 'Dynamic Content', 'max-addons' ),
			'tab'   => 'content',
			'required' => [ 'source', '=', 'query' ],
		];

		$this->control_groups['headerStyle'] = [
			'title' => esc_html__( 'Header Style', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['rows'] = [
			'title' => esc_html__( 'Rows Style', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['cells'] = [
			'title' => esc_html__( 'Cells Style', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['footerStyle'] = [
			'title' => esc_html__( 'Footer Style', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['iconStyle'] = [
			'title' => esc_html__( 'Icon Style', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['columns'] = [
			'title' => esc_html__( 'Columns', 'max-addons' ),
			'tab'   => 'content',
		];
	}

	// Set builder controls
	public function set_controls() {

		$this->set_general_controls();

		$this->set_header_controls();

		$this->set_body_controls();

		$this->set_footer_controls();

		$this->set_dynamic_content_controls();

		$this->set_header_style_controls();

		$this->set_rows_controls();

		$this->set_cells_controls();

		$this->set_footer_style_controls();

		$this->set_icon_controls();

		$this->set_columns_controls();
	}

	// Set general controls
	public function set_general_controls() {

		//$this->controls = array_replace_recursive( $this->controls, $this->get_loop_builder_controls( 'general' ) );

		$this->controls['source'] = [
			'tab'       => 'content',
			'group'     => 'general',
			'label'     => esc_html__( 'Source', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'custom' => esc_html__( 'Custom', 'max-addons' ),
				'query'  => esc_html__( 'Query', 'max-addons' ),
			],
			'inline'    => true,
			'clearable' => false,
			'default'   => 'custom',
		];

		$this->controls['query'] = [
			'tab'      => 'content',
			'group'    => 'general',
			'label'    => esc_html__( 'Query', 'max-addons' ),
			'type'     => 'query',
			'popup'    => true,
			'inline'   => true,
			'required' => [ 'source', '=', 'query' ],
		];

		$this->controls['tableType'] = [
			'tab'       => 'content',
			'group'     => 'general',
			'label'     => esc_html__( 'Table Type', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'normal'     => esc_html__( 'Normal', 'max-addons' ),
				'responsive' => esc_html__( 'Responsive', 'max-addons' ),
			],
			'inline'    => true,
			'clearable' => false,
			'default'   => 'responsive',
			'required'  => [ 'hideTableHeader', '=', '' ],
		];

		$this->controls['hideTableHeader'] = [
			'tab'   => 'content',
			'group' => 'general',
			'label' => esc_html__( 'Hide Table Header', 'max-addons' ),
			'type'  => 'checkbox',
		];

		$this->controls['sortable'] = [
			'tab'         => 'content',
			'group'       => 'general',
			'label'       => esc_html__( 'Sortable Table', 'max-addons' ),
			'description' => esc_html__( 'Enable sorting the table data by clicking on the table headers', 'max-addons' ),
			'type'        => 'checkbox',
			'required'    => [
				[ 'tableType', '=', 'responsive' ],
				[ 'hideTableHeader', '=', '' ],
			],
		];

		$this->controls['sortableDropdown'] = [
			'tab'         => 'content',
			'group'       => 'general',
			'label'       => esc_html__( 'Sortable Dropdown', 'max-addons' ),
			'description' => esc_html__( 'This will show dropdown menu to sort the table by columns', 'max-addons' ),
			'type'        => 'checkbox',
			'required'    => [
				[ 'tableType', '=', 'responsive' ],
				[ 'sortable', '!=', '' ],
				[ 'hideTableHeader', '=', '' ],
			],

		];

		$this->controls['scrollable'] = [
			'tab'         => 'content',
			'group'       => 'general',
			'label'       => esc_html__( 'Scrollable Table', 'max-addons' ),
			'description' => esc_html__( 'This will disable stacking and enable swipe/scroll when below breakpoint', 'max-addons' ),
			'type'        => 'checkbox',
			'required'    => [
				[ 'tableType', '=', 'responsive' ],
				[ 'hideTableHeader', '=', '' ],
			],
		];

		$this->controls['breakpoint'] = [
			'tab'      => 'content',
			'group'    => 'general',
			'type'     => 'text',
			'label'    => esc_html__( 'Breakpoint', 'max-addons' ),
			'inline'   => true,
			'required' => [
				[ 'tableType', '=', 'responsive' ],
				[ 'scrollable', '!=', '' ],
				[ 'hideTableHeader', '=', '' ],
			],
		];
	}

	// Set header controls
	public function set_header_controls() {
		$this->controls['headerRows'] = [
			'tab'           => 'content',
			'group'         => 'header',
			'placeholder'   => esc_html__( 'Column', 'max-addons' ),
			'type'          => 'repeater',
			'titleProperty' => 'cellText',
			'fields'        => [
				'cellText'         => [
					'label'          => esc_html__( 'Text', 'max-addons' ),
					'type'           => 'text',
					'hasDynamicData' => 'text',
				],
				'cellIconType'     => [
					'label'     => esc_html__( 'Icon Type', 'max-addons' ),
					'type'      => 'select',
					'options'   => [
						'icon'  => esc_html__( 'Icon', 'max-addons' ),
						'image' => esc_html__( 'Image', 'max-addons' ),
					],
					'inline'    => true,
					'clearable' => true,
					'default'   => '',
				],
				'cellIcon'         => [
					'label'    => esc_html__( 'Icon', 'max-addons' ),
					'type'     => 'icon',
					'default'  => [
						'library' => 'themify',
						'icon'    => 'ti-wordpress',
					],
					'required' => [ 'cellIconType', '=', [ 'icon' ] ],
				],
				'cellImage'        => [
					'label'    => esc_html__( 'Image', 'max-addons' ),
					'type'     => 'image',
					'required' => [ 'cellIconType', '=', [ 'image' ] ],
				],
				'cellIconPosition' => [
					'label'     => esc_html__( 'Icon Position', 'max-addons' ),
					'type'      => 'select',
					'options'   => [
						'before' => esc_html__( 'Before', 'max-addons' ),
						'after'  => esc_html__( 'After', 'max-addons' ),
					],
					'inline'    => true,
					'clearable' => true,
					'default'   => 'before',
					'required'  => [ 'cellIconType', '=', [ 'image', 'icon' ] ],
				],
				'colspan'          => [
					'label'  => esc_html__( 'Column Span', 'max-addons' ),
					'type'   => 'number',
					'min'    => 1,
					'max'    => 20,
					'step'   => 1,
					'inline' => true,
				],
				'rowspan'          => [
					'label'  => esc_html__( 'Row Span', 'max-addons' ),
					'type'   => 'number',
					'min'    => 1,
					'max'    => 20,
					'step'   => 1,
					'inline' => true,
				],
				'cssId'            => [
					'label'  => esc_html__( 'CSS ID', 'max-addons' ),
					'type'   => 'text',
					'inline' => true,
				],
				'cssClasses'       => [
					'label'  => esc_html__( 'CSS Classes', 'max-addons' ),
					'type'   => 'text',
					'inline' => true,
				],
			],
			'default'       => [
				[
					'cellText'         => esc_html__( 'Header Column #1', 'max-addons' ),
					'cellIconType'     => '',
					'cellIcon'         => [
						'library' => 'themify',
						'icon'    => 'ti-wordpress',
					],
					'cellIconPosition' => 'before',
				],
				[
					'cellText'         => esc_html__( 'Header Column #2', 'max-addons' ),
					'cellIconType'     => '',
					'cellIcon'         => [
						'library' => 'themify',
						'icon'    => 'ti-wordpress',
					],
					'cellIconPosition' => 'before',
				],
				[
					'cellText'         => esc_html__( 'Header Column #3', 'max-addons' ),
					'cellIconType'     => '',
					'cellIcon'         => [
						'library' => 'themify',
						'icon'    => 'ti-wordpress',
					],
					'cellIconPosition' => 'before',
				],
			],
			'required'    => [ 'hideTableHeader', '=', '' ],
		];
	}

	// Set body controls
	public function set_body_controls() {
		$this->controls['bodyRows'] = [
			'tab'           => 'content',
			'group'         => 'body',
			'placeholder'   => esc_html__( 'Item', 'max-addons' ),
			'type'          => 'repeater',
			'titleProperty' => 'tableBodyElement',
			'fields'        => [
				'tableBodyElement' => [
					'label'     => esc_html__( 'Start New', 'max-addons' ),
					'type'      => 'select',
					'options'   => [
						'row'  => esc_html__( 'Row', 'max-addons' ),
						'cell' => esc_html__( 'Cell', 'max-addons' ),
					],
					'inline'    => true,
					'clearable' => false,
				],
				'cellType'         => [
					'label'     => esc_html__( 'Cell Type', 'max-addons' ),
					'type'      => 'select',
					'options'   => [
						'td' => esc_html__( 'Default', 'max-addons' ),
						'th' => esc_html__( 'Header', 'max-addons' ),
					],
					'inline'    => true,
					'clearable' => true,
					'default'   => 'td',
					'required'  => [ 'tableBodyElement', '=', [ 'cell' ] ],
				],
				'cellText'         => [
					'label'          => esc_html__( 'Text', 'max-addons' ),
					'type'           => 'text',
					'hasDynamicData' => 'text',
					'required'       => [ 'tableBodyElement', '=', [ 'cell' ] ],
				],
				'link'             => [
					'label'    => esc_html__( 'Link', 'max-addons' ),
					'type'     => 'link',
					'required' => [ 'tableBodyElement', '=', [ 'cell' ] ],
				],
				'cellIconType'     => [
					'label'     => esc_html__( 'Icon Type', 'max-addons' ),
					'type'      => 'select',
					'options'   => [
						'icon'  => esc_html__( 'Icon', 'max-addons' ),
						'image' => esc_html__( 'Image', 'max-addons' ),
					],
					'inline'    => true,
					'clearable' => true,
					'default'   => '',
					'required'  => [ 'tableBodyElement', '=', [ 'cell' ] ],
				],
				'cellIcon'         => [
					'label'    => esc_html__( 'Icon', 'max-addons' ),
					'type'     => 'icon',
					'default'  => [
						'library' => 'themify',
						'icon'    => 'ti-wordpress',
					],
					'required' => [ 'cellIconType', '=', [ 'icon' ] ],
				],
				'cellImage'        => [
					'label'    => esc_html__( 'Image', 'max-addons' ),
					'type'     => 'image',
					'required' => [ 'cellIconType', '=', [ 'image' ] ],
				],
				'cellIconPosition' => [
					'label'     => esc_html__( 'Icon Position', 'max-addons' ),
					'type'      => 'select',
					'options'   => [
						'before' => esc_html__( 'Before', 'max-addons' ),
						'after'  => esc_html__( 'After', 'max-addons' ),
					],
					'inline'    => true,
					'clearable' => true,
					'default'   => 'before',
					'required'  => [ 'cellIconType', '=', [ 'image', 'icon' ] ],
				],
				'colspan'          => [
					'label'    => esc_html__( 'Column Span', 'max-addons' ),
					'type'     => 'number',
					'min'      => 1,
					'max'      => 20,
					'step'     => 1,
					'inline'   => true,
					'required' => [ 'tableBodyElement', '=', [ 'cell' ] ],
				],
				'rowspan'          => [
					'label'    => esc_html__( 'Row Span', 'max-addons' ),
					'type'     => 'number',
					'min'      => 1,
					'max'      => 20,
					'step'     => 1,
					'inline'   => true,
					'required' => [ 'tableBodyElement', '=', [ 'cell' ] ],
				],
				'cssId'            => [
					'label'    => esc_html__( 'CSS ID', 'max-addons' ),
					'type'     => 'text',
					'inline'   => true,
					'required' => [ 'tableBodyElement', '=', [ 'cell' ] ],
				],
				'cssClasses'       => [
					'label'    => esc_html__( 'CSS Classes', 'max-addons' ),
					'type'     => 'text',
					'inline'   => true,
					'required' => [ 'tableBodyElement', '=', [ 'cell' ] ],
				],
			],
			'default'       => [
				[
					'tableBodyElement' => 'row',
				],
				[
					'tableBodyElement' => 'cell',
					'cellText'         => esc_html__( 'Column #1', 'max-addons' ),
					'cellIconType'     => '',
					'cellIcon'         => [
						'library' => 'themify',
						'icon'    => 'ti-wordpress',
					],
					'cellType'         => 'td',
					'cellIconPosition' => 'before',
				],
				[
					'tableBodyElement' => 'cell',
					'cellText'         => esc_html__( 'Column #2', 'max-addons' ),
					'cellIconType'     => '',
					'cellIcon'         => [
						'library' => 'themify',
						'icon'    => 'ti-wordpress',
					],
					'cellType'         => 'td',
					'cellIconPosition' => 'before',
				],
				[
					'tableBodyElement' => 'cell',
					'cellText'         => esc_html__( 'Column #3', 'max-addons' ),
					'cellIconType'     => '',
					'cellIcon'         => [
						'library' => 'themify',
						'icon'    => 'ti-wordpress',
					],
					'cellType'         => 'td',
					'cellIconPosition' => 'before',
				],
			],
		];
	}

	// Set dynamic content controls
	public function set_dynamic_content_controls() {
		$this->controls['dynamicRows'] = [
			'tab'           => 'content',
			'group'         => 'dynamicContent',
			'placeholder'   => esc_html__( 'Item', 'max-addons' ),
			'type'          => 'repeater',
			'titleProperty' => 'cell',
			'fields'        => [
				'columnHeading'         => [
					'label'          => esc_html__( 'Column Heading', 'max-addons' ),
					'type'           => 'text',
					'hasDynamicData' => 'text',
				],
				'columnData'         => [
					'label'          => esc_html__( 'Column Data', 'max-addons' ),
					'type'           => 'text',
					'hasDynamicData' => 'text',
				],
				'link'             => [
					'label'    => esc_html__( 'Link', 'max-addons' ),
					'type'     => 'link',
				],
				'cellIconType'     => [
					'label'     => esc_html__( 'Icon Type', 'max-addons' ),
					'type'      => 'select',
					'options'   => [
						'icon'  => esc_html__( 'Icon', 'max-addons' ),
						'image' => esc_html__( 'Image', 'max-addons' ),
					],
					'inline'    => true,
					'clearable' => true,
					'default'   => '',
				],
				'cellIcon'         => [
					'label'    => esc_html__( 'Icon', 'max-addons' ),
					'type'     => 'icon',
					'default'  => [
						'library' => 'themify',
						'icon'    => 'ti-wordpress',
					],
					'required' => [ 'cellIconType', '=', [ 'icon' ] ],
				],
				'cellImage'        => [
					'label'    => esc_html__( 'Image', 'max-addons' ),
					'type'     => 'image',
					'required' => [ 'cellIconType', '=', [ 'image' ] ],
				],
				'cellIconPosition' => [
					'label'     => esc_html__( 'Icon Position', 'max-addons' ),
					'type'      => 'select',
					'options'   => [
						'before' => esc_html__( 'Before', 'max-addons' ),
						'after'  => esc_html__( 'After', 'max-addons' ),
					],
					'inline'    => true,
					'clearable' => true,
					'default'   => 'before',
					'required' => [ 'cellIconType', '!=', '' ],
				],
			],
			'default'       => [
				[
					'columnHeading'    => esc_html__( 'Post Title', 'max-addons' ),
					'columnData'       => '{post_title}',
				],
				[
					'columnHeading'    => esc_html__( 'Post Author', 'max-addons' ),
					'columnData'       => '{author_name}',
				],
				[
					'columnHeading'    => esc_html__( 'Post Date', 'max-addons' ),
					'columnData'       => '{post_date}',
				],
			],
			'required' => [ 'source', '=', 'query' ],
		];
	}

	// Set body controls
	public function set_footer_controls() {
		$this->controls['footerRows'] = [
			'tab'           => 'content',
			'group'         => 'footer',
			'placeholder'   => esc_html__( 'Column', 'max-addons' ),
			'type'          => 'repeater',
			'titleProperty' => 'cellText',
			'fields'        => [
				'cellText'         => [
					'label'          => esc_html__( 'Text', 'max-addons' ),
					'type'           => 'text',
					'hasDynamicData' => 'text',
				],
				'cellIconType'     => [
					'label'     => esc_html__( 'Icon Type', 'max-addons' ),
					'type'      => 'select',
					'options'   => [
						'icon'  => esc_html__( 'Icon', 'max-addons' ),
						'image' => esc_html__( 'Image', 'max-addons' ),
					],
					'inline'    => true,
					'clearable' => true,
					'default'   => '',
				],
				'cellIcon'         => [
					'label'    => esc_html__( 'Icon', 'max-addons' ),
					'type'     => 'icon',
					'default'  => [
						'library' => 'themify',
						'icon'    => 'ti-wordpress',
					],
					'required' => [ 'cellIconType', '=', [ 'icon' ] ],
				],
				'cellImage'        => [
					'label'    => esc_html__( 'Image', 'max-addons' ),
					'type'     => 'image',
					'required' => [ 'cellIconType', '=', [ 'image' ] ],
				],
				'cellIconPosition' => [
					'label'     => esc_html__( 'Icon Position', 'max-addons' ),
					'type'      => 'select',
					'options'   => [
						'before' => esc_html__( 'Before', 'max-addons' ),
						'after'  => esc_html__( 'After', 'max-addons' ),
					],
					'inline'    => true,
					'clearable' => true,
					'default'   => 'before',
					'required'  => [ 'cellIconType', '=', [ 'image', 'icon' ] ],
				],
				'colspan'          => [
					'label'  => esc_html__( 'Column Span', 'max-addons' ),
					'type'   => 'number',
					'min'    => 1,
					'max'    => 20,
					'step'   => 1,
					'inline' => true,
				],
				'rowspan'          => [
					'label'  => esc_html__( 'Row Span', 'max-addons' ),
					'type'   => 'number',
					'min'    => 1,
					'max'    => 20,
					'step'   => 1,
					'inline' => true,
				],
				'cssId'            => [
					'label'  => esc_html__( 'CSS ID', 'max-addons' ),
					'type'   => 'text',
					'inline' => true,
				],
				'cssClasses'       => [
					'label'  => esc_html__( 'CSS Classes', 'max-addons' ),
					'type'   => 'text',
					'inline' => true,
				],
			],
		];
	}

	// Set header style controls
	public function set_header_style_controls() {

		$this->controls['headerPadding'] = [
			'tab'   => 'content',
			'group' => 'headerStyle',
			'type'  => 'spacing',
			'label' => esc_html__( 'Padding', 'max-addons' ),
			'css'   => [
				[
					'property' => 'padding',
					'selector' => '.mab-table th.mab-table-cell',
				],
			],
		];

		$this->controls['headerTextAlign'] = [
			'tab'         => 'content',
			'group'       => 'headerStyle',
			'label'       => esc_html__( 'Text align', 'max-addons' ),
			'type'        => 'text-align',
			'css'         => [
				[
					'property' => 'text-align',
					'selector' => '.mab-table thead .mab-table-cell',
				],
			],
			'placeholder' => esc_html__( 'Center', 'max-addons' ),
			'inline'      => true,
		];

		$this->controls['headerTypography'] = [
			'tab'    => 'content',
			'group'  => 'headerStyle',
			'label'  => esc_html__( 'Typography', 'max-addons' ),
			'type'   => 'typography',
			'css'    => [
				[
					'property' => 'font',
					'selector' => '.mab-table th.mab-table-cell',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['headerSortableIconColor'] = [
			'tab'      => 'content',
			'group'    => 'headerStyle',
			'type'     => 'color',
			'label'    => esc_html__( 'Sortable Icon Color', 'max-addons' ),
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.mab-table th.mab-table-cell .tablesaw-sortable-arrow',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [
				[ 'tableType', '=', 'responsive' ],
				[ 'sortable', '!=', '' ],
				[ 'hideTableHeader', '=', '' ],
			],
		];

		$this->controls['headerBgColor'] = [
			'tab'     => 'content',
			'group'   => 'headerStyle',
			'type'    => 'color',
			'label'   => esc_html__( 'Background Color', 'max-addons' ),
			'css'     => [
				[
					'property' => 'background-color',
					'selector' => '.mab-table th.mab-table-cell',
				],
			],
			'default' => [
				'hex' => '#e6e6e6',
			],
			'inline'  => true,
			'small'   => true,
		];

		$this->controls['headerBorder'] = [
			'tab'     => 'content',
			'group'   => 'headerStyle',
			'label'   => esc_html__( 'Border', 'max-addons' ),
			'type'    => 'border',
			'css'     => [
				[
					'property' => 'border',
					'selector' => '.mab-table th.mab-table-cell',
				],
			],
			'inline'  => true,
			'small'   => true,
			'default' => [
				'width' => [
					'top'    => 0,
					'right'  => 1,
					'bottom' => 1,
					'left'   => 0,
				],
				'style' => 'solid',
				'color' => [
					'rgb' => 'rgba(0, 0, 0, 0.1)',
				],
			],
		];

		$this->controls['headerBoxShadow'] = [
			'tab'    => 'content',
			'group'  => 'headerStyle',
			'label'  => esc_html__( 'Box Shadow', 'max-addons' ),
			'type'   => 'box-shadow',
			'css'    => [
				[
					'property' => 'box-shadow',
					'selector' => '.mab-table thead',
				],
			],
			'inline' => true,
			'small'  => true,
		];
	}

	// Set rows controls
	public function set_rows_controls() {
		$this->controls['stripedRows'] = [
			'tab'   => 'content',
			'group' => 'rows',
			'label' => esc_html__( 'Striped Rows', 'max-addons' ),
			'type'  => 'checkbox',
		];

		$this->controls['rowsBgColor'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'type'     => 'color',
			'label'    => esc_html__( 'Background Color', 'max-addons' ),
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.mab-table tbody tr td',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'stripedRows', '=', '' ],
		];

		$this->controls['rowsTextColor'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'type'     => 'color',
			'label'    => esc_html__( 'Text Color', 'max-addons' ),
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.mab-table tbody tr .mab-table-cell-content',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'stripedRows', '=', '' ],
		];

		$this->controls['rowsBorder'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.mab-table tr',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'stripedRows', '=', '' ],
		];

		// Even Rows Style
		$this->controls['evenRowsColSeparator'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'type'     => 'separator',
			'label'    => esc_html__( 'Even Rows', 'max-addons' ),
			'required' => [ 'stripedRows', '!=', '' ],
		];

		$this->controls['evenRowsBgColor'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'type'     => 'color',
			'label'    => esc_html__( 'Background Color', 'max-addons' ),
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.mab-table tbody tr:nth-child(even)',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'stripedRows', '!=', '' ],
		];

		$this->controls['evenRowsTextColor'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'type'     => 'color',
			'label'    => esc_html__( 'Text Color', 'max-addons' ),
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.mab-table tbody tr:nth-child(even) .mab-table-cell-content',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'stripedRows', '!=', '' ],
		];

		$this->controls['evenRowsBorder'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.mab-table tr:nth-child(even)',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'stripedRows', '!=', '' ],
		];

		// Odd Rows Style
		$this->controls['oddRowsColSeparator'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'type'     => 'separator',
			'label'    => esc_html__( 'Odd Rows', 'max-addons' ),
			'required' => [ 'stripedRows', '!=', '' ],
		];

		$this->controls['oddRowsBgColor'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'type'     => 'color',
			'label'    => esc_html__( 'Background Color', 'max-addons' ),
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.mab-table tbody tr:nth-child(odd)',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'stripedRows', '!=', '' ],
		];

		$this->controls['oddRowsTextColor'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'type'     => 'color',
			'label'    => esc_html__( 'Text Color', 'max-addons' ),
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.mab-table tbody tr:nth-child(odd) .mab-table-cell-content',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'stripedRows', '!=', '' ],
		];

		$this->controls['oddRowsborder'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.mab-table tr:nth-child(odd)',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'stripedRows', '!=', '' ],
		];
	}

	// Set cells style controls
	public function set_cells_controls() {

		$this->controls['cellPadding'] = [
			'tab'   => 'content',
			'group' => 'cells',
			'type'  => 'spacing',
			'label' => esc_html__( 'Padding', 'max-addons' ),
			'css'   => [
				[
					'property' => 'padding',
					'selector' => '.mab-table tbody td.mab-table-cell',
				],
			],
		];

		$this->controls['cellTextAlign'] = [
			'tab'    => 'content',
			'group'  => 'cells',
			'label'  => esc_html__( 'Text align', 'max-addons' ),
			'type'   => 'text-align',
			'css'    => [
				[
					'property' => 'text-align',
					'selector' => '.mab-table tbody .mab-table-cell',
				],
			],
			'inline' => true,
		];

		$this->controls['cellTypography'] = [
			'tab'    => 'content',
			'group'  => 'cells',
			'label'  => esc_html__( 'Typography', 'max-addons' ),
			'type'   => 'typography',
			'css'    => [
				[
					'property' => 'font',
					'selector' => '.mab-table tr .mab-table-cell',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['cellBgColor'] = [
			'tab'    => 'content',
			'group'  => 'cells',
			'type'   => 'color',
			'label'  => esc_html__( 'Background Color', 'max-addons' ),
			'css'    => [
				[
					'property' => 'background-color',
					'selector' => '.mab-table .mab-table-cell',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['cellBorder'] = [
			'tab'     => 'content',
			'group'   => 'cells',
			'label'   => esc_html__( 'Border', 'max-addons' ),
			'type'    => 'border',
			'css'     => [
				[
					'property' => 'border',
					'selector' => '.mab-table .mab-table-cell',
				],
			],
			'default' => [
				'width' => [
					'top'    => 0,
					'right'  => 1,
					'bottom' => 1,
					'left'   => 0,
				],
				'style' => 'solid',
				'color' => [
					'rgb' => 'rgba(0, 0, 0, 0.1)',
				],
			],
			'inline'  => true,
			'small'   => true,
		];
	}

	// Set footer style controls
	public function set_footer_style_controls() {
		$this->controls['footerTypography'] = [
			'tab'    => 'content',
			'group'  => 'footerStyle',
			'label'  => esc_html__( 'Typography', 'max-addons' ),
			'type'   => 'typography',
			'css'    => [
				[
					'property' => 'font',
					'selector' => '.mab-table tfoot td.mab-table-cell',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['footerBgColor'] = [
			'tab'    => 'content',
			'group'  => 'footerStyle',
			'type'   => 'color',
			'label'  => esc_html__( 'Background Color', 'max-addons' ),
			'css'    => [
				[
					'property' => 'background-color',
					'selector' => '.mab-table tfoot td.mab-table-cell',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['footerBorder'] = [
			'tab'    => 'content',
			'group'  => 'footerStyle',
			'label'  => esc_html__( 'Border', 'max-addons' ),
			'type'   => 'border',
			'css'    => [
				[
					'property' => 'border',
					'selector' => '.mab-table tfoot td.mab-table-cell',
				],
			],
			'inline' => true,
			'small'  => true,
		];
	}

	// Set columns controls
	public function set_icon_controls() {
		$this->controls['iconSpacing'] = [
			'tab'   => 'content',
			'group' => 'iconStyle',
			'label' => esc_html__( 'Icon Spacing', 'max-addons' ),
			'type'  => 'number',
			'units' => true,
			'css'   => [
				[
					'property' => 'margin-right',
					'selector' => '.mab-table-cell-icon-before',
				],
				[
					'property' => 'margin-left',
					'selector' => '.mab-table-cell-icon-after',
				],
			],
		];

		$this->controls['iconTypography'] = [
			'tab'     => 'content',
			'group'   => 'iconStyle',
			'label'   => esc_html__( 'Icon Typography', 'max-addons' ),
			'type'    => 'typography',
			'css'     => [
				[
					'property' => 'font',
					'selector' => '.mab-table-cell .mab-table-cell-icon',
				],
			],
			'exclude' => [
				'font-family',
				'font-weight',
				'font-style',
				'text-align',
				'letter-spacing',
				'line-height',
				'text-transform',
				'text-decoration',
			],
			'inline'  => true,
			'small'   => true,
		];

		$this->controls['imageSize'] = [
			'tab'     => 'content',
			'group'   => 'iconStyle',
			'label'   => esc_html__( 'Image Size', 'max-addons' ),
			'type'    => 'number',
			'units'   => true,
			'css'     => [
				[
					'property' => 'width',
					'selector' => '.mab-table-cell-icon img',
				],
			],
			'default' => '100px',
		];
	}

	// Set columns controls
	public function set_columns_controls() {
		$this->controls['tableColumnStyles'] = [
			'tab'         => 'content',
			'group'       => 'columns',
			'placeholder' => esc_html__( 'Column Span', 'max-addons' ),
			'type'        => 'repeater',
			'fields'      => [
				'tableColSpan'    => [
					'label'   => esc_html__( 'Span', 'max-addons' ),
					'type'    => 'number',
					'min'     => 0,
					'max'     => 999,
					'step'    => 1,
					'default' => 1,
					'inline'  => true,
				],
				'tableColBgColor' => [
					'label'  => esc_html__( 'Background', 'max-addons' ),
					'type'   => 'color',
					'inline' => true,
					'small'  => true,
				],
				'tableColWidth'   => [
					'label' => esc_html__( 'Width', 'max-addons' ),
					'type'  => 'number',
					'units' => true,
				],
			],
			'default'     => [
				[
					'tableColSpan' => 1,
				],
			],
		];
	}

	public function get_normalized_image_settings( $settings ) {
		if ( ! isset( $settings['cellImage'] ) ) {
			$settings['cellImage'] = [
				'id'  => 0,
				'url' => '',
			];
			return $settings;
		}

		$image = $settings['cellImage'];

		if ( isset( $image['useDynamicData']['name'] ) ) {
			$images = $this->render_dynamic_data_tag( $image['useDynamicData']['name'], $image );
			$image['id'] = empty( $images ) ? 0 : $images[0];
		} else {
			$image['id'] = isset( $image['id'] ) ? $image['id'] : 0;
		}

		// Image Size
		$image['size'] = isset( $image['size'] ) ? $settings['cellImage']['size'] : BRICKS_DEFAULT_IMAGE_SIZE;

		// Image URL
		if ( ! isset( $image['url'] ) ) {
			$image['url'] = wp_get_attachment_image_url( $image['id'], $image['size'] );
		}

		$settings['cellImage'] = $image;

		return $settings;
	}

	public function render_table_image( $settings ) {

		if ( isset( $settings['cellImage']['useDynamicData']['name'] ) ) {

			if ( empty( $settings['cellImage']['id'] ) ) {

				if ( 'ACF' === $settings['cellImage']['useDynamicData']['group'] && ! class_exists( 'ACF' ) ) {
					$message = esc_html__( 'Can\'t render element, as the selected ACF field is not available. Please activate ACF or edit the element to select different data.', 'max-addons' );
				} elseif ( '{featured_image}' == $settings['cellImage']['useDynamicData']['name'] ) {
					$message = esc_html__( 'No featured image set.', 'max-addons' );
				} else {
					$message = esc_html__( 'Dynamic Data %1$s (%2$s) is empty.', 'max-addons' );
				}

				return $this->render_element_placeholder( [
					'icon-class' => 'ti-image',
					'title'      => sprintf(
						$message,
						$settings['cellImage']['useDynamicData']['label'],
						$settings['cellImage']['useDynamicData']['group']
					),
				] );
			}
		}

		// Image id is empty or doesn't exist
		else {

			// No image
			if ( empty( $settings['cellImage']['id'] ) ) {
				return $this->render_element_placeholder( [
					'icon-class' => 'ti-image',
					'title'      => esc_html__( 'No image selected.', 'max-addons' ),
				] );
			}

			// Return if image ID does not exist
			if ( ! wp_get_attachment_image_src( $settings['cellImage']['id'] ) ) {
				return $this->render_element_placeholder( [
					'icon-class' => 'ti-image',
					'title'      => sprintf( esc_html__( 'Image ID (%s) no longer exist. Please select another image.', 'max-addons' ), $settings['cellImage']['id'] ),
				] );
			}
		}

		// Render
		$img_html = '';
		$image_atts = [];
		$image_atts['id'] = 'image-' . $settings['cellImage']['id'];

		$image_wrapper_classes = [ 'mab-table-icon-image' ];
		$img_classes = [ 'post-thumbnail', 'css-filter' ];

		$img_classes[] = 'size-' . $settings['cellImage']['size'];
		$image_atts['class'] = join( ' ', $img_classes );

		$this->set_attribute( 'image-wrapper', 'class', $image_wrapper_classes );

		$img_html .= '<div ' . $this->render_attributes( 'image-wrapper' ) . '>';

		// Lazy load atts set via 'wp_get_attachment_image_attributes' filter
		if ( isset( $settings['cellImage']['id'] ) ) {
			$img_html .= wp_get_attachment_image( $settings['cellImage']['id'], $settings['cellImage']['size'], false, $image_atts );
		} elseif ( ! empty( $settings['cellImage']['url'] ) ) {
			$img_html .= '<img src="' . $settings['cellImage']['url'] . '">';
		}

		$img_html .= '</div>';

		return $img_html;
	}

	protected function render_column_styles() {
		$settings = $this->settings;

		if ( isset( $settings['tableColumnStyles'] ) ) { ?>
			<colgroup>
				<?php foreach ( $settings['tableColumnStyles'] as $index => $col_style ) { ?>
					<?php
					$col_key = "col_key_$index";

					if ( isset( $col_style['tableColSpan'] ) ) {
						$this->set_attribute( $col_key, 'span', $col_style['tableColSpan'] );
					}

					$this->set_attribute( $col_key, 'class', 'repeater-item' );

					if ( isset( $col_style['tableColBgColor'] ) || isset( $col_style['tableColWidth'] ) ) {
						if ( isset( $col_style['tableColBgColor'] ) ) {
							$this->set_attribute( $col_key, 'style', 'background-color: ' . $col_style['tableColBgColor']['hex'] . ';' );
						}

						if ( isset( $col_style['tableColWidth'] ) ) {
							$this->set_attribute( $col_key, 'style', 'width: ' . $col_style['tableColWidth'] . ';' );
						}
					}
					?>
					<col <?php echo wp_kses_post( $this->render_attributes( $col_key ) ); ?>>
				<?php } ?>
			</colgroup>
			<?php
		}
	}

	// Table Header
	public function render_header_content( $item, $loop_index ) {
		$settings           = $this->settings;
		$cell_key           = "header_cell_key_$loop_index";
		$cell_text_key      = "header_cell_text_key_$loop_index";
		$cell_icon_wrap_key = "header_icon_wrap_key_$loop_index";
		$cell_icon_key      = "header_icon_key_$loop_index";

		$this->set_attribute( $cell_text_key, 'class', 'mab-table-cell-text' );
		$this->set_attribute( $cell_key, 'class', 'mab-table-cell' );

		if ( isset( $item['cssId'] ) ) {
			$this->set_attribute( $cell_key, 'id', $item['cssId'] );
		}

		if ( isset( $item['cssClasses'] ) ) {
			$this->set_attribute( $cell_key, 'class', $item['cssClasses'] );
		}

		if ( isset( $item['cellIconType'] ) && '' !== $item['cellIconType'] ) {
			$this->set_attribute( $cell_icon_wrap_key, 'class', 'mab-table-cell-icon' );

			$icon_position = isset( $item['cellIconPosition'] ) ? esc_attr( $item['cellIconPosition'] ) : 'before';
			$this->set_attribute( $cell_icon_wrap_key, 'class', 'mab-table-cell-icon-' . $icon_position );

			if ( isset( $item['cellIcon']['icon'] ) ) {
				$this->set_attribute( "icon-$loop_index", 'class', [
					'mab-icon',
					$item['cellIcon']['icon'], // Dont' sanitize_html_class (Font Awesome uses two class names, so we need spaces)
				] );
			}
		}

		$colspan = isset( $item['colspan'] ) ? intval( $item['colspan'] ) : '';
		$rowspan = isset( $item['rowspan'] ) ? intval( $item['rowspan'] ) : '';

		if ( $colspan > 1 ) {
			$this->set_attribute( $th_key, 'colspan', $item['colspan'] );
		}

		if ( $rowspan > 1 ) {
			$this->set_attribute( $th_key, 'rowspan', $item['rowspan'] );
		}

		if ( isset( $settings['tableType'] ) && 'responsive' === $settings['tableType'] && isset( $settings['sortable'] ) ) {
			$sortable_header = ' data-tablesaw-sortable-col';
		} else {
			$sortable_header = '';
		}

		echo '<th ' . $this->render_attributes( $cell_key ) . $sortable_header . '>';
		echo '<span class="mab-table-cell-content">';
		if ( isset( $item['cellIconType'] ) && '' != $item['cellIconType'] ) {
			echo '<span ' . $this->render_attributes( $cell_icon_wrap_key ) . '>';
			if ( 'image' === $item['cellIconType'] ) {
				$image_settings = $this->get_normalized_image_settings( $item );
				echo $this->render_table_image( $image_settings );
			} elseif ( 'icon' === $item['cellIconType'] ) {
				if ( isset( $item['cellIcon']['icon'] ) ) {
					echo '<i ' . wp_kses_post( $this->render_attributes( "icon-$loop_index" ) ) . '></i>';
				}
			}
			echo '</span>';
		}
		echo '<span ' . $this->render_attributes( $cell_text_key ) . '>';
		if ( isset( $settings['source'] ) && 'query' === $settings['source'] ) {
			if ( isset( $item['columnHeading'] ) ) {
				echo $item['columnHeading'];
			}
		} else {
			if ( isset( $item['cellText'] ) ) {
				echo $this->render_dynamic_data( $item['cellText'] );
			}
		}
		echo '</span>';
		echo '</span>';
		echo '</th>';
	}

	// Table Header
	public function render_header_custom( $item ) {
		$settings   = $this->settings;
		$loop_index = $this->header_loop_index;
		ob_start();

		self::render_header_content( $item, $loop_index );

		$html = ob_get_contents();
		ob_end_clean();
		$this->header_loop_index++;

		return $html;
	}

	// Table Header
	public function render_header_dynamic( $item ) {
		$settings   = $this->settings;
		$loop_index = $this->header_loop_index;
		ob_start();

		$rows = empty( $settings['dynamicRows'] ) ? false : $settings['dynamicRows'];

		foreach ( $rows as $index => $item ) {

			if ( 0 == $loop_index ) {

				if ( isset( $item['columnHeading'] ) || '' !== $item['cellIconType'] ) {
					self::render_header_content( $item, $index );
				}
			}
		}
		
		$html = ob_get_contents();
		ob_end_clean();
		$this->header_loop_index++;

		return $html;
	}

	// Table Header
	protected function render_header() {
		$settings = $this->settings;
		$i = 1;
		$this->set_attribute( 'row', 'class', 'mab-table-row' );

		$hide_table_header = isset( $settings['hideTableHeader'] ) ? $settings['hideTableHeader'] : false;

		if ( $hide_table_header === false ) {
			$rows = empty( $settings['dynamicRows'] ) ? false : $settings['dynamicRows'];
			?>
			<thead>
				<tr <?php echo $this->render_attributes( 'row' ); ?>>
					<?php
					$output = '';

					// Query Loop
					if ( isset( $settings['source'] ) && 'query' === $settings['source'] ) {

						$query = new \Bricks\Query( [
							'id'       => $this->id,
							'settings' => $settings,
						] );

						$item = $rows[0];

						$output .= $query->render( [ $this, 'render_header_dynamic' ], compact( 'item' ) );

						// We need to destroy the Query to explicitly remove it from the global store
						$query->destroy();
						unset( $query );
					} else {
						foreach ( $settings['headerRows'] as $index => $item ) {
							$output .= self::render_header_custom( $item );
						}
					}

					echo $output;
					?>
				</tr>
			</thead>
			<?php
		}
	}

	// Table Body
	protected function render_body() {
		$settings = $this->settings;
		$i = 1;

		$rows = empty( $settings['dynamicRows'] ) ? false : $settings['dynamicRows'];
		?>
		<tbody>
			<?php
			$output = '';

			// Query Loop
			if ( isset( $settings['source'] ) && 'query' === $settings['source'] ) {

				$query = new \Bricks\Query( [
					'id'       => $this->id,
					'settings' => $settings,
				] );

				$item = $rows[0];

				$output .= $query->render( [ $this, 'render_body_content_dynamic' ], compact( 'item' ) );

				// We need to destroy the Query to explicitly remove it from the global store
				$query->destroy();
				unset( $query );
			} else {
				foreach ( $settings['bodyRows'] as $index => $item ) {
					$output .= self::render_body_content_custom( $item );
				}
			}

			echo $output;
			echo '</tr>';
			?>
		</tbody>
		<?php
	}

	// Table Body
	public function render_body_content_custom( $item ) {
		$settings = $this->settings;
		$loop_index = $this->loop_index;
		$i = $loop_index + 1;

		ob_start();
			$row_key            = "body_row_key_$loop_index";
			$cell_key           = "body_cell_key_$loop_index";
			$cell_text_key      = "body_cell_text_key_$loop_index";
			$cell_icon_wrap_key = "body_icon_wrap_key_$loop_index";
			$cell_icon_key      = "body_icon_key_$loop_index";

			$this->set_attribute( $cell_text_key, 'class', 'mab-table-cell-text' );
			$this->set_attribute( $row_key, 'class', 'mab-table-row' );
			$this->set_attribute( $cell_key, 'class', 'mab-table-cell' );

			if ( isset( $item['cssId'] ) ) {
				$this->set_attribute( $cell_key, 'id', $item['cssId'] );
			}

			if ( isset( $item['cssClasses'] ) ) {
				$this->set_attribute( $cell_key, 'class', $item['cssClasses'] );
			}

			if ( isset( $item['tableBodyElement'] ) && 'cell' === $item['tableBodyElement'] ) {

				if ( isset( $item['cellIconType'] ) && '' !== $item['cellIconType'] ) {
					$this->set_attribute( $cell_icon_wrap_key, 'class', 'mab-table-cell-icon' );

					$icon_position = isset( $item['cellIconPosition'] ) ? esc_attr( $item['cellIconPosition'] ) : 'before';
					$this->set_attribute( $cell_icon_wrap_key, 'class', 'mab-table-cell-icon-' . $icon_position );

					if ( isset( $item['cellIcon']['icon'] ) ) {
						$this->set_attribute( "icon-$loop_index", 'class', [
							'mab-icon',
							$item['cellIcon']['icon'], // Dont' sanitize_html_class (Font Awesome uses two class names, so we need spaces)
						] );
					}
				}

				$colspan = isset( $item['colspan'] ) ? intval( $item['colspan'] ) : '';
				$rowspan = isset( $item['rowspan'] ) ? intval( $item['rowspan'] ) : '';

				if ( $colspan > 1 ) {
					$this->set_attribute( $cell_key, 'colspan', $item['colspan'] );
				}

				if ( $rowspan > 1 ) {
					$this->set_attribute( $cell_key, 'rowspan', $item['rowspan'] );
				}

				if ( isset( $item['link'] ) ) {
					$this->set_link_attributes( 'col-link-' . $loop_index, $item['link'] );
				}

				$convert_cell_th = isset( $item['cellType'] ) ? esc_attr( $item['cellType'] ) : 'td';

				if ( isset( $item['cellText'] ) || '' !== $item['cellIconType'] ) {
					echo '<' . esc_attr( $convert_cell_th ) . ' ' . $this->render_attributes( $cell_key ) . '>';
					if ( isset( $item['link'] ) ) { ?>
						<a <?php echo $this->render_attributes( 'col-link-' . $loop_index ); ?>>
						<?php
					}
					echo '<span class="mab-table-cell-content">';
					if ( isset( $item['cellIconType'] ) && '' !== $item['cellIconType'] ) {
						echo '<span ' . $this->render_attributes( $cell_icon_wrap_key ) . '>';
						if ( 'icon' === $item['cellIconType'] ) {
							if ( isset( $item['cellIcon']['icon'] ) ) {
								echo '<i ' . wp_kses_post( $this->render_attributes( "icon-$loop_index" ) ) . '></i>';
							}
						} elseif ( 'image' === $item['cellIconType'] ) {
							$image_settings = $this->get_normalized_image_settings( $item );
							echo $this->render_table_image( $image_settings );
						}
						echo '</span>';
					}
					if ( isset( $item['cellText'] ) ) {
						echo '<span ' . $this->render_attributes( $cell_text_key ) . '>';
						echo $this->render_dynamic_data( $item['cellText'] );
						echo '</span>';
					}
					echo '</span>';
					if ( isset( $item['link'] ) ) { ?>
						</a>
						<?php
					}
					echo '</' . esc_attr( $convert_cell_th ) . '>';
				}
			} else {
				if ( 1 === $i && isset( $item['tableBodyElement'] ) && 'row' === $item['tableBodyElement'] ) {
					echo '<tr ' . $this->render_attributes( $row_key ) . '>';
				} elseif ( $i > 1 ) {
					echo '</tr><tr ' . $this->render_attributes( $row_key ) . '>';
				}
			}
		
			$html = ob_get_contents();
		ob_end_clean();
		$this->loop_index++;
		return $html;
	}

	// Table Body
	public function render_body_content_dynamic( $item ) {
		$settings = $this->settings;
		$i = 1;
		$loop_index = $this->loop_index;
		ob_start();

		$row_key            = "body_row_key_$loop_index";
		$cell_key           = "body_cell_key_$loop_index";
		$cell_text_key      = "body_cell_text_key_$loop_index";
		$cell_icon_wrap_key = "body_icon_wrap_key_$loop_index";
		$cell_icon_key      = "body_icon_key_$loop_index";
		$link_key           = "body_link_key_$loop_index";

		$this->set_attribute( $cell_text_key, 'class', 'mab-table-cell-text' );
		$this->set_attribute( $row_key, 'class', 'mab-table-row' );
		$this->set_attribute( $cell_key, 'class', 'mab-table-cell' );

		if ( isset( $item['cssId'] ) ) {
			$this->set_attribute( $cell_key, 'id', $item['cssId'] );
		}

		if ( isset( $item['cssClasses'] ) ) {
			$this->set_attribute( $cell_key, 'class', $item['cssClasses'] );
		}

		$rows = empty( $settings['dynamicRows'] ) ? false : $settings['dynamicRows'];

		if ( 1 === $i ) {
			echo '<tr ' . $this->render_attributes( $row_key ) . '>';
		} elseif ( $i > 1 ) {
			echo '</tr><tr ' . $this->render_attributes( $row_key ) . '>';
		}

		foreach ( $rows as $index => $item ) {

			if ( isset( $item['cellIconType'] ) && '' !== $item['cellIconType'] ) {
				$this->set_attribute( $cell_icon_wrap_key, 'class', 'mab-table-cell-icon' );

				$icon_position = isset( $item['cellIconPosition'] ) ? esc_attr( $item['cellIconPosition'] ) : 'before';
				$this->set_attribute( $cell_icon_wrap_key, 'class', 'mab-table-cell-icon-' . $icon_position );

				if ( isset( $item['cellIcon']['icon'] ) ) {
					$this->set_attribute( $cell_icon_key, 'class', [
						'mab-icon',
						$item['cellIcon']['icon'], // Dont' sanitize_html_class (Font Awesome uses two class names, so we need spaces)
					] );
				}
			}

			$colspan = isset( $item['colspan'] ) ? intval( $item['colspan'] ) : '';
			$rowspan = isset( $item['rowspan'] ) ? intval( $item['rowspan'] ) : '';

			if ( $colspan > 1 ) {
				$this->set_attribute( $cell_key, 'colspan', $item['colspan'] );
			}

			if ( $rowspan > 1 ) {
				$this->set_attribute( $cell_key, 'rowspan', $item['rowspan'] );
			}

			if ( isset( $item['link'] ) ) {
				$this->set_link_attributes( $link_key, $item['link'] );
			}

			$convert_cell_th = isset( $item['cellType'] ) ? esc_attr( $item['cellType'] ) : 'td';

			if ( isset( $item['columnData'] ) || '' !== $item['cellIconType'] ) {
				echo '<' . esc_attr( $convert_cell_th ) . ' ' . $this->render_attributes( $cell_key ) . '>';
				if ( isset( $item['link'] ) ) { ?>
					<a <?php echo $this->render_attributes( $link_key ); ?>>
					<?php
				}
				echo '<span class="mab-table-cell-content">';
				if ( isset( $item['cellIconType'] ) && '' !== $item['cellIconType'] ) {
					echo '<span ' . $this->render_attributes( $cell_icon_wrap_key ) . '>';
					if ( 'icon' === $item['cellIconType'] ) {
						if ( isset( $item['cellIcon']['icon'] ) ) {
							echo '<i ' . wp_kses_post( $this->render_attributes( $cell_icon_key ) ) . '></i>';
						}
					} elseif ( 'image' === $item['cellIconType'] ) {
						$image_settings = $this->get_normalized_image_settings( $item );
						echo $this->render_table_image( $image_settings );
					}
					echo '</span>';
				}
				echo '<span ' . $this->render_attributes( $cell_text_key ) . '>';
				echo $item['columnData'];
				echo '</span>';
				echo '</span>';
				if ( isset( $item['link'] ) ) { ?>
					</a>
					<?php
				}
				echo '</' . esc_attr( $convert_cell_th ) . '>';
			}

			$i++;
		}
		
		$html = ob_get_contents();
		ob_end_clean();
		$this->loop_index++;

		return $html;
	}

	// Table Footer
	protected function render_footer() {
		$settings = $this->settings;
		$i = 1;

		ob_start();
		if ( ! empty( $settings['footerRows'] ) ) {
			echo '<tfoot>';
			echo '<tr>';
			foreach ( $settings['footerRows'] as $index => $item ) {
				$footer_cell_key = "table_footer_content_$index";

				$this->set_attribute( $footer_cell_key, 'class', 'mab-table-cell-text' );
				$this->set_attribute( 'footer-' . $i, 'class', 'mab-table-cell' );

				if ( isset( $item['cssId'] ) ) {
					$this->set_attribute( 'footer-' . $i, 'id', $item['cssId'] );
				}

				if ( isset( $item['cssClasses'] ) ) {
					$this->set_attribute( 'footer-' . $i, 'class', $item['cssClasses'] );
				}

				if ( isset( $item['cellIconType'] ) && '' !== $item['cellIconType'] ) {
					$this->set_attribute( 'footer-icon-' . $i, 'class', 'mab-table-cell-icon' );

					$icon_position = isset( $item['cellIconPosition'] ) ? esc_attr( $item['cellIconPosition'] ) : 'before';
					$this->set_attribute( 'footer-icon-' . $i, 'class', 'mab-table-cell-icon-' . $icon_position );

					if ( isset( $item['cellIcon']['icon'] ) ) {
						$this->set_attribute( "icon-$i", 'class', [
							'mab-icon',
							$item['cellIcon']['icon'], // Dont' sanitize_html_class (Font Awesome uses two class names, so we need spaces)
						] );
					}
				}

				$colspan = isset( $item['colspan'] ) ? intval( $item['colspan'] ) : '';
				$rowspan = isset( $item['rowspan'] ) ? intval( $item['rowspan'] ) : '';

				if ( $colspan > 1 ) {
					$this->set_attribute( 'footer-' . $i, 'colspan', $item['colspan'] );
				}

				if ( $rowspan > 1 ) {
					$this->set_attribute( 'footer-' . $i, 'rowspan', $item['rowspan'] );
				}

				if ( $item['cellText'] || ( isset( $item['cellIconType'] ) && '' !== $item['cellIconType'] ) ) {
					echo '<td ' . $this->render_attributes( 'footer-' . $i ) . '">';
					echo '<span class="mab-table-cell-content">';
					if ( isset( $item['cellIconType'] ) && '' !== $item['cellIconType'] ) {
						echo '<span ' . $this->render_attributes( 'footer-icon-' . $i ) . '>';
						if ( 'icon' === $item['cellIconType'] ) {
							if ( isset( $item['cellIcon']['icon'] ) ) {
								echo '<i ' . wp_kses_post( $this->render_attributes( "icon-$i" ) ) . '></i>';
							}
						} elseif ( 'image' === $item['cellIconType'] ) {
							$image_settings = $this->get_normalized_image_settings( $item );
							echo $this->render_table_image( $image_settings );
						}
						echo '</span>';
					}
					echo '<span ' . $this->render_attributes( $footer_cell_key ) . '>';
					echo $this->render_dynamic_data( $item['cellText'] );
					echo '</span>';
					echo '</span>';
					echo '</td>';
				}
				$i++;
			}
			echo '</tr>';
			echo '</tfoot>';
		}
		$html = ob_get_contents();
		ob_end_clean();
		echo $html;
	}

	// Render element HTML
	public function render() {
		$settings = $this->settings;

		// Set element attributes
		$wrapper_classes = [ 'mab-table-container' ];

		if ( isset( $settings['tableAlign'] ) ) {
			$wrapper_classes[] = 'mab-table-' . $settings['tableAlign'];
		}

		// Set attribute tag for '_root'
		$this->set_attribute( '_root', 'class', $wrapper_classes );

		$table_options = [
			'table_type' => isset( $settings['tableType'] ) ? $settings['tableType'] : 'responsive',
			'scrollable' => isset( $settings['scrollable'] ) ? $settings['scrollable'] : false,
			'breakpoint' => isset( $settings['breakpoint'] ) ? $settings['breakpoint'] : '',
		];

		$this->set_attribute( 'table', 'data-settings', wp_json_encode( $table_options ) );

		$this->set_attribute( 'table', 'class', [ 'mab-table' ] );

		$hide_table_header = isset( $settings['hideTableHeader'] ) ? $settings['hideTableHeader'] : false;

		if ( $hide_table_header === false ) {
			$this->set_attribute( 'table', 'class', [ 'tablesaw' ] );

			if ( isset( $settings['sortable'] ) && ! isset( $settings['sortableDropdown'] ) && ! isset( $settings['hideTableHeader'] ) ) {
				$this->set_attribute( '_root', 'class', 'mab-table-sortable-dd-hide' );
			}

			if ( isset( $settings['tableType'] ) && 'responsive' === $settings['tableType'] ) {
				if ( isset( $settings['scrollable'] ) && false === $hide_table_header ) {
					$this->set_attribute( 'table', 'data-tablesaw-mode', 'swipe' );
				} else {
					$this->set_attribute( 'table', 'data-tablesaw-mode', 'stack' );
				}
			}
		}

		?>
		<div <?php echo wp_kses_post( $this->render_attributes( '_root' ) ); ?>>
			<table <?php echo $this->render_attributes( 'table' );
			if ( isset( $settings['tableType'] ) && 'responsive' === $settings['tableType'] && isset( $settings['sortable'] ) ) {
				echo ' data-tablesaw-sortable data-tablesaw-sortable-switch'; } ?>>
				<?php
				/*
				if ( 'file' === $settings['source'] ) {
					$csv = $this->parse_csv();
					echo $csv['html'];
				} else {*/
					$this->render_column_styles();

					$this->render_header();

					$this->render_footer();

					$this->render_body();
				//}
				?>
			</table>
		</div>
		<?php
	}
}
