<?php
namespace MaxAddonsPro\Elements;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Reviews_Element extends \Bricks\Element {
	// Element properties
	public $category     = 'max-addons-elements'; // Use predefined element category 'general'
	public $name         = 'max-reviews'; // Make sure to prefix your elements
	public $icon         = 'ti-comment-alt max-element'; // Themify icon font class
	public $css_selector = '.mab-review'; // Default CSS selector
	public $scripts      = [ 'bricksSwiper' ]; // Script(s) run when element is rendered on frontend or updated in builder
	public $tag          = 'ul';

	public function get_label() {
		return esc_html__( 'Reviews', 'max-addons' );
	}

	public function get_keywords() {
		return [ 'google', 'yelp' ];
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		$layout = ! empty( $this->settings['layout'] ) ? $this->settings['layout'] : 'grid';

		wp_enqueue_style( 'mab-reviews' );

		if ( $layout === 'carousel' ) {
			wp_enqueue_script( 'bricks-swiper' );
		}

		wp_enqueue_script( 'mab-frontend' );
	}

	// Set builder control groups
	public function set_control_groups() {
		$this->control_groups['reviewSettings'] = [
			'title' => esc_html__( 'Reviews', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['settings'] = [
			'title' => esc_html__( 'Layout', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['filters'] = [
			'title' => esc_html__( 'Filters', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['reviewerDetails'] = [
			'title' => esc_html__( 'Review Details', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['image'] = [
			'title'    => esc_html__( 'Image', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'reviewerImage', '!=', '' ],
		];

		$this->control_groups['rating'] = [
			'title'    => esc_html__( 'Rating', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'starRating', '!=', '' ],
		];

		$this->control_groups['arrows'] = [
			'title'    => esc_html__( 'Arrows', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'layout', '=', 'carousel' ],
		];

		$this->control_groups['dots'] = [
			'title'    => esc_html__( 'Dots', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'layout', '=', 'carousel' ],
		];
	}

	public function set_controls() {

		$this->set_reviews_controls();
		$this->set_layout_controls();
		$this->set_filters_controls();
		$this->set_reviewer_details_controls();
		$this->set_image_controls();
		$this->set_star_rating_controls();
		$this->set_review_text_controls();
		$this->set_navigation_controls();
	}

	// Set items controls
	public function set_reviews_controls() {

		$this->controls['reviewsSource'] = [
			'tab'       => 'content',
			'group'     => 'reviewSettings',
			'label'     => esc_html__( 'Source', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'google' => __( 'Google Places', 'max-addons' ),
				'yelp'   => __( 'Yelp', 'max-addons' ),
				'all'    => __( 'Google + Yelp', 'max-addons' ),
			],
			'default'   => 'google',
			'inline'    => true,
			'clearable' => false,
		];

		$this->controls['googlePlaceId'] = [
			'tab'      => 'content',
			'group'    => 'reviewSettings',
			'label'    => esc_html__( 'Google Place ID', 'max-addons' ),
			'type'     => 'text',
			'inline'   => false,
			'required' => [ 'reviewsSource', '!=', 'yelp' ],
			'default'  => 'ChIJw____96GhYARCVVwg5cT7c0',
		];

		$this->controls['googlePlaceIdCodeInfo'] = [
			'tab'      => 'content',
			'group'    => 'reviewSettings',
			'content'  => sprintf( __( 'Click %1$s here %2$s to get your Google Place ID.', 'max-addons' ), '<a href="https://developers.google.com/places/place-id/" target="_blank" rel="noopener">', '</a>' ),
			'type'     => 'info',
			'required' => [ 'reviewsSource', '!=', 'yelp' ],
		];

		$this->controls['languageId'] = [
			'tab'      => 'content',
			'group'    => 'reviewSettings',
			'label'    => esc_html__( 'Language Code', 'max-addons' ),
			'type'     => 'text',
			'inline'   => false,
			'required' => [ 'reviewsSource', '!=', 'yelp' ],
			'default'  => '',
		];

		$this->controls['languageCodeInfo'] = [
			'tab'      => 'content',
			'group'    => 'reviewSettings',
			'content'  => sprintf( __( 'Click %1$s here %2$s to check your Language code.', 'max-addons' ), '<a href="https://developers.google.com/admin-sdk/directory/v1/languages" target="_blank" rel="noopener">', '</a>' ),
			'type'     => 'info',
			'required' => [ 'reviewsSource', '!=', 'yelp' ],
		];

		$this->controls['yelpBusinessId'] = [
			'tab'      => 'content',
			'group'    => 'reviewSettings',
			'label'    => esc_html__( 'Yelp Business ID', 'max-addons' ),
			'type'     => 'text',
			'inline'   => false,
			'required' => [ 'reviewsSource', '!=', 'google' ],
			'default'  => 'golden-gate-bridge-san-francisco',
		];

		$this->controls['yelpBusinessIdInfo'] = [
			'tab'      => 'content',
			'group'    => 'reviewSettings',
			'content'  => sprintf( __( 'Click %1$s here %2$s to get your Yelp Business ID.', 'max-addons' ), '<a href="#" target="_blank" rel="noopener">', '</a>' ),
			'type'     => 'info',
			'required' => [ 'reviewsSource', '!=', 'google' ],
		];

		$this->controls['reviewsRefreshTime'] = [
			'tab'       => 'content',
			'group'     => 'reviewSettings',
			'label'     => esc_html__( 'Refresh Reviews after', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'hour'  => __( 'Hour', 'max-addons' ),
				'day'   => __( 'Day', 'max-addons' ),
				'week'  => __( 'Week', 'max-addons' ),
				'month' => __( 'Month', 'max-addons' ),
				'year'  => __( 'Year', 'max-addons' ),
			],
			'default'   => 'day',
			'inline'    => true,
			'clearable' => false,
		];
	}

	// Set layout controls
	public function set_layout_controls() {

		$this->controls['skin'] = [
			'tab'       => 'content',
			'group'     => 'settings',
			'label'     => esc_html__( 'Skin', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'classic' => __( 'Classic', 'max-addons' ),
				'card'    => __( 'Card', 'max-addons' ),
			],
			'default'   => 'classic',
			'inline'    => true,
			'clearable' => false,
		];

		$this->controls['layout'] = [
			'tab'       => 'content',
			'group'     => 'settings',
			'label'     => esc_html__( 'Layout', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'grid'     => __( 'Grid', 'max-addons' ),
				'carousel' => __( 'Carousel', 'max-addons' ),
			],
			'inline'    => true,
			'clearable' => false,
			'default'   => 'grid',
		];

		$this->controls['googleReviewsCount'] = [
			'tab'      => 'content',
			'group'    => 'settings',
			'label'    => esc_html__( 'Number of Reviews', 'max-addons' ),
			'type'     => 'number',
			'min'      => 1,
			'max'      => 5,
			'step'     => 1,
			'inline'   => true,
			'default'  => 3,
			'required' => [ 'reviewsSource', '=', 'google' ],
		];

		$this->controls['googleMaxReviewsInfo'] = [
			'tab'      => 'content',
			'group'    => 'settings',
			'content'  => sprintf( __( 'Google allows maximum 5 reviews. Click <a href="%s" target="_blank" rel="noopener">here</a> to know more.', 'max-addons' ), 'https://wpbricksaddons.com/docs/elements/reviews/how-many-numbers-of-reviews-can-be-shown-for-google-and-yelp/' ),
			'type'     => 'info',
			'required' => [ 'reviewsSource', '=', 'google' ],
		];

		$this->controls['yelpReviewsCount'] = [
			'tab'      => 'content',
			'group'    => 'settings',
			'label'    => esc_html__( 'Number of Reviews', 'max-addons' ),
			'type'     => 'number',
			'min'      => 1,
			'max'      => 3,
			'step'     => 1,
			'inline'   => true,
			'default'  => 3,
			'required' => [ 'reviewsSource', '=', 'yelp' ],
		];

		$this->controls['yelpMaxReviewsInfo'] = [
			'tab'      => 'content',
			'group'    => 'settings',
			'content'  => sprintf( __( 'Yelp allows maximum 3 reviews. Click <a href="%s" target="_blank" rel="noopener">here</a> to know more.', 'max-addons' ), 'https://wpbricksaddons.com/docs/elements/reviews/how-many-numbers-of-reviews-can-be-shown-for-google-and-yelp/' ),
			'type'     => 'info',
			'required' => [ 'reviewsSource', '=', 'yelp' ],
		];

		$this->controls['reviewsCount'] = [
			'tab'      => 'content',
			'group'    => 'settings',
			'label'    => esc_html__( 'Number of Reviews', 'max-addons' ),
			'type'     => 'number',
			'min'      => 1,
			'max'      => 8,
			'step'     => 1,
			'inline'   => true,
			'default'  => 3,
			'required' => [ 'reviewsSource', '=', 'all' ],
		];

		$this->controls['allMaxReviewsInfo'] = [
			'tab'      => 'content',
			'group'    => 'settings',
			'content'  => sprintf( __( 'We can fetch only up to 5 Google and 3 Yelp Reviews. So, a maximum of 8 reviews can be displayed. Click <a href="%s" target="_blank" rel="noopener">here</a> to know more.', 'max-addons' ), 'https://wpbricksaddons.com/docs/elements/reviews/how-many-numbers-of-reviews-can-be-shown-for-google-and-yelp/' ),
			'type'     => 'info',
			'required' => [ 'reviewsSource', '=', 'all' ],
		];

		$this->controls['columns'] = [
			'tab'         => 'content',
			'group'       => 'settings',
			'label'       => esc_html__( 'Columns', 'max-addons' ),
			'type'        => 'number',
			'min'         => 1,
			'css'         => [
				[
					'property' => '--columns',
					'selector' => '.mab-reviews-wrapper',
				],
			],
			'rerender'    => true,
			'placeholder' => 3,
			'required'    => [ 'layout', '=', 'grid' ],
		];

		$this->controls['spacing'] = [
			'tab'     => 'content',
			'group'   => 'settings',
			'label'   => esc_html__( 'Spacing', 'max-addons' ),
			'type'    => 'number',
			'units'   => true,
			'css'     => [
				[
					'property' => '--gutter',
					'selector' => '.mab-reviews-wrapper',
				],
			],
			'default'  => 30,
			'required' => [ 'layout', '=', 'grid' ],
		];

		$this->controls['slidesToShow'] = [
			'tab'         => 'content',
			'group'       => 'settings',
			'label'       => esc_html__( 'Items to show', 'max-addons' ),
			'type'        => 'number',
			'min'         => 1,
			'max'         => 10,
			'breakpoints' => true, // NOTE: Undocumented (allows to set non-CSS settings per breakpoint: Carousel, Slider, etc.)
			'placeholder' => 1,
			'required'    => [ 'layout', '=', 'carousel' ],
		];

		$this->controls['slidesToScroll'] = [
			'tab'         => 'content',
			'group'       => 'settings',
			'label'       => esc_html__( 'Items to scroll', 'max-addons' ),
			'type'        => 'number',
			'min'         => 1,
			'max'         => 10,
			'breakpoints' => true, // NOTE: Undocumented
			'placeholder' => 1,
			'required'    => [ 'layout', '=', 'carousel' ],
		];

		$this->controls['gutter'] = [
			'tab'         => 'content',
			'group'       => 'settings',
			'label'       => esc_html__( 'Spacing', 'max-addons' ) . ' (px)',
			'type'        => 'number',
			'placeholder' => 0,
			'required'    => [ 'layout', '=', 'carousel' ],
		];

		$this->controls['infinite'] = [
			'tab'     => 'content',
			'group'   => 'settings',
			'label'   => esc_html__( 'Loop', 'max-addons' ),
			'type'    => 'checkbox',
			'default' => true,
			'inline'  => true,
			'required' => [ 'layout', '=', 'carousel' ],
		];

		$this->controls['random']   = [
			'tab'   => 'content',
			'group' => 'settings',
			'label' => esc_html__( 'Random order', 'max-addons' ),
			'type'  => 'checkbox',
		];

		$this->controls['autoplay'] = [
			'tab'   => 'content',
			'group' => 'settings',
			'label' => esc_html__( 'Autoplay', 'max-addons' ),
			'type'  => 'checkbox',
			'required' => [ 'layout', '=', 'carousel' ],
		];

		$this->controls['autoplaySpeed'] = [
			'tab'         => 'content',
			'group'       => 'settings',
			'label'       => esc_html__( 'Autoplay speed in ms', 'max-addons' ),
			'type'        => 'number',
			'required'    => [
				[ 'layout', '=', 'carousel' ],
				[ 'autoplay', '!=', '' ]
			],
			'placeholder' => 3000,
		];

		$this->controls['pauseOnHover'] = [
			'tab'      => 'content',
			'group'    => 'settings',
			'label'    => esc_html__( 'Pause on hover', 'max-addons' ),
			'type'     => 'checkbox',
			'required' => [
				[ 'layout', '=', 'carousel' ],
				[ 'autoplay', '!=', '' ]
			],
			'inline'   => true,
		];

		$this->controls['speed'] = [
			'tab'         => 'content',
			'group'       => 'settings',
			'label'       => esc_html__( 'Animation speed in ms', 'max-addons' ),
			'type'        => 'number',
			'min'         => 1,
			'placeholder' => 300,
			'required'    => [ 'layout', '=', 'carousel' ],
		];
	}

	// Set filters controls
	public function set_filters_controls() {

		$this->controls['reviewsFilterBy'] = [
			'tab'       => 'content',
			'group'     => 'filters',
			'label'     => esc_html__( 'Filter By', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'default' => esc_html__( 'None', 'max-addons' ),
				'rating'  => esc_html__( 'Minimum Rating', 'max-addons' ),
				'date'    => esc_html__( 'Review Date', 'max-addons' ),
			],
			'inline'    => false,
			'clearable' => false,
			'default'   => 'rating',
		];

		$this->controls['reviewsMinRating'] = [
			'tab'       => 'content',
			'group'     => 'filters',
			'label'     => esc_html__( 'Minimum Rating', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'no' => esc_html__( 'No Minimum Rating', 'max-addons' ),
				'2'  => esc_html__( '2 star', 'max-addons' ),
				'3'  => esc_html__( '3 star', 'max-addons' ),
				'4'  => esc_html__( '4 star', 'max-addons' ),
				'5'  => esc_html__( '5 star', 'max-addons' ),
			],
			'inline'    => false,
			'clearable' => false,
			'default'   => 'no',
			'required'  => [ 'reviewsFilterBy', '=', [ 'rating' ] ],
		];

		$this->controls['minimumRatingInfo'] = [
			'tab'      => 'content',
			'group'    => 'filters',
			'content'  => __( 'Display reviews of ratings greater than or equal to minimum rating. For example, choosing 3 star will skip the reviews with less than 3 rating from displaying.', 'max-addons' ),
			'type'     => 'info',
			'required' => [ 'reviewsFilterBy', '=', [ 'rating' ] ],
		];
	}

	// Set reviewer details controls
	public function set_reviewer_details_controls() {

		$this->controls['headingReviewerImage'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Reviewer Image', 'max-addons' ),
			'type'      => 'separator',
		];

		$this->controls['reviewerImage'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Show Image', 'max-addons' ),
			'type'      => 'checkbox',
			'default'   => true,
			'inline'    => true,
		];

		$this->controls['imageAlign'] = [
			'tab'         => 'content',
			'group'       => 'reviewerDetails',
			'label'       => __( 'Image Position', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'top'      => __( 'Above Name', 'max-addons' ),
				'left'     => __( 'Left of Name', 'max-addons' ),
				'all_left' => __( 'Left of all content', 'max-addons' ),
			],
			'inline'      => false,
			'clearable'   => true,
			'default'     => 'left',
			'placeholder' => __( 'Left of Name', 'max-addons' ),
			'required'    => [ 'reviewerImage', '!=', '' ],
		];

		$this->controls['textAlign'] = [
			'tab'     => 'content',
			'group'   => 'reviewerDetails',
			'label'   => esc_html__( 'Text align', 'max-addons' ),
			'type'    => 'text-align',
			'exclude' => 'justify',
			'inline'  => true,
			'css'     => [
				[
				'property' => 'text-align',
				'selector' => '.mab-review',
				],
			],
			'required' => [ 'imageAlign', '=', 'top' ],
		  ];

		$this->controls['headingreviewerName'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Reviewer Name', 'max-addons' ),
			'type'      => 'separator',
		];

		$this->controls['reviewerName'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Show Name', 'max-addons' ),
			'type'      => 'checkbox',
			'default'   => true,
			'inline'    => true,
		];

		$this->controls['reviewerNameLink'] = [
			'tab'      => 'content',
			'group'    => 'reviewerDetails',
			'label'    => __( 'Link Name', 'max-addons' ),
			'type'     => 'checkbox',
			'default'  => true,
			'inline'   => true,
			'required' => [ 'reviewerName', '!=', '' ],
		];

		$this->controls['headingStarRating'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Star Rating', 'max-addons' ),
			'type'      => 'separator',
		];

		$this->controls['starRating'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Show Star Rating', 'max-addons' ),
			'type'      => 'checkbox',
			'default'   => true,
			'inline'    => true,
		];

		$this->controls['headingReviewSourceIcon'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Review Source Icon', 'max-addons' ),
			'type'      => 'separator',
		];

		$this->controls['reviewSourceIcon'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Show Review Source Icon', 'max-addons' ),
			'type'      => 'checkbox',
			'default'   => true,
			'inline'    => true,
		];

		$this->controls['headingReviewDate'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Review Date', 'max-addons' ),
			'type'      => 'separator',
		];

		$this->controls['reviewDate'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Show Review Date', 'max-addons' ),
			'type'      => 'checkbox',
			'default'   => true,
			'inline'    => true,
		];

		$this->controls['reviewDateType'] = [
			'tab'         => 'content',
			'group'       => 'reviewerDetails',
			'label'       => __( 'Select Type', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'default'  => __( 'Numeric', 'max-addons' ),
				'relative' => __( 'Relative', 'max-addons' ),
			],
			'inline'      => true,
			'clearable'   => true,
			'default'     => 'relative',
			'placeholder' => __( 'Relative', 'max-addons' ),
			'required'    => [
				[ 'reviewDate', '!=', '' ],
				[ 'reviewsSource', '=', 'google' ]
			],
		];

		$this->controls['headingReviewText'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Review Text', 'max-addons' ),
			'type'      => 'separator',
		];

		$this->controls['reviewContent'] = [
			'tab'       => 'content',
			'group'     => 'reviewerDetails',
			'label'     => __( 'Show Review Text', 'max-addons' ),
			'type'      => 'checkbox',
			'default'   => true,
			'inline'    => true,
		];

		$this->controls['reviewContentLength'] = [
			'tab'      => 'content',
			'group'    => 'reviewerDetails',
			'label'    => esc_html__( 'Text Length', 'max-addons' ),
			'type'     => 'number',
			'step'     => 1,
			'inline'   => true,
			'default'  => 25,
			'required' => [ 'reviewContent', '!=', '' ],
		];

		$this->controls['readMore'] = [
			'tab'      => 'content',
			'group'    => 'reviewerDetails',
			'label'    => esc_html__( 'Read More Text', 'max-addons' ),
			'type'     => 'text',
			'inline'   => true,
			'required' => [ 'reviewContent', '!=', '' ],
			'default'  => esc_html__( 'Read More »', 'max-addons' ),
		];
	}

	// Set image controls
	public function set_image_controls() {

		$this->controls['imageSize'] = [
			'tab'         => 'content',
			'group'       => 'image',
			'label'       => esc_html__( 'Image size', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'css'         => [
				[
					'property' => 'width',
					'selector' => '.mab-review-image',
				],
				[
					'property' => 'height',
					'selector' => '.mab-review-image',
				],
			],
			'placeholder' => 60,
		];

		$this->controls['imageBorder'] = [
			'tab'   => 'content',
			'group' => 'image',
			'label' => esc_html__( 'Image border', 'max-addons' ),
			'type'  => 'border',
			'css'   => [
				[
					'property' => 'border',
					'selector' => '.mab-review-image',
				],
			],
		];

		$this->controls['imageBoxShadow'] = [
			'tab'   => 'content',
			'group' => 'image',
			'label' => esc_html__( 'Image box shadow', 'max-addons' ),
			'type'  => 'box-shadow',
			'css'   => [
				[
					'property' => 'box-shadow',
					'selector' => '.mab-review-image',
				],
			],
		];
	}

	// Set rating controls
	public function set_star_rating_controls() {

		$this->controls['starStyle'] = [
			'tab'         => 'content',
			'group'       => 'rating',
			'label'       => esc_html__( 'Star Icon Style', 'max-addons' ),
			'type'        => 'select',
			'options'     => array(
				'default' => __( 'Default', 'max-addons' ),
				'custom'  => __( 'Custom', 'max-addons' ),
			),
			'inline'      => true,
			'clearable'   => false,
			'default'     => 'custom',
			'required'    => [ 'starRating', '!=', '' ],
		];

		$this->controls['headingMarkedStarIcon'] = [
			'tab'      => 'content',
			'group'    => 'rating',
			'label'    => __( 'Marked Icon', 'max-addons' ),
			'type'     => 'separator',
			'required' => [
				[ 'starRating', '!=', '' ],
				[ 'starStyle', '=', 'custom' ]
			],
		];

		$this->controls['markedStarIcon'] = [
			'tab'     => 'content',
			'group'   => 'rating',
			'label'   => esc_html__( 'Icon', 'max-addons' ),
			'type'    => 'icon',
			'default' => [
				'library' => 'fontawesomeSolid',
				'icon'    => 'fas fa-star',
			],
			'required' => [
				[ 'starRating', '!=', '' ],
				[ 'starStyle', '=', 'custom' ]
			],
		];

		$this->controls['markedStarIconColor'] = [
			'tab'      => 'content',
			'group'    => 'rating',
			'label'    => esc_html__( 'Color', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.mab-star-full',
				],
			],
			'required' => [
				[ 'starRating', '!=', '' ],
				[ 'starStyle', '=', 'custom' ]
			],
		];

		$this->controls['headingUnmarkedStarIcon'] = [
			'tab'      => 'content',
			'group'    => 'rating',
			'label'    => __( 'Unmarked Icon', 'max-addons' ),
			'type'     => 'separator',
			'required' => [
				[ 'starRating', '!=', '' ],
				[ 'starStyle', '=', 'custom' ]
			],
		];

		$this->controls['unmarkedStarIcon'] = [
			'tab'     => 'content',
			'group'   => 'rating',
			'label'   => esc_html__( 'Icon', 'max-addons' ),
			'type'    => 'icon',
			'default' => [
				'library' => 'fontawesomeSolid',
				'icon'    => 'fas fa-star',
			],
			'required' => [
				[ 'starRating', '!=', '' ],
				[ 'starStyle', '=', 'custom' ]
			],
		];

		$this->controls['unmarkedStarIconColor'] = [
			'tab'      => 'content',
			'group'    => 'rating',
			'label'    => esc_html__( 'Color', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.mab-star-empty',
				],
			],
			'required' => [
				[ 'starRating', '!=', '' ],
				[ 'starStyle', '=', 'custom' ]
			],
		];
	}

	// Set review text controls
	public function set_review_text_controls() {

		unset( $this->controls['_typography'] );

		$this->controls['reviewNameTypography'] = [
			'tab'      => 'style',
			'group'    => '_typography',
			'label'    => esc_html__( 'Name', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-reviewer-name, .mab-reviewer-name a',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'reviewContent', '!=', '' ],
		];

		$this->controls['reviewDateTypography'] = [
			'tab'      => 'style',
			'group'    => '_typography',
			'label'    => esc_html__( 'Date', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-review-time',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'reviewContent', '!=', '' ],
		];

		$this->controls['reviewTextTypography'] = [
			'tab'      => 'style',
			'group'    => '_typography',
			'label'    => esc_html__( 'Review Text', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-review-content',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'reviewContent', '!=', '' ],
		];

		$this->controls['readMoreTypography'] = [
			'tab'      => 'style',
			'group'    => '_typography',
			'label'    => esc_html__( 'Read More', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-review-content .mab-reviews-read-more',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'reviewContent', '!=', '' ],
		];
	}

	// Set navigation controls
	public function set_navigation_controls() {
		if ( method_exists( '\Bricks\Setup', "get_carousel_controls" ) ) {
			$carousel_controls = \Bricks\Setup::get_carousel_controls();
		} else {
			$carousel_controls = self::get_swiper_controls();
		}

		// Arrows

		$this->controls['arrows']          = $carousel_controls['arrows'];
		$this->controls['arrowHeight']     = $carousel_controls['arrowHeight'];
		$this->controls['arrowWidth']      = $carousel_controls['arrowWidth'];
		$this->controls['arrowBackground'] = $carousel_controls['arrowBackground'];
		$this->controls['arrowBorder']     = $carousel_controls['arrowBorder'];
		$this->controls['arrowTypography'] = $carousel_controls['arrowTypography'];
		$this->controls['arrowTypography']['default'] = [
			'color' => [
				'hex' => \Bricks\Setup::get_default_color( 'body' ),
			],
		];

		$this->controls['prevArrowSeparator'] = $carousel_controls['prevArrowSeparator'];
		$this->controls['prevArrow']          = $carousel_controls['prevArrow'];
		$this->controls['prevArrowTop']       = $carousel_controls['prevArrowTop'];
		$this->controls['prevArrowRight']     = $carousel_controls['prevArrowRight'];
		$this->controls['prevArrowBottom']    = $carousel_controls['prevArrowBottom'];
		$this->controls['prevArrowLeft']      = $carousel_controls['prevArrowLeft'];

		$this->controls['nextArrowSeparator'] = $carousel_controls['nextArrowSeparator'];
		$this->controls['nextArrow']          = $carousel_controls['nextArrow'];
		$this->controls['nextArrowTop']       = $carousel_controls['nextArrowTop'];
		$this->controls['nextArrowRight']     = $carousel_controls['nextArrowRight'];
		$this->controls['nextArrowBottom']    = $carousel_controls['nextArrowBottom'];
		$this->controls['nextArrowLeft']      = $carousel_controls['nextArrowLeft'];

		// Dots

		$this->controls['dots']            = $carousel_controls['dots'];
		$this->controls['dotsDynamic']     = $carousel_controls['dotsDynamic'];
		$this->controls['dotsVertical']    = $carousel_controls['dotsVertical'];
		$this->controls['dotsHeight']      = $carousel_controls['dotsHeight'];
		$this->controls['dotsWidth']       = $carousel_controls['dotsWidth'];
		$this->controls['dotsTop']         = $carousel_controls['dotsTop'];
		$this->controls['dotsRight']       = $carousel_controls['dotsRight'];
		$this->controls['dotsBottom']      = $carousel_controls['dotsBottom'];
		$this->controls['dotsLeft']        = $carousel_controls['dotsLeft'];
		$this->controls['dotsBorder']      = $carousel_controls['dotsBorder'];
		$this->controls['dotsSpacing']     = $carousel_controls['dotsSpacing'];
		$this->controls['dotsColor']       = $carousel_controls['dotsColor'];
		$this->controls['dotsActiveColor'] = $carousel_controls['dotsActiveColor'];
	}

	/**
	 * Get Google Places API from Max Addons options.
	 *
	 * @since 1.2.0
	 * @return string
	 */
	public function get_google_places_api() {
		return \MaxAddonsPro\Classes\Admin_Settings::get_option( 'mab_google_places_api_key' );
	}

	/**
	 * Get Yelp API from Max Addons options.
	 *
	 * @since 1.2.0
	 * @return string
	 */
	public function get_yelp_api() {
		return \MaxAddonsPro\Classes\Admin_Settings::get_option( 'mab_yelp_api_key' );
	}

	public function slider_settings() {
		$settings = $this->settings;

		$slidesToShow = isset( $settings['slidesToShow'] ) ? intval( $settings['slidesToShow'] ) : 3;

		$options = [
			'speed'          => isset( $settings['speed'] ) ? intval( $settings['speed'] ) : 300,
			'autoHeight'     => true, // isset( $settings['adaptiveHeight'] ),
			'spaceBetween'   => isset( $settings['gutter'] ) ? intval( $settings['gutter'] ) : 0,
			'slidesPerView'  => $slidesToShow,
			'slidesPerGroup' => isset( $settings['slidesToScroll'] ) ? intval( $settings['slidesToScroll'] ) : 1,
			'loop'           => isset( $settings['infinite'] ),
		];

		if ( isset( $settings['autoplay'] ) ) {
			$options['autoplay'] = [
				'delay'                => isset( $settings['autoplaySpeed'] ) ? intval( $settings['autoplaySpeed'] ) : 3000,
				'disableOnInteraction' => true,
			];

			// Custom as SwiperJS does not provide this option
			if ( isset( $settings['pauseOnHover'] ) ) {
				$options['pauseOnHover'] = true;
			}
		}

		// Arrow navigation
		if ( isset( $settings['arrows'] ) ) {
			$options['navigation'] = true;
		}

		// Dots
		if ( isset( $settings['dots'] ) ) {
			$options['pagination'] = true;

			if ( isset( $settings['dotsDynamic'] ) && ! isset( $settings['dotsVertical'] ) ) {
				$options['dynamicBullets'] = true;
			}
		}

		$breakpoints = [];

		// Breakpoint values compare to ">="
		if ( isset( $settings['responsive'] ) ) {
			// Custom breakpoints
			foreach ( $settings['responsive'] as $responsive_option ) {
				if ( isset( $responsive_option['breakpoint'] ) ) {
					$breakpoints[ $responsive_option['breakpoint'] ] = [
						'slidesPerView'  => isset( $responsive_option['slidesToShow'] ) ? intval( $responsive_option['slidesToShow'] ) : 1,
						'slidesPerGroup' => isset( $responsive_option['slidesToScroll'] ) ? intval( $responsive_option['slidesToScroll'] ) : 1,
					];
				}
			}
		} else {
			// Default breakpoints
			$breakpoints[992] = [
				'slidesPerView'  => $options['slidesPerView'],
				'slidesPerGroup' => $options['slidesPerGroup'],
			];

			$breakpoints[768] = [
				'slidesPerView'  => $slidesToShow > 1 ? 2 : 1,
				'slidesPerGroup' => 1,
			];

			$breakpoints[1] = [
				'slidesPerView'  => 1,
				'slidesPerGroup' => 1,
			];
		}

		$options['breakpoints'] = $breakpoints;

		$this->set_attribute( 'container', 'class', 'bricks-swiper-container' );
		$this->set_attribute( 'container', 'data-script-args', wp_json_encode( $options ) );
	}

	public function render_star_icon( $icon, $icon_class = '' ) {
		$settings = $this->settings;

		$icon = ! empty( $icon ) ? self::render_icon( $icon, ['mab-rating-star', $icon_class] ) : false;

		return $icon;
	}

	/**
	 * Gets the layout of five star.
	 *
	 * @since 1.2.0
	 * @access public
	 *
	 * @param array $total_rating total_rating.
	 * @param array $review data of single review.
	 * @param array $settings The settings array.
	 * @return the layout of Google reviews star rating.
	 */
	public function render_stars( $total_rating, $review, $settings ) {
		$settings   = $this->settings;
		$rating     = $total_rating;
		$stars_html = '';
		$flag       = 0;

		if ( isset( $settings['starStyle'] ) && 'default' === $settings['starStyle'] ) {

			if ( 'google' === $review['source'] ) {
				$marked_icon_html   = self::render_star_icon( array( 'library' => 'fontawesomeSolid', 'icon' => 'fas fa-star' ), 'mab-star-full mab-star-default');
				$unmarked_icon_html = self::render_star_icon( array( 'library' => 'fontawesomeSolid', 'icon' => 'fas fa-star' ), 'mab-star-empty mab-star-default');
				$flag               = 1;
			} else {
				$stars_html = '<svg version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px"
	 width="100px" height="18px" viewBox="-1 0.054 32 5.642" enable-background="new -1 0.054 32 5.642" xml:space="preserve" class="mab-yelp-rating-svg-' . $rating . '">
<g>
	<path fill="#CECECE" d="M4.075,0.055h-4.511C-0.744,0.055-1,0.307-1,0.626v4.497c0,0.314,0.256,0.572,0.564,0.572h4.511
		c0.308,0,0.557-0.258,0.557-0.572V0.626C4.632,0.307,4.383,0.055,4.075,0.055z M3.973,2.486L2.889,3.434l0.322,1.396
		C3.241,4.927,3.13,5.004,3.05,4.945L1.82,4.214L0.59,4.945C0.501,5,0.399,4.926,0.42,4.829l0.33-1.396l-1.086-0.947
		c-0.08-0.061-0.041-0.187,0.062-0.19l1.432-0.123l0.56-1.327c0.03-0.088,0.161-0.088,0.205,0L2.48,2.173l1.433,0.123
		C4.003,2.302,4.046,2.428,3.973,2.486z" class="mab-yelp-rating-1"/>
	<path fill="#CECECE" d="M10.663,0.055H6.159c-0.311,0-0.571,0.252-0.571,0.571v4.497c0,0.314,0.26,0.572,0.571,0.572h4.504
		c0.315,0,0.564-0.258,0.564-0.572V0.626C11.227,0.307,10.978,0.055,10.663,0.055z M10.567,2.486L9.483,3.434l0.322,1.396
		C9.83,4.927,9.717,5.004,9.64,4.945L8.414,4.214l-1.23,0.731C7.096,5,6.994,4.925,7.008,4.829l0.329-1.396L6.25,2.486
		C6.172,2.426,6.216,2.3,6.319,2.296l1.425-0.123l0.565-1.327c0.032-0.088,0.164-0.088,0.208,0l0.557,1.327L10.5,2.296
		C10.597,2.302,10.641,2.428,10.567,2.486z" class="mab-yelp-rating-2"/>
	<path fill="#CECECE" d="M17.246,0.055h-4.497c-0.318,0-0.571,0.252-0.571,0.571v4.497c0,0.314,0.253,0.572,0.571,0.572h4.497
		c0.32,0,0.572-0.258,0.572-0.572V0.626C17.818,0.307,17.566,0.055,17.246,0.055z M17.158,2.486l-1.084,0.947l0.322,1.396
		c0.018,0.098-0.088,0.175-0.172,0.116l-1.228-0.73l-1.225,0.732c-0.086,0.054-0.191-0.021-0.174-0.117l0.322-1.396l-1.084-0.944
		c-0.073-0.062-0.029-0.188,0.073-0.191l1.421-0.123l0.562-1.325c0.039-0.09,0.172-0.09,0.211,0l0.561,1.325l1.422,0.123
		C17.188,2.302,17.232,2.428,17.158,2.486z" class="mab-yelp-rating-3"/>
	<path fill="#CECECE" d="M23.838,0.055h-4.503c-0.315,0-0.565,0.252-0.565,0.571v4.497c0,0.314,0.25,0.572,0.565,0.572h4.503
		c0.314,0,0.572-0.258,0.572-0.572V0.626C24.41,0.307,24.152,0.055,23.838,0.055z M23.742,2.486l-1.083,0.947l0.323,1.396
		c0.026,0.098-0.082,0.175-0.17,0.116l-1.229-0.731l-1.226,0.731C20.279,5,20.168,4.925,20.191,4.829l0.322-1.396L19.43,2.486
		C19.355,2.426,19.4,2.3,19.496,2.296l1.426-0.123l0.563-1.327c0.037-0.088,0.17-0.088,0.205,0l0.559,1.327l1.438,0.123
		C23.773,2.302,23.824,2.428,23.742,2.486z" class="mab-yelp-rating-4"/>
	<path fill="#CECECE" d="M30.43,0.055h-4.505c-0.3,0-0.563,0.252-0.563,0.571v4.497c0,0.314,0.266,0.572,0.563,0.572h4.505
		c0.321,0,0.57-0.258,0.57-0.572V0.626C31,0.307,30.751,0.055,30.43,0.055z M30.34,2.486l-1.083,0.947l0.323,1.396
		c0.027,0.098-0.09,0.175-0.176,0.116l-1.229-0.731l-1.229,0.731C26.868,5,26.764,4.925,26.791,4.829l0.326-1.396l-1.086-0.945
		c-0.088-0.062-0.035-0.188,0.059-0.191l1.438-0.123l0.557-1.326c0.031-0.089,0.169-0.089,0.207,0l0.557,1.326l1.436,0.123
		C30.371,2.302,30.416,2.428,30.34,2.486z" class="mab-yelp-rating-5"/>
</g>
</svg>';
			}
		} else {
			$marked_icon_html   = self::render_star_icon( $settings['markedStarIcon'], 'mab-star-full mab-star-custom' );
			//$marked_icon_html   = ! empty( $settings['markedStarIcon'] ) ? self::render_icon( $settings['markedStarIcon'], ['mab-rating-star', 'mab-star-full', 'mab-star-custom'] ) : false;
			$unmarked_icon_html = self::render_star_icon( $settings['unmarkedStarIcon'], 'mab-star-empty mab-star-custom' );
			//$unmarked_icon_html = ! empty( $settings['unmarkedStarIcon'] ) ? self::render_icon( $settings['unmarkedStarIcon'], ['mab-rating-star', 'mab-star-empty', 'mab-star-custom'] ) : false;
			$flag               = 1;
		}

		if ( $flag ) {
			for ( $stars = 1; $stars <= 5; $stars++ ) {
				if ( $stars <= $rating ) {
					$stars_html .= $marked_icon_html;
				} else {
					$stars_html .= $unmarked_icon_html;
				}
			}
		}

		return $stars_html;
	}

	/**
	 * Get API data.
	 * 
	 * Handles review source remote API calls.
	 *
	 * @since 1.2.0
	 * @param string 	$source		Review source.
	 *
	 * @return array	$response	API response.
	 */
	public function get_api_data( $source ) {
		$settings = $this->settings;

		$api_args = array(
			'method'      => 'POST',
			'timeout'     => 60,
			'httpversion' => '1.0',
			'sslverify'   => false,
		);

		if ( 'google' === $source ) {
			$api_key = $this->get_google_places_api();
			$place_id = ! empty( $settings['googlePlaceId'] ) ? $this->render_dynamic_data( $settings['googlePlaceId'] ) : '';

			if ( empty( $api_key ) ) {
				return new \WP_Error( 'missing_api_key', __( 'To display Google Reviews, you need to setup API key.', 'max-addons' ) );
			}
			if ( empty( $place_id ) ) {
				return new \WP_Error( 'missing_place_id', __( 'To display Google Reviews, you need to provide valid Place ID.', 'max-addons' ) );
			}

			$url = add_query_arg(
				array(
					'key'      => $api_key,
					'placeid'  => $place_id,
					'language' => isset( $settings['languageId'] ) ? $settings['languageId'] : get_locale(),
				),
				'https://maps.googleapis.com/maps/api/place/details/json'
			);
		}

		if ( 'yelp' === $source ) {
			$business_id = ! empty( $settings['yelpBusinessId'] ) ? $this->render_dynamic_data( $settings['yelpBusinessId'] ) : '';

			if ( empty( $business_id ) ) {
				return new \WP_Error( 'missing_business_id', __( 'To display Yelp Reviews, you need to provide valid Business ID.', 'max-addons' ) );
			}

			$url = 'https://api.yelp.com/v3/businesses/' . $business_id . '/reviews';

			$yelp_api_key = $this->get_yelp_api();
			
			$api_args['method'] = 'GET';
			$api_args['user-agent'] = '';
			$api_args['headers'] = array(
				'Authorization' => 'Bearer ' . $yelp_api_key,
			);
		}

		$response = wp_remote_post(
			esc_url_raw( $url ),
			$api_args
		);

		if ( ! is_wp_error( $response ) ) {
			$body = json_decode( wp_remote_retrieve_body( $response ) );
			if ( isset( $body->error_message ) && ! empty( $body->error_message ) ) {
				$status = isset( $body->status ) ? $body->status : $source . '_api_error';
				return new \WP_Error( $status, $body->error_message );
			}
		}

		return $response;
	}

	/**
	 * Get google reviews.
	 * 
	 * Get reviews from Google Place API and store it in transient.
	 *
	 * @since 1.2.0
	 *
	 * @return array $response Reviews data.
	 */
	private function get_google_reviews( $settings ) {

		$response = array(
			'data'	=> array(),
			'error' => false,
		);

		if ( empty( $settings['googlePlaceId'] ) ) {
			return;
		};

		$language_id = isset( $settings['languageId'] ) ? $settings['languageId'] : '';

		$transient_name = 'mab_reviews_' . $this->render_dynamic_data( $settings['googlePlaceId'] ) . '&language=' . $language_id;

		$response['data'] = get_transient( $transient_name );

		if ( empty( $response['data'] ) ) {
			$api_data = $this->get_api_data( 'google' );

			if ( is_wp_error( $api_data ) ) {
				
				$response['error'] = $api_data;

			} else {
				if ( 200 === wp_remote_retrieve_response_code( $api_data ) ) {

					$data = json_decode( wp_remote_retrieve_body( $api_data ) );

					if ( 'OK' !== $data->status ) {
						$response['error'] = isset( $data->error_message ) ? $data->error_message : __( 'No reviews found.', 'max-addons' );
					} else {
						if ( isset( $data->result ) && isset( $data->result->reviews ) ) {
							$response['data'] = array(
								'reviews'  => $data->result->reviews,
								'location' => array(),
							);

							if ( isset( $data->result->geometry->location ) ) {
								$response['data']['location'] = $data->result->geometry->location;
							}

							set_transient( $transient_name, $response['data'], $this->get_transient_expire( $settings ) );

							$response['error'] = false;
						} else {
							$response['error'] = __( 'This place doesn\'t have any reviews.', 'max-addons' );
						}
					}
				}
			}	
		}

		return $response;
	}

	/**
	 * Get yelp reviews.
	 * 
	 * Get reviews from Yelp Business API and store it in transient.
	 *
	 * @since 1.2.0
	 *
	 * @return array $response Reviews data.
	 */
	private function get_yelp_reviews( $settings ) {

		$response = array(
			'data'	=> array(),
			'error' => false,
		);

		if ( empty( $settings['yelpBusinessId'] ) ) {
			return;
		};

		$transient_name = 'mab_reviews_' . $this->render_dynamic_data( $settings['yelpBusinessId'] );

		$response['data'] = get_transient( $transient_name );

		if ( empty( $response['data'] ) ) {
			$api_data = $this->get_api_data( 'yelp' );

			if ( is_wp_error( $api_data ) ) {
				
				$response['error'] = $api_data;

			} else {
				if ( 200 !== wp_remote_retrieve_response_code( $api_data ) ) {
					$data = json_decode( wp_remote_retrieve_body( $api_data ) );

					if ( isset( $data->error ) ) {
						if ( 'VALIDATION_ERROR' === $data->error->code ) {
							$response['error'] = __( 'Yelp Reviews Error: Invalid or empty API key.', 'max-addons' );
						}
						if ( 'BUSINESS_NOT_FOUND' === $data->error->code ) {
							$response['error'] = __( 'Yelp Reviews Error: Incorrect or empty Business ID.', 'max-addons' );
						}
						if ( 'INTERNAL_SERVER_ERROR' === $data->error->code ) {
							$response['error'] = __( 'Yelp Reviews Error: Something is wrong with Yelp.', 'max-addons' );
						}
					} else {
						$response['error'] = __( 'Yelp Reviews Error: Unknown error occurred.', 'max-addons' );
					}
				} else {
					$data = json_decode( wp_remote_retrieve_body( $api_data ) );

					if ( empty( $data ) || ! isset( $data->reviews ) || empty( $data->reviews ) ) {
						$response['error'] = __( 'This business doesn\'t have any reviews.', 'max-addons' );
					} else {
						$response['data'] = $data->reviews;

						set_transient( $transient_name, $response['data'], $this->get_transient_expire( $settings ) );

						$response['error'] = false;
					}
				}
			}	
		}

		return $response;
	}

	/**
	 * Gets expire time of transient.
	 *
	 * @since 1.2.0
	 * @param array $settings The settings array.
	 * @return the reviews transient expire time.
	 * @access public
	 */
	public function get_transient_expire( $settings ) {

		$expire_value = isset( $settings['reviewsRefreshTime'] ) ? $settings['reviewsRefreshTime'] : 'day';
		$expire_time  = 24 * HOUR_IN_SECONDS;

		if ( 'hour' === $expire_value ) {
			$expire_time = 60 * MINUTE_IN_SECONDS;
		} elseif ( 'week' === $expire_value ) {
			$expire_time = 7 * DAY_IN_SECONDS;
		} elseif ( 'month' === $expire_value ) {
			$expire_time = 30 * DAY_IN_SECONDS;
		} elseif ( 'year' === $expire_value ) {
			$expire_time = 365 * DAY_IN_SECONDS;
		}

		return $expire_time;
	}

	/**
	 * Get Reviews array with the same key for Google & Yelp.
	 *
	 * @since 1.2.0
	 * @param string $type The reviews source.
	 * @param array  $reviews The reviews array.
	 * @param array  $settings The settings array.
	 * @return the merged array of Google & Yelp reviews.
	 * @access public
	 */
	public function parse_reviews( $type, $reviews, $settings ) {
		if ( is_wp_error( $reviews['error'] ) ) {
			return $reviews['error'];
		}

		if ( empty( $reviews['data'] ) ) {
			return;
		}

		$parsed_reviews       = array();
		$filter_by_min_rating = false;

		if ( isset( $settings['reviewsMinRating'] ) && 'no' !== $settings['reviewsMinRating'] ) {
			$filter_by_min_rating = true;
		}

		$data = $reviews['data'];

		if ( 'google' === $type ) {
			$data = $data['reviews'];
		}

		foreach ( $data as $review ) {
			$_review = array();

			if ( 'google' === $type ) {
				if ( ! empty( $settings['googlePlaceId'] ) ) {
					$user_review_url = explode( '/reviews', $review->author_url );
					array_pop( $user_review_url );
					$review_url = $user_review_url[0] . '/place/' . $this->render_dynamic_data( $settings['googlePlaceId'] );

					$_review['source']                    = 'google';
					$_review['author_name']               = $review->author_name;
					$_review['author_url']                = $review->author_url;
					$_review['profile_photo_url']         = $review->profile_photo_url;
					$_review['rating']                    = $review->rating;
					$_review['relative_time_description'] = $review->relative_time_description;
					$_review['text']                      = $review->text;
					$_review['time']                      = $review->time;
					$_review['review_url']                = $review_url;
				}
			}
			if ( 'yelp' === $type ) {
				$_review['source']                    = 'yelp';
				$_review['author_name']               = $review->user->name;
				$_review['author_url']                = $review->user->profile_url;
				$_review['profile_photo_url']         = $review->user->image_url;
				$_review['rating']                    = $review->rating;
				$_review['relative_time_description'] = '';
				$_review['text']                      = $review->text;
				$_review['time']                      = $review->time_created;
				$_review['review_url']                = $review->url;
			}

			if ( $filter_by_min_rating ) {
				if ( $review->rating >= $settings['reviewsMinRating'] ) {
					array_push( $parsed_reviews, $_review );
				}
			} else {
				array_push( $parsed_reviews, $_review );
			}
		}

		return $parsed_reviews;

	}
	/**
	 * Get Reviews array with the same key for Google & Yelp.
	 *
	 * @since 1.2.0
	 * @param array $settings The settings array.
	 * @return the layout of Google reviews.
	 * @access public
	 */
	public function get_reviews( $settings ) {
		$reviews = array();

		if ( isset( $settings['reviewsSource'] ) ) {
			if ( 'google' === $settings['reviewsSource'] ) {

				$reviews = $this->get_google_reviews( $settings );
				$reviews = $this->parse_reviews( 'google', $reviews, $settings );

			} elseif ( 'yelp' === $settings['reviewsSource'] ) {

				$reviews = $this->get_yelp_reviews( $settings );
				$reviews = $this->parse_reviews( 'yelp', $reviews, $settings );

			} elseif ( 'all' === $settings['reviewsSource'] ) {

				$google_reviews = $this->get_google_reviews( $settings );
				$yelp_reviews = $this->get_yelp_reviews( $settings );

				$google_reviews = $this->parse_reviews( 'google', $google_reviews, $settings );
				$yelp_reviews   = $this->parse_reviews( 'yelp', $yelp_reviews, $settings );

				if ( empty( $google_reviews ) || empty( $yelp_reviews ) ) {
					return;
				}

				$count = count( $google_reviews );

				/* Merge reviews array elements inalternative order */
				for ( $i = 0; $i < $count; $i++ ) {
					$reviews[] = $google_reviews[ $i ];
					if ( $i < count( $yelp_reviews ) ) {
						$reviews[] = $yelp_reviews[ $i ];
					}
				}
				$reviews = array_filter( $reviews );
			}
		}

		return $reviews;
	}

	/**
	 * Get sorted array of reviews by rating.
	 *
	 * @since 1.2.0
	 * @access public
	 * @param string $review1 represents review1 to compare.
	 * @param string $review2 represents review2 to compare.
	 * @return string of compared reviews.
	 */
	public function filter_by_rating( $review1, $review2 ) {
		return strcmp( $review2['rating'], $review1['rating'] );
	}

	/**
	 * Get sorted array of reviews by date.
	 *
	 * @since 1.2.0
	 * @access public
	 * @param string $review1 represents review1 to compare.
	 * @param string $review2 represents review2 to compare.
	 * @return string of compared reviews.
	 */
	public function filter_by_date( $review1, $review2 ) {
		return strcmp( $review2['time'], $review1['time'] );
	}

	/**
	 * Get reviewer name section.
	 *
	 * @since 1.2.0
	 * @access public
	 * @param string $review represents single review.
	 * @param array  $settings represents settings array.
	 */
	public function get_reviewer_name( $review, $settings ) {
		if ( isset( $settings['reviewerName'] ) ) {
			?>
			<?php if ( isset( $settings['reviewerNameLink'] ) ) { ?>
				<span class="mab-reviewer-name"><?php echo wp_kses_post( "<a href={$review['author_url']} target='_blank'>{$review['author_name']}</a>" ); ?></span>
			<?php } else { ?>
				<span class="mab-reviewer-name"><?php echo wp_kses_post( "{$review['author_name']}" ); ?></span>
				<?php
			}
		}
	}

	/**
	 * Get review header.
	 *
	 * @since 1.2.0
	 * @access public
	 * @param string $review represents single review.
	 * @param string $photolink represents reviewer image link.
	 * @param array  $settings represents settings array.
	 */
	public function get_reviews_header( $review, $photolink, $settings ) {
		$skin         = isset( $settings['skin'] ) ? $settings['skin'] : 'classic';
		$total_rating = $review['rating'];
		$timestamp    = ( 'google' === $review['source'] ) ? $review['time'] : strtotime( $review['time'] );
		$date         = gmdate( 'd-m-Y', $timestamp );
		$image_align  = ( isset( $settings['reviewerImage'] ) ) ? ( isset( $settings['imageAlign'] ) ? $settings['imageAlign'] : 'left' ) : '';

		$google_svg = '<svg version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px"
	 width="18px" height="18px" viewBox="0 0 100 100" enable-background="new 0 0 100 100" xml:space="preserve">
		<g>
			<path id="XMLID_5_" fill="#FFFFFF" d="M34.963,3.686C23.018,7.777,12.846,16.712,7.206,28.002
				c-1.963,3.891-3.397,8.045-4.258,12.315C0.78,50.961,2.289,62.307,7.2,72.002c3.19,6.328,7.762,11.951,13.311,16.361
				c5.236,4.175,11.336,7.256,17.806,8.979c8.163,2.188,16.854,2.14,25.068,0.268c7.426-1.709,14.452-5.256,20.061-10.436
				c5.929-5.449,10.158-12.63,12.399-20.342c2.441-8.415,2.779-17.397,1.249-26.011c-15.373-0.009-30.744-0.004-46.113-0.002
				c0.003,6.375-0.007,12.749,0.006,19.122c8.9-0.003,17.802-0.006,26.703,0c-1.034,6.107-4.665,11.696-9.813,15.135
				c-3.236,2.176-6.954,3.587-10.787,4.26c-3.861,0.661-7.846,0.746-11.696-0.035c-3.914-0.781-7.649-2.412-10.909-4.711
				c-5.212-3.662-9.189-9.018-11.23-15.048c-2.088-6.132-2.103-12.954,0.009-19.08c1.466-4.316,3.907-8.305,7.112-11.551
				c3.955-4.048,9.095-6.941,14.633-8.128c4.742-1.013,9.745-0.819,14.389,0.586c3.947,1.198,7.584,3.359,10.563,6.206
				c3.012-2.996,6.011-6.008,9.014-9.008c1.579-1.615,3.236-3.161,4.763-4.819C79.172,9.52,73.819,6.123,67.97,3.976
				C57.438,0.1,45.564,0.018,34.963,3.686z"/>
			<g>
				<path id="XMLID_4_" fill="#EA4335" d="M34.963,3.686C45.564,0.018,57.438,0.1,67.97,3.976c5.85,2.147,11.202,5.544,15.769,9.771
					c-1.526,1.659-3.184,3.205-4.763,4.819c-3.003,3-6.002,6.012-9.014,9.008c-2.979-2.846-6.616-5.008-10.563-6.206
					c-4.645-1.405-9.647-1.599-14.389-0.586c-5.539,1.187-10.679,4.08-14.633,8.128c-3.206,3.246-5.646,7.235-7.112,11.551
					c-5.353-4.152-10.703-8.307-16.058-12.458C12.846,16.712,23.018,7.777,34.963,3.686z"/>
			</g>
			<g>
				<path id="XMLID_3_" fill="#FBBC05" d="M2.947,40.317c0.861-4.27,2.295-8.424,4.258-12.315c5.355,4.151,10.706,8.306,16.058,12.458
					c-2.112,6.126-2.097,12.948-0.009,19.08C17.903,63.695,12.557,67.856,7.2,72.002C2.289,62.307,0.78,50.961,2.947,40.317z"/>
			</g>
			<g>
				<path id="XMLID_2_" fill="#4285F4" d="M50.981,40.818c15.369-0.002,30.74-0.006,46.113,0.002
					c1.53,8.614,1.192,17.596-1.249,26.011c-2.241,7.712-6.471,14.893-12.399,20.342c-5.18-4.039-10.386-8.057-15.568-12.099
					c5.147-3.438,8.778-9.027,9.813-15.135c-8.9-0.006-17.803-0.003-26.703,0C50.974,53.567,50.984,47.194,50.981,40.818z"/>
			</g>
			<g>
				<path id="XMLID_1_" fill="#34A853" d="M7.2,72.002c5.356-4.146,10.703-8.307,16.055-12.461c2.041,6.03,6.018,11.386,11.23,15.048
					c3.26,2.299,6.995,3.93,10.909,4.711c3.851,0.781,7.835,0.696,11.696,0.035c3.833-0.673,7.551-2.084,10.787-4.26
					c5.183,4.042,10.389,8.06,15.568,12.099c-5.608,5.18-12.635,8.727-20.061,10.436c-8.215,1.872-16.906,1.921-25.068-0.268
					c-6.469-1.723-12.57-4.804-17.806-8.979C14.962,83.953,10.39,78.33,7.2,72.002z"/>
			</g>
		</g>
		</svg>';

		if ( isset( $settings['reviewDate'] ) ) {
			if ( 'google' === $settings['reviewsSource'] ) {
				$date_type  = isset( $settings['reviewDateType'] ) ? $settings['reviewDateType'] : 'relative';
				$date_value = ( 'default' === $date_type ) ? $date : $review['relative_time_description'];
			} else {
				$date_value = $date;
			}
		}
		?>
		<div class="mab-review-header">
			<?php if ( 'all_left' !== $image_align ) { ?>
				<div class="mab-review-image" style="background-image:url( <?php echo wp_kses_post( $photolink ); ?> );"></div>
			<?php } ?>
			<div class="mab-review-details">
				<?php
				if ( 'classic' === $skin ) {
					$this->get_reviewer_name( $review, $settings );
				}
				?>
				<?php if ( isset( $settings['starRating'] ) ) { ?>
					<?php
					$star_rating_wrap_class = '';

					if ( 'yelp' !== $review['source'] || 'custom' === $settings['starStyle'] ) {
						$star_rating_wrap_class = 'mab-review-stars';
					}
					?>
					<span class="mab-star-rating-wrapper <?php echo esc_attr( $star_rating_wrap_class ); ?>">
						<span class="mab-star-rating"><?php echo $this->render_stars( $total_rating, $review, $settings ); ?></span>
					</span>
				<?php } ?>
				<?php
				if ( isset( $settings['reviewDate'] ) ) {
					$review_source = ( 'google' === $review['source'] ) ? 'Google' : 'Yelp';
					$via_source    = ' via ' . $review_source;
					if ( isset( $settings['reviewSourceIcon'] ) ) {
						$via_source = '';
					}
					?>
					<span class="mab-review-time"><?php echo esc_attr( $date_value ) . esc_attr( $via_source ); ?></span>
				<?php } ?>
				<?php
				if ( 'card' === $skin ) {
					$this->get_reviewer_name( $review, $settings );
				}
				?>
			</div>
			<?php if ( isset( $settings['reviewSourceIcon'] ) && ( 'all_left' === $image_align || 'left' === $image_align ) ) { ?>
				<div class="mab-review-icon-wrap">
					<?php if ( 'yelp' === $review['source'] ) {
						self::render_yelp_icon(array( 'library' => 'fontawesomeBrands', 'icon' => 'fab fa-yelp' ));
					} else {
						echo $google_svg; // phpcs:ignore
					}
					?>
				</div>
			<?php } ?>
		</div>
		<?php
	}

	private static function render_yelp_icon( $network_name ) {
		$icon = ! empty( $network_name ) ? self::render_icon( $network_name ) : false;

		$icon = '<span class="mab-icon">' . $icon . '</span>';

		echo $icon;
	}

	/**
	 * Render classic skin output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @access public
	 */
	public function render_skin_classic( $review ) {
		$settings    = $this->settings;
		$photolink   = ( null !== $review['profile_photo_url'] ) ? $review['profile_photo_url'] : ( MAB_PRO_URL . 'assets/images/user.png' );
		$image_align = ( isset( $settings['reviewerImage'] ) ) ? ( isset( $settings['imageAlign'] ) ? $settings['imageAlign'] : 'left' ) : '';
		?>
		<?php if ( 'all_left' === $image_align ) { ?>
			<div class="mab-review-image" style="background-image:url( <?php echo esc_url( $photolink ); ?> );"></div>
		<?php } ?>
		<div class="mab-review-inner-wrap">
			<?php $this->get_reviews_header( $review, $photolink, $settings ); ?>
			<?php if ( isset( $settings['reviewContent'] ) ) { ?>
				<?php
				$the_content = $review['text'];
				if ( isset( $settings['reviewContentLength'] ) ) {
					$the_content    = wp_strip_all_tags( $review['text'] ); // Strips tags.
					$content_length = $settings['reviewContentLength']; // Sets content length by word count.
					$words          = explode( ' ', $the_content, $content_length + 1 );
					if ( count( $words ) > $content_length ) {
						array_pop( $words );
						$the_content  = implode( ' ', $words ); // put in content only the number of word that is set in $content_length.
						$the_content .= '...';
						if ( isset( $settings['readMore'] ) ) {
							$the_content .= '<a href="' . apply_filters( 'mab_reviews_read_more', $review['review_url'] ) . '"  target="_blank" rel="noopener noreferrer" class="mab-reviews-read-more">' . $settings['readMore'] . '</a>';
						}
					}
				}
				?>
				<div class="mab-review-content"><?php echo wp_kses_post( $the_content ); ?></div>
			<?php } ?>
		</div>
		<?php
	}

	/**
	 * Render card skin output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @access public
	 */
	public function render_skin_card( $review ) {
		$settings    = $this->settings;
		$photolink   = ( null !== $review['profile_photo_url'] ) ? $review['profile_photo_url'] : ( MAB_PRO_URL . 'assets/images/user.png' );
		$image_align = ( isset( $settings['reviewerImage'] ) ) ? ( isset( $settings['imageAlign'] ) ? $settings['imageAlign'] : 'left' ) : '';
		?>
		<?php if ( 'all_left' === $image_align ) { ?>
			<div class="mab-review-image" style="background-image:url( <?php echo esc_url( $photolink ); ?> );"></div>
		<?php } ?>
		<div class="mab-review-inner-wrap">
			<?php if ( isset( $settings['reviewContent'] ) ) { ?>
				<?php
				$the_content = $review['text'];
				if ( isset( $settings['reviewContentLength'] ) ) {
					$the_content    = wp_strip_all_tags( $review['text'] ); // Strips tags.
					$content_length = $settings['reviewContentLength']; // Sets content length by word count.
					$words          = explode( ' ', $the_content, $content_length + 1 );
					if ( count( $words ) > $content_length ) {
						array_pop( $words );
						$the_content  = implode( ' ', $words ); // put in content only the number of word that is set in $content_length.
						$the_content .= '...';
						if ( isset( $settings['readMore'] ) ) {
							$the_content .= '<a href="' . apply_filters( 'mab_reviews_read_more', $review['review_url'] ) . '"  target="_blank" rel="noopener noreferrer" class="mab-reviews-read-more">' . $settings['readMore'] . '</a>';
						}
					}
				}
				?>
				<div class="mab-review-content"><?php echo wp_kses_post( $the_content ); ?></div>
				<?php $this->get_reviews_header( $review, $photolink, $settings ); ?>
			<?php } ?>
		</div>
		<?php
	}

	public function render() {
		$settings = $this->settings;

		$is_editor          = ! $this->is_frontend;
		$skin               = isset( $settings['skin'] ) ? $settings['skin'] : 'classic';
		$reviews            = '';
		$reviews_max        = 8;
		$disply_num_reviews = 8;
		$layout             = isset( $settings['layout'] ) ? $settings['layout'] : 'grid';
		$image_align        = ( isset( $settings['reviewerImage'] ) ) ? ( isset( $settings['imageAlign'] ) ? 'mab-review-image-' . $settings['imageAlign'] : 'mab-review-image-left' ) : '';
		$item_tag           = 'li';

		$item_sizer_classes = [
			'bricks-layout-item',
		];

		$reviews = $this->get_reviews( $settings );

		if ( is_wp_error( $reviews ) ) {
			$error_message = $reviews->get_error_message();

			if ( $is_editor ) {
				return $this->render_element_placeholder(
					[
						'title' => $error_message,
					]
				);
			}
			return;
		}

		if ( ! is_array( $reviews ) || empty( $reviews ) ) {
			return;
		}

		$wrapper_classes = [
			'mab-reviews-wrapper',
			'mab-reviews-' . $skin,
			$image_align
		];

		if ( 'carousel' === $layout ) {
			$this->tag = 'div';
			$item_tag  = 'div';

			$this->slider_settings();

			$wrapper_classes[] = 'swiper-container-wrap';

			$this->set_attribute( 'review-wrap', 'class', 'swiper-slide' );
		} else {
			$wrapper_classes[] = 'bricks-layout-wrapper';

			$this->set_attribute( 'review-wrap', 'class', $item_sizer_classes );
		}

		$this->set_attribute( 'reviews-wrap', 'class', $wrapper_classes );
		$this->set_attribute( 'reviews-wrap', 'data-layout', $layout );
		$this->set_attribute( 'reviews-wrap', 'data-review-skin', $skin );
		?>
		<div <?php echo wp_kses_post( $this->render_attributes( '_root' ) ); ?>>
			<<?php echo $this->tag; ?> <?php echo wp_kses_post( $this->render_attributes( 'reviews-wrap' ) ); ?>>
				<?php if ( 'carousel' === $layout ) { ?>
				<div <?php echo wp_kses_post( $this->render_attributes( 'container' ) ); ?>>
					<div class="swiper-wrapper">
				<?php } ?>
						<?php
							if ( isset( $settings['reviewsFilterBy'] ) ) {
								if ( 'rating' === $settings['reviewsFilterBy'] ) {
									usort( $reviews, array( $this, 'filter_by_rating' ) );
								} elseif ( 'date' === $settings['reviewsFilterBy'] ) {
									usort( $reviews, array( $this, 'filter_by_date' ) );
								}
							}

							if ( isset( $settings['reviewsSource'] ) ) {
								if ( 'google' === $settings['reviewsSource'] ) {
									$reviews_max        = 5;
									$disply_num_reviews = isset( $settings['googleReviewsCount'] ) ? $settings['googleReviewsCount'] : 3;
								} elseif ( 'yelp' === $settings['reviewsSource'] ) {
									$reviews_max        = 3;
									$disply_num_reviews = isset( $settings['yelpReviewsCount'] ) ? $settings['yelpReviewsCount'] : 3;
								} elseif ( 'all' === $settings['reviewsSource'] ) {
									$reviews_max        = 8;
									$disply_num_reviews = isset( $settings['reviewsCount'] ) ? $settings['reviewsCount'] : 3;
								}
							}

							$disply_num_reviews = ( '' !== $disply_num_reviews ) ? $disply_num_reviews : $reviews_max;

							if ( $reviews_max !== $disply_num_reviews ) {
								$display_number = (int) $disply_num_reviews;
								$items          = array_slice( $reviews, 0, $display_number );

								if ( isset( $settings['random'] ) ) {
									$review_keys = array_keys( $items );
									shuffle( $review_keys );
									$reviews = [];
						
									foreach ( $review_keys as $key ) {
										$reviews[ $key ] = $items[ $key ];
									}
								} else {
									$reviews = $items;
								}
							}

							foreach ( $reviews as $key => $review ) {
								?>
								<<?php echo $item_tag; ?> <?php echo wp_kses_post( $this->render_attributes( 'review-wrap' ) ); ?>>
									<div class="mab-review mab-review-type-<?php echo esc_attr( $review['source'] ); ?>" >
										<?php
										if ( 'card' === $skin ) {
											$this->render_skin_card( $review );
										} else {
											$this->render_skin_classic( $review );
										}
										?>
									</div>
								</<?php echo $item_tag; ?>>
								<?php
							}
						?>
					<?php if ( 'carousel' === $layout ) { ?>
					</div>
				</div>
				<?php echo $this->render_swiper_nav(); ?>
				<?php } ?>
			</<?php echo $this->tag; ?>>
		</div>
		<?php
	}
}
