<?php
namespace MaxAddonsPro\Elements;

use MaxAddonsPro\Classes\Helper;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Off_Canvas_Element extends \Bricks\Element {
	// Element properties
	public $category     = 'max-addons-elements'; // Use predefined element category 'general'
	public $name         = 'max-off-canvas'; // Make sure to prefix your elements
	public $icon         = 'ti-direction-alt max-element'; // Themify icon font class
	public $css_selector = ''; // Default CSS selector
	public $scripts      = [ 'mabOffCanvas' ]; // Script(s) run when element is rendered on frontend or updated in builder

	// Return localized element label
	public function get_label() {
		return esc_html__( 'Off Canvas', 'max-addons' );
	}

	public function get_keywords() {
		return [ 'menu', 'mobile', 'offcanvas' ];
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		wp_enqueue_style( 'mab-offcanvas' );
		wp_enqueue_script( 'mab-offcanvas' );
	}

	// Set builder control groups
	public function set_control_groups() {
		$this->control_groups['offcanvasContent'] = [
			'title' => esc_html__( 'Content', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['trigger'] = [
			'title' => esc_html__( 'Trigger', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['labelSettings'] = [
			'title' => esc_html__( 'Label', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['settings'] = [
			'title' => esc_html__( 'Settings', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['offCanvasStyle'] = [
			'title' => esc_html__( 'Off Canvas Style', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['contentStyle'] = [
			'title' => esc_html__( 'Content Style', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['closeButton'] = [
			'title' => esc_html__( 'Close Button', 'max-addons' ),
			'tab'   => 'content',
		];
	}

	public function set_controls() {

		$this->set_offcanvas_controls();
		$this->set_trigger_controls();
		$this->set_label_settings_controls();
		$this->set_settings_controls();
		$this->set_offcanvas_style_controls();
		$this->set_content_style_controls();
		$this->set_close_button_controls();
	}

	// Set primary controls
	public function set_offcanvas_controls() {

		$menus = [];

		if ( bricks_is_builder() ) {
			$menus = $this->get_menus_list();
		}

		$this->controls['contentList'] = [
			'tab'           => 'content',
			'group'         => 'offcanvasContent',
			'label'         => esc_html__( 'Off Canvas Content', 'max-addons' ),
			'type'          => 'repeater',
			'placeholder'   => esc_html__( 'Content', 'max-addons' ),
			'titleProperty' => 'label',
			'fields'        => [
				'label'          => [
					'label'          => esc_html__( 'Label', 'max-addons' ),
					'type'           => 'text',
					'hasDynamicData' => 'text',
				],

				'contentType'   => [
					'label'     => esc_html__( 'Content Type', 'max-addons' ),
					'type'      => 'select',
					'options'   => [
						'content'  => esc_html__( 'Content', 'max-addons' ),
						'menu'     => esc_html__( 'Menu', 'max-addons' ),
						'template' => esc_html__( 'Template', 'max-addons' ),
					],
					'inline'    => true,
					'clearable' => false,
				],

				'content'  => [
					'label'    => esc_html__( 'Content', 'max-addons' ),
					'type'     => 'editor',
					'required' => [ 'contentType', '=', [ 'content' ] ],
				],

				'menu' => [
					'label'       => esc_html__( 'Menu', 'max-addons' ),
					'type'        => 'select',
					'options'     => $menus,
					'placeholder' => esc_html__( 'Select Menu', 'max-addons' ),
					'description' => '',
					'searchable'  => true,
					'required'    => [ 'contentType', '=', [ 'menu' ] ],
				],

				'template' => [
					'label'       => esc_html__( 'Template', 'max-addons' ),
					'type'        => 'select',
					'options'     => bricks_is_builder() ? Helper::get_templates_list( get_the_ID() ) : array(),
					'placeholder' => esc_html__( 'Select Template', 'max-addons' ),
					'description' => esc_html__( 'Please refresh the page to properly load the styles of the template.', 'max-addons' ),
					'searchable'  => true,
					'required'    => [ 'contentType', '=', [ 'template' ] ],
				],
			],
			'default'       => [
				[
					'label'       => esc_html__( 'Primary', 'max-addons' ),
					'contentType' => 'content',
					'content'     => 'Primary Content.',
				],
				[
					'label'       => esc_html__( 'Secondary', 'max-addons' ),
					'contentType' => 'content',
					'content'     => 'Secondary Content.',
				],
			],
		];

	}

	// Set trigger controls
	public function set_trigger_controls() {
		$this->controls['triggerOn'] = [
			'tab'       => 'content',
			'group'     => 'trigger',
			'label'     => esc_html__( 'Trigger On', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'button' => esc_html__( 'Button', 'max-addons' ),
				'icon'   => esc_html__( 'Icon', 'max-addons' ),
				'element-class' => __( 'Element Class', 'max-addons' ),
				'element-id'    => __( 'Element ID', 'max-addons' ),
			],
			'inline'    => true,
			'clearable' => false,
			'default'   => 'button',
		];

		$this->controls['elementClass'] = [
			'tab'            => 'content',
			'group'          => 'trigger',
			'label'          => esc_html__( 'Element CSS Class', 'max-addons' ),
			'type'           => 'text',
			'default'        => '',
			'inline'         => true,
			'required'       => [ 'triggerOn', '=', 'element-class' ],
		];

		$this->controls['elementId'] = [
			'tab'            => 'content',
			'group'          => 'trigger',
			'label'          => esc_html__( 'Element CSS ID', 'max-addons' ),
			'type'           => 'text',
			'default'        => '',
			'inline'         => true,
			'required'       => [ 'triggerOn', '=', 'element-id' ],
		];

		$this->controls['buttonText'] = [
			'tab'            => 'content',
			'group'          => 'trigger',
			'label'          => esc_html__( 'Button Text', 'max-addons' ),
			'type'           => 'text',
			'default'        => esc_html__( 'Click Here', 'max-addons' ),
			'hasDynamicData' => 'text',
			'inline'         => true,
			'required'       => [ 'triggerOn', '=', 'button' ],
		];

		$this->controls['buttonIcon'] = [
			'tab'      => 'content',
			'group'    => 'trigger',
			'label'    => esc_html__( 'Button Icon', 'max-addons' ),
			'type'     => 'icon',
			'default'  => [
				'library' => 'themify',
				'icon'    => 'ti-arrow-circle-right',
			],
			'css' 	=> [
				[
					'selector' => '.icon-svg',
				],
			],
			'required' => [ 'triggerOn', '=', 'button' ],
		];

		$this->controls['triggerIcon'] = [
			'tab'      => 'content',
			'group'    => 'trigger',
			'label'    => esc_html__( 'Icon', 'max-addons' ),
			'type'     => 'icon',
			'default'  => [
				'library' => 'themify',
				'icon'    => 'ti-menu',
			],
			'css' 	=> [
				[
					'selector' => '.icon-svg',
				],
			],
			'required' => [ 'triggerOn', '=', 'icon' ],
		];

		$this->controls['buttonStyle'] = [
			'tab'         => 'content',
			'group'       => 'trigger',
			'label'       => esc_html__( 'Style', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['styles'],
			'inline'      => true,
			'reset'       => true,
			'default'     => 'primary',
			'placeholder' => esc_html__( 'None', 'max-addons' ),
			'required'    => [ 'triggerOn', '=', 'button' ],
		];

		$this->controls['buttonCircle'] = [
			'tab'   => 'content',
			'group' => 'trigger',
			'label' => esc_html__( 'Circle', 'max-addons' ),
			'type'  => 'checkbox',
			'reset' => true,
			'required' => [ 'triggerOn', '=', 'button' ],
		];

		$this->controls['buttonOutline'] = [
			'tab'   => 'content',
			'group' => 'trigger',
			'label' => esc_html__( 'Outline', 'max-addons' ),
			'type'  => 'checkbox',
			'reset' => true,
			'required' => [ 'triggerOn', '=', 'button' ],
		];

		$this->controls['buttonBorder'] = [
			'tab'    => 'content',
			'group'  => 'trigger',
			'type'   => 'border',
			'label'  => esc_html__( 'Border', 'max-addons' ),
			'css'    => [
				[
					'property' => 'border',
					'selector' => '.mab-offcanvas-button',
				],
			],
			'inline' => true,
			'small'  => true,
			'required' => [ 'triggerOn', '=', 'button' ],
		];

		$this->controls['buttonBoxShadow'] = [
			'tab'    => 'content',
			'group'  => 'trigger',
			'label'  => esc_html__( 'Box Shadow', 'max-addons' ),
			'type'   => 'box-shadow',
			'css'    => [
				[
					'property' => 'box-shadow',
					'selector' => '.mab-offcanvas-button',
				],
			],
			'inline' => true,
			'small'  => true,
			'required' => [ 'triggerOn', '=', 'button' ],
		];

		$this->controls['buttonTypography'] = [
			'tab'    => 'content',
			'group'  => 'trigger',
			'type'   => 'typography',
			'label'  => esc_html__( 'Typography', 'max-addons' ),
			'css'    => [
				[
					'property' => 'font',
					'selector' => '.mab-offcanvas-button .mab-offcanvas-toggle-text',
				],
			],
			'inline' => true,
			'small'  => true,
			'required' => [ 'triggerOn', '=', 'button' ],
		];

		$this->controls['buttonBackground'] = [
			'tab'    => 'content',
			'group'  => 'trigger',
			'type'   => 'color',
			'label'  => esc_html__( 'Background', 'max-addons' ),
			'css'    => [
				[
					'property' => 'background-color',
					'selector' => '.mab-offcanvas-button',
				],
			],
			'inline' => true,
			'small'  => true,
			'required' => [ 'triggerOn', '=', 'button' ],
		];

		$this->controls['buttonPadding'] = [
			'tab'   => 'content',
			'group' => 'trigger',
			'label' => esc_html__( 'Padding', 'max-addons' ),
			'type'  => 'spacing',
			'css'   => [
				[
					'property' => 'padding',
					'selector' => '.mab-offcanvas-button',
				],
			],
			'required' => [ 'triggerOn', '=', 'button' ],
		];

		$this->controls['triggerIconTypography'] = [
			'tab'      => 'content',
			'group'    => 'trigger',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-offcanvas-icon i',
				],
			],
			'exclude'  => [
				'font-family',
				'font-weight',
				'font-style',
				'text-align',
				'text-decoration',
				'text-transform',
				'line-height',
				'letter-spacing',
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'triggerOn', '=', 'icon' ],
		];
	}

	// Set label settings controls
	public function set_label_settings_controls() {
		$this->controls['showLabel'] = [
			'tab'     => 'content',
			'group'   => 'labelSettings',
			'label'   => esc_html__( 'Show Label', 'max-addons' ),
			'type'    => 'checkbox',
			'info'    => __( 'Show labels in the offcanvas panel.', 'max-addons' ),
			'default' => false,
			'reset'   => true,
		];

		$this->controls['labelTypography'] = [
			'tab'    => 'content',
			'group'  => 'labelSettings',
			'type'   => 'typography',
			'label'  => esc_html__( 'Typography', 'max-addons' ),
			'css'    => [
				[
					'property' => 'font',
					'selector' => '.mab-offcanvas-section-label',
				],
			],
			'inline' => true,
			'small'  => true,
			'required' => [ 'showLabel', '!=', '' ],
		];

		$this->controls['labelBackground'] = [
			'tab'     => 'content',
			'group'   => 'labelSettings',
			'label'   => esc_html__( 'Background', 'max-addons' ),
			'type'    => 'color',
			'css'     => [
				[
					'property' => 'background-color',
					'selector' => '.mab-offcanvas-section-label',
				],
			],
			'inline'  => true,
			'small'   => true,
			'required' => [ 'showLabel', '!=', '' ],
		];

		$this->controls['labelMargin'] = [
			'tab'   => 'content',
			'group' => 'labelSettings',
			'label' => esc_html__( 'Margin', 'max-addons' ),
			'type'  => 'spacing',
			'css'   => [
				[
					'property' => 'margin',
					'selector' => '.mab-offcanvas-section-label',
				],
			],
			'required' => [ 'showLabel', '!=', '' ],
		];

		$this->controls['labelPadding'] = [
			'tab'   => 'content',
			'group' => 'labelSettings',
			'label' => esc_html__( 'Padding', 'max-addons' ),
			'type'  => 'spacing',
			'css'   => [
				[
					'property' => 'padding',
					'selector' => '.mab-offcanvas-section-label',
				],
			],
			'required' => [ 'showLabel', '!=', '' ],
		];
	}

	// Set settings controls
	public function set_settings_controls() {
		$this->controls['direction'] = [
			'tab'       => 'content',
			'group'     => 'settings',
			'label'     => esc_html__( 'Direction', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'left'   => __( 'Left', 'max-addons' ),
				'right'  => __( 'Right', 'max-addons' ),
				'top'    => __( 'Top', 'max-addons' ),
				'bottom' => __( 'Bottom', 'max-addons' ),
			],
			'inline'    => true,
			'clearable' => false,
			'default'   => 'left',
		];

		$this->controls['transition'] = [
			'tab'       => 'content',
			'group'     => 'settings',
			'label'     => esc_html__( 'Transition', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'slide' => __( 'Slide', 'max-addons' ),
				'push'  => __( 'Push', 'max-addons' ),
			],
			'inline'    => true,
			'clearable' => false,
			'default'   => 'slide',
		];

		$this->controls['escClose'] = [
			'tab'     => 'content',
			'group'   => 'settings',
			'label'   => esc_html__( 'Close on Esc Key', 'max-addons' ),
			'type'    => 'checkbox',
			'default' => true,
			'reset'   => true,
		];

		$this->controls['overlayClose'] = [
			'tab'   => 'content',
			'group' => 'settings',
			'label' => esc_html__( 'Close on Overlay Click', 'max-addons' ),
			'type'  => 'checkbox',
			'default' => true,
			'reset' => true,
		];
	}

	// Set toggle container style controls
	public function set_offcanvas_style_controls() {

		$this->controls['offCanvasSize'] = [
			'tab'   => 'content',
			'group' => 'offCanvasStyle',
			'label' => esc_html__( 'Size', 'max-addons' ),
			'type'  => 'number',
			'units' => true,
			'css'   => [
				[
					'property' => 'width',
					'selector' => '.mab-offcanvas-content-left',
				],
				[
					'property' => 'width',
					'selector' => '.mab-offcanvas-content-right',
				],
				[
					'property' => 'height',
					'selector' => '.mab-offcanvas-content-top',
				],
				[
					'property' => 'height',
					'selector' => '.mab-offcanvas-content-bottom',
				],
			],
		];

		$this->controls['offCanvasPadding'] = [
			'tab'   => 'content',
			'group' => 'offCanvasStyle',
			'label' => esc_html__( 'Padding', 'max-addons' ),
			'type'  => 'spacing',
			'css'   => [
				[
					'property' => 'padding',
					'selector' => '.mab-offcanvas-content',
				],
			],
		];

		$this->controls['offCanvasBackground'] = [
			'tab'     => 'content',
			'group'   => 'offCanvasStyle',
			'label'   => esc_html__( 'Background', 'max-addons' ),
			'type'    => 'background',
			'css'     => [
				[
					'property' => 'background',
					'selector' => '.mab-offcanvas-content',
				],
			],
			'inline'  => true,
			'small'   => true,
		];
	}

	// Set content style controls
	public function set_content_style_controls() {

		$this->controls['contentPadding'] = [
			'tab'   => 'content',
			'group' => 'contentStyle',
			'label' => esc_html__( 'Padding', 'max-addons' ),
			'type'  => 'spacing',
			'css'   => [
				[
					'property' => 'padding',
					'selector' => '.mab-offcanvas-content-section',
				],
			],
		];

		$this->controls['contentAlign'] = [
			'tab'     => 'content',
			'group'   => 'contentStyle',
			'label'   => esc_html__( 'Alignment', 'max-addons' ),
			'type'    => 'text-align',
			'default' => '',
			'css'     => [
				[
					'property' => 'text-align',
					'selector' => '.mab-offcanvas-content-section',
				],
			],
			'inline'  => true,
		];

		$this->controls['contentTypography'] = [
			'tab'    => 'content',
			'group'  => 'contentStyle',
			'label'  => esc_html__( 'Typography', 'max-addons' ),
			'type'   => 'typography',
			'css'    => [
				[
					'property' => 'font',
					'selector' => '.mab-offcanvas-content-section',
				],
			],
			'exclude'  => [
				'text-align',
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['contentBackground'] = [
			'tab'    => 'content',
			'group'  => 'contentStyle',
			'label'  => esc_html__( 'Background', 'max-addons' ),
			'type'   => 'color',
			'css'    => [
				[
					'property' => 'background-color',
					'selector' => '.mab-offcanvas-content-section',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['contentBorder'] = [
			'tab'    => 'content',
			'group'  => 'contentStyle',
			'label'  => esc_html__( 'Border', 'max-addons' ),
			'type'   => 'border',
			'css'    => [
				[
					'property' => 'border',
					'selector' => '.mab-offcanvas-content-section',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['contentBoxShadow'] = [
			'tab'    => 'content',
			'group'  => 'contentStyle',
			'label'  => esc_html__( 'Box Shadow', 'max-addons' ),
			'type'   => 'box-shadow',
			'css'    => [
				[
					'property' => 'box-shadow',
					'selector' => '.mab-offcanvas-content-section',
				],
			],
			'inline' => true,
			'small'  => true,
		];
	}

	// Set close button controls
	public function set_close_button_controls() {
		$this->controls['showCloseButton'] = [
			'tab'     => 'content',
			'group'   => 'closeButton',
			'label'   => esc_html__( 'Close Button', 'max-addons' ),
			'type'    => 'checkbox',
			'default' => true,
			'reset'   => true,
		];

		$this->controls['closeButtonIcon'] = [
			'tab'      => 'content',
			'group'    => 'closeButton',
			'label'    => esc_html__( 'Button Icon', 'max-addons' ),
			'type'     => 'icon',
			'default'  => [
				'library' => 'themify',
				'icon'    => 'ti-close',
			],
			'css' 	=> [
				[
					'selector' => '.icon-svg',
				],
			],
			'required' => [ 'showCloseButton', '!=', '' ],
		];

		$this->controls['closeIconTypography'] = [
			'tab'      => 'content',
			'group'    => 'closeButton',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => 'i',
				],
			],
			'exclude'  => [
				'font-family',
				'font-weight',
				'font-style',
				'text-align',
				'text-decoration',
				'text-transform',
				'line-height',
				'letter-spacing',
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'showCloseButton', '!=', '' ],
		];
	}

	/**
	 * Get available menus list
	 *
	 * @since 1.0.0
	 * @return array Array of menu
	 * @access public
	 */
	public function get_menus_list() {
		$menus = wp_get_nav_menus();

		$options = array();

		foreach ( $menus as $menu ) {
			$options[ $menu->slug ] = $menu->name;
		}

		return $options;
	}

	/**
	 * Render Menu HTML.
	 *
	 * @since 1.0.0
	 * @param array $settings The settings array.
	 * @return string menu HTML
	 * @access public
	 */
	public function get_menu_html( $settings ) {

		$menus = $this->get_menus_list();

		if ( ! empty( $menus ) ) {
			$args = array(
				'echo'        => false,
				'menu'        => $settings['menu'],
				'menu_class'  => 'mab-offcanvas-menu',
				'fallback_cb' => '__return_empty_string',
				'container'   => '',
			);

			$menu_html = wp_nav_menu( $args );

			return $menu_html;
		}
	}

	/**
	 * Get toggle content.
	 *
	 * @since 1.0.0
	 */
	protected function get_toggle_content( $item ) {
		$settings     = $this->settings;
		$content_type = ( isset( $item['contentType'] ) ? $item['contentType'] : '' );
		$output       = '';

		switch ( $content_type ) {
			case 'content':
				$output = wp_kses_post( $item['content'] );
				break;

			case 'menu':
				$menu_content = $this->get_menu_html( $item );
				$output       = $menu_content;
				break;

			case 'template':
				$template_id = ! empty( $item['template'] ) ? intval( $item['template'] ) : false;

				if ( ! $template_id ) {
					$output = esc_html__( 'No template selected.', 'max-addons' );
				}

				if ( $template_id == $this->post_id || $template_id == get_the_ID() ) {
					$output = esc_html__( 'Not allowed: Infinite template loop.', 'max-addons' );
				}

				$output = do_shortcode( '[bricks_template id="' . $template_id . '" ]' );
				break;

			default:
				return;
		}

		if ( isset( $settings['showLabel'] ) && isset( $item['label'] ) ) {
			$output = '<h4 class="mab-offcanvas-section-label">' . $item['label'] . '</h4>' . $output;
		}

		return $output;
	}

	/**
	 * Render close button output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @access protected
	 */
	protected function render_close_button() {
		$settings = $this->settings;

		if ( ! isset( $settings['showCloseButton'] ) ) {
			return;
		}

		$this->set_attribute( 'close-button', 'class', [
			'mab-offcanvas-close',
			'mab-offcanvas-close-' . $this->id,
			'mab-icon',
		] );

		$this->set_attribute( 'close-button', 'role', 'button' );

		/* if ( isset( $settings['closeButtonIcon']['icon'] ) ) {
			$this->set_attribute( 'icon', 'class', $settings['closeButtonIcon']['icon'] );
		} */

		if ( isset( $settings['closeButtonIcon'] ) ) {
			?>
			<div class="mab-offcanvas-header">
				<div <?php echo wp_kses_post( $this->render_attributes( 'close-button' ) ); ?>>
					<?php
						echo self::render_icon( $settings['closeButtonIcon'] );
					?>
				</div>
			</div>
			<?php
		}
	}

	/**
	 * Render trigger output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @access protected
	 */
	protected function render_trigger() {
		$settings = $this->settings;

		if ( isset( $settings['triggerOn'] ) ) {
			if ( 'button' == $settings['triggerOn'] ) {

				$button_classes[] = 'bricks-button mab-offcanvas-toggle mab-offcanvas-button';
		
				if ( isset( $settings['size'] ) ) {
					$button_classes[] = $settings['size'];
				}
		
				if ( isset( $settings['buttonStyle'] ) ) {
					// Outline
					if ( isset( $settings['buttonOutline'] ) ) {
						$button_classes[] = 'outline';
						$button_classes[] = 'bricks-color-' . $settings['buttonStyle'];
					} else {
						// Fill (default)
						$button_classes[] = 'bricks-background-' . $settings['buttonStyle'];
					}
				}
		
				// Button circle
				if ( isset( $settings['buttonCircle'] ) ) {
					$button_classes[] = 'circle';
				}

				$this->set_attribute( 'toggle-button', 'class', $button_classes );

				$icon = ! empty( $settings['buttonIcon'] ) ? self::render_icon( $settings['buttonIcon'] ) : false;
				?>
				<div <?php echo wp_kses_post( $this->render_attributes( 'toggle-button' ) ); ?>>
					<span class="mab-offcanvas-toggle-text">
						<?php echo wp_kses_post( $settings['buttonText'] ); ?>
					</span>
					<?php
					if ( $icon ) {
						echo $icon;
					}
					?>
				</div>
				<?php
			} else if ( 'icon' == $settings['triggerOn'] ) {
				if ( isset( $settings['triggerIcon']['icon'] ) ) {
					$this->set_attribute( 'toggle-icon', 'class', [ 'mab-offcanvas-toggle', 'mab-offcanvas-icon' ] );
					?>
					<div <?php echo wp_kses_post( $this->render_attributes( 'toggle-icon' ) ); ?>>
						<?php echo self::render_icon( $settings['triggerIcon'] ); ?>
					</div>
					<?php
				}
			}
		}
	}

	public function render() {
		$settings = $this->settings;

		$settings_attr = array(
			'toggle_source'   => isset( $settings['triggerOn'] ) ? esc_attr( $settings['triggerOn'] ) : '',
			'transition'      => isset( $settings['transition'] ) ? esc_attr( $settings['transition'] ) : 'slide',
			'direction'       => isset( $settings['direction'] ) ? esc_attr( $settings['direction'] ) : '',
			'esc_close'       => isset( $settings['escClose'] ) ? esc_attr( $settings['escClose'] ) : '',
			'overlay_close'   => isset( $settings['overlayClose'] ) ? esc_attr( $settings['overlayClose'] ) : '',
			'offcanvas_width' => isset( $settings['offCanvasSize'] ) ? esc_attr( $settings['offCanvasSize'] ) : '',
		);

		if ( isset( $settings['elementClass'] ) && ( 'element-class' === $settings['triggerOn'] || 'element-id' === $settings['triggerOn'] ) ) {
			echo $this->render_element_placeholder(
				[
					'title' => esc_html__( 'You have selected to open offcanvas using another element.', 'max-addons' ),
				]
			);
		}

		if ( isset( $settings['triggerOn'] ) ) {
			if ( 'element-class' === $settings['triggerOn'] && isset( $settings['elementClass'] ) ) {
				$settings_attr['toggle_class'] = $settings['elementClass'];
			} elseif ( 'element-id' === $settings['triggerOn'] && isset( $settings['elementId'] ) ) {
				$settings_attr['toggle_id'] = $settings['elementId'];
			}
		}

		$primary = ( isset( $settings['contentList'][0] ) ? $settings['contentList'][0] : '' );
		$secondary = ( isset( $settings['contentList'][1] ) ? $settings['contentList'][1] : '' );

		if ( isset( $settings['togglePosition'] ) ) {
			$wrapper_classes[] = 'mab-toggle-switch-' . $settings['togglePosition'];
		}

		$this->set_attribute( '_root', 'data-settings', htmlspecialchars( wp_json_encode( $settings_attr ) ) );

		$this->set_attribute( 'content', 'class', [
			'mab-offcanvas-content',
			'mab-offcanvas-content-' . $this->id,
			'mab-offcanvas-' . $settings_attr['transition']
		] );

		if ( isset( $settings['direction'] ) ) {
			$this->set_attribute( 'content', 'class', 'mab-offcanvas-content-' . $settings['direction'] );
		}
		?>
		<div <?php echo wp_kses_post( $this->render_attributes( '_root' ) ); ?>>
			<?php $this->render_trigger(); ?>

			<div <?php echo wp_kses_post( $this->render_attributes( 'content' ) ); ?> style="visibility: hidden;">
				<?php $this->render_close_button(); ?>

				<?php foreach ( $settings['contentList'] as $i => $item ) : ?>
					<div class="mab-offcanvas-content-section">
						<?php echo $this->get_toggle_content( $item ); //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
					</div>
				<?php endforeach; ?>
			</div>

			<div class="mab-offcanvas-overlay"></div>
		</div>
		<?php
	}
}
