<?php
namespace MaxAddonsPro\Elements;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Modal_Popup_Element extends \Bricks\Element {
	// Element properties
	public $category     = 'max-addons-elements'; // Use predefined element category 'general'
	public $name         = 'max-modal-popup'; // Make sure to prefix your elements
	public $icon         = 'ti-layers max-element'; // Themify icon font class
	public $css_selector = ''; // Default CSS selector
	public $scripts      = []; // Script(s) run when element is rendered on frontend or updated in builder
	public $nestable = true;

	public function get_label() {
		return esc_html__( 'Modal Popup', 'max-addons' );
	}

	public function get_keywords() {
		return [ 'modal', 'popup', 'lightbox', 'trigger' ];
	}

	public function enqueue_scripts() {
		wp_enqueue_style( 'mab-modal-popup' );
		wp_enqueue_script( 'mab-modal-popup' );
	}

	// Set builder control groups
	public function set_control_groups() {
		$this->control_groups['triggerSettings'] = [
			'title' => esc_html__( 'Triggers', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['triggerConditions'] = [
			'title' => esc_html__( 'Conditions', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['closeButtonSettings'] = [
			'title' => esc_html__( 'Close Button', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['advancedSettings'] = [
			'title' => esc_html__( 'Advanced', 'max-addons' ),
			'tab'   => 'content',
		];
	}

	public function set_controls() {
		$this->controls['_margin']['default'] = [
			'top'    => 20,
			'bottom' => 20,
		];

		$this->controls['_padding']['default'] = [
			'top'    => 30,
			'right'  => 30,
			'bottom' => 30,
			'left'   => 30,
		];

		$this->controls['_alignSelf']['default'] = 'center';

		$this->controls['_background']['default'] = [
			'color' => [
				'hex' => '#ffffff',
			],
		];

		$this->controls['onPageLoad'] = [
			'tab'            => 'content',
			'group'          => 'triggerSettings',
			'label'          => esc_html__( 'On Page Load', 'max-addons' ),
			'type'           => 'checkbox',
			'reset'          => true,
		];

		$this->controls['onPageLoadDelay'] = [
			'tab'            => 'content',
			'group'          => 'triggerSettings',
			'label'          => esc_html__( 'Delay (seconds)', 'max-addons' ),
			'type'           => 'number',
			'required'       => [ 'onPageLoad', '!=', false ],
		];

		$this->controls['_onPageLoadSeparator'] = [
			'tab'   => 'content',
			'group' => 'triggerSettings',
			'type'  => 'separator',
			'label' => '',
		];

		$this->controls['onScroll'] = [
			'tab'            => 'content',
			'group'          => 'triggerSettings',
			'label'          => esc_html__( 'On Scroll', 'max-addons' ),
			'type'           => 'checkbox',
			'reset'          => true,
		];

		$this->controls['onScrollDirection'] = [
			'tab'            => 'content',
			'group'          => 'triggerSettings',
			'label'          => esc_html__( 'Scroll Direction', 'max-addons' ),
			'type'           => 'select',
			'options'        => [
				'up'   => esc_html__( 'Up', 'max-addons' ),
				'down' => esc_html__( 'Down', 'max-addons' ),
			],
			'clearable' => false,
			'inline'    => true,
			'required'  => [ 'onScroll', '!=', false ],
		];

		$this->controls['onScrollOffset'] = [
			'tab'            => 'content',
			'group'          => 'triggerSettings',
			'label'          => esc_html__( 'Scroll Offset (%)', 'max-addons' ),
			'type'           => 'number',
			'required'       => [ 'onScroll', '!=', false ],
		];

		$this->controls['_onScrollSeparator'] = [
			'tab'   => 'content',
			'group' => 'triggerSettings',
			'type'  => 'separator',
			'label' => '',
		];

		$this->controls['onScrollTo'] = [
			'tab'            => 'content',
			'group'          => 'triggerSettings',
			'label'          => esc_html__( 'On Scroll To Element', 'max-addons' ),
			'type'           => 'checkbox',
			'reset'          => true,
		];

		$this->controls['onScrollToSelector'] = [
			'tab'            => 'content',
			'group'          => 'triggerSettings',
			'label'          => esc_html__( 'Element Selector', 'max-addons' ),
			'type'           => 'text',
			'placeholder'    => '.my-element-class or #my-element-id',
			'required'       => [ 'onScrollTo', '!=', false ],
		];

		$this->controls['_onScrollToSeparator'] = [
			'tab'   => 'content',
			'group' => 'triggerSettings',
			'type'  => 'separator',
			'label' => '',
		];

		$this->controls['onClick'] = [
			'tab'            => 'content',
			'group'          => 'triggerSettings',
			'label'          => esc_html__( 'On Click', 'max-addons' ),
			'type'           => 'checkbox',
			'reset'          => true,
		];

		$this->controls['onClickSelector'] = [
			'tab'            => 'content',
			'group'          => 'triggerSettings',
			'label'          => esc_html__( 'Element Selector', 'max-addons' ),
			'type'           => 'text',
			'placeholder'    => '.my-element-class or #my-element-id',
			'required'       => [ 'onClick', '!=', false ],
		];

		$this->controls['_onClickSeparator'] = [
			'tab'   => 'content',
			'group' => 'triggerSettings',
			'type'  => 'separator',
			'label' => '',
		];

		$this->controls['onURLParameter'] = [
			'tab'            => 'content',
			'group'          => 'triggerSettings',
			'label'          => esc_html__( 'On URL Parameter', 'max-addons' ),
			'type'           => 'checkbox',
			'reset'          => true,
		];

		$this->controls['urlParameterName'] = [
			'tab'            => 'content',
			'group'          => 'triggerSettings',
			'label'          => esc_html__( 'Parameter Name', 'max-addons' ),
			'type'           => 'text',
			'required'       => [ 'onURLParameter', '!=', false ],
		];

		$this->controls['urlParameterValue'] = [
			'tab'            => 'content',
			'group'          => 'triggerSettings',
			'label'          => esc_html__( 'Parameter Value (optional)', 'max-addons' ),
			'type'           => 'text',
			'required'       => [ 'onURLParameter', '!=', false ],
		];

		$this->controls['_onURLParamSeparator'] = [
			'tab'   => 'content',
			'group' => 'triggerSettings',
			'type'  => 'separator',
			'label' => '',
		];

		$this->controls['onUserInactivity'] = [
			'tab'            => 'content',
			'group'          => 'triggerSettings',
			'label'          => esc_html__( 'On User Inactivity', 'max-addons' ),
			'type'           => 'checkbox',
			'reset'          => true,
		];

		$this->controls['inactiveDuration'] = [
			'tab'            => 'content',
			'group'          => 'triggerSettings',
			'label'          => esc_html__( 'Inactive Duration (seconds)', 'max-addons' ),
			'type'           => 'number',
			'placeholder'    => 30,
			'required'       => [ 'onUserInactivity', '!=', false ],
		];

		$this->controls['_onUserInactivitySeparator'] = [
			'tab'   => 'content',
			'group' => 'triggerSettings',
			'type'  => 'separator',
			'label' => '',
		];

		$this->controls['onExitIntent'] = [
			'tab'            => 'content',
			'group'          => 'triggerSettings',
			'label'          => esc_html__( 'On Page Exit Intent', 'max-addons' ),
			'type'           => 'checkbox',
			'reset'          => true,
		];

		$this->controls['_triggerSeparator'] = [
			'tab'   => 'content',
			'group' => 'triggerSettings',
			'type'  => 'separator',
			'label' => esc_html__( 'Re-trigger', 'max-addons' ),
		];

		$this->controls['reTriggerAfter'] = [
			'tab'            => 'content',
			'group'          => 'triggerSettings',
			'label'          => esc_html__( 'Re-trigger after these days', 'max-addons' ),
			'description'    => esc_html__( 'Once closed, Modal Popup will only show after the given day(s). Not applicable for "On Click" trigger.', 'max-addons' ),
			'type'           => 'number',
		];

		// Conditions.
		$this->controls['triggerPageViews'] = [
			'tab'            => 'content',
			'group'          => 'triggerConditions',
			'label'          => esc_html__( 'Trigger after X page views', 'max-addons' ),
			'type'           => 'checkbox',
			'reset'          => true,
		];

		$this->controls['pageViews'] = [
			'tab'            => 'content',
			'group'          => 'triggerConditions',
			'label'          => esc_html__( 'Page views', 'max-addons' ),
			'type'           => 'number',
			'required'       => [ 'triggerPageViews', '!=', false ],
		];

		$this->controls['_pageViewsSeparator'] = [
			'tab'   => 'content',
			'group' => 'triggerConditions',
			'type'  => 'separator',
			'label' => '',
		];

		$this->controls['triggerTimes'] = [
			'tab'            => 'content',
			'group'          => 'triggerConditions',
			'label'          => esc_html__( 'Trigger up to X times', 'max-addons' ),
			'type'           => 'checkbox',
			'reset'          => true,
		];

		$this->controls['numberOfTimes'] = [
			'tab'            => 'content',
			'group'          => 'triggerConditions',
			'label'          => esc_html__( 'Number of Times', 'max-addons' ),
			'type'           => 'number',
			'required'       => [ 'triggerTimes', '!=', false ],
		];

		$this->controls['resetTimes'] = [
			'tab'            => 'content',
			'group'          => 'triggerConditions',
			'label'          => esc_html__( 'Reset when re-trigger', 'max-addons' ),
			'description'    => esc_html__( 'Use this option in conjunction with the "Re-Trigger" setting. This will reset the number of times the Modal Popup displayed and it will display again after the days you provided for re-trigger. Or keep it disabled to never trigger again.', 'max-addons' ),
			'type'           => 'checkbox',
			'reset'          => true,
		];

		// Close button.
		$this->controls['hideCloseButton'] = [
			'tab'            => 'content',
			'group'          => 'closeButtonSettings',
			'label'          => esc_html__( 'Hide Close Button', 'max-addons' ),
			'type'           => 'checkbox',
		];

		$this->controls['closeButtonIcon'] = [
			'tab'      => 'content',
			'group'    => 'closeButtonSettings',
			'label'    => esc_html__( 'Icon', 'max-addons' ),
			'type'     => 'icon',
			'default'  => [
				'library' => 'themify',
				'icon'    => 'ti-close',
			],
			'required'  => [ 'hideCloseButton', '!=', true ],
		];

		$this->controls['closeButtonLabel'] = [
			'tab'   => 'content',
			'group' => 'closeButtonSettings',
			'label' => esc_html__( 'ARIA Label', 'max-addons' ),
			'type'  => 'text',
			'inline' => true,
			'required'  => [ 'hideCloseButton', '!=', true ],
		];

		$this->controls['_closeButtonLabelSeparator'] = [
			'tab'   => 'content',
			'group' => 'closeButtonSettings',
			'type'  => 'separator',
			'label' => '',
			'required'  => [ 'hideCloseButton', '!=', true ],
		];

		$this->controls['closeButtonAlign'] = [
			'tab'   => 'content',
			'group' => 'closeButtonSettings',
			'label' => esc_html__( 'Align', 'max-addons' ),
			'type'  => 'select',
			'options' => [
				'left'  => esc_html__( 'Left', 'max-addons' ),
				'right' => esc_html__( 'Right', 'max-addons' ),
			],
			'inline' => true,
			'required'  => [ 'hideCloseButton', '!=', true ],
		];

		$this->controls['closeButtonVPos'] = [
			'tab'   => 'content',
			'group' => 'closeButtonSettings',
			'label' => esc_html__( 'Vertical Position', 'max-addons' ),
			'type'  => 'number',
			'units' => true,
			'css'   => [
				[
					'property' => 'top',
					'selector' => '.max-modal-popup-close',
				],
			],
			'required'  => [ 'hideCloseButton', '!=', true ],
		];

		$this->controls['closeButtonHPos'] = [
			'tab'   => 'content',
			'group' => 'closeButtonSettings',
			'label' => esc_html__( 'Horizontal Position', 'max-addons' ),
			'type'  => 'number',
			'units' => true,
			'css'   => [
				[
					'property' => 'right',
					'selector' => '.max-modal-popup-close.is-align-right',
				],
				[
					'property' => 'left',
					'selector' => '.max-modal-popup-close.is-align-left',
				],
			],
			'required'  => [ 'hideCloseButton', '!=', true ],
		];

		$this->controls['closeButtonSize'] = [
			'tab'   => 'content',
			'group' => 'closeButtonSettings',
			'label' => esc_html__( 'Size', 'max-addons' ),
			'type'  => 'number',
			'units' => true,
			'css'   => [
				[
					'property' => 'width',
					'selector' => '.max-modal-popup-close svg',
				],
				[
					'property' => 'height',
					'selector' => '.max-modal-popup-close svg',
				],
			],
			'required'  => [ 'hideCloseButton', '!=', true ],
		];

		$this->controls['closeButtonSize'] = [
			'tab'   => 'content',
			'group' => 'closeButtonSettings',
			'label' => esc_html__( 'Inner Spacing', 'max-addons' ),
			'type'  => 'number',
			'units' => true,
			'css'   => [
				[
					'property' => 'padding',
					'selector' => '.max-modal-popup-close',
				],
			],
			'required'  => [ 'hideCloseButton', '!=', true ],
		];

		$this->controls['closeButtonColor'] = [
			'tab'    => 'content',
			'group'  => 'closeButtonSettings',
			'type'   => 'color',
			'label'  => esc_html__( 'Color', 'max-addons' ),
			'css'    => [
				[
					'property' => 'fill',
					'selector' => '.max-modal-popup-close svg',
				],
				[
					'property' => 'stroke',
					'selector' => '.max-modal-popup-close svg',
				],
			],
			'inline' => true,
			'small'  => true,
			'required'  => [ 'hideCloseButton', '!=', true ],
		];

		$this->controls['closeButtonBgColor'] = [
			'tab'    => 'content',
			'group'  => 'closeButtonSettings',
			'type'   => 'color',
			'label'  => esc_html__( 'Background Color', 'max-addons' ),
			'css'    => [
				[
					'property' => 'background-color',
					'selector' => '.max-modal-popup-close',
				],
			],
			'inline' => true,
			'small'  => true,
			'required'  => [ 'hideCloseButton', '!=', true ],
		];

		$this->controls['showCloseButtonAfter'] = [
			'tab'      => 'content',
			'group'    => 'advancedSettings',
			'label'    => esc_html__( 'Show Close Button after (seconds)', 'max-addons' ),
			'type'     => 'number',
			'required' => [ 'hideCloseButton', '!=', true ],
		];

		$this->controls['autoCloseAfter'] = [
			'tab'      => 'content',
			'group'    => 'advancedSettings',
			'label'    => esc_html__( 'Automatically close after (seconds)', 'max-addons' ),
			'type'     => 'number',
		];

		$this->controls['preventCloseOutsideClick'] = [
			'tab'            => 'content',
			'group'          => 'advancedSettings',
			'label'          => esc_html__( 'Prevent closing on clicking outside', 'max-addons' ),
			'type'           => 'checkbox',
			'reset'          => true,
		];

		$this->controls['preventCloseEscKey'] = [
			'tab'            => 'content',
			'group'          => 'advancedSettings',
			'label'          => esc_html__( 'Prevent closing on ESC key press', 'max-addons' ),
			'type'           => 'checkbox',
			'reset'          => true,
		];

		$this->controls['disablePageScroll'] = [
			'tab'            => 'content',
			'group'          => 'advancedSettings',
			'label'          => esc_html__( 'Disable page scrolling', 'max-addons' ),
			'type'           => 'checkbox',
			'reset'          => true,
		];

		$this->controls['hideForLoggedIn'] = [
			'tab'            => 'content',
			'group'          => 'advancedSettings',
			'label'          => esc_html__( 'Hide for logged in users', 'max-addons' ),
			'type'           => 'checkbox',
			'reset'          => true,
		];

		$this->controls['hideForRoles'] = [
			'tab'            => 'content',
			'group'          => 'advancedSettings',
			'label'          => esc_html__( 'Hide for selected roles', 'max-addons' ),
			'placeholder'    => esc_html__( 'Select role(s)', 'max-addons' ),
			'type'           => 'select',
			'options'        => $this->get_user_roles(),
			'inline'         => false,
			'multiple' => true,
			'required' => [ 'hideForLoggedIn', '!=', false ],
		];

		// Add backdrop color option under the Style tab > Background group.
		$this->controls['backdropColor'] = [
			'tab'    => 'style',
			'group'  => '_background',
			'type'   => 'color',
			'label'  => esc_html__( 'Backdrop color', 'max-addons' ),
			'css'    => [
				[
					'property' => 'background-color',
					'selector' => '.max-modal-popup-backdrop',
				],
			],
			'inline' => true,
			'small'  => true,
		];
	}

	public static function inject_custom_controls( $controls ) {
		// Add exit animation option under the Style tab > Layout > Animation group.
		$exit_anim = $controls['_animation'];
		$exit_anim['label'] = esc_html__( 'Exit animation', 'max-addons' );

		$index = array_search( '_animation', array_keys( $controls ) );
		$index += 1;

		$controls = array_merge( array_slice( $controls, 0, $index ), array( '_exitAnimation' => $exit_anim ), array_slice( $controls, $index ) );

		return $controls;
	}

	public function get_nestable_item() {
		return [
			'name'     => 'block',
			'label'    => esc_html__( 'Content', 'max-addons' ),
			'settings' => [
				'_alignItems'     => 'center',
				'_justifyContent' => 'center',
			],
			'children' => [
				[
					'name'     => 'heading',
					'settings' => [
						'text' => esc_html__( 'Modal Popup', 'max-addons' ),
						'_alignSelf' => 'center',
					],
				],
				[
					'name'     => 'text-basic',
					'settings' => [
						'text' => esc_html__( 'Drag & Drop any elements here.', 'max-addons' ),
						'_margin'    => [
							'top'    => 10,
							'bottom' => 20
						],
						'_alignSelf' => 'center',
						'_typography' => [
							'text-align' => 'center'
						]
					],
				],
				[
					'name'     => 'button',
					'settings' => [
						'text'  => esc_html__( 'I am a button', 'max-addons' ),
						'size'  => 'lg',
						'style' => 'primary',
						'_alignSelf' => 'center',
					],
				],
			]
		];
	}

	public function get_nestable_children() {
		$children = [];

		for ( $i = 0; $i < 1; $i++ ) {
			$item = $this->get_nestable_item();

			// Replace {item_index} with $index
			$item       = json_encode( $item );
			$item       = str_replace( '{item_index}', $i + 1, $item );
			$item       = json_decode( $item, true );
			$children[] = $item;
		}

		return $children;
	}

	public function get_settings_attrs() {
		$settings = $this->settings;

		$settings_attrs = [
			'triggers' => []
		];

		// Triggers.
		if ( isset( $settings['onPageLoad'] ) ) {
			$trigger = [
				'type' => 'onPageLoad',
			];
			if ( isset( $settings['onPageLoadDelay'] ) ) {
				$trigger['delay'] = $settings['onPageLoadDelay'];
			}
			$settings_attrs['triggers'][] = $trigger;
		}
		if ( isset( $settings['onScroll'] ) ) {
			$trigger = [
				'type' => 'onScroll',
			];
			if ( isset( $settings['onScrollOffset'] ) ) {
				$trigger['offset'] = $settings['onScrollOffset'];
			}
			if ( isset( $settings['onScrollDirection'] ) ) {
				$trigger['direction'] = $settings['onScrollDirection'];
			}
			$settings_attrs['triggers'][] = $trigger;
		}
		if ( isset( $settings['onScrollTo'] ) ) {
			$trigger = [
				'type' => 'onScrollTo',
			];
			if ( isset( $settings['onScrollToSelector'] ) ) {
				$trigger['selector'] = $settings['onScrollToSelector'];
			}
			$settings_attrs['triggers'][] = $trigger;
		}
		if ( isset( $settings['onClick'] ) ) {
			$trigger = [
				'type' => 'onClick',
			];
			if ( isset( $settings['onClickSelector'] ) ) {
				$trigger['selector'] = $settings['onClickSelector'];
			}
			$settings_attrs['triggers'][] = $trigger;
		}
		if ( isset( $settings['onURLParameter'] ) ) {
			$trigger = [
				'type' => 'onURLParameter',
			];
			if ( isset( $settings['urlParameterName'] ) ) {
				$trigger['parameter'] = $settings['urlParameterName'];
			}
			if ( isset( $settings['urlParameterValue'] ) ) {
				$trigger['value'] = $settings['urlParameterValue'];
			}
			$settings_attrs['triggers'][] = $trigger;
		}
		if ( isset( $settings['onUserInactivity'] ) ) {
			$trigger = [
				'type' => 'onUserInactivity',
			];
			if ( isset( $settings['inactiveDuration'] ) ) {
				$trigger['duration'] = $settings['inactiveDuration'];
			}
			$settings_attrs['triggers'][] = $trigger;
		}
		if ( isset( $settings['onExitIntent'] ) ) {
			$trigger = [
				'type' => 'onExitIntent',
			];
			$settings_attrs['triggers'][] = $trigger;
		}

		// Re-trigger.
		if ( isset( $settings['reTriggerAfter'] ) ) {
			$settings_attrs['reTriggerAfter'] = $settings['reTriggerAfter'];
		}

		// Conditions.
		$conditions = [];
		if ( isset( $settings['triggerPageViews'] ) ) {
			if ( isset( $settings['pageViews'] ) && ! empty( $settings['pageViews'] ) ) {
				$conditions['pageViews'] = $settings['pageViews'];
			}
		}
		if ( isset( $settings['triggerTimes'] ) ) {
			if ( isset( $settings['numberOfTimes'] ) && ! empty( $settings['numberOfTimes'] ) ) {
				$conditions['times'] = $settings['numberOfTimes'];
			}
			if ( isset( $settings['resetTimes'] ) ) {
				$conditions['resetTimes'] = $settings['resetTimes'];
			}
		}
		if ( ! empty( $conditions ) ) {
			$settings_attrs['conditions'] = $conditions;
		}

		// Animation.
		$animation = [];
		if ( isset( $settings['_animation'] ) ) {
			$animation['entrance'] = $settings['_animation'];
		}
		if ( isset( $settings['_exitAnimation'] ) ) {
			$animation['exit'] = $settings['_exitAnimation'];
		}
		if ( isset( $settings['_animationDuration'] ) ) {
			if ( 'custom' === $settings['_animationDuration'] && isset( $settings['_animationDurationCustom'] ) ) {
				$animation['duration'] = $settings['_animationDurationCustom'];
			} else {
				$animation['duration'] = $settings['_animationDuration'];
			}
		}
		if ( ! empty( $animation ) ) {
			$settings_attrs['animation'] = $animation;
		}

		// Advanced.
		$advancedRules = [];
		if ( isset( $settings['preventCloseOutsideClick'] ) ) {
			$advancedRules['preventCloseOutsideClick'] = $settings['preventCloseOutsideClick'];
		}
		if ( isset( $settings['preventCloseEscKey'] ) ) {
			$advancedRules['preventCloseEscKey'] = $settings['preventCloseEscKey'];
		}
		if ( isset( $settings['showCloseButtonAfter'] ) ) {
			$advancedRules['showCloseButtonAfter'] = $settings['showCloseButtonAfter'];
		}
		if ( isset( $settings['autoCloseAfter'] ) ) {
			$advancedRules['autoCloseAfter'] = $settings['autoCloseAfter'];
		}
		if ( isset( $settings['disablePageScroll'] ) ) {
			$advancedRules['disablePageScroll'] = $settings['disablePageScroll'];
		}
		if ( ! empty( $advancedRules ) ) {
			$settings_attrs['advancedRules'] = $advancedRules;
		}

		return $settings_attrs;
	}

	public function get_user_roles() {
		global $wp_roles;

		return $wp_roles->get_names();
	}

	public function is_hidden_for_current_role() {
		$settings  = $this->settings;
		$is_hidden = false;

		if ( isset( $settings['hideForLoggedIn'] ) && isset( $settings['hideForRoles'] ) ) {
			$hide_for_roles = $settings['hideForRoles'];
			$current_user   = wp_get_current_user();
			$user_roles     = $current_user->roles;

			if ( is_multisite() && is_super_admin() ) {
				$user_roles[] = 'super_admin';
			}

			$compare_roles = array_intersect( $user_roles, $hide_for_roles );

			if ( ! empty( $compare_roles ) ) {
				$is_hidden = true;
			}
		}

		return $is_hidden;
	}

	public function filter_classes( $attributes, $element ) {
		$classes = $attributes['class'];
		$new_classes = [];

		foreach ( $classes as $class ) {
			if ( 'brx-animated' !== $class ) {
				$new_classes[] = $class;
			}
		}

		$attributes['class'] = $new_classes;

		return $attributes;
	}

	public function render() {
		if ( $this->is_hidden_for_current_role() && $this->is_frontend ) {
			return;
		}

		$settings = $this->settings;

		$this->set_attribute( '_root', 'data-settings', htmlspecialchars( wp_json_encode( $this->get_settings_attrs() ) ) );
		$this->set_attribute( '_root', 'data-element-id', $this->id );

		if ( isset( $settings['backdropColor'] ) && ! empty( $settings['backdropColor'] ) ) {
			$this->set_attribute( 'backdrop', 'style', 'background-color: ' . $settings['backdropColor']['rgb'] );
		}

		$has_position_set = isset( $settings['_position'] ) && in_array( $settings['_position'], array( 'absolute', 'fixed' ) );
		$modal_wrapper_class = $has_position_set ? ' has-custom-position' : '';

		$close_btn_label = isset( $settings['closeButtonLabel'] ) ? esc_attr( $settings['closeButtonLabel'] ) : esc_attr__( 'Close', 'max-addons' );
		$close_btn_align = isset( $settings['closeButtonAlign'] ) ? $settings['closeButtonAlign'] : 'right';
		?>

		<div <?php echo $this->render_attributes( '_root' ); ?>>
			<?php if ( ! isset( $settings['hideCloseButton'] ) || ! $settings['hideCloseButton'] ) { ?>
			<div class="max-modal-popup-close is-align-<?php echo $close_btn_align; ?>" tabindex="0" aria-label="<?php echo $close_btn_label; ?>">
				<?php if ( ! isset( $settings['closeButtonIcon'] ) || empty( $settings['closeButtonIcon'] ) ) { ?>
					<i class="ti-close" aria-hidden="true"></i>
				<?php } else {
					echo self::render_icon( $settings['closeButtonIcon'] );
				}
				?>
			</div>
			<?php } ?>
			<?php
				// Render children elements (= individual items)
				echo \Bricks\Frontend::render_children( $this );
			?>
		</div>
		<?php if ( $this->is_frontend ) { ?>
			<div id="modal-popup-<?php echo $this->id; ?>" class="max-modal-popup<?php echo $modal_wrapper_class; ?>" data-modal-id="<?php echo $this->id; ?>" tabindex="-1">
				<div class="max-modal-popup-backdrop" <?php echo $this->render_attributes( 'backdrop' ); ?>></div>
				<div class="max-modal-popup-content"></div>
			</div>
			<script>
				(function(d, w) {
					if ( 'undefined' === typeof w.mabPopupCount ) {
						w.mabPopupCount = 0;
					}
					d.querySelectorAll('#modal-popup-<?php echo $this->id; ?>:not(.appended)').forEach( function(n) {
						var id = '<?php echo $this->id; ?>-' + mabPopupCount;
						var e = d.querySelectorAll('.brxe-max-modal-popup[data-element-id="<?php echo $this->id; ?>"]')[0];
						e.setAttribute( 'id', 'brxe-<?php echo $this->id; ?>' );
						e.setAttribute( 'data-element-id', id );
						n.setAttribute( 'id', 'modal-popup-' + id );
						n.setAttribute( 'data-modal-id', id );
						n.getElementsByClassName('max-modal-popup-content')[0].append(e);
						n.classList.add( 'appended' );
						d.body.appendChild( n );
						mabPopupCount++;
					} );
				})(document, window);
			</script>
		<?php } ?>
		<?php
	}
}