<?php
namespace MaxAddonsPro\Elements;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Instagram_Feed_Element extends \Bricks\Element {
	// Element properties
	public $category     = 'max-addons-elements'; // Use predefined element category 'general'
	public $name         = 'max-instagram-feed'; // Make sure to prefix your elements
	public $icon         = 'ion-logo-instagram max-element'; // Iconicons icon font class
	public $css_selector = ''; // Default CSS selector
	public $scripts      = [ 'bricksIsotope', 'bricksSwiper' ]; // Script(s) run when element is rendered on frontend or updated in builder
	public $tag          = 'ul';

	/**
	 * Instagram Access token.
	 *
	 * @since 1.2.0
	 * @var   string
	 */
	private $insta_access_token = null;

	/**
	 * Instagram API URL.
	 *
	 * @since 1.2.0
	 * @var   string
	 */
	private $insta_api_url = 'https://www.instagram.com/';

	/**
	 * Official Instagram API URL.
	 *
	 * @since 1.2.0
	 * @var   string
	 */
	private $insta_official_api_url = 'https://graph.instagram.com/';

	public function get_label() {
		return esc_html__( 'Instagram Feed', 'max-addons' );
	}

	public function get_keywords() {
		return [ 'social' ];
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		$settings = $this->settings;

		$lightbox_enabled = ( isset( $this->settings['link'] ) && $this->settings['link'] === 'lightbox' ) ? true : false;
		$layout           = isset( $settings['layout'] ) ? $settings['layout'] : 'grid';

		wp_enqueue_style( 'mab-instagram-feed');

		if ( 'masonry' === $layout ) {
			wp_enqueue_script( 'bricks-isotope' );
			wp_enqueue_style( 'bricks-isotope' );
		}

		if ( 'carousel' === $layout ) {
			wp_enqueue_script( 'bricks-swiper' );
			wp_enqueue_style( 'bricks-swiper' );
		}

		if ( $lightbox_enabled ) {
			wp_enqueue_script( 'bricks-photoswipe' );
			wp_enqueue_script( 'bricks-photoswipe-lightbox' );
			wp_enqueue_style( 'bricks-photoswipe' );
		}
	}

	// Set builder control groups
	public function set_control_groups() {
		$this->control_groups['settings'] = [
			'title' => esc_html__( 'Settings', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['thumbnail'] = [
			'title' => esc_html__( 'Thumbnails', 'max-addons' ),
			'tab'   => 'content',
		];
		
		$this->control_groups['caption'] = [
			'title' => esc_html__( 'Caption', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['overlay'] = [
			'title' => esc_html__( 'Overlay', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['arrows'] = [
			'title'    => esc_html__( 'Arrows', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'layout', '=', 'carousel' ],
		];

		$this->control_groups['dots'] = [
			'title'    => esc_html__( 'Dots', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'layout', '=', 'carousel' ],
		];
	}

	public function set_controls() {

		$this->set_settings_controls();
		$this->set_thumbnail_style_controls();
		$this->set_caption_style_controls();
		$this->set_overlay_style_controls();
		$this->set_navigation_controls();
	}

	// Set settings controls
	public function set_settings_controls() {
		$this->controls['globalAccessToken'] = [
			'tab'     => 'content',
			'group'   => 'settings',
			'label'   => esc_html__( 'Global Access Token', 'max-addons' ),
			'type'    => 'checkbox',
			'default' => true
		];

		if ( ! $this->get_insta_global_access_token() ) {
			$this->controls['accessTokenMissingInfo'] = [
				'tab'      => 'content',
				'group'    => 'settings',
				'content'  => sprintf(
					__( 'Your Instagram Access Token is missing, %1$sclick here%2$s to configure.', 'max-addons' ),
					'<a href="' . admin_url( 'admin.php?page=mab-settings&tab=integration' ) . '"><strong>',
					'</strong></a>'
				),
				'type'     => 'info',
				'required' => [ 'globalAccessToken', '!=', '' ],
			];
		}

		$this->controls['accessToken'] = [
			'tab'         => 'content',
			'group'       => 'settings',
			'label'       => esc_html__( 'Custom Access Token', 'max-addons' ),
			'description' => esc_html__( 'Overrides global Instagram access token', 'max-addons' ),
			'type'        => 'text',
			'inline'      => false,
			'default'     => '',
			'required'    => [ 'globalAccessToken', '=', '' ],
		];

		$this->controls['cacheTimeout'] = [
			'tab'         => 'content',
			'group'       => 'settings',
			'label'       => esc_html__( 'Cache Timeout', 'max-addons' ),
			'placeholder' => esc_html__( 'Hour', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'none'   => esc_html__( 'None', 'max-addons' ),
				'minute' => esc_html__( 'Minute', 'max-addons' ),
				'hour'   => esc_html__( 'Hour', 'max-addons' ),
				'day'    => esc_html__( 'Day', 'max-addons' ),
				'week'   => esc_html__( 'Week', 'max-addons' ),
			],
			'default'     => 'hour',
			'inline'      => true,
		];

		$this->controls['imagesCount'] = [
			'tab'         => 'content',
			'group'       => 'settings',
			'label'       => esc_html__( 'Images Count', 'max-addons' ),
			'type'        => 'number',
			'min'         => 1,
			'max'         => 50,
			'breakpoints' => false, // NOTE: Undocumented
			'inline'      => true,
			'small'       => true,
			'placeholder' => 5,
			'default' 	  => 5,
		];

		$this->controls['layout'] = [
			'tab'         => 'content',
			'group'       => 'settings',
			'label'       => esc_html__( 'Layout', 'max-addons' ),
			'placeholder' => esc_html__( 'Grid', 'max-addons' ),
			'type'    => 'select',
			'options'     => [
				'grid'     	=> esc_html__( 'Grid', 'max-addons' ),
				'masonry'  	=> esc_html__( 'Masonry', 'max-addons' ),
				'carousel'  => esc_html__( 'Carousel', 'max-addons' ),
			],
			'default'     => 'grid',
			'inline'      => true,
		];

		$this->controls['columns'] = [
			'tab'         => 'content',
			'group'       => 'settings',
			'label'       => esc_html__( 'Columns', 'max-addons' ),
			'type'        => 'number',
			'min'         => 1,
			'css'         => [
				[
					'property' => '--columns',
					'selector' => '',
				],
			],
			'rerender'    => true,
			'inline'      => true,
			'small'       => true,
			'placeholder' => 3,
			'default' 	  => 3,
		];

		$this->controls['gutter'] = [
			'tab'         => 'content',
			'group'       => 'settings',
			'label'       => esc_html__( 'Spacing', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'css'         => [
				[
					'property' => '--gutter',
					'selector' => '',
				],
			],
			'placeholder' => 0,
		];

		$carousel_controls = self::get_swiper_controls();

		$this->controls['adaptiveHeight']                = $carousel_controls['adaptiveHeight'];
		$this->controls['adaptiveHeight']['required']    = [ 'layout', '=', 'carousel' ];
		$this->controls['initialSlide']                  = $carousel_controls['initialSlide'];
		$this->controls['initialSlide']['required']      = [ 'layout', '=', 'carousel' ];
		$this->controls['slidesToShow']                  = $carousel_controls['slidesToShow'];
		$this->controls['slidesToShow']['placeholder']   = 3;
		$this->controls['slidesToShow']['required']      = [ 'layout', '=', 'carousel' ];
		$this->controls['slidesToScroll']                = $carousel_controls['slidesToScroll'];
		$this->controls['slidesToScroll']['placeholder'] = 1;
		$this->controls['slidesToScroll']['required']    = [ 'layout', '=', 'carousel' ];
		$this->controls['effect']                        = $carousel_controls['effect'];
		$this->controls['effect']['required']            = [ 'layout', '=', 'carousel' ];
		$this->controls['infinite']                      = $carousel_controls['infinite'];
		$this->controls['infinite']['required']          = [ 'layout', '=', 'carousel' ];
		$this->controls['centerMode']                    = $carousel_controls['centerMode'];
		$this->controls['centerMode']['required']        = [ 'layout', '=', 'carousel' ];

		$this->controls['autoplay'] = [
			'tab'      => 'content',
			'group'    => 'settings',
			'label'    => esc_html__( 'Autoplay', 'max-addons' ),
			'type'     => 'checkbox',
			'required' => [ 'layout', '=', 'carousel' ],
		];

		$this->controls['pauseOnHover'] = [
			'tab'      => 'content',
			'group'    => 'settings',
			'label'    => esc_html__( 'Pause on hover', 'max-addons' ),
			'type'     => 'checkbox',
			'required' => [
				[ 'layout', '=', 'carousel' ],
				[ 'autoplay', '!=', '' ]
			],
			'inline'   => true,
		];

		$this->controls['stopOnLastSlide'] = [
			'tab'      => 'content',
			'group'    => 'settings',
			'label'    => esc_html__( 'Stop on last slide', 'max-addons' ),
			'type'     => 'checkbox',
			'info'     => esc_html__( 'No effect with loop enabled', 'max-addons' ),
			'required' => [
				[ 'layout', '=', 'carousel' ],
				[ 'autoplay', '!=', '' ]
			],
			'inline'   => true,
		];

		$this->controls['autoplaySpeed'] = [
			'tab'         => 'content',
			'group'       => 'settings',
			'label'       => esc_html__( 'Autoplay speed in ms', 'max-addons' ),
			'type'        => 'number',
			'required'    => [
				[ 'layout', '=', 'carousel' ],
				[ 'autoplay', '!=', '' ]
			],
			'placeholder' => 3000,
		];

		$this->controls['speed']                  = $carousel_controls['speed'];
		$this->controls['speed']['required']      = [ 'layout', '=', 'carousel' ];
	}

	// Set thumbnail controls
	public function set_thumbnail_style_controls() {
		$this->controls['link'] = [
			'tab'         => 'content',
			'group'   	  => 'thumbnail',
			'label'       => esc_html__( 'Link to', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'lightbox' => esc_html__( 'Lightbox', 'max-addons' ),
				'post'     => esc_html__( 'Instagram Post', 'max-addons' ),
			],
			'inline'      => true,
			'placeholder' => esc_html__( 'None', 'max-addons' ),
		];

		$this->controls['imageBorder'] = [
			'tab'      => 'content',
			'group'    => 'thumbnail',
			'type'     => 'border',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.mab-image',
				],
			],
			'inline'   => true,
			'small'    => true,
		];

		$this->controls['imageBoxShadow'] = [
			'tab'   => 'content',
			'group' => 'thumbnail',
			'label' => esc_html__( 'Box shadow', 'max-addons' ),
			'type'  => 'box-shadow',
			'css'   => [
				[
					'property' => 'box-shadow',
					'selector' => '.mab-image',
				],
			],
		];

		$this->controls['thumbnailFilter'] = [
			'tab'     => 'content',
			'group'   => 'thumbnail',
			'label'   => esc_html__( 'Image Filter', 'max-addons' ),
			'type'    => 'select',
			'options' => $this->image_filters(),
			'default' => '',
			'inline'  => true,
		];

		$this->controls['thumbnailHoverFilter'] = [
			'tab'     => 'content',
			'group'   => 'thumbnail',
			'label'   => esc_html__( 'Image Hover Filter', 'max-addons' ),
			'type'    => 'select',
			'options' => $this->image_filters(),
			'default' => '',
			'inline'  => true,
		];

	}

	// Set caption controls
	public function set_caption_style_controls() {
		$this->controls['caption'] = [
			'tab'   => 'content',
			'group' => 'caption',
			'label' => esc_html__( 'Caption', 'max-addons' ),
			'type'  => 'checkbox',
			'default' => true
		];

		$this->controls['captionPosition'] = [
			'tab'     => 'content',
			'group'   => 'caption',
			'label'   => esc_html__( 'Caption Position', 'max-addons' ),
			'type'    => 'select',
			'options' => [
				'over_image'  => __( 'Over Image', 'max-addons' ),
				'below_image' => __( 'Below Image', 'max-addons' ),
			],
			'default' => 'over_image',
			'inline'  => true,
			'required' => [ 'caption', '=', true ],
		];

		$this->controls['captionTypography'] = [
			'tab'      => 'content',
			'group'    => 'caption',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.bricks-image-caption',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'caption', '=', true ],
		];

		$this->controls['captionJustifyContent'] = [
			'tab'     => 'content',
			'group'   => 'caption',
			'label'   => esc_html__( 'Vertical Align', 'max-addons' ),
			'type'    => 'justify-content',
			'css'     => [
				[
					'property' => 'justify-content',
					'selector' => '.mab-gallery-image-content',
				],
			],
			'default' => 'flex-end',
			'exclude' => [
				'space',
			],
			'inline'  	=> true,
			'required' 	=> [
				[ 'caption', '=', true ],
				[ 'captionPosition', '!=', 'below_image' ],
			],
		];

		$this->controls['captionAlignContent'] = [
			'tab'     => 'content',
			'group'   => 'caption',
			'label'   => esc_html__( 'Horizontal Align', 'max-addons' ),
			'type'    => 'align-items',
			'css'     => [
				[
					'property' => 'align-items',
					'selector' => '.mab-gallery-image-content',
				],
			],
			'default' => '',
			'exclude' => [
				'stretch',
			],
			'inline'  => true,
			'required' => [ 'caption', '=', true ],
		];

		$this->controls['captionMargin'] = [
			'tab'      => 'content',
			'group'    => 'caption',
			'label'    => esc_html__( 'Margin', 'max-addons' ),
			'type'     => 'spacing',
			'css'      => [
				[
					'property' => 'margin',
					'selector' => '.bricks-image-caption',
				],
			],
			'required'	=> [ 'caption', '=', true ],
		];

		$this->controls['captionPadding'] = [
			'tab'      => 'content',
			'group'    => 'caption',
			'label'    => esc_html__( 'Padding', 'max-addons' ),
			'type'     => 'spacing',
			'css'      => [
				[
					'property' => 'padding',
					'selector' => '.bricks-image-caption',
				],
			],
			'required'	=> [ 'caption', '=', true ],
		];

		$this->controls['captionBackground'] = [
			'tab'      => 'content',
			'group'    => 'caption',
			'label'    => esc_html__( 'Background', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'background',
					'selector' => '.bricks-image-caption',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'caption', '=', true ],
		];

		$this->controls['captionBorder'] = [
			'tab'      => 'content',
			'group'    => 'caption',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.bricks-image-caption',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'caption', '=', true ],
		];
	}

	// Set overlay controls
	public function set_overlay_style_controls() {
		$this->controls['overlayBackground'] = [
			'tab'      => 'content',
			'group'    => 'overlay',
			'label'    => esc_html__( 'Background', 'max-addons' ),
			'type'     => 'gradient',
			'css'      => [
				[
					'property' => 'background-image',
					'selector' => '.mab-media-overlay',
				],
			],
		];
	}

	// Set navigation controls
	public function set_navigation_controls() {
		if ( method_exists( '\Bricks\Setup', "get_carousel_controls" ) ) {
			$carousel_controls = \Bricks\Setup::get_carousel_controls();
		} else {
			$carousel_controls = self::get_swiper_controls();
		}

		// Arrows

		$this->controls['arrows']          = $carousel_controls['arrows'];
		$this->controls['arrowHeight']     = $carousel_controls['arrowHeight'];
		$this->controls['arrowWidth']      = $carousel_controls['arrowWidth'];
		$this->controls['arrowBackground'] = $carousel_controls['arrowBackground'];
		$this->controls['arrowBorder']     = $carousel_controls['arrowBorder'];
		$this->controls['arrowTypography'] = $carousel_controls['arrowTypography'];
		$this->controls['arrowTypography']['default'] = [
			'color' => [
				'hex' => \Bricks\Setup::get_default_color( 'body' ),
			],
		];

		$this->controls['prevArrowSeparator'] = $carousel_controls['prevArrowSeparator'];
		$this->controls['prevArrow']          = $carousel_controls['prevArrow'];
		$this->controls['prevArrowTop']       = $carousel_controls['prevArrowTop'];
		$this->controls['prevArrowRight']     = $carousel_controls['prevArrowRight'];
		$this->controls['prevArrowBottom']    = $carousel_controls['prevArrowBottom'];
		$this->controls['prevArrowLeft']      = $carousel_controls['prevArrowLeft'];

		$this->controls['nextArrowSeparator'] = $carousel_controls['nextArrowSeparator'];
		$this->controls['nextArrow']          = $carousel_controls['nextArrow'];
		$this->controls['nextArrowTop']       = $carousel_controls['nextArrowTop'];
		$this->controls['nextArrowRight']     = $carousel_controls['nextArrowRight'];
		$this->controls['nextArrowBottom']    = $carousel_controls['nextArrowBottom'];
		$this->controls['nextArrowLeft']      = $carousel_controls['nextArrowLeft'];

		// Dots

		$this->controls['dots']            = $carousel_controls['dots'];
		$this->controls['dotsDynamic']     = $carousel_controls['dotsDynamic'];
		$this->controls['dotsVertical']    = $carousel_controls['dotsVertical'];
		$this->controls['dotsHeight']      = $carousel_controls['dotsHeight'];
		$this->controls['dotsWidth']       = $carousel_controls['dotsWidth'];
		$this->controls['dotsTop']         = $carousel_controls['dotsTop'];
		$this->controls['dotsRight']       = $carousel_controls['dotsRight'];
		$this->controls['dotsBottom']      = $carousel_controls['dotsBottom'];
		$this->controls['dotsLeft']        = $carousel_controls['dotsLeft'];
		$this->controls['dotsBorder']      = $carousel_controls['dotsBorder'];
		$this->controls['dotsSpacing']     = $carousel_controls['dotsSpacing'];
		$this->controls['dotsColor']       = $carousel_controls['dotsColor'];
		$this->controls['dotsActiveColor'] = $carousel_controls['dotsActiveColor'];
	}

	/**
	 * Image filters.
	 *
	 * @access public
	 * @param boolean $inherit if inherit option required.
	 * @return array Filters.
	 */
	protected function image_filters( $inherit = false ) {

		$inherit_opt = array();

		if ( $inherit ) {
			$inherit_opt = array(
				'' => __( 'Inherit', 'max-addons' ),
			);
		}

		$mab_image_filters = array(
			'filter-1977'      => __( '1977', 'max-addons' ),
			'filter-aden'      => __( 'Aden', 'max-addons' ),
			'filter-amaro'     => __( 'Amaro', 'max-addons' ),
			'filter-ashby'     => __( 'Ashby', 'max-addons' ),
			'filter-brannan'   => __( 'Brannan', 'max-addons' ),
			'filter-brooklyn'  => __( 'Brooklyn', 'max-addons' ),
			'filter-charmes'   => __( 'Charmes', 'max-addons' ),
			'filter-clarendon' => __( 'Clarendon', 'max-addons' ),
			'filter-crema'     => __( 'Crema', 'max-addons' ),
			'filter-dogpatch'  => __( 'Dogpatch', 'max-addons' ),
			'filter-earlybird' => __( 'Earlybird', 'max-addons' ),
			'filter-gingham'   => __( 'Gingham', 'max-addons' ),
			'filter-ginza'     => __( 'Ginza', 'max-addons' ),
			'filter-hefe'      => __( 'Hefe', 'max-addons' ),
			'filter-helena'    => __( 'Helena', 'max-addons' ),
			'filter-hudson'    => __( 'Hudson', 'max-addons' ),
			'filter-inkwell'   => __( 'Inkwell', 'max-addons' ),
			'filter-juno'      => __( 'Juno', 'max-addons' ),
			'filter-kelvin'    => __( 'Kelvin', 'max-addons' ),
			'filter-lark'      => __( 'Lark', 'max-addons' ),
			'filter-lofi'      => __( 'Lofi', 'max-addons' ),
			'filter-ludwig'    => __( 'Ludwig', 'max-addons' ),
			'filter-maven'     => __( 'Maven', 'max-addons' ),
			'filter-mayfair'   => __( 'Mayfair', 'max-addons' ),
			'filter-moon'      => __( 'Moon', 'max-addons' ),
		);

		return array_merge( $inherit_opt, $mab_image_filters );
	}

	/**
	 * Get Instagram access token from Max ddons options.
	 *
	 * @since 1.2.0
	 * @return string
	 */
	public function get_insta_global_access_token() {
		return \MaxAddonsPro\Classes\Admin_Settings::get_option( 'mab_instagram_access_token' );
	}

	/**
	 * Get Instagram access token.
	 *
	 * @since 1.2.0
	 * @return string
	 */
	public function get_insta_access_token() {
		$settings = $this->settings;

		if ( ! $this->insta_access_token ) {
			$global_access_token = isset( $settings['globalAccessToken'] ) ? $settings['globalAccessToken'] : '';
			$custom_access_token = isset( $settings['accessToken'] ) ? $settings['accessToken'] : '';

			if ( ! isset( $settings['globalAccessToken'] ) ) {
				if ( '' !== trim( $custom_access_token ) ) {
					$this->insta_access_token = $custom_access_token;
				}
			} else {
				$this->insta_access_token = $this->get_insta_global_access_token();
			}
		}

		return $this->insta_access_token;
	}

	/**
	 * Retrieve a URL for own photos.
	 *
	 * @since  1.2.0
	 * @return string
	 */
	public function get_feed_endpoint() {
		return $this->insta_official_api_url . 'me/media/';
	}

	/**
	 * Retrieve a grab URL.
	 *
	 * @since  1.2.0
	 * @return string
	 */
	public function get_fetch_url() {
		$settings = $this->settings;

		$url = $this->get_feed_endpoint();
		$url = add_query_arg( [
			'fields'       => 'id,media_type,media_url,thumbnail_url,permalink,caption',
			'access_token' => $this->get_insta_access_token(),
		], $url );

		return $url;
	}

	/**
	 * Get thumbnail data from response data
	 *
	 * @param $post
	 * @since 1.2.0
	 *
	 * @return array
	 */
	public function get_insta_feed_thumbnail_data( $post ) {
		$thumbnail = array(
			'thumbnail' => false,
			'low'       => false,
			'standard'  => false,
			'high'      => false,
		);

		if ( ! empty( $post['images'] ) && is_array( $post['images'] ) ) {
			$data = $post['images'];

			$thumbnail['thumbnail'] = [
				'src'           => $data['thumbnail']['url'],
				'config_width'  => $data['thumbnail']['width'],
				'config_height' => $data['thumbnail']['height'],
			];

			$thumbnail['low'] = [
				'src'           => $data['low_resolution']['url'],
				'config_width'  => $data['low_resolution']['width'],
				'config_height' => $data['low_resolution']['height'],
			];

			$thumbnail['standard'] = [
				'src'           => $data['standard_resolution']['url'],
				'config_width'  => $data['standard_resolution']['width'],
				'config_height' => $data['standard_resolution']['height'],
			];

			$thumbnail['high'] = $thumbnail['standard'];
		}

		return $thumbnail;
	}

	/**
	 * Get data from response
	 *
	 * @param  $response
	 * @since  1.2.0
	 *
	 * @return array
	 */
	public function get_insta_feed_response_data( $response ) {
		$settings = $this->settings;

		if ( ! array_key_exists( 'data', $response ) ) { // Avoid PHP notices
			return;
		}

		$response_posts = $response['data'];

		if ( empty( $response_posts ) ) {
			return array();
		}

		$return_data  = array();
		$images_count = isset( $settings['imagesCount'] ) ? $settings['imagesCount'] : 5;
		$posts = array_slice( $response_posts, 0, $images_count, true );

		foreach ( $posts as $post ) {
			$_post              = array();

			$_post['id']        = $post['id'];
			$_post['link']      = $post['permalink'];
			$_post['caption']   = '';
			$_post['image']     = 'VIDEO' === $post['media_type'] ? $post['thumbnail_url'] : $post['media_url'];
			//$_post['comments']  = ! empty( $post['comments_count'] ) ? $post['comments_count'] : 0;
			//$_post['likes']     = ! empty( $post['likes_count'] ) ? $post['likes_count'] : 0;

			$_post['thumbnail'] = $this->get_insta_feed_thumbnail_data( $post );

			if ( ! empty( $post['caption'] ) ) {
				$insta_caption_length = isset( $settings['insta_caption_length'] ) ? $settings['insta_caption_length'] : 30;
				$_post['caption'] = wp_html_excerpt( $post['caption'], $insta_caption_length, '&hellip;' );
			}

			$return_data[] = $_post;
		}

		return $return_data;
	}

	/**
	 * Generate thumbnail resources.
	 *
	 * @since 1.2.0
	 * @param $post_data
	 *
	 * @return array
	 */
	public function get_insta_tags_thumbnail_data( $post ) {
		$post = $post['node'];

		$thumbnail = array(
			'thumbnail' => false,
			'low'       => false,
			'standard'  => false,
			'high'      => false,
		);

		if ( is_array( $post['thumbnail_resources'] ) && ! empty( $post['thumbnail_resources'] ) ) {
			foreach ( $post['thumbnail_resources'] as $key => $resources_data ) {

				if ( 150 === $resources_data['config_width'] ) {
					$thumbnail['thumbnail'] = $resources_data;
					continue;
				}

				if ( 320 === $resources_data['config_width'] ) {
					$thumbnail['low'] = $resources_data;
					continue;
				}

				if ( 640 === $resources_data['config_width'] ) {
					$thumbnail['standard'] = $resources_data;
					continue;
				}
			}
		}

		if ( ! empty( $post['display_url'] ) ) {
			$thumbnail['high'] = array(
				'src'           => $post['display_url'],
				'config_width'  => $post['dimensions']['width'],
				'config_height' => $post['dimensions']['height'],
			);
		}

		return $thumbnail;
	}

	/**
	 * Get Insta Thumbnail Image URL
	 *
	 * @since  1.2.0
	 * @return string   The url of the instagram post image
	 */
	protected function get_insta_image_size() {
		$settings = $this->settings;

		$size = isset( $settings['resolution'] ) ? $settings['resolution'] : 'low_resolution';

		switch ( $size ) {
			case 'thumbnail':
				return 'thumbnail';
			case 'low_resolution':
				return 'low';
			case 'standard_resolution':
				return 'standard';
			default:
				return 'low';
		}
	}

	/**
	 * Retrieve response from API
	 *
	 * @since  1.2.0
	 * @return array|WP_Error
	 */
	public function get_insta_remote( $url ) {
		$response       = wp_remote_get( $url, array(
			'timeout'   => 60,
			'sslverify' => false,
		) );

		$response_code  = wp_remote_retrieve_response_code( $response );
		$result         = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( 200 !== $response_code ) {
			$message = is_array( $result ) && isset( $result['error']['message'] ) ? $result['error']['message'] : __( 'No posts found', 'max-addons' );

			return new \WP_Error( $response_code, $message );
		}

		if ( ! is_array( $result ) ) {
			return new \WP_Error( 'error', __( 'Data Error', 'max-addons' ) );
		}

		return $result;
	}

	/**
	 * Get transient key.
	 *
	 * @since  1.2.0
	 * @return string
	 */
	public function get_transient_key() {
		$settings = $this->settings;

		$insta_caption_length = isset( $settings['insta_caption_length'] ) ? $settings['insta_caption_length'] : 30;
		$images_count = isset( $settings['imagesCount'] ) ? $settings['imagesCount'] : 5;

		return sprintf( 'mab_instagram_feed_posts_count_%s_caption_%s',
			$images_count,
			$insta_caption_length
		);
	}

	/**
	 * Render Title Icon.
	 *
	 * @since  1.2.0
	 */
	public function render_title_icon() {
		$settings = $this->get_settings_for_display();

		if ( ! isset( $settings['insta_title_icon'] ) && ! Icons_Manager::is_migration_allowed() ) {
			// add old default
			$settings['insta_title_icon'] = 'fa fa-instagram';
		}

		$has_icon = ! empty( $settings['insta_title_icon'] );

		if ( $has_icon ) {
			$this->set_attribute( 'i', 'class', $settings['insta_title_icon'] );
			$this->set_attribute( 'i', 'aria-hidden', 'true' );
		}

		if ( ! $has_icon && ! empty( $settings['title_icon']['value'] ) ) {
			$has_icon = true;
		}
		$migrated = isset( $settings['__fa4_migrated']['title_icon'] );
		$is_new   = ! isset( $settings['insta_title_icon'] ) && Icons_Manager::is_migration_allowed();

		if ( $has_icon ) {
			?>
			<span <?php echo wp_kses_post( $this->render_attributes( 'title-icon' ) ); ?>>
				<?php
				if ( $is_new || $migrated ) {
					Icons_Manager::render_icon( $settings['title_icon'], array( 'aria-hidden' => 'true' ) );
				} elseif ( ! empty( $settings['insta_title_icon'] ) ) {
					?>
					<i <?php echo wp_kses_post( $this->render_attributes( 'i' ) ); ?>></i>
					<?php
				}
				?>
			</span>
			<?php
		}
	}

	/**
	 * Render Instagram profile link.
	 *
	 * @since  1.2.0
	 * @param  array $settings
	 * @return array
	 */
	public function get_insta_profile_link() {
		$settings = $this->settings;
		$insta_title_icon_position = isset( $settings['insta_title_icon_position'] ) ? $settings['insta_title_icon_position'] : '';
		$insta_profile_link = isset( $settings['insta_profile_link'] ) ? $settings['insta_profile_link'] : '';
		$insta_link_title = isset( $settings['insta_link_title'] ) ? $settings['insta_link_title'] : '';

		$this->set_attribute( 'title-icon', 'class', 'mab-icon mab-icon-' . $insta_title_icon_position );

		if ( 'yes' === $insta_profile_link && $insta_link_title ) { ?>
			<span class="mab-instagram-feed-title-wrap">
				<a <?php echo wp_kses_post( $this->render_attributes( 'instagram-profile-link' ) ); ?>>
					<span class="mab-instagram-feed-title">
						<?php
						if ( 'before_title' === $insta_title_icon_position ) {
							$this->render_title_icon();
						}

						echo esc_attr( $settings['insta_link_title'] );

						if ( 'after_title' === $insta_title_icon_position ) {
							$this->render_title_icon();
						}
						?>
					</span>
				</a>
			</span>
		<?php }
	}

	protected function get_cache_duration() {
		$settings = $this->settings;
		$cache_duration = isset( $settings['cacheTimeout'] ) ? $settings['cacheTimeout'] : 'hour';
		$duration = 0;

		switch ( $cache_duration ) {
			case 'minute':
				$duration = MINUTE_IN_SECONDS;
				break;
			case 'hour':
				$duration = HOUR_IN_SECONDS;
				break;
			case 'day':
				$duration = DAY_IN_SECONDS;
				break;
			case 'week':
				$duration = WEEK_IN_SECONDS;
				break;
			default:
				break;
		}

		return $duration;
	}

	/**
	 * Retrieve Instagram posts.
	 *
	 * @since  1.2.0
	 * @param  array $settings
	 * @return array
	 */
	public function get_insta_posts( $settings ) {
		$settings = $this->settings;

		$transient_key = md5( $this->get_transient_key() );

		$data = get_transient( $transient_key );
		$cache_timeout = isset( $settings['cacheTimeout'] ) ? $settings['cacheTimeout'] : 'hour';

		if ( ! empty( $data ) && 'none' !== $cache_timeout ) {
			return $data;
		}

		$response = $this->get_insta_remote( $this->get_fetch_url() );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$data = $this->get_insta_feed_response_data( $response );

		if ( empty( $data ) ) {
			return array();
		}

		set_transient( $transient_key, $data, $this->get_cache_duration() );

		return $data;
	}

	/**
	 * Render Image Thumbnail
	 *
	 * @since  1.2.0
	 * @return void
	 */
	protected function render_image_thumbnail( $item, $index ) {
		$settings        = $this->settings;
		$thumbnail_url   = $this->get_insta_image_url( $item, $this->get_insta_image_size() );
		$thumbnail_alt   = $item['caption'];
		$thumbnail_title = $item['caption'];
		//$likes           = $item['likes'];
		//$comments        = $item['comments'];
		$image_key       = 'image_insta_' . $index;
		$link_key        = 'link_image_' . $index;
		$item_link       = '';

		$this->set_attribute( $image_key, 'src', $thumbnail_url );

		if ( '' !== $thumbnail_alt ) {
			$this->set_attribute( $image_key, 'alt', $thumbnail_alt );
		}

		if ( '' !== $thumbnail_title ) {
			$this->set_attribute( $image_key, 'title', $thumbnail_title );
		}

		if ( isset( $settings['link'] ) ) {
			if ( $settings['link'] === 'lightbox' ) {
				$image_size = getimagesize( $thumbnail_url );

				$this->set_attribute( $link_key,'class', 'bricks-lightbox' );
				$this->set_attribute( $link_key, 'data-pswp-src', $thumbnail_url );
				$this->set_attribute( $link_key, 'data-pswp-width', $image_size[0] );
				$this->set_attribute( $link_key, 'data-pswp-height', $image_size[1] );
				//$this->set_attribute( $link_key, 'data-pswp-index', $index );
				$this->set_attribute( $link_key, 'data-pswp-id', $this->id );
				//$this->set_attribute( $link_key, 'data-pswp-title', $thumbnail_alt );

			} elseif ( $settings['link'] === 'post' ) {
				$item_link = $item['link'];

				$this->set_attribute( $link_key, 'href', $item_link );
				$this->set_attribute( $link_key, 'target', '_blank' );
			}
		}

		//$this->set_attribute( $link_key, 'href', $item_link );

		$image_html = '';
		$image_html .= '<div class="mab-gallery-image-content mab-media-overlay">';
		if ( isset( $settings['caption'] ) && $thumbnail_alt ) {
			$image_html .= '<div class="mab-insta-caption bricks-image-caption">' . $thumbnail_alt . '</div>';
		}
		$image_html .= '</div>';
		$image_html .= '<img ' . $this->render_attributes( $image_key ) . '/>';

		if ( isset( $settings['link'] ) ) {
			$image_html = '<a ' . $this->render_attributes( $link_key ) . '></a>' . $image_html;
		}

		echo wp_kses_post( $image_html );
	}

	/**
	 * Get Insta Thumbnail Image URL
	 *
	 * @since  1.2.0
	 * @return string   The url of the instagram post image
	 */
	protected function get_insta_image_url( $item, $size = 'high' ) {
		$thumbnail  = $item['thumbnail'];

		if ( ! empty( $thumbnail[ $size ] ) ) {
			$image_url = $thumbnail[ $size ]['src'];
		} else {
			$image_url = isset( $item['image'] ) ? $item['image'] : '';
		}

		return $image_url;
	}

	/**
	 * Render load more button output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since  1.2.0
	 * @access protected
	 */
	protected function render_api_images() {
		$settings = $this->settings;
		$gallery  = $this->get_insta_posts( $settings );
		$count    = 0;

		$layout            = isset( $settings['layout'] ) ? $settings['layout'] : 'grid';
		$filter            = isset( $settings['thumbnailFilter'] ) ? $settings['thumbnailFilter'] : '';
		$hover_filter      = isset( $settings['thumbnailHoverFilter'] ) ? $settings['thumbnailHoverFilter'] : '';
		$item_wrap_classes = [];
		$item_tag          = 'li';

		if ( 'carousel' == $layout ) {
			$item_wrap_classes[] = 'swiper-slide';
			$item_tag            = 'div';
		} else {
			$item_wrap_classes[] = 'bricks-layout-item';
		}

		if ( $hover_filter ) {
			$item_wrap_classes[] = 'mab-ins-filter-hover';
		}

		$item_classes[] = 'mab-image';

		if ( $filter || $hover_filter ) {
			$item_classes[] = 'mab-ins-filter-target';
		}

		// Item sizer (Isotope requirement)
		$item_sizer_classes = [ 'bricks-isotope-sizer' ];

		$this->set_attribute( 'item-sizer', 'class', $item_sizer_classes );

		if ( empty( $gallery ) || is_wp_error( $gallery ) ) {
			$placeholder = sprintf( __( 'Click here to edit the "%1$s" settings and update the access token.', 'max-addons' ), 'Instagram Feed' );

			return $this->render_element_placeholder(
				[
					'title' => $placeholder,
				]
			);
		}

		foreach ( $gallery as $index => $item ) {
			$this->set_attribute( "item-wrap-{$index}", 'class', $item_wrap_classes );
			$this->set_attribute( "item-{$index}", 'class', $item_classes );
			?>
			<<?php echo $item_tag; ?> <?php echo wp_kses_post( $this->render_attributes( "item-wrap-{$index}" ) ); ?>>
				<div <?php echo wp_kses_post( $this->render_attributes( "item-{$index}" ) ); ?>>
					<?php $this->render_image_thumbnail( $item, $index ); ?>
				</div>
			</<?php echo $item_tag; ?>>
			<?php
		}

		if ( 'masonry' == $layout ) {
			echo "<li {$this->render_attributes( 'item-sizer' )}></li>";
			echo '<li class="bricks-gutter-sizer"></li>';
		}
	}

	public function slider_settings() {
		$settings = $this->settings;

		$options = [
			'slidesPerView'  => isset( $settings['slidesToShow'] ) ? intval( $settings['slidesToShow'] ) : 3,
			'slidesPerGroup' => isset( $settings['slidesToScroll'] ) ? intval( $settings['slidesToScroll'] ) : 1,
			'speed'          => isset( $settings['speed'] ) ? intval( $settings['speed'] ) : 300,
			'autoHeight'     => isset( $settings['adaptiveHeight'] ),
			'effect'         => isset( $settings['effect'] ) ? $settings['effect'] : 'slide',
			'spaceBetween'   => isset( $settings['gutter'] ) ? intval( $settings['gutter'] ) : 0,
			'initialSlide'   => isset( $settings['initialSlide'] ) ? intval( $settings['initialSlide'] ) : 0,
			'loop'           => isset( $settings['infinite'] ),
			'centeredSlides' => isset( $settings['centerMode'] ),
		];

		if ( isset( $settings['autoplay'] ) ) {
			$options['autoplay'] = [
				'delay'                => isset( $settings['autoplaySpeed'] ) ? intval( $settings['autoplaySpeed'] ) : 3000,

				// Set to false if pauseOnHover is true, otherwise it stops after the first hover
				'disableOnInteraction' => ! isset( $settings['pauseOnHover'] ),

				// Pause autoplay on mouse enter (new in v6.6: autoplay.pauseOnMouseEnter)
				'pauseOnMouseEnter'    => isset( $settings['pauseOnHover'] ),

				// Stop autoplay on last slide (@since 1.4)
				'stopOnLastSlide'      => isset( $settings['stopOnLastSlide'] ),
			];
		}

		// Arrow navigation
		if ( isset( $settings['arrows'] ) ) {
			$options['navigation'] = true;
		}

		// Dots
		if ( isset( $settings['dots'] ) ) {
			$options['pagination'] = true;

			if ( isset( $settings['dotsDynamic'] ) && ! isset( $settings['dotsVertical'] ) ) {
				$options['dynamicBullets'] = true;
			}
		}

		$breakpoint_options = \Bricks\Helpers::generate_swiper_breakpoint_data_options( $settings );

		// Has slidesPerView/slidesPerGroup set on non-base breakpoints
		if ( is_array( $breakpoint_options ) && count( $breakpoint_options ) > 1 ) {
			unset( $options['slidesPerView'] );
			unset( $options['slidesPerGroup'] );
			$options['breakpoints'] = $breakpoint_options;
		}

		//$this->set_attribute( '_root', 'data-slider-options', wp_json_encode( $options ) );
		$this->set_attribute( 'container', 'class', 'bricks-swiper-container' );
		$this->set_attribute( 'container', 'data-script-args', wp_json_encode( $options ) );
	}

	public function render() {
		$settings = $this->settings;

		if ( isset( $settings['access_token'] ) ) {
			if ( $this->get_insta_global_access_token() !== $settings['access_token'] ) {
				$access_token = $this->get_insta_access_token();
				$widget_id    = $this->id;

				\MaxAddonsPro\Classes\Admin_Settings::refresh_instagram_access_token( $access_token, $widget_id );
			}
		}

		$layout = isset( $settings['layout'] ) ? $settings['layout'] : 'grid';
		$filter = isset( $settings['thumbnailFilter'] ) ? $settings['thumbnailFilter'] : '';
		$hover_filter = isset( $settings['thumbnailHoverFilter'] ) ? $settings['thumbnailHoverFilter'] : '';
		$root_classes = [];

		if ( 'carousel' === $layout ) {
			$this->tag = 'div';

			$this->slider_settings();
		} else {
			$root_classes[] = 'bricks-layout-wrapper';
		}

		if ( $filter ) {
			$root_classes[] = 'mab-ins-' . $filter;
		}

		if ( $hover_filter ) {
			$root_classes[] = 'mab-ins-hover-' . $hover_filter;
		}

		if ( 'masonry' === $layout ) {
			$root_classes[] = 'isotope';
		}

		$this->set_attribute( '_root', 'class', $root_classes );
		$this->set_attribute( '_root', 'data-layout', $layout );
		?>
		<<?php echo $this->tag; ?> <?php echo wp_kses_post( $this->render_attributes( '_root' ) ); ?>>
			<?php
			$this->get_insta_profile_link();

			if ( 'carousel' === $settings['layout'] ) { ?>
				<div <?php echo wp_kses_post( $this->render_attributes( 'container' ) ); ?>>
					<div class="swiper-wrapper">
			<?php }

			$this->render_api_images();

			if ( 'carousel' === $settings['layout'] ) { ?>
			</div>
				</div>
				<?php echo $this->render_swiper_nav(); ?>
			<?php }

			//$this->render_dots();

			//$this->render_arrows();
			?>
		</<?php echo $this->tag; ?>>
		<?php
	}
}
