<?php
namespace MaxAddonsPro\Elements;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Image_Gallery_Element extends \Bricks\Element {
	// Element properties
	public $category     = 'max-addons-elements'; // Use predefined element category 'general'
	public $name         = 'max-image-gallery'; // Make sure to prefix your elements
	public $icon         = 'ti-gallery max-element'; // Themify icon font class
	public $css_selector = ''; // Default CSS selector
	public $scripts      = [ 'bricksIsotope', 'mabImageGallery' ]; // Script(s) run when element is rendered on frontend or updated in builder

	public function get_label() {
		return esc_html__( 'Image Gallery', 'max-addons' );
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		$settings = $this->settings;

		$lightbox_enabled  = ( isset( $this->settings['link_to'] ) && $this->settings['link_to'] === 'lightbox' ) ? true : false;
		$tilt_enabled      = isset( $settings['tilt'] ) ? $this->settings['tilt'] : '';
		$layout            = isset( $settings['layout'] ) ? $settings['layout'] : '';
		$filters_enabled   = ( isset( $settings['filtersEnable'] ) && isset( $settings['galleryType'] ) && ( 'filterable' === $settings['galleryType'] ) ) ? true : false;
		$filters_animation = $filters_enabled && ( isset( $settings['pointer'] ) ) && 'none' !== $settings['pointer'] ? true : false;
		$isotope_enabled   = ( 'masonry' === $layout || $filters_enabled ) ? true : false;
		$gallery_script    = ( $isotope_enabled || $tilt_enabled || $lightbox_enabled ) ? true : false;

		wp_enqueue_style( 'mab-image-gallery');

		if ( $isotope_enabled ) {
			wp_enqueue_script( 'imagesloaded' );
			wp_enqueue_script( 'bricks-isotope' );
			wp_enqueue_style( 'bricks-isotope' );
		}

		if ( $tilt_enabled ) {
			wp_enqueue_script( 'tilt' );
		}

		if ( $lightbox_enabled ) {
			if ( isset( $settings['lightboxLibrary'] ) && $settings['lightboxLibrary'] === 'fancybox' ) {
				wp_enqueue_script( 'fancybox' );
				wp_enqueue_style( 'fancybox' );
			} else {
				wp_enqueue_script( 'bricks-photoswipe' );
				wp_enqueue_script( 'bricks-photoswipe-lightbox' );
				wp_enqueue_style( 'bricks-photoswipe' );
			}
		}

		if ( $filters_animation ) {
			wp_enqueue_style( 'mab-frontend-filters');
		}

		if ( $gallery_script ) {
			wp_enqueue_script( 'mab-image-gallery' );
		}
	}

	// Set builder control groups
	public function set_control_groups() {
		$this->control_groups['gallery'] = [
			'title' => esc_html__( 'Gallery', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['gallerySettings'] = [
			'title' => esc_html__( 'Settings', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['filterTabs'] = [
			'title' => esc_html__( 'Filters', 'max-addons' ),
			'tab'   => 'content',
			'required' => [ 'galleryType', '=', 'filterable' ],
		];

		$this->control_groups['thumbnail'] = [
			'title' => esc_html__( 'Thumbnails', 'max-addons' ),
			'tab'   => 'content',
		];
		
		$this->control_groups['caption'] = [
			'title' => esc_html__( 'Caption', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['download'] = [
			'title' => esc_html__( 'Download Button', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['overlay'] = [
			'title' => esc_html__( 'Overlay', 'max-addons' ),
			'tab'   => 'content',
		];
	}

	public function set_controls() {
		$this->set_items_controls();
		$this->set_settings_controls();
		$this->set_filters_controls();
		$this->set_thumbnail_style_controls();
		$this->set_caption_controls();
		$this->set_download_button_controls();
		$this->set_overlay_style_controls();
	}

	// Set items controls
	public function set_items_controls() {

		$this->controls['galleryType'] = [
			'tab'       => 'content',
			'group'     => 'gallery',
			'label'     => esc_html__( 'Gallery Type', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'standard'   => esc_html__( 'Standard', 'max-addons' ),
				'filterable' => esc_html__( 'Filterable', 'max-addons' ),
			],
			'default'   => 'filterable',
			'inline'    => true,
			'clearable' => false,
		];

		$this->controls['galleryImages'] = [
			'tab'           => 'content',
			'group'         => 'gallery',
			'placeholder'   => esc_html__( 'Gallery', 'max-addons' ),
			'type'          => 'repeater',
			'titleProperty' => 'filterLabel',
			'fields'        => [
				'imageGroup'  => [
					'label' => esc_html__( 'Select Images', 'max-addons' ),
					'type'	=> 'image-gallery',
				],

				'filterLabel' => [
					'label' => esc_html__( 'Filter Label', 'max-addons' ),
					'type'  => 'text',
				],

				'filterId'    => [
					'label'       => esc_html__( 'Filter ID', 'max-addons' ),
					'description' => __( 'To filter the gallery using URL parameters, specify an ID here', 'max-addons' ),
					'type'        => 'text',
				],
			],
			'required'      => [ 'galleryType', '=', [ 'filterable' ] ],
		];

		$this->controls['imageGroupStandard'] = [
			'tab'  		=> 'content',
			'group'		=> 'gallery',
			'type' 		=> 'image-gallery',
			'required' 	=> [ 'galleryType', '=', [ 'standard' ] ],
		];
	}

	// Set settings controls
	public function set_settings_controls() {
		$this->controls['layout'] = [
			'tab'     => 'content',
			'group'   => 'gallerySettings',
			'label'   => esc_html__( 'Layout', 'max-addons' ),
			'type'    => 'select',
			'options' => [
				'grid'    => esc_html__( 'Grid', 'max-addons' ),
				'masonry' => esc_html__( 'Masonry', 'max-addons' ),
				'metro'   => esc_html__( 'Metro', 'max-addons' ),
			],
			'default' => 'grid',
			'inline'  => true,
		];

		$this->control_options['imageRatio']['custom'] = esc_html__( 'Custom', 'max-addons' );

		$this->controls['imageRatio'] = [
			'tab'         => 'content',
			'group'       => 'gallerySettings',
			'label'       => esc_html__( 'Image ratio', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['imageRatio'],
			'inline'      => true,
			'placeholder' => esc_html__( 'Square', 'max-addons' ),
			'required'    => [ 'layout', '!=', [ 'masonry', 'metro' ] ],
		];

		/**
		 * Custom aspect ratio (remove control from style tab)
		 *
		 * @since 1.7.0
		 */
		unset( $this->controls['_aspectRatio'] );
		$this->controls['_aspectRatio'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Aspect ratio', 'max-addons' ),
			'type'     => 'text',
			'inline'   => true,
			'dd'       => false,
			'css'      => [
				[
					'selector' => '.image',
					'property' => 'aspect-ratio',
				],
			],
			'required' => [ 'imageRatio', '=', 'custom' ],
		];

		$this->controls['imageHeight'] = [
			'tab'         => 'content',
			'group'       => 'gallerySettings',
			'label'       => esc_html__( 'Image height', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'css'         => [
				[
					'property'  => 'height',
					'selector'  => '.image',
					'important' => true,
				],
			],
			'placeholder' => '',
			'required'    => [ 'layout', '!=', [ 'masonry', 'metro' ] ],
		];

		$this->controls['columns'] = [
			'tab'         => 'content',
			'group'       => 'gallerySettings',
			'label'       => esc_html__( 'Columns', 'max-addons' ),
			'type'        => 'number',
			'min'         => 1,
			'css'         => [
				[
					'property' => '--columns',
					'selector' => '.mab-image-gallery',
				],
			],
			'rerender'    => true,
			'inline'      => true,
			'small'       => true,
			'placeholder' => 3,
			'default' 	  => 3,
		];

		$this->controls['gutter'] = [
			'tab'         => 'content',
			'group'       => 'gallerySettings',
			'label'       => esc_html__( 'Spacing', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'css'         => [
				[
					'property' => '--gutter',
					'selector' => '.mab-image-gallery',
				],
			],
			'placeholder' => 0,
		];

		$this->controls['ordering'] = [
			'tab'     => 'content',
			'group'   => 'gallerySettings',
			'label'   => esc_html__( 'Order', 'max-addons' ),
			'type'    => 'select',
			'options' => [
				'default' => __( 'Default', 'max-addons' ),
				'date'    => __( 'Date', 'max-addons' ),
				'random'  => __( 'Random', 'max-addons' ),
			],
			'default' => 'default',
			'inline'  => true,
		];

		$this->controls['linkToSeparator'] = [
			'tab'   => 'content',
			'group' => 'gallerySettings',
			'type'  => 'separator',
			'label' => esc_html__( 'Link To', 'max-addons' ),
		];

		$this->controls['link_to'] = [
			'tab'         => 'content',
			'group'       => 'gallerySettings',
			'label'       => esc_html__( 'Link to', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'lightbox'   => esc_html__( 'Lightbox', 'max-addons' ),
				'attachment' => esc_html__( 'Attachment Page', 'max-addons' ),
				'media'      => esc_html__( 'Media File', 'max-addons' ),
				'custom'     => esc_html__( 'Custom URL', 'max-addons' ),
			],
			'inline'      => true,
			'placeholder' => esc_html__( 'None', 'max-addons' ),
		];

		$this->controls['lightboxLibrary'] = [
			'tab'         => 'content',
			'group'       => 'gallerySettings',
			'label'       => esc_html__( 'Lightbox Library', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'bricks'   => esc_html__( 'Bricks', 'max-addons' ),
				'fancybox' => esc_html__( 'Fancybox', 'max-addons' ),
			],
			'inline'      => true,
			'placeholder' => esc_html__( 'Bricks', 'max-addons' ),
			'required'    => [ 'link_to', '=', 'lightbox' ],
		];

		$this->controls['lightboxImageSize'] = [
			'tab'         => 'content',
			'group'       => 'gallerySettings',
			'label'       => esc_html__( 'Lightbox image size', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['imageSizes'],
			'placeholder' => esc_html__( 'Full', 'max-addons' ),
			'required'    => [ 'link_to', '=', 'lightbox' ],
		];

		$this->controls['lightboxAnimationType'] = [
			'tab'         => 'content',
			'group'       => 'gallerySettings',
			'label'       => esc_html__( 'Lightbox animation type', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['lightboxAnimationTypes'],
			'placeholder' => esc_html__( 'Zoom', 'max-addons' ),
			'required'    => [
				[ 'link_to', '=', 'lightbox' ],
				[ 'lightboxLibrary', '=', 'bricks' ],
			],
		];

		$this->controls['lightboxCaptionType'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Caption Type', 'max-addons' ),
			'type'     => 'select',
			'options'  => [
				'title'       => esc_html__( 'Title', 'max-addons' ),
				'caption'     => esc_html__( 'Caption', 'max-addons' ),
				'description' => esc_html__( 'Description', 'max-addons' ),
			],
			'default'  => 'caption',
			'inline'   => true,
			'required' => [
				[ 'link_to', '=', 'lightbox' ],
				[ 'lightboxLibrary', '=', 'fancybox' ],
			],
		];

		$this->controls['lightboxArrows'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Arrows', 'max-addons' ),
			'type'     => 'checkbox',
			'default'  => true,
			'required' => [
				[ 'link_to', '=', 'lightbox' ],
				[ 'lightboxLibrary', '=', 'fancybox' ],
			],
		];

		$this->controls['lightboxToolbar'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Toolbar', 'max-addons' ),
			'type'     => 'checkbox',
			'default'  => true,
			'required' => [
				[ 'link_to', '=', 'lightbox' ],
				[ 'lightboxLibrary', '=', 'fancybox' ],
			],
		];

		$toolbar_buttons = [
			'infobar'    => __( 'Slides Counter', 'max-addons' ),
			'zoomIn'     => __( 'Zoom In', 'max-addons' ),
			'zoomOut'    => __( 'Zoom Out', 'max-addons' ),
			'toggle1to1' => __( 'Toggle 1:1', 'max-addons' ),
			'fullScreen' => __( 'Full Screen', 'max-addons' ),
			'rotateCCW'  => __( 'Rotate Counterclockwise', 'max-addons' ),
			'rotateCW'   => __( 'Rotate Clockwise', 'max-addons' ),
			'flipX'      => __( 'Flip Horizontally', 'max-addons' ),
			'flipY'      => __( 'Flip Vertically', 'max-addons' ),
			'slideshow'  => __( 'Slideshow', 'max-addons' ),
			'thumbs'     => __( 'Thumbs', 'max-addons' ),
			'close'      => __( 'Close', 'max-addons' ),
		];

		$this->controls['lightboxToolbarButtonsLeft'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Toolbar Buttons Left', 'max-addons' ),
			'type'     => 'select',
			'options'  => $toolbar_buttons,
			'default'  => [ 'infobar' ],
			'inline'   => false,
			'multiple' => true,
			'required' => [
				[ 'link_to', '=', 'lightbox' ],
				[ 'lightboxLibrary', '=', 'fancybox' ],
				[ 'lightboxToolbar', '!=', '' ],
			],
		];

		$this->controls['lightboxToolbarButtonsMiddle'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Toolbar Buttons Middle', 'max-addons' ),
			'type'     => 'select',
			'options'  => $toolbar_buttons,
			'default'  => [ 'zoomIn', 'zoomOut', 'rotateCCW', 'rotateCW' ],
			'inline'   => false,
			'multiple' => true,
			'required' => [
				[ 'link_to', '=', 'lightbox' ],
				[ 'lightboxLibrary', '=', 'fancybox' ],
				[ 'lightboxToolbar', '!=', '' ],
			],
		];

		$this->controls['lightboxToolbarButtonsRight'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Toolbar Buttons Right', 'max-addons' ),
			'type'     => 'select',
			'options'  => $toolbar_buttons,
			'default'  => [ 'slideshow', 'thumbs', 'close' ],
			'inline'   => false,
			'multiple' => true,
			'required' => [
				[ 'link_to', '=', 'lightbox' ],
				[ 'lightboxLibrary', '=', 'fancybox' ],
				[ 'lightboxToolbar', '!=', '' ],
			],
		];

		$this->controls['lightboxThumbsAutoStart'] = [
			'tab'         => 'content',
			'group'       => 'gallerySettings',
			'label'       => esc_html__( 'Thumbs Auto Start', 'max-addons' ),
			'description' => __( 'Display thumbnails on lightbox opening', 'max-addons' ),
			'type'        => 'checkbox',
			'required'    => [
				[ 'link_to', '=', 'lightbox' ],
				[ 'lightboxLibrary', '=', 'fancybox' ],
			],
		];

		$this->controls['lightboxThumbstype'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Thumbs Type', 'max-addons' ),
			'type'     => 'select',
			'options'  => [
				'classic' => __( 'Classic', 'max-addons' ),
				'modern'  => __( 'Modern', 'max-addons' ),
			],
			'inline'   => true,
			'required' => [
				[ 'link_to', '=', 'lightbox' ],
				[ 'lightboxLibrary', '=', 'fancybox' ],
				[ 'lightboxToolbar', '!=', '' ],
			],
		];

		/* $this->controls['lightboxAnimation'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Animation', 'max-addons' ),
			'type'     => 'select',
			'options'  => [
				'fade'        => __( 'Fade', 'powerpack' ),
				'zoom'        => __( 'Zoom', 'powerpack' ),
				'zoom-in-out' => __( 'Zoom in Out', 'powerpack' ),
			],
			'inline'   => true,
			'required' => [
				[ 'link_to', '=', 'lightbox' ],
				[ 'lightboxLibrary', '=', 'fancybox' ],
				[ 'lightboxToolbar', '!=', '' ],
			],
		]; */

		/* $this->controls['lightboxTransitionEffect'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Transition Effect', 'max-addons' ),
			'type'     => 'select',
			'options'  => [
				'fade'        => __( 'Fade', 'powerpack' ),
				'slide'       => __( 'Slide', 'powerpack' ),
				'circular'    => __( 'Circular', 'powerpack' ),
				'tube'        => __( 'Tube', 'powerpack' ),
				'zoom-in-out' => __( 'Zoom in Out', 'powerpack' ),
				'rotate'      => __( 'Rotate', 'powerpack' ),
			],
			'inline'   => true,
			'required' => [
				[ 'link_to', '=', 'lightbox' ],
				[ 'lightboxLibrary', '=', 'fancybox' ],
				[ 'lightboxToolbar', '!=', '' ],
			],
		]; */

		$this->controls['linkCustom'] = [
			'tab'         => 'content',
			'group'   	  => 'gallerySettings',
			'label'       => esc_html__( 'Custom links', 'max-addons' ),
			'type'        => 'repeater',
			'fields'      => [
				'link' => [
					'label'   => esc_html__( 'Link', 'max-addons' ),
					'type'    => 'link',
					'exclude' => [
						'lightboxImage',
						'lightboxVideo',
					],
				],
			],
			'placeholder' => esc_html__( 'Custom link', 'max-addons' ),
			'required'    => [ 'link_to', '=', 'custom' ],
		];

		$this->controls['linkIconSeparator'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'type'     => 'separator',
			'label'    => esc_html__( 'Link Icon', 'max-addons' ),
			'required' => [ 'link_to', '!=', '' ],
		];

		$this->controls['linkIcon'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Icon', 'max-addons' ),
			'type'     => 'icon',
			'inline'   => true,
			'small'    => true,
			'rerender' => true,
			'required' => [ 'link_to', '!=', '' ],
		];

		$this->controls['linkIconTypography'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '&{pseudo} .icon',
				],
			],
			'exclude'  => [
				'font-family',
				'font-weight',
				'font-style',
				'text-align',
				'text-decoration',
				'text-transform',
				'line-height',
				'letter-spacing',
			],
			'inline'   => true,
			'small'    => true,
			'required' => [
				[ 'link_to', '!=', '' ],
				[ 'linkIcon', '!=', '' ],
			],
		];

		$this->controls['linkIconBackgroundColor'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Icon background color', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '&{pseudo} .icon',
				],
			],
			'required' => [
				[ 'link_to', '!=', '' ],
				[ 'linkIcon', '!=', '' ],
			],
		];

		$this->controls['linkIconBorder'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Icon border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '&{pseudo} .icon',
				],
			],
			'required' => [
				[ 'link_to', '!=', '' ],
				[ 'linkIcon', '!=', '' ],
			],
		];

		$this->controls['linkIconBoxShadow'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Icon box shadow', 'max-addons' ),
			'type'     => 'box-shadow',
			'css'      => [
				[
					'property' => 'box-shadow',
					'selector' => '&{pseudo} .icon',
				],
			],
			'required' => [
				[ 'link_to', '!=', '' ],
				[ 'linkIcon', '!=', '' ],
			],
		];

		$this->controls['linkIconHeight'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Icon height', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'line-height',
					'selector' => '&{pseudo} .icon',
				],
			],
			'required' => [
				[ 'link_to', '!=', '' ],
				[ 'linkIcon', '!=', '' ],
			],
		];

		$this->controls['linkIconWidth'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Icon width', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'width',
					'selector' => '&{pseudo} .icon',
				],
			],
			'required' => [
				[ 'link_to', '!=', '' ],
				[ 'linkIcon', '!=', '' ],
			],
		];

		$this->controls['linkIconOpacity'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Opacity', 'max-addons' ),
			'type'     => 'number',
			'inline'   => true,
			'min'      => 0,
			'max'      => 1,
			'step'     => 0.1,
			'css'      => [
				[
					'property' => 'opacity',
					'selector' => '&{pseudo} .icon',
				],
			],
			'required' => [
				[ 'link_to', '!=', '' ],
				[ 'linkIcon', '!=', '' ],
			],
		];

		$this->controls['linkIconOpacityHover'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Opacity', 'max-addons' ) . ' (' . esc_html__( 'Hover', 'max-addons' ) . ')',
			'type'     => 'number',
			'inline'   => true,
			'min'      => 0,
			'max'      => 1,
			'step'     => 0.1,
			'css'      => [
				[
					'property' => 'opacity',
					'selector' => '.mab-image-gallery-thumbnail-wrap:hover .icon',
				],
			],
			'required' => [
				[ 'link_to', '!=', '' ],
				[ 'linkIcon', '!=', '' ],
			],
		];

		$this->controls['linkIconTransition'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Icon transition', 'max-addons' ),
			'type'     => 'text',
			'inline'   => true,
			'css'      => [
				[
					'property' => 'transition',
					'selector' => '&{pseudo} .icon',
				],
			],
			'required' => [
				[ 'link_to', '!=', '' ],
				[ 'linkIcon', '!=', '' ],
			],
		];

		$this->controls['tiltSeparator'] = [
			'tab'   => 'content',
			'group' => 'gallerySettings',
			'type'  => 'separator',
			'label' => esc_html__( 'Tilt', 'max-addons' ),
		];

		$this->controls['tilt'] = [
			'tab'   => 'content',
			'group' => 'gallerySettings',
			'label' => esc_html__( 'Tilt', 'max-addons' ),
			'type'  => 'checkbox',
		];

		$this->controls['tilt_axis'] = [
			'tab'      => 'content',
			'group'    => 'gallerySettings',
			'label'    => esc_html__( 'Tilt Axis', 'max-addons' ),
			'type'     => 'select',
			'options'  => [
				'null' => __( 'Both', 'max-addons' ),
				'x'    => __( 'X Axis', 'max-addons' ),
				'y'    => __( 'Y Axis', 'max-addons' ),
			],
			'default'  => 'null',
			'inline'   => true,
			'required' => [ 'tilt', '=', true ],
		];

		$this->controls['tilt_amount'] = [
			'tab'         => 'content',
			'group'       => 'gallerySettings',
			'label'       => esc_html__( 'Amount', 'max-addons' ),
			'type'        => 'number',
			'min'         => 10,
			'max'         => 50,
			'inline'      => true,
			'placeholder' => 20,
			'default' 	  => 20,
			'required' 	  => [ 'tilt', '=', true ],
		];

		$this->controls['tilt_scale'] = [
			'tab'         => 'content',
			'group'       => 'gallerySettings',
			'label'       => esc_html__( 'Scale', 'max-addons' ),
			'type'        => 'number',
			'min'         => 1,
			'max'         => 1.4,
			'step'        => 0.01,
			'inline'      => true,
			'placeholder' => 1.06,
			'default' 	  => 1.06,
			'required' 	  => [ 'tilt', '=', true ],
		];

		$this->controls['tilt_speed'] = [
			'tab'         => 'content',
			'group'       => 'gallerySettings',
			'label'       => esc_html__( 'Speed', 'max-addons' ),
			'type'        => 'number',
			'min'         => 100,
			'max'         => 1000,
			'step'        => 20,
			'inline'      => true,
			'placeholder' => 800,
			'default' 	  => 800,
			'required' 	  => [ 'tilt', '=', true ],
		];
	}

	// Set filters controls
	public function set_filters_controls() {
		$this->controls['filtersEnable'] = [
			'tab'   => 'content',
			'group' => 'filterTabs',
			'label' => esc_html__( 'Enable Filters', 'max-addons' ),
			'type'  => 'checkbox',
		];

		$this->controls['filterAllLabel'] = [
			'tab'            => 'content',
			'group'          => 'filterTabs',
			'label'          => esc_html__( '"All" Filter Label', 'max-addons' ),
			'default'        => esc_html__( 'All', 'max-addons' ),
			'type'           => 'text',
			'hasDynamicData' => 'text',
			'inline'         => true,
			'required'       => [ 'filtersEnable', '!=', '' ],
		];

		$this->controls['defaultFilterSelect'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Default Filter', 'max-addons' ),
			'type'     => 'select',
			'options'  => [
				'first'  => __( 'First', 'max-addons' ),
				'custom' => __( 'Custom', 'max-addons' ),
			],
			'default'  => 'first',
			'inline'   => true,
			'required' => [ 'filtersEnable', '!=', '' ],
		];

		$this->controls['defaultFilter'] = [
			'tab'           => 'content',
			'group'         => 'filterTabs',
			'label'         => esc_html__( 'Default Filter Name', 'max-addons' ),
			'type'          => 'text',
			'spellcheck'    => true,
			'inline'        => true,
			'inlineEditing' => false,
			'required'      => [ 'defaultFilterSelect', '=', [ 'custom' ] ],
		];

		/* $this->controls['responsiveSupport'] = [
			'tab'     		=> 'content',
			'group'   		=> 'filterTabs',
			'label'   		=> esc_html__( 'Responsive Support', 'max-addons' ),
			'description' 	=> esc_html__( 'Enable this option to display filters in a dropdown on responsive devices.', 'max-addons' ),
			'type'    		=> 'select',
			'options' 	=> [
				'none'		=> __( 'None', 'max-addons' ),
				'tablet' 	=> __( 'Tablet & Mobile', 'max-addons' ),
				'mobile' 	=> __( 'Mobile', 'max-addons' ),
			],
			'default' 		=> 'none',
			'inline'  		=> true,
			'required'      => [ 
				['galleryType', '=', [ 'filterable' ] ],
				['filtersEnable', '!=', '' ]
			],
		]; */

		$this->controls['pointer'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Pointer', 'max-addons' ),
			'type'     => 'select',
			'options'  => [
				'none'        => __( 'None', 'max-addons' ),
				'underline'   => __( 'Underline', 'max-addons' ),
				'overline'    => __( 'Overline', 'max-addons' ),
				'double-line' => __( 'Double Line', 'max-addons' ),
				'framed'      => __( 'Framed', 'max-addons' ),
				'background'  => __( 'Background', 'max-addons' ),
				'text'        => __( 'Text', 'max-addons' ),
			],
			'default'  => 'underline',
			'inline'   => true,
			'required' => [ 
				['galleryType', '=', [ 'filterable' ] ],
				['filtersEnable', '!=', '' ]
			],
		];

		$this->controls['animationLine'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Animation', 'max-addons' ),
			'type'     => 'select',
			'options'  => [
				'none'     => __( 'None', 'max-addons' ),
				'fade'     => __( 'Fade', 'max-addons' ),
				'slide'    => __( 'Slide', 'max-addons' ),
				'grow'     => __( 'Grow', 'max-addons' ),
				'drop-in'  => __( 'Drop In', 'max-addons' ),
				'drop-out' => __( 'Drop Out', 'max-addons' ),
			],
			'default'  => 'fade',
			'inline'   => true,
			'required' => [ 
				['galleryType', '=', [ 'filterable' ] ],
				['filtersEnable', '!=', '' ],
				['pointer', '=', [ 'underline', 'overline', 'double-line' ] ], 
			],
		];

		$this->controls['animationFramed'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Animation', 'max-addons' ),
			'type'     => 'select',
			'options'  => [
				'none'    => __( 'None', 'max-addons' ),
				'fade'    => __( 'Fade', 'max-addons' ),
				'grow'    => __( 'Grow', 'max-addons' ),
				'shrink'  => __( 'Shrink', 'max-addons' ),
				'draw'    => __( 'Draw', 'max-addons' ),
				'corners' => __( 'Corners', 'max-addons' ),
			],
			'default'  => 'fade',
			'inline'   => true,
			'required' => [ 
				[ 'galleryType', '=', [ 'filterable' ] ],
				[ 'filtersEnable', '!=', '' ],
				[ 'pointer', '=', [ 'framed' ] ], 
			],
		];

		$this->controls['animationBackground'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Animation', 'max-addons' ),
			'type'     => 'select',
			'options'  => [
				'none'			=> __( 'None', 'max-addons' ),
				'fade'			=> __( 'Fade', 'max-addons' ),
				'grow' 			=> __( 'Grow', 'max-addons' ),
				'shrink' 		=> __( 'Shrink', 'max-addons' ),
				'sweep-left' 	=> __( 'Sweep Left', 'max-addons' ),
				'sweep-right' 	=> __( 'Sweep Right', 'max-addons' ),
				'sweep-up' 		=> __( 'Sweep Up', 'max-addons' ),
				'sweep-down' 	=> __( 'Sweep Down', 'max-addons' ),
				'shutter-in-vertical' => __( 'Shutter In Vertical', 'max-addons' ),
				'shutter-out-vertical' => __( 'Shutter Out Vertical', 'max-addons' ),
				'shutter-in-horizontal' => __( 'Shutter In Horizontal', 'max-addons' ),
				'shutter-out-horizontal' => __( 'Shutter Out Horizontal', 'max-addons' ),
			],
			'default'  => 'fade',
			'inline'   => true,
			'required' => [ 
				[ 'galleryType', '=', [ 'filterable' ] ],
				[ 'filtersEnable', '!=', '' ],
				[ 'pointer', '=', [ 'background' ] ], 
			],
		];

		$this->controls['animationText'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Animation', 'max-addons' ),
			'type'     => 'select',
			'options'  => [
				'none'   => __( 'None', 'max-addons' ),
				'grow'   => __( 'Grow', 'max-addons' ),
				'shrink' => __( 'Shrink', 'max-addons' ),
				'sink'   => __( 'sink', 'max-addons' ),
				'float'  => __( 'Float', 'max-addons' ),
				'skew'   => __( 'Skew', 'max-addons' ),
				'rotate' => __( 'Rotate', 'max-addons' ),
			],
			'default'  => 'grow',
			'inline'   => true,
			'required' => [ 
				[ 'galleryType', '=', [ 'filterable' ] ],
				[ 'filtersEnable', '!=', '' ],
				[ 'pointer', '=', [ 'text' ] ], 
			],
		];

		$this->controls['filtersTypography'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.bricks-isotope-filters',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'filtersEnable', '!=', '' ],
		];

		$this->controls['activeFiltersTypography'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Typography Active', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.bricks-isotope-filters .active',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'filtersEnable', '!=', '' ],
		];

		$this->controls['filtersBorder'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.bricks-isotope-filters li',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'filtersEnable', '!=', '' ],
		];

		$this->controls['filtersBackground'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Background', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.bricks-isotope-filters li',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'filtersEnable', '!=', '' ],
		];

		$this->controls['activeFiltersBackground'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Background Active', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.bricks-isotope-filters .active',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'filtersEnable', '!=', '' ],
		];

		$this->controls['filtersMargin'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Margin', 'max-addons' ),
			'type'     => 'spacing',
			'css'      => [
				[
					'property' => 'margin',
					'selector' => '.bricks-isotope-filters li',
				],
			],
			'required' => [ 'filtersEnable', '!=', '' ],
		];

		$this->controls['filtersPadding'] = [
			'tab'      => 'content',
			'group'    => 'filterTabs',
			'label'    => esc_html__( 'Padding', 'max-addons' ),
			'type'     => 'spacing',
			'css'      => [
				[
					'property' => 'padding',
					'selector' => '.bricks-isotope-filters li',
				],
			],
			'required' => [ 'filtersEnable', '!=', '' ],
		];
	}

	// Set thumbnail controls
	public function set_thumbnail_style_controls() {
		$this->controls['imageBorder'] = [
			'tab'      => 'content',
			'group'    => 'thumbnail',
			'type'     => 'border',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.mab-image-gallery-thumbnail-wrap',
				],
			],
			'inline'   => true,
			'small'    => true,
		];

		$this->controls['thumbnailFilter'] = [
			'tab'     => 'content',
			'group'   => 'thumbnail',
			'label'   => esc_html__( 'Image Filter', 'max-addons' ),
			'type'    => 'select',
			'options' => $this->image_filters(),
			'inline'  => true,
		];

		$this->controls['thumbnailHoverFilter'] = [
			'tab'     => 'content',
			'group'   => 'thumbnail',
			'label'   => esc_html__( 'Image Hover Filter', 'max-addons' ),
			'type'    => 'select',
			'options' => $this->image_filters(),
			'inline'  => true,
		];
	}

	// Set caption controls
	public function set_caption_controls() {
		$this->controls['caption'] = [
			'tab'   => 'content',
			'group' => 'caption',
			'label' => esc_html__( 'Caption', 'max-addons' ),
			'type'  => 'checkbox',
			'default' => true
		];

		$this->controls['caption_type'] = [
			'tab'     => 'content',
			'group'   => 'caption',
			'label'   => esc_html__( 'Caption Type', 'max-addons' ),
			'type'    => 'select',
			'options' => [
				'title'       => esc_html__( 'Title', 'max-addons' ),
				'caption'     => esc_html__( 'Caption', 'max-addons' ),
				'description' => esc_html__( 'Description', 'max-addons' ),
			],
			'default' => 'caption',
			'inline'  => true,
			'required' => [ 'caption', '=', true ],
		];

		$this->controls['caption_position'] = [
			'tab'         => 'content',
			'group'       => 'caption',
			'label'       => esc_html__( 'Caption Position', 'max-addons' ),
			'placeholder' => esc_html__( 'Over Image', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'over_image'  => esc_html__( 'Over Image', 'max-addons' ),
				'below_image' => esc_html__( 'Below Image', 'max-addons' ),
			],
			'default'     => 'over_image',
			'inline'      => true,
			'clearable'   => false,
			'required'    => [ 'caption', '=', true ],
		];

		$this->controls['captionHoverEffect'] = [
			'tab'     => 'content',
			'group'   => 'caption',
			'label'   => esc_html__( 'Hover Effect', 'max-addons' ),
			'type'    => 'select',
			'options' => [
				'fade-in'           => __( 'Fade In', 'max-addons' ),
				'fade-out'          => __( 'Fade Out', 'max-addons' ),
				'fade-from-top'     => __( 'Fade From Top', 'max-addons' ),
				'fade-from-bottom'  => __( 'Fade From Bottom', 'max-addons' ),
				'fade-from-left'    => __( 'Fade From Left', 'max-addons' ),
				'fade-from-right'   => __( 'Fade From Right', 'max-addons' ),
				'slide-from-top'    => __( 'Slide From Top', 'max-addons' ),
				'slide-from-bottom' => __( 'Slide From Bottom', 'max-addons' ),
				'slide-from-left'   => __( 'Slide From Left', 'max-addons' ),
				'slide-from-right'  => __( 'Slide From Right', 'max-addons' ),
				'fade-to-top'       => __( 'Fade To Top', 'max-addons' ),
				'fade-to-bottom'    => __( 'Fade To Bottom', 'max-addons' ),
				'fade-to-left'      => __( 'Fade To Left', 'max-addons' ),
				'fade-to-right'     => __( 'Fade To Right', 'max-addons' ),
				'slide-to-top'      => __( 'Slide To Top', 'max-addons' ),
				'slide-to-bottom'   => __( 'Slide To Bottom', 'max-addons' ),
				'slide-to-left'     => __( 'Slide To Left', 'max-addons' ),
				'slide-to-right'    => __( 'Slide To Right', 'max-addons' ),
			],
			'default' => '',
			'inline'  => true,
			'required' => [ 
				[ 'caption', '=', true ],
				[ 'caption_position', '!=', 'below_image' ],
				[ 'tilt', '=', '' ]
			],
		];

		$this->controls['captionJustifyContent'] = [
			'tab'      => 'content',
			'group'    => 'caption',
			'label'    => esc_html__( 'Vertical align', 'max-addons' ),
			'type'     => 'justify-content',
			'css'      => [
				[
					'property' => 'justify-content',
					'selector' => '.mab-gallery-image-content',
				],
			],
			'default'  => 'flex-end',
			'exclude'  => 'space',
			'inline'   => true,
			'required' => [
				[ 'caption', '=', true ],
				[ 'caption_position', '!=', 'below_image' ],
			],
		];

		$this->controls['captionAlignContent'] = [
			'tab'      => 'content',
			'group'    => 'caption',
			'label'    => esc_html__( 'Horizontal align', 'max-addons' ),
			'type'     => 'align-items',
			'css'      => [
				[
					'property' => 'align-items',
					'selector' => '.mab-gallery-image-content',
				],
			],
			'default'  => 'flex-start',
			'exclude'  => 'stretch',
			'inline'   => true,
			'required' => [ 'caption', '=', true ],
		];

		$this->controls['captionTypography'] = [
			'tab'      => 'content',
			'group'    => 'caption',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-gallery-image-caption',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'caption', '=', true ],
		];

		$this->controls['captionBackground'] = [
			'tab'      => 'content',
			'group'    => 'caption',
			'label'    => esc_html__( 'Background', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.mab-gallery-image-caption',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'caption', '=', true ],
		];

		$this->controls['captionBorder'] = [
			'tab'      => 'content',
			'group'    => 'caption',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.mab-gallery-image-caption',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'caption', '=', true ],
		];

		$this->controls['captionTextShadow'] = [
			'tab'      => 'content',
			'group'    => 'caption',
			'label'    => esc_html__( 'Text Shadow', 'max-addons' ),
			'type'     => 'text-shadow',
			'css'      => [
				[
					'property' => 'text-shadow',
					'selector' => '.mab-gallery-image-caption',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'caption', '=', true ],
		];

		$this->controls['captionMargin'] = [
			'tab'      => 'content',
			'group'    => 'caption',
			'label'    => esc_html__( 'Margin', 'max-addons' ),
			'type'     => 'spacing',
			'css'      => [
				[
					'property' => 'margin',
					'selector' => '.mab-gallery-image-caption',
				],
			],
			'required'	=> [ 'caption', '=', true ],
		];

		$this->controls['captionPadding'] = [
			'tab'      => 'content',
			'group'    => 'caption',
			'label'    => esc_html__( 'Padding', 'max-addons' ),
			'type'     => 'spacing',
			'css'      => [
				[
					'property' => 'padding',
					'selector' => '.mab-gallery-image-caption',
				],
			],
			'required'	=> [ 'caption', '=', true ],
		];
	}

	// Set download button controls
	public function set_download_button_controls() {
		$this->controls['downloadButtonInfo'] = [
			'tab'     => 'content',
			'group'   => 'download',
			'type'    => 'info',
			'content' => esc_html__( 'Allow users to download full size images with download button', 'max-addons' ),
		];

		$this->controls['downloadButton'] = [
			'tab'   => 'content',
			'group' => 'download',
			'label' => esc_html__( 'Show download button', 'max-addons' ),
			'type'  => 'checkbox',
		];

		$this->controls['downloadIcon'] = [
			'tab'      => 'content',
			'group'    => 'download',
			'label'    => esc_html__( 'Download icon', 'max-addons' ),
			'type'     => 'icon',
			'inline'   => true,
			'small'    => true,
			'rerender' => true,
			'required' => [ 'downloadButton', '!=', '' ],
		];

		$this->controls['downloadIconJustifyContent'] = [
			'tab'      => 'content',
			'group'    => 'download',
			'label'    => esc_html__( 'Vertical align', 'max-addons' ),
			'type'     => 'justify-content',
			'css'      => [
				[
					'property' => 'justify-content',
					'selector' => '.mab-gallery-download-wrap',
				],
			],
			'exclude'  => 'space',
			'inline'   => true,
			'required' => [
				[ 'downloadButton', '!=', '' ],
				[ 'downloadIcon', '!=', '' ],
			],
		];

		$this->controls['downloadIconAlignContent'] = [
			'tab'      => 'content',
			'group'    => 'download',
			'label'    => esc_html__( 'Horizontal align', 'max-addons' ),
			'type'     => 'align-items',
			'css'      => [
				[
					'property' => 'align-items',
					'selector' => '.mab-gallery-download-wrap',
				],
			],
			'exclude'  => 'stretch',
			'inline'   => true,
			'required' => [
				[ 'downloadButton', '!=', '' ],
				[ 'downloadIcon', '!=', '' ],
			],
		];

		$this->controls['downloadIconTypography'] = [
			'tab'      => 'content',
			'group'    => 'download',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '&{pseudo} .mab-download-icon',
				],
			],
			'exclude'  => [
				'font-family',
				'font-weight',
				'font-style',
				'text-align',
				'text-decoration',
				'text-transform',
				'line-height',
				'letter-spacing',
			],
			'inline'   => true,
			'small'    => true,
			'required' => [
				[ 'downloadButton', '!=', '' ],
				[ 'downloadIcon', '!=', '' ],
			],
		];

		$this->controls['downloadIconBackgroundColor'] = [
			'tab'      => 'content',
			'group'    => 'download',
			'label'    => esc_html__( 'Icon background color', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '&{pseudo} .mab-download-icon',
				],
			],
			'required' => [
				[ 'downloadButton', '!=', '' ],
				[ 'downloadIcon', '!=', '' ],
			],
		];

		$this->controls['downloadIconBorder'] = [
			'tab'      => 'content',
			'group'    => 'download',
			'label'    => esc_html__( 'Icon border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '&{pseudo} .mab-download-icon',
				],
			],
			'required' => [
				[ 'downloadButton', '!=', '' ],
				[ 'downloadIcon', '!=', '' ],
			],
		];

		$this->controls['downloadIconBoxShadow'] = [
			'tab'      => 'content',
			'group'    => 'download',
			'label'    => esc_html__( 'Icon box shadow', 'max-addons' ),
			'type'     => 'box-shadow',
			'css'      => [
				[
					'property' => 'box-shadow',
					'selector' => '&{pseudo} .mab-download-icon',
				],
			],
			'required' => [
				[ 'downloadButton', '!=', '' ],
				[ 'downloadIcon', '!=', '' ],
			],
		];

		$this->controls['downloadIconHeight'] = [
			'tab'      => 'content',
			'group'    => 'download',
			'label'    => esc_html__( 'Icon height', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'line-height',
					'selector' => '&{pseudo} .mab-download-icon',
				],
			],
			'required' => [
				[ 'downloadButton', '!=', '' ],
				[ 'downloadIcon', '!=', '' ],
			],
		];

		$this->controls['downloadIconWidth'] = [
			'tab'      => 'content',
			'group'    => 'download',
			'label'    => esc_html__( 'Icon width', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'width',
					'selector' => '&{pseudo} .mab-download-icon',
				],
			],
			'required' => [
				[ 'downloadButton', '!=', '' ],
				[ 'downloadIcon', '!=', '' ],
			],
		];

		$this->controls['downloadButtonPadding'] = [
			'tab'      => 'content',
			'group'    => 'download',
			'label'    => esc_html__( 'Padding', 'max-addons' ),
			'type'     => 'spacing',
			'css'      => [
				[
					'property' => 'padding',
					'selector' => '.mab-gallery-download-wrap a',
				],
			],
			'required' => [
				[ 'downloadButton', '!=', '' ],
				[ 'downloadIcon', '!=', '' ],
			],
		];
	}

	// Set overlay controls
	public function set_overlay_style_controls() {
		$this->controls['overlayBackground'] = [
			'tab'      => 'content',
			'group'    => 'overlay',
			'label'    => esc_html__( 'Background', 'max-addons' ),
			'type'     => 'gradient',
			'css'      => [
				[
					'property' => 'background-image',
					'selector' => '.mab-image-overlay',
				],
			],
		];
	}

	/**
	 * Image filters.
	 *
	 * @access public
	 * @param boolean $inherit if inherit option required.
	 * @return array Filters.
	 */
	protected function image_filters( $inherit = false ) {

		$inherit_opt = array();

		if ( $inherit ) {
			$inherit_opt = array(
				'' => __( 'Inherit', 'max-addons' ),
			);
		}

		$mab_image_filters = array(
			'normal'           => __( 'Normal', 'max-addons' ),
			'filter-1977'      => __( '1977', 'max-addons' ),
			'filter-aden'      => __( 'Aden', 'max-addons' ),
			'filter-amaro'     => __( 'Amaro', 'max-addons' ),
			'filter-ashby'     => __( 'Ashby', 'max-addons' ),
			'filter-brannan'   => __( 'Brannan', 'max-addons' ),
			'filter-brooklyn'  => __( 'Brooklyn', 'max-addons' ),
			'filter-charmes'   => __( 'Charmes', 'max-addons' ),
			'filter-clarendon' => __( 'Clarendon', 'max-addons' ),
			'filter-crema'     => __( 'Crema', 'max-addons' ),
			'filter-dogpatch'  => __( 'Dogpatch', 'max-addons' ),
			'filter-earlybird' => __( 'Earlybird', 'max-addons' ),
			'filter-gingham'   => __( 'Gingham', 'max-addons' ),
			'filter-ginza'     => __( 'Ginza', 'max-addons' ),
			'filter-hefe'      => __( 'Hefe', 'max-addons' ),
			'filter-helena'    => __( 'Helena', 'max-addons' ),
			'filter-hudson'    => __( 'Hudson', 'max-addons' ),
			'filter-inkwell'   => __( 'Inkwell', 'max-addons' ),
			'filter-juno'      => __( 'Juno', 'max-addons' ),
			'filter-kelvin'    => __( 'Kelvin', 'max-addons' ),
			'filter-lark'      => __( 'Lark', 'max-addons' ),
			'filter-lofi'      => __( 'Lofi', 'max-addons' ),
			'filter-ludwig'    => __( 'Ludwig', 'max-addons' ),
			'filter-maven'     => __( 'Maven', 'max-addons' ),
			'filter-mayfair'   => __( 'Mayfair', 'max-addons' ),
			'filter-moon'      => __( 'Moon', 'max-addons' ),
		);

		return array_merge( $inherit_opt, $mab_image_filters );
	}

	public function is_masonry_layout() {
		$settings = $this->settings;

		if ( ( isset( $settings['layout'] ) && 'masonry' === $settings['layout'] ) || ( isset( $settings['filtersEnable'] ) && $settings['galleryType'] === 'filterable' ) ) {
			return true;
		}

		return false;
	}

	public function render() {
		$settings = $this->settings;
		$layout   = ! empty( $settings['layout'] ) ? $settings['layout'] : 'grid';
		$lightbox_enabled  = ( isset( $this->settings['link_to'] ) && $this->settings['link_to'] === 'lightbox' ) ? true : false;

		$classes = array(
			'mab-image-gallery',
			'mab-image-gallery-' . $layout,
			'bricks-layout-wrapper',
		);

		if ( $this->is_masonry_layout() ) {
			$classes[] = 'isotope';
		}

		$this->set_attribute( 'gallery', 'class', $classes );
		$this->set_attribute( 'gallery', 'data-layout', $layout );

		$mab_gallery_settings = array();
		$mab_tilt_options     = array();
		$fancybox_settings    = array();

		if ( isset( $settings['tilt'] ) ) {
			$mab_tilt_options = array(
				'tilt_enable' => 'yes',
				'tilt_axis'   => isset( $settings['tilt_axis'] ) ? $settings['tilt_axis'] : '',
				'tilt_amount' => isset( $settings['tilt_amount'] ) ? $settings['tilt_amount'] : '20',
				'tilt_scale'  => isset( $settings['tilt_scale'] ) ? $settings['tilt_scale'] : '1.06',
				'tilt_speed'  => isset( $settings['tilt_speed'] ) ? $settings['tilt_speed'] : '800',
			);

			$mab_gallery_settings = array_merge( $mab_gallery_settings, $mab_tilt_options );
		}

		if ( $lightbox_enabled && isset( $settings['lightboxLibrary'] ) && $settings['lightboxLibrary'] === 'bricks' ) {
			if ( ! empty( $settings['lightboxAnimationType'] ) ) {
				$this->set_attribute( '_root', 'data-animation-type', esc_attr( $settings['lightboxAnimationType'] ) );
			}
		}

		if ( isset( $settings['lightboxLibrary'] ) && $settings['lightboxLibrary'] === 'fancybox' ) {
			$fancybox_settings = $this->fancybox_settings();

			$this->set_attribute( '_root', 'data-fancybox-settings', wp_json_encode( $fancybox_settings ) );

			//$mab_gallery_settings = array_merge( $mab_gallery_settings, $fancybox_settings );
		}

		$filter = isset( $settings['thumbnailFilter'] ) ? $settings['thumbnailFilter'] : '';
		$hover_filter = isset( $settings['thumbnailHoverFilter'] ) ? $settings['thumbnailHoverFilter'] : '';
		$caption_hover_effect = isset( $settings['captionHoverEffect'] ) ? $settings['captionHoverEffect'] : '';

		$root_classes = array();

		if ( $filter ) {
			$root_classes[] = 'mab-ins-' . $filter;
		}

		if ( $hover_filter ) {
			$root_classes[] = 'mab-ins-hover-' . $hover_filter;
		}

		if ( $caption_hover_effect ) {
			$root_classes[] = 'mab-caption-hover-effect-' . $caption_hover_effect;
		}

		$this->set_attribute( '_root', 'class', $root_classes );

		if ( ! empty( $mab_gallery_settings ) ) {
			$this->set_attribute( '_root', 'data-settings', wp_json_encode( $mab_gallery_settings ) );
		}

		$image_gallery = $this->get_photos();

		if ( $this->is_masonry_layout() ) {	
			$item_sizer_classes = [ 'bricks-isotope-sizer' ];

			$this->set_attribute( 'item-sizer', 'class', $item_sizer_classes );
		}
		?>
		<div <?php echo wp_kses_post( $this->render_attributes( '_root' ) ); ?>>
			<?php if ( ! empty( $image_gallery ) ) { ?>
				<?php $this->render_filters(); ?>
				<ul <?php echo wp_kses_post( $this->render_attributes( 'gallery' ) ); ?>>
					<?php $this->render_gallery_items(); ?>
				</ul>
			<?php
			} else {
				echo wp_kses_post( $this->render_element_placeholder(
					array(
						'icon-class' => 'ti-gallery',
						'title' => __( 'Gallery is empty!', 'max-addons' ),
					)
				) );
			}
			?>
		</div>
		<?php
	}

	/**
	 * Render filters
	 */
	protected function render_filters() {
		$settings = $this->settings;

		if ( isset( $settings['filtersEnable'] ) && $settings['galleryType'] === 'filterable' ) {
			$all_text = ( '' !== $settings['filterAllLabel'] ) ? $settings['filterAllLabel'] : esc_html__( 'All', 'max-addons' );
			$gallery  = $settings['galleryImages'];
			$default_filter_select = isset($settings['defaultFilterSelect']) ? $settings['defaultFilterSelect'] : 'first';
			$default_filter = isset($settings['defaultFilter']) ? $settings['defaultFilter'] : '';

			$this->set_attribute( 'filters-container', 'class', 'mab-gallery-filters bricks-isotope-filters' );
			$this->set_attribute( 'filters-container', 'id', 'bricks-isotope-filters-' . sanitize_html_class( $this->id ) );

			if ( isset( $settings['pointer'] ) ) {
				if ( 'underline' === $settings['pointer'] || 'overline' === $settings['pointer'] || 'double-line' === $settings['pointer'] ) {
					$this->set_attribute( 'filters-container', 'class', 'mab-pointer-line' );
				}

				$this->set_attribute( 'filters-container', 'class', 'mab-pointer-' . $settings['pointer'] );
				$value = '';
				foreach ( $settings as $key => $value ) {
					
					if ( 0 === strpos( $key, 'animation' ) && $value ) {
						$this->set_attribute( 'filters-container', 'class', 'mab-animation-' . $value );
						break;
					}
				}
			}
			?>
			<ul <?php echo wp_kses_post( $this->render_attributes( 'filters-container' ) ); ?>>
				<?php
				$this->set_attribute( 'all-filter', 'class', 'mab-gallery-filter' );
				$this->set_attribute( 'all-filter', 'data-filter', '*' );
				$this->set_attribute( 'all-filter', 'data-filter-index', 'all' );

				if ( 'first' === $default_filter_select || '' === $default_filter ) {
					$this->set_attribute( 'all-filter', 'class', 'active' );
				}
				?>
				<li <?php echo wp_kses_post( $this->render_attributes( 'all-filter' ) ); ?>>
					<?php echo wp_kses_post( $all_text ); ?>
				</li>
				<?php
				foreach ( $gallery as $index => $item ) {
					$filter_label = isset( $item['filterLabel'] ) ? $item['filterLabel'] : '';
					$filter_id = isset( $item['filterId'] ) ? $item['filterId'] : '';

					if ( empty( $filter_label ) ) {
						$filter_label  = __( 'Group ', 'max-addons' );
						$filter_label .= ( $index + 1 );
					}

					$filter_key = 'filter-'.$index;

					$this->set_attribute( $filter_key, 'class', 'mab-gallery-filter' );
					$this->set_attribute( $filter_key, 'data-filter', '.mab-group-' . ( $index + 1 ) );
					$this->set_attribute( $filter_key, 'data-filter-index', 'mab-group-' . ( $index + 1 ) );

					if ( '' !== $filter_id ) {
						$this->set_attribute( $filter_key, 'id', $filter_id );
					}
					
					if ( 'custom' === $default_filter_select && $filter_label === $default_filter ) {
						
						$this->set_attribute( $filter_key, 'class', 'active' );
						$this->set_attribute( $filter_key, 'data-default', '.mab-group-' . ( $index + 1 ) );
					}
					?>
					<li <?php echo wp_kses_post( $this->render_attributes( $filter_key ) ); ?>><?php echo wp_kses_post( $filter_label ); ?></li>
				<?php } ?>
			</ul>
			<?php
		}
	}

	/**
	 * Render gallery items
	 */
	protected function render_gallery_items() {
		$settings       = $this->settings;
		$photos         = $this->get_photos();
		$count          = 0;
		$thumb_wrap_tag = 'div';

		$layout           = ! empty( $settings['layout'] ) ? $settings['layout'] : 'grid';
		$filter           = isset( $settings['thumbnailFilter'] ) ? $settings['thumbnailFilter'] : '';
		$hover_filter     = isset( $settings['thumbnailHoverFilter'] ) ? $settings['thumbnailHoverFilter'] : '';
		$caption_position = isset( $settings['caption_position'] ) ? $settings['caption_position'] : 'over_image';

		foreach ( $photos as $index => $photo ) {
			$image_html     = '';
			$image_styles   = [];
			$thumb_wrap_key = 'thumb-wrap-' . $count;

			$this->set_attribute( $thumb_wrap_key, 'class', 'mab-image-gallery-thumbnail-wrap' );

			if ( $filter || $hover_filter ) {
				$this->set_attribute( $thumb_wrap_key, 'class', 'mab-ins-filter-hover' );
			}

			if ( isset( $settings['tilt'] ) ) {
				$this->set_attribute( $thumb_wrap_key, 'class', 'mab-gallery-tilt' );
			}

			if ( isset( $settings['filtersEnable'] ) && 'filterable' === $settings['galleryType'] ) {
				$filter_labels      = $this->get_filter_ids( $settings['galleryImages'], true );
				$filter_label       = $filter_labels[ $photo->id ];
				$final_filter_label = preg_replace( '/[^\sA-Za-z0-9]/', '-', $filter_label );
				
				$this->set_attribute( "item-{$index}", 'class', $final_filter_label );
			} else {
				$final_filter_label = '';
			}

			$this->set_attribute( "item-{$index}", 'class', 'bricks-layout-item' );

			if ( isset( $settings['link_to'] ) ) {
				$link      = '';
				$link_attr = '';

				if ( isset( $settings['link_to'] ) ) {
					$thumb_wrap_tag = 'a';

					if ( $settings['link_to'] === 'lightbox' ) {

						$lightbox_image_size = ! empty( $settings['lightboxImageSize'] ) ? $settings['lightboxImageSize'] : 'full';
						$lightbox_image      = wp_get_attachment_image_src( $photo->id, $lightbox_image_size );
						$lightbox_image      = ! empty( $lightbox_image ) && is_array( $lightbox_image ) ? $lightbox_image : $image_src;

						// $caption  = isset( $settings['caption'] ) ? wp_get_attachment_caption( $photo->id ) : false;
						$caption_type = isset( $settings['lightboxCaptionType'] ) ? $settings['lightboxCaptionType'] : 'caption';
						$caption      = $this->get_image_caption( $photo->id, $caption_type );
					
						if ( isset( $settings['lightboxLibrary'] ) && $settings['lightboxLibrary'] === 'fancybox' ) {
							$this->set_attribute( $thumb_wrap_key, 'data-fancybox', 'mab-image-gallery-' . $this->id );
							$this->set_attribute( $thumb_wrap_key, 'data-caption', $caption );
							$this->set_attribute( $thumb_wrap_key, 'href', $lightbox_image[0] );
						} else {
							$this->set_attribute( $thumb_wrap_key, 'class', 'bricks-lightbox' );
							$this->set_attribute( $thumb_wrap_key, 'data-pswp-src', $lightbox_image[0] );
							$this->set_attribute( $thumb_wrap_key, 'data-pswp-width', $lightbox_image[1] );
							$this->set_attribute( $thumb_wrap_key, 'data-pswp-height', $lightbox_image[2] );
							// $this->set_attribute( $thumb_wrap_key, 'data-bricks-lightbox-index', $count );
							$this->set_attribute( $thumb_wrap_key, 'data-pswp-id', $this->id );
							// $this->set_attribute( $thumb_wrap_key, 'data-bricks-lightbox-title', $caption );
						}

					} elseif ( $settings['link_to'] === 'custom' && isset( $settings['linkCustom'][ $count ]['link'] ) ) {

						$this->set_link_attributes( $thumb_wrap_key, $settings['linkCustom'][ $count ]['link'] );

					} elseif ( $settings['link_to'] === 'attachment' ) {

						$link      = get_attachment_link( $photo->id );
						$link_attr = 'href';

						$this->set_attribute( $thumb_wrap_key, $link_attr, $link );

					} elseif ( $settings['link_to'] === 'media' ) {

						$link      = esc_attr( $photo->src );
						$link_attr = 'href';
						
						$this->set_attribute( $thumb_wrap_key, $link_attr, $link );
					}
				}
			}
			?>
			<li <?php echo wp_kses_post( $this->render_attributes( "item-{$index}" ) ); ?>>
				<<?php echo esc_html( $thumb_wrap_tag ); ?> <?php echo wp_kses_post( $this->render_attributes( $thumb_wrap_key ) ); ?>>
					<?php
					$image_classes = [ 'image', 'mab-image-gallery-thumbnail' ];

					if ( isset( $settings['thumbnailFilter'] ) || isset( $settings['thumbnailHoverFilter'] ) ) {
						$image_classes[] = 'mab-ins-filter-target';
					}

					// Image lazy load
					if ( $this->lazy_load() ) {
						$image_classes[] = 'bricks-lazy-hidden';
						$image_classes[] = 'bricks-lazy-load-isotope';
					}

					if ( $layout !== 'masonry' ) {
						$image_classes[] = 'bricks-layout-inner';
					}

					// Grid: Image ratio
					if ( $layout === 'grid' && ! empty( $settings['imageRatio'] ) ) {
						$image_classes[] = "bricks-aspect-{$settings['imageRatio']}";
					}

					$image_atts = [ 'class' => implode( ' ', $image_classes ) ];

					if ( isset( $settings['link_to'] ) && isset( $settings['linkIcon'] ) ) {
						// Link Icon
						$image_html .= isset( $settings['linkIcon'] ) ? self::render_icon( $settings['linkIcon'], [ 'icon' ] ) : false;
					}

					$image_html .= wp_get_attachment_image( $photo->id, $photo->size, false, $image_atts );

					if ( isset( $settings['overlayBackground'] ) ) {
						$image_html .= $this->render_image_overlay( $count );
					}

					if ( '' != $this->render_image_caption( $photo->id ) && 'over_image' === $caption_position ) {
						$image_html .= '<div class="mab-gallery-image-content mab-media-content">';
							// Image Caption.
							$image_html .= $this->render_image_caption( $photo->id );
						$image_html .= '</div>';
					}

					echo $image_html;

					if ( 'below_image' === $caption_position ) {
						?>
						<div class="mab-gallery-image-content">
							<?php
								// Image Caption.
								echo wp_kses_post( $this->render_image_caption( $photo->id ) );
							?>
						</div>
						<?php
					}
					?>
				</<?php echo esc_html( $thumb_wrap_tag ); ?>>
				<?php
				if ( isset( $settings['downloadButton'] ) && isset( $settings['downloadIcon'] ) ) { ?>
					<div class="mab-gallery-download-wrap mab-media-content">
						<?php
							$download_button_key = 'download-' . $count;
							$this->set_attribute( $download_button_key, 'href', $photo->src );
							$this->set_attribute( $download_button_key, 'download', '' );
						?>
						<a <?php echo wp_kses_post( $this->render_attributes( $download_button_key ) ); ?>>
							<?php echo self::render_icon( $settings['downloadIcon'], ['mab-download-icon'] ); ?>
						</a>
					</div>
					<?php
				}
				?>
			</li>
			<?php
			$count++;
		}

		if ( $this->is_masonry_layout() ) {	
			echo "<li {$this->render_attributes( 'item-sizer' )}></li>";
			echo '<li class="bricks-gutter-sizer"></li>';
		}
	}

	/**
	 * Fancybox Settings.
	 *
	 * @return array
	 */
	protected function fancybox_settings() {
		$settings = $this->settings;

		$base_class = 'pp-gallery-fancybox pp-gallery-fancybox-' . esc_attr( $this->id ) . ' ';

		$fancybox_options = array();

		if ( ! isset( $settings['lightboxArrows'] ) ) {
			$fancybox_options['Carousel'] = array(
				'Navigation' => ( isset( $settings['lightboxArrows'] ) ),
			);
		}

		if ( isset( $settings['lightboxToolbar'] ) ) {
			if ( isset( $settings['lightboxToolbarButtonsLeft'] ) ) {
				$toolbar_buttons_left = $settings['lightboxToolbarButtonsLeft'];
			} else {
				$toolbar_buttons_left = array();
			}

			if ( isset( $settings['lightboxToolbarButtonsMiddle'] ) ) {
				$toolbar_buttons_middle = $settings['lightboxToolbarButtonsMiddle'];
			} else {
				$toolbar_buttons_middle = array();
			}

			if ( isset( $settings['lightboxToolbarButtonsRight'] ) ) {
				$toolbar_buttons_right = $settings['lightboxToolbarButtonsRight'];
			} else {
				$toolbar_buttons_right = array();
			}

			$fancybox_options['Toolbar'] = array(
				'display' => array(
					'left'   => $toolbar_buttons_left,
					'middle' => $toolbar_buttons_middle,
					'right'  => $toolbar_buttons_right,
				),
			);
		} else {
			$fancybox_options['Toolbar'] = array(
				'enabled' => false
			);
		}

		$fancybox_options['Thumbs'] = array(
			'showOnStart' => ( isset( $settings['lightboxThumbsAutoStart'] ) ),
			'type'        => ( isset( $settings['lightboxThumbstype'] ) ? $settings['lightboxThumbstype'] : 'modern' ),
		);

		return $fancybox_options;
	}

	/**
	 * Render image caption
	 *
	 * @param  int $id image ID.
	 * @return $html
	 */
	protected function render_image_caption( $id ) {
		$settings = $this->settings;

		if ( ! isset ($settings['caption'] ) ) {
			return '';
		}

		$caption_type = isset($settings[ 'caption_type' ]) ? $settings[ 'caption_type' ] : '';

		$caption = $this->get_image_caption( $id, $caption_type );
		//$caption  = isset( $settings['caption'] ) ? wp_get_attachment_caption( $id ) : false;

		if ( '' === $caption ) {
			return '';
		}

		ob_start();
		?>
		<div class="mab-gallery-image-caption">
			<?php echo wp_kses_post( $caption ); ?>
		</div>
		<?php
		$html = ob_get_contents();
		ob_end_clean();
		return $html;
	}

	/**
	 * Render image overlay
	 *
	 * @param int $count image count
	 */
	protected function render_image_overlay( $count ) {
		$overlay_key = 'overlay-'.$count;

		$this->set_attribute( $overlay_key, 'class', 'mab-image-overlay' );

		return '<div ' . $this->render_attributes( $overlay_key ) . '></div>';
	}

	protected function unique_multidim_array( $array, $key ) {
		$i = 0;
		$temp_array = array();
		$key_array = array();

		foreach( $array as $val ) {
			if ( ! in_array( $val[$key], $key_array ) ) {
				$key_array[$i] = $val[$key];
				$temp_array[$i] = $val;
			}

			$i++;
		}

		return $temp_array;
	}

	/**
	 * Get filter ids
	 *
	 * @param  array $items gallery items array.
	 * @param  bool $get_labels get labels or not.
	 * @return $unique_ids
	 */
	protected function get_filter_ids( $items = array(), $get_labels = false ) {
		$ids    = array();
		$labels = array();

		if ( ! count( $items ) ) {
			return $ids;
		}

		foreach ( $items as $index => $item ) {
			$image_group  = $item['imageGroup'];
			$filter_ids   = array();
			$filter_label = '';

			$size = $image_group['size'];
			if ( ! empty( $image_group['useDynamicData'] ) ) {
				$images = $this->render_dynamic_data_tag( $image_group['useDynamicData'], 'image' );

				if ( is_array( $images ) ) {
					foreach ( $images as $index_1 => $image_id ) {
						$ids[$index . $index_1]['id']   = $image_id;
						$ids[$index . $index_1]['size'] = $size;
						$filter_ids[] = $image_id;
						$filter_label = 'mab-group-' . ( $index + 1 );
					}
				}	
			} else {
				foreach ( $image_group['images'] as $index_1 => $group ) {
					$ids[$index . $index_1]['id']   = $group['id'];
					$ids[$index . $index_1]['size'] = $size;
					$filter_ids[] = $group['id'];
					$filter_label = 'mab-group-' . ( $index + 1 );
				}
			}

			$labels[ $filter_label ] = $filter_ids;
		}

		if ( ! count( $ids ) ) {
			return $ids;
		}

		//$unique_ids = $this->array_unique( $ids );
		$unique_ids = $this->unique_multidim_array( $ids, 'id' );

		if ( $get_labels ) {
			$filter_labels = array();

			foreach ( $unique_ids as $unique_id ) {
				if ( empty( $unique_id ) ) {
					continue;
				}

				foreach ( $labels as $key => $filter_ids ) {
					$unique_id_new = $unique_id['id'];

					if ( in_array( $unique_id_new, $filter_ids ) ) {
						if ( isset( $filter_labels[ $unique_id_new ] ) ) {
							$filter_labels[ $unique_id_new ] = $filter_labels[ $unique_id_new ] . ' ' . str_replace( ' ', '-', strtolower( $key ) );
						} else {
							$filter_labels[ $unique_id_new ] = str_replace( ' ', '-', strtolower( $key ) );
						}
					}
				}
			}

			return $filter_labels;
		}

		return $unique_ids;
	}

	/**
	 * Get WordPress photos
	 *
	 * @return $photos
	 */
	protected function get_wordpress_photos() {
		$settings   = $this->settings;
		$image_size = 'full';
		$photos     = array();
		$ids        = array();
		$photo_ids  = array();

		if ( 'standard' === $settings['galleryType'] && isset( $settings['imageGroupStandard'] ) ) {
			if ( ! count( $settings['imageGroupStandard'] ) ) {
				return $photos;
			}

			$items = isset( $settings['imageGroupStandard'] ) ? $settings['imageGroupStandard'] : [];

			$size = ! empty( $items['size'] ) ? $items['size'] : BRICKS_DEFAULT_IMAGE_SIZE;

			// Dynamic Data
			if ( ! empty( $items['useDynamicData'] ) ) {
				$settings['imageGroupStandard']['images'] = [];

				if ( BRICKS_DB_TEMPLATE_SLUG !== get_post_type( $this->post_id ) ) {
					$size = isset( $settings['imageGroupStandard']['size'] ) ? $settings['imageGroupStandard']['size'] : BRICKS_DEFAULT_IMAGE_SIZE;

					$images = $this->render_dynamic_data_tag( $items['useDynamicData'], 'image' );

					if ( is_array( $images ) ) {
						foreach ( $images as $image_id ) {
							$settings['imageGroupStandard']['images'][] = [
								'id'     => $image_id,
								'size'   => $size,
								'full'   => wp_get_attachment_image_url( $image_id, 'full' ),
								'url'    => wp_get_attachment_image_url( $image_id, $size )
							];
						}
					}

				}
			}
			$photos_arr = $settings['imageGroupStandard'];

			foreach ( $photos_arr['images'] as $index => $ids ) {
				//$photo_ids[] = $ids['id'];
				$photo_ids[$index]['id'] = $ids['id'];
				$photo_ids[$index]['size'] = $size;
			}
		} else {

			if ( ! isset( $settings['galleryImages'] ) ) {
				return $photos;
			}

			$size = ! empty( $settings['galleryImages']['size'] ) ? $settings['galleryImages']['size'] : BRICKS_DEFAULT_IMAGE_SIZE;
			$photo_ids = $this->get_filter_ids( $settings['galleryImages'] );
		}

		if ( 'date' === $settings['ordering'] ) {
			$photo_ids_by_date = array();

			foreach ( $photo_ids as $id ) {
				$date = get_post_time( 'U', '', $id['id'] );
				$photo_ids_by_date[ $date ] = $id;
			}

			$photo_ids = $photo_ids_by_date;

			krsort( $photo_ids );
		}

		foreach ( $photo_ids as $id ) {
			if ( empty( $id ) ) {
				continue;
			}

			$photo_id     = $id['id'];
			$photo_size   = $id['size'];
			$photo        = $this->get_attachment_data( $photo_id );
			$image_src    = wp_get_attachment_image_src( $photo_id, $photo_size );
			$image_width  = ! empty( $image_src[1] ) ? $image_src[1] : 200;
			$image_height = ! empty( $image_src[2] ) ? $image_src[2] : 200;

			if ( ! $photo ) {
				continue;
			}

			// Only use photos who have the sizes object.
			if ( isset( $photo->sizes ) ) {
				$data = new \stdClass();

				// Photo data object.
				$data->id          = $photo_id;
				$data->alt         = $photo->alt;
				$data->caption     = $photo->caption;
				$data->description = $photo->description;
				$data->title       = $photo->title;
				$data->width       = $image_width;
				$data->height      = $image_height;
				$data->size        = $photo_size;

				// Collage photo src.
				if ( 'masonry' === $settings['layout'] ) {
					if ( 'thumbnail' === $image_size && isset( $photo->sizes->thumbnail ) ) {
						$data->src = $photo->sizes->thumbnail->url;
					} elseif ( 'medium' === $image_size && isset( $photo->sizes->medium ) ) {
						$data->src = $photo->sizes->medium->url;
					} else {
						$data->src = $photo->sizes->full->url;
					}
				} else {
					// Grid photo src.
					if ( 'thumbnail' === $image_size && isset( $photo->sizes->thumbnail ) ) {
						$data->src = $photo->sizes->thumbnail->url;
					} elseif ( 'medium' === $image_size && isset( $photo->sizes->medium ) ) {
						$data->src = $photo->sizes->medium->url;
					} else {
						$data->src = $photo->sizes->full->url;
					}
				}

				// Photo Link.
				if ( isset( $photo->sizes->large ) ) {
					$data->link = $photo->sizes->large->url;
				} else {
					$data->link = $photo->sizes->full->url;
				}

				$photos[ $photo_id ] = $data;
			}
		}

		return $photos;
	}

	/**
	 * Get photos
	 *
	 * @return $ordered
	 */
	protected function get_photos() {
		$settings = $this->settings;
		$photos   = $this->get_wordpress_photos();
		$order    = $settings['ordering'];
		$ordered  = array();

		if ( is_array( $photos ) && 'random' === $order ) {
			$keys = array_keys( $photos );
			shuffle( $keys );

			foreach ( $keys as $key ) {
				$ordered[ $key ] = $photos[ $key ];
			}
		} else {
			$ordered = $photos;
		}

		return $ordered;
	}

	/**
	 * Get attachment data
	 *
	 * @param  int $id attachment id.
	 * @return $data
	 */
	protected function get_attachment_data( $id ) {
		$data = wp_prepare_attachment_for_js( $id );

		if ( gettype( $data ) == 'array' ) {
			return json_decode( json_encode( $data ) );
		}

		return $data;
	}

	/**
	 * Get Image Caption.
	 *
	 * @since 1.0.0
     *
	 * @access public
	 *
	 * @return string image caption.
	 */
    public static function get_image_caption( $id, $caption_type = 'caption' ) {

        $attachment = get_post( $id );
        
        $attachment_caption = '';

        if ( $caption_type == 'title' ) {
            $attachment_caption = $attachment->post_title;
        }
        elseif ( $caption_type == 'caption' ) {
            $attachment_caption = wp_get_attachment_caption( $id );
        }

        return $attachment_caption;
	}
        
}
