<?php
namespace MaxAddonsPro\Elements;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Dynamic_Table_Element extends \Bricks\Element {
	// Element properties
	public $category     = 'max-addons-elements'; // Use predefined element category 'general'
	public $name         = 'max-dynamic-table'; // Make sure to prefix your elements
	public $icon         = 'ti-widgetized'; // Themify icon font class
	public $css_selector = ''; // Default CSS selector
	public $header_loop_index = 0;
	public $loop_index   = 0;
	public $scripts      = [ 'mabDynamicTable' ]; // Script(s) run when element is rendered on frontend or updated in builder

	// Return localized element label
	public function get_label() {
		return esc_html__( 'Dynamic Table', 'max-addons' );
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		wp_enqueue_style( 'mab-dynamic-table' );
		wp_enqueue_script( 'gridjs' );
		wp_enqueue_script( 'mab-dynamic-table' );
		wp_enqueue_script( 'mab-frontend' );
	}

	// Set builder control groups
	public function set_control_groups() {
		$this->control_groups['general'] = [ // Unique group identifier (lowercase, no spaces)
			'title' => esc_html__( 'General', 'max-addons' ), // Localized control group title
			'tab'   => 'content', // Set to either "content" or "style"
		];

		$this->control_groups['header'] = [
			'title'    => esc_html__( 'Header', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'source', '!=', 'query' ],
		];

		$this->control_groups['body'] = [
			'title'    => esc_html__( 'Body', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'source', '!=', 'query' ],
		];

		$this->control_groups['dynamicContent'] = [
			'title' => esc_html__( 'Dynamic Content', 'max-addons' ),
			'tab'   => 'content',
			'required' => [ 'source', '=', 'query' ],
		];

		$this->control_groups['search'] = [
			'title' => esc_html__( 'Search', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['pagination'] = [
			'title' => esc_html__( 'Pagination', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['headerStyle'] = [
			'title' => esc_html__( 'Header Style', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['rows'] = [
			'title' => esc_html__( 'Rows Style', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['cells'] = [
			'title' => esc_html__( 'Cells Style', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['iconStyle'] = [
			'title' => esc_html__( 'Icon Style', 'max-addons' ),
			'tab'   => 'content',
		];
	}

	// Set builder controls
	public function set_controls() {

		$this->set_general_controls();

		$this->set_header_controls();

		$this->set_body_controls();

		$this->set_dynamic_content_controls();

		$this->set_search_controls();

		$this->set_pagination_controls();

		$this->set_header_style_controls();

		$this->set_rows_controls();

		$this->set_cells_controls();

		$this->set_icon_controls();
	}

	// Set general controls
	public function set_general_controls() {

		$this->controls['source'] = [
			'tab'       => 'content',
			'group'     => 'general',
			'label'     => esc_html__( 'Source', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'custom' => esc_html__( 'Custom', 'max-addons' ),
				'query'  => esc_html__( 'Query', 'max-addons' ),
			],
			'inline'    => true,
			'clearable' => false,
			'default'   => 'custom',
		];

		$this->controls['query'] = [
			'tab'      => 'content',
			'group'    => 'general',
			'label'    => esc_html__( 'Query', 'max-addons' ),
			'type'     => 'query',
			'popup'    => true,
			'inline'   => true,
			'required' => [ 'source', '=', 'query' ],
		];

		$this->controls['sortable'] = [
			'tab'         => 'content',
			'group'       => 'general',
			'label'       => esc_html__( 'Sortable table', 'max-addons' ),
			'description' => esc_html__( 'Enable sorting the table data by clicking on the table headers', 'max-addons' ),
			'type'        => 'checkbox',
		];

		$breakpoints        = \Bricks\Breakpoints::$breakpoints;
		$breakpoint_options = [];

		foreach ( $breakpoints as $index => $breakpoint ) {
			$breakpoint_options[ $breakpoint['key'] ] = $breakpoint['label'] . ' (<= ' . $breakpoint['width'] . 'px)';
		}

		$breakpoint_options['never']  = esc_html__( 'Never', 'max-addons' );

		$this->controls['stackTable'] = [
			'tab'         => 'content',
			'group'       => 'general',
			'label'       => esc_html__( 'Stack at breakpoint', 'max-addons' ),
			'type'        => 'select',
			'options'     => $breakpoint_options,
			'rerender'    => true,
			'placeholder' => esc_html__( 'Never', 'max-addons' ),
		];

		$this->controls['stackedDisplay'] = [
			'tab'         => 'content',
			'group'       => 'general',
			'label'       => esc_html__( 'Stacked content display', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'row'    => esc_html__( 'Row', 'max-addons' ),
				'column' => esc_html__( 'Column', 'max-addons' ),
			],
			'inline'      => true,
			'clearable'   => true,
			'default'     => 'column',
			'placeholder' => esc_html__( 'Column', 'max-addons' ),
			'required'    => [ 'stackTable', '!=', [ '', 'never' ] ],
		];
	}

	// Set header controls
	public function set_header_controls() {
		$this->controls['columns'] = [
			'tab'           => 'content',
			'group'         => 'header',
			'placeholder'   => esc_html__( 'Column', 'max-addons' ),
			'type'          => 'repeater',
			'titleProperty' => 'cellText',
			'fields'        => [
				'cellText'         => [
					'label'          => esc_html__( 'Text', 'max-addons' ),
					'type'           => 'text',
					'hasDynamicData' => 'text',
				],
				'cellIconType'     => [
					'label'     => esc_html__( 'Icon Type', 'max-addons' ),
					'type'      => 'select',
					'options'   => [
						'icon'  => esc_html__( 'Icon', 'max-addons' ),
						'image' => esc_html__( 'Image', 'max-addons' ),
					],
					'inline'    => true,
					'clearable' => true,
					'default'   => '',
				],
				'cellIcon'         => [
					'label'    => esc_html__( 'Icon', 'max-addons' ),
					'type'     => 'icon',
					'default'  => [
						'library' => 'themify',
						'icon'    => 'ti-wordpress',
					],
					'required' => [ 'cellIconType', '=', [ 'icon' ] ],
				],
				'cellImage'        => [
					'label'    => esc_html__( 'Image', 'max-addons' ),
					'type'     => 'image',
					'required' => [ 'cellIconType', '=', [ 'image' ] ],
				],
				'cellIconPosition' => [
					'label'     => esc_html__( 'Icon Position', 'max-addons' ),
					'type'      => 'select',
					'options'   => [
						'before' => esc_html__( 'Before', 'max-addons' ),
						'after'  => esc_html__( 'After', 'max-addons' ),
					],
					'inline'    => true,
					'clearable' => true,
					'default'   => 'before',
					'required'  => [ 'cellIconType', '=', [ 'image', 'icon' ] ],
				],
				'columnAttributes' => [
					'label'         => esc_html__( 'Attributes', 'max-addons' ),
					'type'          => 'repeater',
					'titleProperty' => 'name',
					'placeholder'   => esc_html__( 'Attribute', 'max-addons' ),
					'fields'        => [
						'name'  => [
							'label'    => esc_html__( 'Name', 'max-addons' ),
							'type'     => 'text',
							'rerender' => false,
						],
						'value' => [
							'label'    => esc_html__( 'Value', 'max-addons' ),
							'type'     => 'text',
							'rerender' => false,
						],
					],
				],
			],
			'default'       => [
				[
					'cellText'         => esc_html__( 'Header Column #1', 'max-addons' ),
					'cellIconType'     => '',
					'cellIcon'         => [
						'library' => 'themify',
						'icon'    => 'ti-wordpress',
					],
					'cellIconPosition' => 'before',
				],
				[
					'cellText'         => esc_html__( 'Header Column #2', 'max-addons' ),
					'cellIconType'     => '',
					'cellIcon'         => [
						'library' => 'themify',
						'icon'    => 'ti-wordpress',
					],
					'cellIconPosition' => 'before',
				],
				[
					'cellText'         => esc_html__( 'Header Column #3', 'max-addons' ),
					'cellIconType'     => '',
					'cellIcon'         => [
						'library' => 'themify',
						'icon'    => 'ti-wordpress',
					],
					'cellIconPosition' => 'before',
				],
			],
		];
	}

	// Set body controls
	public function set_body_controls() {
		$this->controls['bodyRows'] = [
			'tab'         => 'content',
			'group'       => 'body',
			'placeholder' => esc_html__( 'Row', 'max-addons' ),
			'type'        => 'repeater',
			'fields'      => [
				'cell' => [
					'label'         => esc_html__( 'Cells', 'max-addons' ),
					'type'          => 'repeater',
					'placeholder'   => esc_html__( 'Cell', 'max-addons' ),
					'titleProperty' => 'cellText',
					'fields'        => [
						'cellTextType' => [
							'label'     => esc_html__( 'Text type', 'max-addons' ),
							'type'      => 'select',
							'options'   => [
								'basic' => esc_html__( 'Basic text', 'max-addons' ),
								'rich'  => esc_html__( 'Rich text', 'max-addons' ),
							],
							'inline'    => true,
							'clearable' => true,
							'default'   => 'basic',
						],
						'cellText' => [
							'label'          => esc_html__( 'Text', 'max-addons' ),
							'type'           => 'text',
							'hasDynamicData' => 'text',
							'required'       => [ 'cellTextType', '!=', 'rich' ],
						],
						'cellTextRich' => [
							'label'          => esc_html__( 'Text', 'max-addons' ),
							'type'           => 'editor',
							'hasDynamicData' => 'text',
							'required'       => [ 'cellTextType', '=', 'rich' ],
						],
						'link' => [
							'label' => esc_html__( 'Link', 'max-addons' ),
							'type'  => 'link',
						],
						'cellIconType' => [
							'label'     => esc_html__( 'Icon type', 'max-addons' ),
							'type'      => 'select',
							'options'   => [
								'icon'  => esc_html__( 'Icon', 'max-addons' ),
								'image' => esc_html__( 'Image', 'max-addons' ),
							],
							'inline'    => true,
							'clearable' => true,
							'default'   => '',
						],
						'cellIcon' => [
							'label'    => esc_html__( 'Icon', 'max-addons' ),
							'type'     => 'icon',
							'default'  => [
								'library' => 'themify',
								'icon'    => 'ti-wordpress',
							],
							'required' => [ 'cellIconType', '=', [ 'icon' ] ],
						],
						'cellImage' => [
							'label'    => esc_html__( 'Image', 'max-addons' ),
							'type'     => 'image',
							'required' => [ 'cellIconType', '=', [ 'image' ] ],
						],
						'cellIconPosition' => [
							'label'     => esc_html__( 'Icon Position', 'max-addons' ),
							'type'      => 'select',
							'options'   => [
								'before' => esc_html__( 'Before', 'max-addons' ),
								'after'  => esc_html__( 'After', 'max-addons' ),
							],
							'inline'    => true,
							'clearable' => true,
							'default'   => 'before',
							'required'  => [ 'cellIconType', '=', [ 'image', 'icon' ] ],
						],
					],
					'default'       => [
						[
							'cellText' => esc_html__( 'Column 1', 'max-addons' ),
						],
						[
							'cellText' => esc_html__( 'Column 2', 'max-addons' ),
						],
						[
							'cellText' => esc_html__( 'Column 3', 'max-addons' ),
						],
					],
				],
			],
			'default'       => [
				[
					'cell' => '',
				],
			],
		];
	}

	// Set dynamic content controls
	public function set_dynamic_content_controls() {
		$this->controls['dynamicColumns'] = [
			'tab'           => 'content',
			'group'         => 'dynamicContent',
			'placeholder'   => esc_html__( 'Item', 'max-addons' ),
			'type'          => 'repeater',
			'titleProperty' => 'cell',
			'fields'        => [
				'columnHeading'         => [
					'label'          => esc_html__( 'Column Heading', 'max-addons' ),
					'type'           => 'text',
					'hasDynamicData' => 'text',
				],
				'columnData'         => [
					'label'          => esc_html__( 'Column Data', 'max-addons' ),
					'type'           => 'text',
					'hasDynamicData' => 'text',
				],
				'link'             => [
					'label'    => esc_html__( 'Link', 'max-addons' ),
					'type'     => 'link',
				],
				'cellIconType'     => [
					'label'     => esc_html__( 'Icon Type', 'max-addons' ),
					'type'      => 'select',
					'options'   => [
						'icon'  => esc_html__( 'Icon', 'max-addons' ),
						'image' => esc_html__( 'Image', 'max-addons' ),
					],
					'inline'    => true,
					'clearable' => true,
					'default'   => '',
				],
				'cellIcon'         => [
					'label'    => esc_html__( 'Icon', 'max-addons' ),
					'type'     => 'icon',
					'default'  => [
						'library' => 'themify',
						'icon'    => 'ti-wordpress',
					],
					'required' => [ 'cellIconType', '=', [ 'icon' ] ],
				],
				'cellImage'        => [
					'label'    => esc_html__( 'Image', 'max-addons' ),
					'type'     => 'image',
					'required' => [ 'cellIconType', '=', [ 'image' ] ],
				],
				'cellIconPosition' => [
					'label'     => esc_html__( 'Icon Position', 'max-addons' ),
					'type'      => 'select',
					'options'   => [
						'before' => esc_html__( 'Before', 'max-addons' ),
						'after'  => esc_html__( 'After', 'max-addons' ),
					],
					'inline'    => true,
					'clearable' => true,
					'default'   => 'before',
					'required' => [ 'cellIconType', '!=', '' ],
				],
			],
			'default'       => [
				[
					'columnHeading'    => esc_html__( 'Post Title', 'max-addons' ),
					'columnData'       => '{post_title}',
				],
				[
					'columnHeading'    => esc_html__( 'Post Author', 'max-addons' ),
					'columnData'       => '{author_name}',
				],
				[
					'columnHeading'    => esc_html__( 'Post Date', 'max-addons' ),
					'columnData'       => '{post_date}',
				],
			],
			'required' => [ 'source', '=', 'query' ],
		];
	}

	// Set search controls
	public function set_search_controls() {
		$this->controls['search'] = [
			'tab'   => 'content',
			'group' => 'search',
			'label' => esc_html__( 'Search form', 'max-addons' ),
			'type'  => 'checkbox',
		];

		$this->controls['searchInputWidth'] = [
			'tab'      => 'content',
			'group'    => 'search',
			'label'    => esc_html__( 'Width', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'width',
					'selector' => '.gridjs-search',
				],
			],
			'required' => [ 'search', '!=', '' ],
		];

		$this->controls['searchFormAlign'] = [
			'tab'         => 'content',
			'group'       => 'search',
			'label'       => esc_html__( 'Align', 'max-addons' ),
			'type'        => 'text-align',
			'css'         => [
				[
					'property' => 'float',
					'selector' => '.gridjs-search',
				],
			],
			'placeholder' => esc_html__( 'Left', 'max-addons' ),
			'exclude'     => [
				'center',
				'justify'
			],
			'inline'      => true,
			'required'    => [ 'search', '!=', '' ],
		];

		$this->controls['searchPlaceholder'] = [
			'tab'         => 'content',
			'group'       => 'search',
			'type'        => 'text',
			'label'       => esc_html__( 'Placefolder', 'max-addons' ),
			'placeholder' => esc_html__( 'Search...', 'max-addons' ),
			'inline'      => true,
			'required'    => [ 'search', '!=', '' ],
		];

		$this->controls['searchPlaceholderColor'] = [
			'tab'      => 'content',
			'group'    => 'search',
			'label'    => esc_html__( 'Placeholder color', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.gridjs-search-input::placeholder',
				],
			],
			'required' => [ 'search', '!=', '' ],
		];

		$this->controls['searchTypography'] = [
			'tab'      => 'content',
			'group'    => 'search',
			'type'     => 'typography',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.gridjs-search-input',
				],
			],
			'exclude'  => [
				'text-align',
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'search', '!=', '' ],
		];

		$this->controls['searchInputBackgroundColor'] = [
			'tab'      => 'content',
			'group'    => 'search',
			'label'    => esc_html__( 'Background color', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.gridjs-search-input',
				],
			],
			'required' => [ 'search', '!=', '' ],
		];

		$this->controls['searchInputBorder'] = [
			'tab'      => 'content',
			'group'    => 'search',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.gridjs-search-input',
				],
			],
			'required' => [ 'search', '!=', '' ],
		];

		$this->controls['searchInputBoxShadow'] = [
			'tab'      => 'content',
			'group'    => 'search',
			'label'    => esc_html__( 'Box shadow', 'max-addons' ),
			'type'     => 'box-shadow',
			'css'      => [
				[
					'property' => 'box-shadow',
					'selector' => '.gridjs-search-input',
				],
			],
			'required' => [ 'search', '!=', '' ],
		];

		$this->controls['searchInputMargin'] = [
			'tab'   => 'content',
			'group' => 'search',
			'type'  => 'spacing',
			'label' => esc_html__( 'Margin', 'max-addons' ),
			'css'   => [
				[
					'property' => 'margin',
					'selector' => '.gridjs-search-input',
				],
			],
			'required' => [ 'search', '!=', '' ],
		];

		$this->controls['searchInputPadding'] = [
			'tab'   => 'content',
			'group' => 'search',
			'type'  => 'spacing',
			'label' => esc_html__( 'Padding', 'max-addons' ),
			'css'   => [
				[
					'property' => 'padding',
					'selector' => '.gridjs-search-input',
				],
			],
			'required' => [ 'search', '!=', '' ],
		];
	}

	// Set pagination controls
	public function set_pagination_controls() {
		$this->controls['pagination'] = [
			'tab'   => 'content',
			'group' => 'pagination',
			'label' => esc_html__( 'Pagination', 'max-addons' ),
			'type'  => 'checkbox',
		];

		$this->controls['limit'] = [
			'tab'         => 'content',
			'group'       => 'pagination',
			'label'       => esc_html__( 'Rows per page', 'max-addons' ),
			'type'        => 'number',
			'min'         => 1,
			'step'        => 1,
			'inline'      => true,
			'placeholder' => 10,
			'required'    => [ 'pagination', '!=', '' ],
		];

		$this->controls['buttonsCount'] = [
			'tab'         => 'content',
			'group'       => 'pagination',
			'label'       => esc_html__( 'Buttons count', 'max-addons' ),
			'type'        => 'number',
			'min'         => 1,
			'step'        => 1,
			'inline'      => true,
			'placeholder' => 3,
			'required'    => [ 'pagination', '!=', '' ],
		];

		// Pagination Buttons
		$this->controls['paginationButtonsSep'] = [
			'tab'      => 'content',
			'group'    => 'pagination',
			'type'     => 'separator',
			'label'    => esc_html__( 'Buttons', 'max-addons' ),
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['nextButton'] = [
			'tab'     => 'content',
			'group'   => 'pagination',
			'label'   => esc_html__( 'Next button', 'max-addons' ),
			'type'    => 'checkbox',
			'default' => true,
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['nextButtonText'] = [
			'tab'      => 'content',
			'group'    => 'pagination',
			'type'     => 'text',
			'label'    => esc_html__( 'Next button text', 'max-addons' ),
			'placeholder' => esc_html__( 'Next', 'max-addons' ),
			'inline'   => true,
			'required' => [
				[ 'pagination', '!=', '' ],
				[ 'nextButton', '!=', '' ]
			],
		];

		$this->controls['prevButton'] = [
			'tab'     => 'content',
			'group'   => 'pagination',
			'label'   => esc_html__( 'Previous button', 'max-addons' ),
			'type'    => 'checkbox',
			'default' => true,
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['prevButtonText'] = [
			'tab'      => 'content',
			'group'    => 'pagination',
			'type'     => 'text',
			'label'    => esc_html__( 'Previous button text', 'max-addons' ),
			'placeholder' => esc_html__( 'Previous', 'max-addons' ),
			'inline'   => true,
			'required' => [
				[ 'pagination', '!=', '' ],
				[ 'prevButton', '!=', '' ]
			],
		];

		// Summary
		$this->controls['summarySep'] = [
			'tab'      => 'content',
			'group'    => 'pagination',
			'type'     => 'separator',
			'label'    => esc_html__( 'Summary', 'max-addons' ),
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['summary'] = [
			'tab'     => 'content',
			'group'   => 'pagination',
			'label'   => esc_html__( 'Summary', 'max-addons' ),
			'type'    => 'checkbox',
			'default' => true,
			'required' => [ 'pagination', '!=', '' ],
		];

		$this->controls['showingText'] = [
			'tab'      => 'content',
			'group'    => 'pagination',
			'type'     => 'text',
			'label'    => esc_html__( 'Showing text', 'max-addons' ),
			'placeholder' => esc_html__( 'Showing', 'max-addons' ),
			'inline'   => true,
			'required' => [
				[ 'pagination', '!=', '' ],
				[ 'summary', '!=', '' ]
			],
		];

		$this->controls['resultsText'] = [
			'tab'      => 'content',
			'group'    => 'pagination',
			'type'     => 'text',
			'label'    => esc_html__( 'Results text', 'max-addons' ),
			'placeholder' => esc_html__( 'results', 'max-addons' ),
			'inline'   => true,
			'required' => [
				[ 'pagination', '!=', '' ],
				[ 'summary', '!=', '' ]
			],
		];
	}

	// Set header style controls
	public function set_header_style_controls() {

		$this->controls['headerPadding'] = [
			'tab'   => 'content',
			'group' => 'headerStyle',
			'type'  => 'spacing',
			'label' => esc_html__( 'Padding', 'max-addons' ),
			'css'   => [
				[
					'property' => 'padding',
					'selector' => 'th.gridjs-th',
				]
			],
		];

		$this->controls['headerTypography'] = [
			'tab'    => 'content',
			'group'  => 'headerStyle',
			'label'  => esc_html__( 'Typography', 'max-addons' ),
			'type'   => 'typography',
			'css'    => [
				[
					'property' => 'font',
					'selector' => 'th.gridjs-th',
				],
				[
					'property' => 'font',
					'selector' => '.mab-stacked-table td:before',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['headerBgColor'] = [
			'tab'     => 'content',
			'group'   => 'headerStyle',
			'type'    => 'color',
			'label'   => esc_html__( 'Background Color', 'max-addons' ),
			'css'     => [
				[
					'property' => 'background-color',
					'selector' => 'th.gridjs-th',
				],
				[
					'property' => 'background-color',
					'selector' => '.mab-stacked-table td:before',
				],
			],
			'inline'  => true,
			'small'   => true,
		];

		$this->controls['headerBorder'] = [
			'tab'     => 'content',
			'group'   => 'headerStyle',
			'label'   => esc_html__( 'Border', 'max-addons' ),
			'type'    => 'border',
			'css'     => [
				[
					'property' => 'border',
					'selector' => 'th.gridjs-th',
				],
				[
					'property' => 'border',
					'selector' => '.mab-stacked-table td:before',
				],
			],
			'inline'  => true,
			'small'   => true,
		];
	}

	// Set rows controls
	public function set_rows_controls() {
		$this->controls['stripedRows'] = [
			'tab'   => 'content',
			'group' => 'rows',
			'label' => esc_html__( 'Striped Rows', 'max-addons' ),
			'type'  => 'checkbox',
		];

		$this->controls['oddRowsColSeparator'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'type'     => 'separator',
			'label'    => esc_html__( 'Odd Rows', 'max-addons' ),
			'required' => [ 'stripedRows', '!=', '' ],
		];

		$this->controls['rowsBgColor'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'type'     => 'color',
			'label'    => esc_html__( 'Background Color', 'max-addons' ),
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => 'tr',
				],
			],
			'inline'   => true,
			'small'    => true,
		];

		$this->controls['rowsTextColor'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'type'     => 'color',
			'label'    => esc_html__( 'Text Color', 'max-addons' ),
			'css'      => [
				[
					'property' => 'color',
					'selector' => 'tr .mab-table-cell-content',
				],
			],
			'inline'   => true,
			'small'    => true,
		];

		$this->controls['rowsBorder'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => 'tr',
				],
			],
			'inline'   => true,
			'small'    => true,
		];

		// Even Rows Style
		$this->controls['evenRowsColSeparator'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'type'     => 'separator',
			'label'    => esc_html__( 'Even Rows', 'max-addons' ),
			'required' => [ 'stripedRows', '!=', '' ],
		];

		$this->controls['evenRowsBgColor'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'type'     => 'color',
			'label'    => esc_html__( 'Background Color', 'max-addons' ),
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => 'tr:nth-child(even)',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'stripedRows', '!=', '' ],
		];

		$this->controls['evenRowsTextColor'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'type'     => 'color',
			'label'    => esc_html__( 'Text Color', 'max-addons' ),
			'css'      => [
				[
					'property' => 'color',
					'selector' => 'tr:nth-child(even) .mab-table-cell-content',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'stripedRows', '!=', '' ],
		];

		$this->controls['evenRowsBorder'] = [
			'tab'      => 'content',
			'group'    => 'rows',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => 'tr:nth-child(even)',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'stripedRows', '!=', '' ],
		];
	}

	// Set cells style controls
	public function set_cells_controls() {
		$this->controls['cellPadding'] = [
			'tab'   => 'content',
			'group' => 'cells',
			'type'  => 'spacing',
			'label' => esc_html__( 'Padding', 'max-addons' ),
			'css'   => [
				[
					'property' => 'padding',
					'selector' => 'td.gridjs-td > *',
				],
				[
					'property' => 'padding',
					'selector' => '.mab-stacked-table td.gridjs-td:before',
				],
			],
		];

		$this->controls['cellTypography'] = [
			'tab'    => 'content',
			'group'  => 'cells',
			'label'  => esc_html__( 'Typography', 'max-addons' ),
			'type'   => 'typography',
			'css'    => [
				[
					'property' => 'font',
					'selector' => 'td.gridjs-td',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['cellBgColor'] = [
			'tab'    => 'content',
			'group'  => 'cells',
			'type'   => 'color',
			'label'  => esc_html__( 'Background Color', 'max-addons' ),
			'css'    => [
				[
					'property' => 'background-color',
					'selector' => 'td.gridjs-td',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['cellBorder'] = [
			'tab'     => 'content',
			'group'   => 'cells',
			'label'   => esc_html__( 'Border', 'max-addons' ),
			'type'    => 'border',
			'css'     => [
				[
					'property' => 'border',
					'selector' => 'td.gridjs-td',
				],
			],
			'default' => [
				'width' => [
					'top'    => 0,
					'right'  => 1,
					'bottom' => 1,
					'left'   => 0,
				],
				'style' => 'solid',
				'color' => [
					'rgb' => 'rgba(0, 0, 0, 0.1)',
				],
			],
			'inline'  => true,
			'small'   => true,
		];
	}

	// Set columns controls
	public function set_icon_controls() {
		$this->controls['iconSpacing'] = [
			'tab'   => 'content',
			'group' => 'iconStyle',
			'label' => esc_html__( 'Icon Spacing', 'max-addons' ),
			'type'  => 'number',
			'units' => true,
			'css'   => [
				[
					'property' => 'margin-right',
					'selector' => '.mab-table-cell-icon-before',
				],
				[
					'property' => 'margin-left',
					'selector' => '.mab-table-cell-icon-after',
				],
			],
		];

		$this->controls['iconTypography'] = [
			'tab'     => 'content',
			'group'   => 'iconStyle',
			'label'   => esc_html__( 'Icon Typography', 'max-addons' ),
			'type'    => 'typography',
			'css'     => [
				[
					'property' => 'font',
					'selector' => '.mab-table-cell .mab-table-cell-icon',
				],
			],
			'exclude' => [
				'font-family',
				'font-weight',
				'font-style',
				'text-align',
				'letter-spacing',
				'line-height',
				'text-transform',
				'text-decoration',
			],
			'inline'  => true,
			'small'   => true,
		];

		$this->controls['imageSize'] = [
			'tab'     => 'content',
			'group'   => 'iconStyle',
			'label'   => esc_html__( 'Image Size', 'max-addons' ),
			'type'    => 'number',
			'units'   => true,
			'css'     => [
				[
					'property' => 'width',
					'selector' => '.mab-table-cell-icon img',
				],
			],
			'default' => '100px',
		];
	}

	public function get_normalized_image_settings( $settings ) {
		if ( ! isset( $settings['cellImage'] ) ) {
			$settings['cellImage'] = [
				'id'  => 0,
				'url' => '',
			];
			return $settings;
		}

		$image = $settings['cellImage'];

		if ( isset( $image['useDynamicData']['name'] ) ) {
			$images = $this->render_dynamic_data_tag( $image['useDynamicData']['name'], $image );
			$image['id'] = empty( $images ) ? 0 : $images[0];
		} else {
			$image['id'] = isset( $image['id'] ) ? $image['id'] : 0;
		}

		// Image Size
		$image['size'] = isset( $image['size'] ) ? $settings['cellImage']['size'] : BRICKS_DEFAULT_IMAGE_SIZE;

		// Image URL
		if ( ! isset( $image['url'] ) ) {
			$image['url'] = wp_get_attachment_image_url( $image['id'], $image['size'] );
		}

		$settings['cellImage'] = $image;

		return $settings;
	}

	public function render_table_image( $settings ) {

		if ( isset( $settings['cellImage']['useDynamicData']['name'] ) ) {

			if ( empty( $settings['cellImage']['id'] ) ) {

				if ( 'ACF' === $settings['cellImage']['useDynamicData']['group'] && ! class_exists( 'ACF' ) ) {
					$message = esc_html__( 'Can\'t render element, as the selected ACF field is not available. Please activate ACF or edit the element to select different data.', 'max-addons' );
				} elseif ( '{featured_image}' == $settings['cellImage']['useDynamicData']['name'] ) {
					$message = esc_html__( 'No featured image set.', 'max-addons' );
				} else {
					$message = esc_html__( 'Dynamic Data %1$s (%2$s) is empty.', 'max-addons' );
				}

				return $this->render_element_placeholder( [
					'icon-class' => 'ti-image',
					'text'       => sprintf(
						$message,
						$settings['cellImage']['useDynamicData']['label'],
						$settings['cellImage']['useDynamicData']['group']
					),
				] );
			}
		}

		// Image id is empty or doesn't exist
		else {

			// No image
			if ( empty( $settings['cellImage']['id'] ) ) {
				return $this->render_element_placeholder( [
					'icon-class' => 'ti-image',
					'text'       => esc_html__( 'No image selected.', 'max-addons' ),
				] );
			}

			// Return if image ID does not exist
			if ( ! wp_get_attachment_image_src( $settings['cellImage']['id'] ) ) {
				return $this->render_element_placeholder( [
					'icon-class' => 'ti-image',
					'text'       => sprintf( esc_html__( 'Image ID (%s) no longer exist. Please select another image.', 'max-addons' ), $settings['cellImage']['id'] ),
				] );
			}
		}

		// Render
		$img_html = '';
		$image_atts = [];
		$image_atts['id'] = 'image-' . $settings['cellImage']['id'];

		$image_wrapper_classes = [ 'mab-table-icon-image' ];
		$img_classes = [ 'post-thumbnail', 'css-filter' ];

		$img_classes[] = 'size-' . $settings['cellImage']['size'];
		$image_atts['class'] = join( ' ', $img_classes );

		$this->set_attribute( 'image-wrapper', 'class', $image_wrapper_classes );

		$img_html .= '<div ' . $this->render_attributes( 'image-wrapper' ) . '>';

		// Lazy load atts set via 'wp_get_attachment_image_attributes' filter
		if ( isset( $settings['cellImage']['id'] ) ) {
			$img_html .= wp_get_attachment_image( $settings['cellImage']['id'], $settings['cellImage']['size'], false, $image_atts );
		} elseif ( ! empty( $settings['cellImage']['url'] ) ) {
			$img_html .= '<img src="' . $settings['cellImage']['url'] . '">';
		}

		$img_html .= '</div>';

		return $img_html;
	}

	// Table Header
	public function render_header_content( $item, $loop_index ) {
		$settings           = $this->settings;
		$cell_key           = "header_cell_key_$loop_index";
		$cell_text_key      = "header_cell_text_key_$loop_index";
		$cell_icon_wrap_key = "header_icon_wrap_key_$loop_index";
		$cell_icon_key      = "header_icon_key_$loop_index";

		$this->set_attribute( $cell_text_key, 'class', 'mab-table-cell-text' );
		$this->set_attribute( $cell_key, 'class', 'mab-table-cell' );

		$attributes = isset( $item['columnAttributes'] ) ? $item['columnAttributes'] : [];
		$column_attributes = [];

		if ( ! empty( $attributes ) ) {
			foreach ( $attributes as $attribute ) {
				$name  = isset( $attribute['name'] ) ? $attribute['name'] : '';
				$value = isset( $attribute['value'] ) ? $attribute['value'] : '';

				if ( $name ) {
					$column_attributes[ $name ] = $value;
				}
			}

			if ( ! empty( $column_attributes ) ) {
				$this->set_attribute( $cell_key, 'data-column-attributes', wp_json_encode( $column_attributes ) );
			}
		}

		if ( isset( $item['cellIconType'] ) && '' !== $item['cellIconType'] ) {
			$this->set_attribute( $cell_icon_wrap_key, 'class', 'mab-table-cell-icon' );

			$icon_position = isset( $item['cellIconPosition'] ) ? esc_attr( $item['cellIconPosition'] ) : 'before';
			$this->set_attribute( $cell_icon_wrap_key, 'class', 'mab-table-cell-icon-' . $icon_position );

			if ( isset( $item['cellIcon']['icon'] ) ) {
				$this->set_attribute( "icon-$loop_index", 'class', [
					'mab-icon',
					$item['cellIcon']['icon'], // Dont' sanitize_html_class (Font Awesome uses two class names, so we need spaces)
				] );
			}
		}

		echo '<th ' . $this->render_attributes( $cell_key ) . '>';
		if ( isset( $item['cellIconType'] ) && '' != $item['cellIconType'] ) {
			echo '<span ' . $this->render_attributes( $cell_icon_wrap_key ) . '>';
			if ( 'image' === $item['cellIconType'] ) {
				$image_settings = $this->get_normalized_image_settings( $item );
				echo $this->render_table_image( $image_settings );
			} elseif ( 'icon' === $item['cellIconType'] ) {
				if ( isset( $item['cellIcon']['icon'] ) ) {
					echo '<i ' . wp_kses_post( $this->render_attributes( "icon-$loop_index" ) ) . '></i>';
				}
			}
			echo '</span>';
		}
		echo '<span ' . $this->render_attributes( $cell_text_key ) . '>';
		if ( isset( $settings['source'] ) && 'query' === $settings['source'] ) {
			if ( isset( $item['columnHeading'] ) ) {
				echo $item['columnHeading'];
			}
		} else {
			if ( isset( $item['cellText'] ) ) {
				echo $this->render_dynamic_data( $item['cellText'] );
			}
		}
		echo '</span>';
		echo '</th>';
	}

	// Table Header
	public function render_header_custom( $item ) {
		$settings   = $this->settings;
		$loop_index = $this->header_loop_index;
		ob_start();

		self::render_header_content( $item, $loop_index );

		$html = ob_get_contents();
		ob_end_clean();
		$this->header_loop_index++;

		return $html;
	}

	// Table Header
	public function render_header_dynamic( $item ) {
		$settings   = $this->settings;
		$loop_index = $this->header_loop_index;
		ob_start();

		$columns = empty( $settings['dynamicColumns'] ) ? false : $settings['dynamicColumns'];

		foreach ( $columns as $index => $item ) {

			if ( 0 == $loop_index ) {

				if ( isset( $item['columnHeading'] ) || '' !== $item['cellIconType'] ) {
					self::render_header_content( $item, $index );
				}
			}
		}
		
		$html = ob_get_contents();
		ob_end_clean();
		$this->header_loop_index++;

		return $html;
	}

	// Table Header
	protected function render_header() {
		$settings = $this->settings;
		$i = 1;
		$this->set_attribute( 'row', 'class', 'mab-table-row' );

		$columns = empty( $settings['dynamicColumns'] ) ? false : $settings['dynamicColumns'];
		?>
		<thead>
			<tr <?php echo $this->render_attributes( 'row' ); ?>>
				<?php
				$output = '';

				// Query Loop
				if ( isset( $settings['source'] ) && 'query' === $settings['source'] ) {

					$query = new \Bricks\Query( [
						'id'       => $this->id,
						'settings' => $settings,
					] );

					$item = $columns[0];

					$output .= $query->render( [ $this, 'render_header_dynamic' ], compact( 'item' ) );

					// We need to destroy the Query to explicitly remove it from the global store
					$query->destroy();
					unset( $query );
				} else {
					if ( ! empty( $settings['columns'] ) ) {
						foreach ( $settings['columns'] as $index => $item ) {
							$output .= self::render_header_custom( $item );
						}
					}
				}

				echo $output;
				?>
			</tr>
		</thead>
		<?php
	}

	// Table Body
	protected function render_body() {
		$settings = $this->settings;
		$i = 1;

		$columns_dynamic = empty( $settings['dynamicColumns'] ) ? false : $settings['dynamicColumns'];
		$columns_static  = ! empty( $settings['columns'] ) ? $settings['columns'] : [];
		?>
		<tbody>
			<?php
			$output = '';

			// Query Loop
			if ( isset( $settings['source'] ) && 'query' === $settings['source'] ) {

				$query = new \Bricks\Query( [
					'id'       => $this->id,
					'settings' => $settings,
				] );

				$item = $columns_dynamic[0];

				$output .= $query->render( [ $this, 'render_body_content_dynamic' ], compact( 'item' ) );

				// We need to destroy the Query to explicitly remove it from the global store
				$query->destroy();
				unset( $query );
			} else {
				if ( ! empty( $settings['bodyRows'] ) ) {
					foreach ( $settings['bodyRows'] as $index => $item ) {
						$output .= '<tr class="mab-table-row">';
						if ( ! empty( $item['cell'] ) ) {
							foreach ( array_slice( $item['cell'], 0, sizeof( $columns_static ) ) as $key => $cell ) {
								$output .= self::render_body_content_custom( $cell, $columns_static );
							}

							if ( sizeof( $item['cell'] ) < sizeof( $columns_static ) ) {
								$output .= str_repeat( '<td></td>', sizeof( $columns_static ) - sizeof( $item['cell'] ) );
							}
						} else {
							$output .= str_repeat( '<td></td>', 3 );
						}
						$output .= '</tr>';
					}
				}
			}

			echo $output;
			echo '</tr>';
			?>
		</tbody>
		<?php
	}

	// Table Body
	public function render_body_content_custom( $item, $columns_static ) {
		$settings   = $this->settings;
		$loop_index = $this->loop_index;

		ob_start();
			$row_key            = "body_row_key_$loop_index";
			$cell_key           = "body_cell_key_$loop_index";
			$cell_text_key      = "body_cell_text_key_$loop_index";
			$cell_icon_wrap_key = "body_icon_wrap_key_$loop_index";
			$cell_icon_key      = "body_icon_key_$loop_index";

			$this->set_attribute( $cell_text_key, 'class', 'mab-table-cell-text' );
			$this->set_attribute( $row_key, 'class', 'mab-table-row' );
			$this->set_attribute( $cell_key, 'class', 'mab-table-cell' );

			if ( isset( $item['cellIconType'] ) && '' !== $item['cellIconType'] ) {
				$this->set_attribute( $cell_icon_wrap_key, 'class', 'mab-table-cell-icon' );

				$icon_position = isset( $item['cellIconPosition'] ) ? esc_attr( $item['cellIconPosition'] ) : 'before';
				$this->set_attribute( $cell_icon_wrap_key, 'class', 'mab-table-cell-icon-' . $icon_position );

				if ( isset( $item['cellIcon']['icon'] ) ) {
					$this->set_attribute( "icon-$loop_index", 'class', [
						'mab-icon',
						$item['cellIcon']['icon'], // Dont' sanitize_html_class (Font Awesome uses two class names, so we need spaces)
					] );
				}
			}

			if ( isset( $item['link'] ) ) {
				$this->set_link_attributes( 'col-link-' . $loop_index, $item['link'] );
			}

			if ( isset( $item['cellText'] ) || isset( $item['cellIconType'] ) ) {
				echo '<td ' . $this->render_attributes( $cell_key ) . '>';

				if ( isset( $item['link'] ) ) { ?>
					<a <?php echo $this->render_attributes( 'col-link-' . $loop_index ); ?>>
					<?php
				}

				if ( isset( $item['cellIconType'] ) && '' !== $item['cellIconType'] ) {
					echo '<span ' . $this->render_attributes( $cell_icon_wrap_key ) . '>';
					if ( 'icon' === $item['cellIconType'] ) {
						if ( isset( $item['cellIcon']['icon'] ) ) {
							echo '<i ' . wp_kses_post( $this->render_attributes( "icon-$loop_index" ) ) . '></i>';
						}
					} elseif ( 'image' === $item['cellIconType'] ) {
						$image_settings = $this->get_normalized_image_settings( $item );
						echo $this->render_table_image( $image_settings );
					}
					echo '</span>';
				}

				$cell_text      = '';
				$cell_text_type = isset( $item['cellTextType'] ) ? $item['cellTextType'] : 'basic';

				if ( $cell_text_type === 'rich' && isset( $item['cellTextRich'] ) ) {
					$cell_text = $item['cellTextRich'];
				} elseif ( isset( $item['cellText'] ) ) {
					$cell_text = $item['cellText'];
				}

				if ( $cell_text ) {
					echo '<span ' . $this->render_attributes( $cell_text_key ) . '>';
					echo $this->render_dynamic_data( $cell_text );
					echo '</span>';
				}

				if ( isset( $item['link'] ) ) { ?>
					</a>
					<?php
				}

				echo '</td>';
			}

			$html = ob_get_contents();
		ob_end_clean();

		$this->loop_index++;

		return $html;
	}

	// Table Body
	public function render_body_content_dynamic( $item ) {
		$settings = $this->settings;
		$i = 1;
		$loop_index = $this->loop_index;
		ob_start();

		$row_key            = "body_row_key_$loop_index";
		$cell_key           = "body_cell_key_$loop_index";
		$cell_text_key      = "body_cell_text_key_$loop_index";
		$cell_icon_wrap_key = "body_icon_wrap_key_$loop_index";
		$cell_icon_key      = "body_icon_key_$loop_index";
		$link_key           = "body_link_key_$loop_index";

		$this->set_attribute( $cell_text_key, 'class', 'mab-table-cell-text' );
		$this->set_attribute( $row_key, 'class', 'mab-table-row' );
		$this->set_attribute( $cell_key, 'class', 'mab-table-cell' );

		$columns = empty( $settings['dynamicColumns'] ) ? false : $settings['dynamicColumns'];

		if ( 1 === $i ) {
			echo '<tr ' . $this->render_attributes( $row_key ) . '>';
		} elseif ( $i > 1 ) {
			echo '</tr><tr ' . $this->render_attributes( $row_key ) . '>';
		}

		foreach ( $columns as $index => $item ) {

			if ( isset( $item['cellIconType'] ) && '' !== $item['cellIconType'] ) {
				$this->set_attribute( $cell_icon_wrap_key, 'class', 'mab-table-cell-icon' );

				$icon_position = isset( $item['cellIconPosition'] ) ? esc_attr( $item['cellIconPosition'] ) : 'before';
				$this->set_attribute( $cell_icon_wrap_key, 'class', 'mab-table-cell-icon-' . $icon_position );

				if ( isset( $item['cellIcon']['icon'] ) ) {
					$this->set_attribute( $cell_icon_key, 'class', [
						'mab-icon',
						$item['cellIcon']['icon'], // Dont' sanitize_html_class (Font Awesome uses two class names, so we need spaces)
					] );
				}
			}

			if ( isset( $item['link'] ) ) {
				$this->set_link_attributes( $link_key, $item['link'] );
			}

			if ( isset( $item['columnData'] ) || '' !== $item['cellIconType'] ) {
				echo '<td ' . $this->render_attributes( $cell_key ) . '>';
				if ( isset( $item['link'] ) ) { ?>
					<a <?php echo $this->render_attributes( $link_key ); ?>>
					<?php
				}
				if ( isset( $item['cellIconType'] ) && '' !== $item['cellIconType'] ) {
					echo '<span ' . $this->render_attributes( $cell_icon_wrap_key ) . '>';
					if ( 'icon' === $item['cellIconType'] ) {
						if ( isset( $item['cellIcon']['icon'] ) ) {
							echo '<i ' . wp_kses_post( $this->render_attributes( $cell_icon_key ) ) . '></i>';
						}
					} elseif ( 'image' === $item['cellIconType'] ) {
						$image_settings = $this->get_normalized_image_settings( $item );
						echo $this->render_table_image( $image_settings );
					}
					echo '</span>';
				}
				echo '<span ' . $this->render_attributes( $cell_text_key ) . '>';
				echo $item['columnData'];
				echo '</span>';
				if ( isset( $item['link'] ) ) { ?>
					</a>
					<?php
				}
				echo '</td>';
			}

			$i++;
		}
		
		$html = ob_get_contents();
		ob_end_clean();
		$this->loop_index++;

		return $html;
	}

	// Render element HTML
	public function render() {
		$settings = $this->settings;

		$this->set_attribute( 'table', 'class', [ 'mab-table' ] );

		$table_settings = array();
		
		if ( isset( $settings['sortable'] ) ) {
			$table_settings['sortable'] = true;
		}

		if ( isset( $settings['search'] ) ) {
			$table_settings['search'] = true;

			$table_settings['search_placeholder'] = isset( $settings['searchPlaceholder'] ) ? $settings['searchPlaceholder'] : esc_html__( 'Search...', 'max-addons' );
		}

		if ( isset( $settings['pagination'] ) ) {
			$table_settings['pagination'] = true;
		}

		if ( isset( $settings['limit'] ) ) {
			$table_settings['limit'] = $settings['limit'];
		}

		if ( isset( $settings['buttonsCount'] ) ) {
			$table_settings['buttons_count'] = $settings['buttonsCount'];
		}

		if ( isset( $settings['summary'] ) ) {
			$table_settings['summary'] = $settings['summary'];

			if ( isset( $settings['showingText'] ) ) {
				$table_settings['showing_text'] = $settings['showingText'];
			}

			if ( isset( $settings['resultsText'] ) ) {
				$table_settings['results_text'] = $settings['resultsText'];
			}
		}

		if ( isset( $settings['nextButton'] ) ) {
			$table_settings['next_button'] = $settings['nextButton'];

			if ( isset( $settings['nextButtonText'] ) ) {
				$table_settings['next_button_text'] = $settings['nextButtonText'];
			}
		}

		if ( isset( $settings['prevButton'] ) ) {
			$table_settings['prev_button'] = $settings['prevButton'];

			if ( isset( $settings['prevButtonText'] ) ) {
				$table_settings['prev_button_text'] = $settings['prevButtonText'];
			}
		}

		if ( isset( $settings['stackTable'] ) && $settings['stackTable'] !== 'never' ) {
			$breakpoint = \Bricks\Breakpoints::get_breakpoint_by( 'key', $settings['stackTable'] );

			$table_settings['stack_on'] = $breakpoint['width'];

			if ( isset( $settings['stackedDisplay'] ) && $settings['stackedDisplay'] === 'row' ) {
				$this->set_attribute( '_root', 'class', 'mab-stacked-table-row' );
			}
		}

		$this->set_attribute( '_root', 'data-settings', wp_json_encode( $table_settings ) );
		?>
		<div <?php echo wp_kses_post( $this->render_attributes( '_root' ) ); ?>>
			<div class="mab-dynamic-table-container"></div>
			<table <?php echo $this->render_attributes( 'table' ); ?>>
				<?php
					$this->render_header();

					$this->render_body();
				?>
			</table>
		</div>
		<?php
	}
}
