<?php
namespace MaxAddonsPro\Elements;

use MaxAddonsPro\Classes\Helper;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Breadcrumbs_Element extends \Bricks\Element {
	// Element properties
	public $category     = 'max-addons-elements'; // Use predefined element category 'general'
	public $name         = 'max-breadcrumbs'; // Make sure to prefix your elements
	public $icon         = 'ti-angle-double-right max-element'; // Themify icon font class
	public $css_selector = ''; // Default CSS selector
	public $scripts      = []; // Script(s) run when element is rendered on frontend or updated in builder

	/**
	 * Crumbs
	 *
	 * @since  1.3.0
	 * @var    string
	 */
	protected $crumbs = [];

	/**
	 * Separator
	 *
	 * @since  1.3.0
	 * @var    string
	 */
	private $_separator = null;

	public function get_label() {
		return esc_html__( 'Breadcrumbs', 'max-addons' );
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		wp_enqueue_style( 'mab-breadcrumbs' );
	}

	// Get filtered taxonomies
	public function get_filtered_taxonomies() {
		$taxonomies = Helper::get_taxonomies_options();
		$filtered_taxonomies = array();

		foreach ( $taxonomies as $taxonomy => $label ) {
			if ( 'template_tag' === $taxonomy || 'template_bundle' === $taxonomy ) {
				continue;
			}

			$filtered_taxonomies[$taxonomy] = $label;
		}

		return $filtered_taxonomies;
	}

	public function set_control_groups() {
		$this->control_groups['settings'] = [
			'title' => esc_html__( 'Breadcrumbs', 'max-addons' ),
			'tab'   => 'content',
		];

		$post_types = Helper::get_post_types();

		foreach ( $post_types as $post_type => $label ) {
			$this->control_groups['single_' . $post_type] = [
				'title'    => sprintf( esc_html__( 'Single %s', 'max-addons' ), $label ),
				'tab'      => 'content',
				'required' => [ 'type', '=', [ 'custom' ] ],
			];
		}

		$taxonomies = $this->get_filtered_taxonomies();

		foreach ( $taxonomies as $taxonomy => $label ) {
			$this->control_groups['taxonomy_' . $taxonomy] = [
				'title'    => sprintf( esc_html__( '%s Archive', 'max-addons' ), $label ),
				'tab'      => 'content',
				'required' => [ 'type', '=', [ 'custom' ] ],
			];
		}

		$this->control_groups['separator'] = [
			'title'    => esc_html__( 'Separator', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'type', '=', [ 'custom' ] ],
		];
	}

	public function set_controls() {
		$this->set_general_controls();
		$this->set_single_controls();
		$this->set_archive_controls();
		$this->set_separator_controls();
	}

	public function set_general_controls() {
		$this->controls['type'] = [
			'tab'       => 'content',
			'group'     => 'settings',
			'label'     => esc_html__( 'Type', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'custom'   => __( 'Custom', 'max-addons' ),
				'yoast'    => __( 'Yoast', 'max-addons' ),
				'rankmath' => __( 'Rankmath', 'max-addons' ),
				'seopress' => __( 'SEOPress', 'max-addons' ),
				'navxt'    => __( 'Breadcrumb NavXT', 'max-addons' ),
			],
			'default'   => 'custom',
			'inline'    => true,
			'clearable' => false,
		];

		$this->controls['source'] = [
			'tab'       => 'content',
			'group'     => 'settings',
			'label'     => esc_html__( 'Source', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'query' => __( 'Current Query', 'max-addons' ),
				'id'    => __( 'Custom Selection', 'max-addons' ),
			],
			'default'   => 'query',
			'inline'    => true,
			'clearable' => false,
			'required'  => [ 'type', '=', [ 'custom' ] ],
		];

		$this->controls['sourceId'] = [
			'tab'         => 'content',
			'group'       => 'settings',
			'label'       => esc_html__( 'Select Page or Post', 'max-addons' ),
			'type'        => 'select',
			'optionsAjax' => [
				'action'   => 'bricks_get_posts',
				'postType' => 'any',
			],
			'searchable'  => true,
			'required'    => [ 'source', '=', [ 'id' ] ],
		];

		$this->controls['showHome'] = [
			'tab'      => 'content',
			'group'    => 'settings',
			'label'    => esc_html__( 'Show Home', 'max-addons' ),
			'type'     => 'checkbox',
			'default'  => true,
			'required' => [ 'type', '=', [ 'custom' ] ],
		];

		$this->controls['showCurrent'] = [
			'tab'      => 'content',
			'group'    => 'settings',
			'label'    => esc_html__( 'Show Current', 'max-addons' ),
			'type'     => 'checkbox',
			'default'  => true,
			'required' => [ 'type', '=', [ 'custom' ] ],
		];

		$this->controls['structuredData'] = [
			'tab'      => 'content',
			'group'    => 'settings',
			'label'    => esc_html__( 'Add Structured Data', 'max-addons' ),
			'type'     => 'checkbox',
			'default'  => true,
			'required' => [ 'type', '=', [ 'custom' ] ],
		];

		$this->controls['homeText'] = [
			'tab'      => 'content',
			'group'    => 'settings',
			'label'    => esc_html__( 'Home Text', 'max-addons' ),
			'type'     => 'text',
			'default'  => __( 'Homepage', 'max-addons' ),
			'inline'   => true,
			'required' => [ 'type', '=', [ 'custom' ] ],
		];
	}

	public function set_single_controls() {
		$post_types = Helper::get_post_types();

		foreach ( $post_types as $post_type => $label ) {
			$this->controls['single_' . $post_type . '_show_home'] = [
				'tab'      => 'content',
				'group'    => 'single_' . $post_type,
				'label'    => esc_html__( 'Show Home', 'max-addons' ),
				'type'     => 'checkbox',
				'default'  => true,
				'required' => [ 'type', '=', [ 'custom' ] ],
			];

			$taxonomies_options = Helper::get_taxonomies_options( $post_type );

			foreach ( $taxonomies_options as $taxonomy => $label ) {
				if ( $taxonomy && ! is_taxonomy_hierarchical( $taxonomy ) ) {
					unset( $taxonomies_options[ $taxonomy ] );
				}
			}

			$tax_options_default = ( $taxonomies_options && ! empty( $taxonomies_options[0] ) ) ? array_keys( $taxonomies_options )[0] : '';

			$this->controls['single_' . $post_type . '_show_terms'] = [
				'tab'      => 'content',
				'group'    => 'single_' . $post_type,
				'label'    => esc_html__( 'Taxonomy', 'max-addons' ),
				'type'     => 'select',
				'options'  => array_merge( [
					'none' => __( 'None', 'max-addons' ),
				], $taxonomies_options ),
				'required' => [
					[ 'type', '=', [ 'custom' ] ],
					[ 'showCurrent', '!=', '' ]
				],
			];

			if ( is_post_type_hierarchical( $post_type ) ) {
				$this->controls['single_' . $post_type . '_show_parents'] = [
					'tab'      => 'content',
					'group'    => 'single_' . $post_type,
					'label'    => esc_html__( 'Show Parents', 'max-addons' ),
					'type'     => 'checkbox',
					'default'  => true,
					'required' => [
						[ 'type', '=', [ 'custom' ] ],
						[ 'showCurrent', '!=', '' ]
					],
				];
			}

			$this->controls['single_' . $post_type . '_show_current'] = [
				'tab'      => 'content',
				'group'    => 'single_' . $post_type,
				'label'    => esc_html__( 'Show Current', 'max-addons' ),
				'type'     => 'checkbox',
				'default'  => true,
				'required' => [
					[ 'type', '=', [ 'custom' ] ],
					[ 'showCurrent', '!=', '' ]
				],
			];
		}
	}

	public function set_archive_controls() {
		$taxonomies = $this->get_filtered_taxonomies();

		foreach ( $taxonomies as $taxonomy => $label ) {
			$this->controls['taxonomy_' . $taxonomy . '_show_home'] = [
				'tab'      => 'content',
				'group'    => 'taxonomy_' . $taxonomy,
				'label'    => esc_html__( 'Show Home', 'max-addons' ),
				'type'     => 'checkbox',
				'default'  => true,
				'required' => [ 'type', '=', [ 'custom' ] ],
			];

			$this->controls['taxonomy_' . $taxonomy . '_show_taxonomy'] = [
				'tab'      => 'content',
				'group'    => 'taxonomy_' . $taxonomy,
				'label'    => esc_html__( 'Show Taxonomy', 'max-addons' ),
				'type'     => 'checkbox',
				'default'  => 'category' === $taxonomy || 'post_tag' === $taxonomy ? true : '',
				'required' => [ 'type', '=', [ 'custom' ] ],
			];

			$this->controls['taxonomy_' . $taxonomy . '_taxonomy_link'] = [
				'tab'      => 'content',
				'group'    => 'taxonomy_' . $taxonomy,
				'label'    => esc_html__( 'Taxonomy Link', 'max-addons' ),
				'type'     => 'text',
				'inline'   => true,
				'required' => [
					[ 'type', '=', [ 'custom' ] ],
					[ 'taxonomy_' . $taxonomy . '_show_taxonomy', '!=', '' ]
				],
			];

			if ( is_taxonomy_hierarchical( $taxonomy ) ) {
				$this->controls['taxonomy_' . $taxonomy . '_show_parents'] = [
					'tab'      => 'content',
					'group'    => 'taxonomy_' . $taxonomy,
					'label'    => sprintf( __( 'Show Parent %s', 'max-addons' ), $label ),
					'type'     => 'checkbox',
					'default'  => true,
					'required' => [ 'type', '=', [ 'custom' ] ],
				];
			}

			$this->controls['taxonomy_' . $taxonomy . '_show_current'] = [
				'tab'      => 'content',
				'group'    => 'taxonomy_' . $taxonomy,
				'label'    => esc_html__( 'Show Current', 'max-addons' ),
				'type'     => 'checkbox',
				'default'  => true,
				'required' => [ 'type', '=', [ 'custom' ] ],
			];
		}
	}

	public function set_separator_controls() {
		$this->controls['separatorType'] = [
			'tab'       => 'content',
			'group'     => 'separator',
			'label'     => esc_html__( 'Separator', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'text' => __( 'Text', 'max-addons' ),
				'icon' => __( 'Icon', 'max-addons' ),
			],
			'default'   => 'icon',
			'inline'    => true,
			'clearable' => true,
			'required'  => [ 'type', '=', [ 'custom' ] ],
		];

		$this->controls['separatorText'] = [
			'tab'      => 'content',
			'group'    => 'separator',
			'label'    => esc_html__( 'Text', 'max-addons' ),
			'type'     => 'text',
			'default'  => '>',
			'inline'   => true,
			'required' => [
				[ 'type', '=', 'custom' ],
				[ 'separatorType', '=', 'text' ]
			],
		];

		$this->controls['separatorIcon'] = [
			'tab'      => 'content',
			'group'    => 'separator',
			'label'    => esc_html__( 'Icon', 'max-addons' ),
			'type'     => 'icon',
			'default'  => [
				'library' => 'themify',
				'icon'    => 'ti-angle-right',
			],
			'inline'   => true,
			'required' => [
				[ 'type', '=', 'custom' ],
				[ 'separatorType', '=', 'icon' ]
			],
		];

		$this->controls['separatorTypography'] = [
			'tab'      => 'content',
			'group'    => 'separator',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-breadcrumbs-separator',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'type', '=', 'custom' ],
		];

		$this->controls['separatorBackground'] = [
			'tab'      => 'content',
			'group'    => 'separator',
			'label'    => esc_html__( 'Background', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.mab-breadcrumbs-separator',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'type', '=', 'custom' ],
		];

		$this->controls['separatorBorder'] = [
			'tab'      => 'content',
			'group'    => 'separator',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.mab-breadcrumbs-separator',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'type', '=', 'custom' ],
		];

		$this->controls['separatorPadding'] = [
			'tab'      => 'content',
			'group'    => 'separator',
			'label'    => esc_html__( 'Padding', 'max-addons' ),
			'type'     => 'spacing',
			'css'      => [
				[
					'property' => 'padding',
					'selector' => '.mab-breadcrumbs-separator',
				],
			],
			'required' => [ 'type', '=', 'custom' ],
		];
	}

	/**
	 * Get Query
	 *
	 * @since  1.3.0
	 * @return \WP_Query|bool
	 */
	public function get_query() {
		$settings = $this->settings;

		global $post;

		$_id        = null;
		$_post_type = 'post';

		if ( isset( $settings['source'] ) && 'id' === $settings['source'] && ! empty( $settings['sourceId'] ) ) {
			$_id = $settings['sourceId'];
			$_post_type = 'any';

			$_args = array(
				'p'         => $_id,
				'post_type' => $_post_type,
			);

			// Create custom query
			$_post_query = new \WP_Query( $_args );

			return $_post_query;
		}

		return false;
	}

	/**
	 * Set Separator
	 *
	 * Sets the markup for the breadcrumbs separator
	 *
	 * @since  1.3.0
	 * @return string
	 */
	public function set_separator() {
		$settings  = $this->settings;
		$separator = '';

		if ( isset( $settings['separatorType'] ) && 'icon' === $settings['separatorType'] ) {
			$this->set_attribute( 'icon-wrapper', 'class', 'mab-icon' );

			$separator .= '<span ' . $this->render_attributes( 'icon-wrapper' ) . '>';
			$separator .= isset( $settings['separatorIcon'] ) ? self::render_icon( $settings['separatorIcon'] ) : '';
			$separator .= '</span>';
		} else {
			$this->set_attribute( 'separator_text', 'class', 'mab-breadcrumbs-separator-text' );

			$separator_text = !empty( $settings['separatorText'] ) ? $settings['separatorText'] : '';
			$separator = '<span ' . $this->render_attributes( 'separator_text' ) . '>' . $separator_text . '</span>';
		}

		$this->_separator = $separator;
	}

	/**
	 * Get Separator
	 *
	 * @since  1.3.0
	 * @return var\string
	 */
	public function get_separator() {
		return $this->_separator;
	}

	/**
	 * Render Separator
	 * 
	 * The markup for the separator item
	 *
	 * @since  1.3.0
	 * @param  bool $output Wether to echo or not the markup
	 * @return void
	 */
	public function render_separator( $output = true ) {
		$this->set_attribute( 'separator', 'class', 'mab-breadcrumbs-separator' );

		$separator = $this->get_separator();

		$markup = sprintf( '<li %1$s>%2$s</li>', $this->render_attributes( 'separator' ), $separator );

		if ( $output === true ) {
			echo $markup;
		} else {
			return $markup;
		}
	}

	/**
	 * Add Crumbs
	 * 
	 * Adds a new crumbs to the crumbs list
	 *
	 * @since  1.3.0
	 * @param  string 	$name The name of the crumb
	 * @param  array 	$args The crumbs settings
	 * @return void
	 */
	private function add_crumb( $name, $args = [] ) {
		if ( ! $name || empty( $args['key'] ) ) {
			return;
		}

		$args['name'] = $name;

		$this->crumbs[ $args['key'] ] = $args;
	}

	/**
	 * Add Home Link Crumb
	 * 
	 * The markup for the home link crumb
	 *
	 * @since  1.3.0
	 * @return void
	 */
	protected function add_home_link() {
		$settings = $this->settings;

		if ( ! isset( $settings['showHome'] ) ) {
			return;
		}

		$this->add_crumb( 'home', [
			'key'     => 'home',
			'content' => isset( $settings['homeText'] ) ? $settings['homeText'] : '',
			'link'    => get_home_url(),
		] );
	}

	/**
	 * Add Custom Archive Crumbs
	 *
	 * @since  1.3.0
	 * @return void
	 */
	protected function add_custom_archive() {
		$this->add_crumb( 'archive', [
			'key'     => 'archive',
			'content' => post_type_archive_title( '', false ),
		] );
	}

	/**
	 * Add Custom Taxonomy Archive Crumbs
	 *
	 * @since  2.2.5
	 * @return void
	 */
	protected function add_taxonomy_archive() {
		$term      = get_queried_object();
		$term_name = $term->name;
		$taxonomy  = $term->taxonomy;
		$prefix    = 'taxonomy_' . $taxonomy;
		$settings  = $this->settings;

		if ( ! empty( $settings[ $prefix . '_show_home'] ) ) {
			$this->add_home_link();
		}

		if ( 'post' === get_post_type() ) {
			$this->add_blog();
		}

		if ( ! empty( $settings[ $prefix . '_show_taxonomy'] ) ) {
			$taxonomy_object = get_taxonomy( $taxonomy );
			$taxonomy_link   = ! empty( $settings[ $prefix . '_taxonomy_link' ] ) ? $settings[ $prefix . '_taxonomy_link' ] : '';

			$this->add_crumb( 'taxonomy', [
				'key'     => 'taxonomy',
				'ids'     => [ $taxonomy, $taxonomy_object->name ],
				'content' => $taxonomy_object->labels->name,
				'link'    => ! empty( $taxonomy_link ) ? $taxonomy_link : '',
			] );
		}

		if ( ! empty( $settings[ $prefix . '_show_parents'] ) && '' !== $settings[ $prefix . '_show_parents'] ) {
			$this->add_archive_term_parents( $term->term_id, $term->taxonomy );
		}

		if ( ! empty( $settings[ $prefix . '_show_current'] ) ) {
			$this->add_crumb( 'taxonomy-archive', [
				'key'     => 'taxonomy-archive',
				'ids'     => [ $term->term_id, $term->slug ],
				'content' => $term->name,
				'link'    => '',
			] );
		}
	}

	/**
	 * Add Single Crumbs
	 *
	 * @since  1.3.0
	 * @return void
	 */
	protected function add_single( $query, $post = false ) {
		if ( ! $post ) { global $post; }

		$settings   = $this->settings;
		$post_type  = get_post_type();
		$post_types = get_post_types( [ 'show_in_nav_menus' => true, ], 'names' );
		$prefix     = 'single_' . $post_type;

		if ( ! in_array( $post_type, $post_types ) ) {
			return;
		}

		if ( array_key_exists( $prefix . '_show_home', $settings ) && ! empty( $settings[ $prefix . '_show_home' ] ) ) {
			$this->add_home_link();
		}

		if ( 'post' === $post_type ) {
			$this->add_blog();
		}

		if ( get_object_taxonomies( $post_type, 'objects' ) ) {
			if ( ! empty( $settings[ $prefix . '_show_terms'] ) ) {
				$this->add_single_terms( $post );
			}
		}

		if ( is_post_type_hierarchical( $post_type ) ) {
			if ( ! empty( $settings[ $prefix . '_show_parents'] ) ) {
				$this->add_single_parents( $post );
			}
		}

		if (  array_key_exists( $prefix . '_show_current', $settings ) && ! empty( $settings[ $prefix . '_show_current'] ) ) {
			$this->add_crumb( 'single', [
				'key'     => 'single',
				'ids'     => [ $post->ID ],
				'content' => get_the_title(),
			] );
		}
	}

	/**
	 * Add Single Parents Crumbs
	 *
	 * @since  1.3.0
	 * @return void
	 */
	protected function add_single_parents( $post = false ) {
		if ( ! $post ) { global $post; }

		if ( ! $post->post_parent ) {
			return;
		}
				
		$parents = get_post_ancestors( $post->ID );
		$parents = array_reverse( $parents );
			
		if ( ! isset( $parents ) ) $parents = null;

		foreach ( $parents as $parent ) {
			$this->add_crumb( 'ancestor', [
				'key'     => 'ancestor-' . $parent,
				'ids'     => [ $parent ],
				'content' => get_the_title( $parent ),
				'link'    => get_permalink( $parent ),
			] );
		}
	}

	/**
	 * Add Blog Crumb
	 *
	 * @since  1.3.0
	 * @return void
	 */
	protected function add_blog() {
		$posts_page_id = get_option( 'page_for_posts' );

		if ( $posts_page_id ) {
			$blog = get_post( $posts_page_id );

			$this->add_crumb( 'blog', [
				'key'     => 'blog',
				'ids'     => [ $blog->ID ],
				'content' => $blog->post_title,
				'link'    => get_permalink( $blog->ID ),
			] );
		}
	}

	/**
	 * Add Post Type Archive Crumbs
	 *
	 * @since  1.3.0
	 * @return void
	 */
	protected function add_post_type_archive( $post = false ) {
		$settings  = $this->settings;
		$post_type = get_post_type( $post );

		if ( '' !== $settings['cpt_' . $post_type . '_show_home'] ) {
			$this->add_home_link();
		}

		if ( '' !== $settings['cpt_' . $post_type . '_show_current'] ) {
			$this->add_post_type_archive_link( $post );
		}
	}

	/**
	 * Add Post Type Archive Link Crumb
	 *
	 * @since  1.3.0
	 * @return void
	 */
	protected function add_post_type_archive_link( $post = false ) {
		if ( ! $post ) { global $post; }

		$queried_object = get_queried_object();

		if ( is_post_type_archive() ) {
			$post_type = $queried_object->name;
		} elseif ( is_archive() && ( is_tax() || is_category() || is_tag() ) ) {
			$post_type = get_taxonomy( $queried_object->taxonomy )->object_type[0];
		} elseif ( is_single() || is_page() ) {
			$post_type = get_post_type( $post );
		} else {
			return;
		}

		$post_type_object = get_post_type_object( $post_type );

		switch ( $post_type ) {
			case 'post':
				$page_for_posts = get_option( 'page_for_posts' );
				$link = $page_for_posts ? get_permalink( get_option( 'page_for_posts' ) ) : esc_url( home_url( '/' ) );
				break;
			case 'page':
				$link = esc_url( home_url( '/' ) );
				break;
			default:
				$link = get_post_type_archive_link( $post_type );
				break;
		}

		$this->add_crumb( 'post-type-archive', [
			'key'     => 'post-type-archive',
			'ids'     => [ $post_type ],
			'content' => $post_type_object->labels->name,
			'link'    => $link,
		] );
	}

	/**
	 * Render Single Terms
	 *
	 * @since  1.3.0
	 * @return void
	 */
	protected function add_single_terms( $post = false ) {
		if ( ! $post ) { global $post; }

		$terms      = [];
		$taxonomies = get_post_taxonomies( $post->ID );
		$settings   = $this->settings;
		$show_tax   = $settings['single_' . get_post_type() . '_show_terms'];

		foreach ( $taxonomies as $index => $taxonomy ) {
			if ( '' !== $show_tax && $show_tax !== $taxonomy ) {
				continue;
			}

			$taxonomy_terms = wp_get_post_terms( $post->ID, $taxonomy, [
				'hide_empty' => false,
			] );

			Helper::sort_terms_hierarchicaly( $taxonomy_terms, $terms );
		}

		if ( $terms ) {
			$this->add_terms_recursive( $terms );
		}
	}

	/**
	 * Render Single Terms
	 *
	 * @since  1.3.0
	 * @return void
	 */
	protected function add_archive_term_parents( $term_id, $taxonomy ) {
		$parents = get_ancestors( $term_id, $taxonomy );

		if ( $parents )  {
			$ordered_parents = [];
			$parent_terms    = get_terms( [
				'taxonomy' => $taxonomy,
				'include'  => $parents,
			] );

			Helper::sort_terms_hierarchicaly( $parent_terms, $ordered_parents );

			if ( $ordered_parents ) {
				$this->add_terms_recursive( $ordered_parents );
			}
		}
	}

	/**
	 * Add Terms Crumbs Recursively
	 *
	 * @since  1.3.0
	 * @return void
	 */
	protected function add_terms_recursive( $terms ) {
		foreach ( $terms as $term_id => $term ) {
			$this->add_crumb( 'taxonomy-terms', [
				'key'     => 'term-' . $term_id,
				'ids'     => [ $term->term_id, $term->slug ],
				'content' => $term->name,
				'link'    => get_term_link( $term_id ),
			] );

			if ( $term->children ) {
				$this->add_terms_recursive( $term->children );
			}
		}
	}

	/**
	 * Add Tag Crumb
	 *
	 * @since  1.3.0
	 * @return void
	 */
	protected function add_tag() {
		$term_id  = get_query_var('tag_id');
		$taxonomy = 'post_tag';
		$args     = 'include=' . $term_id;
		$terms    = get_terms([
			'taxonomy' => $taxonomy,
			'include'  => $term_id,
		]);
		$tag      = $terms[0];

		$this->add_crumb( 'tag', [
			'key'     => 'tag',
			'ids'     => [ $tag->term_id, $tag->slug ],
			'content' => sprintf( __( 'Tag: %s', 'max-addons' ), $tag->name ),
		] );
	}

	/**
	 * Add Day Crumb
	 *
	 * @since  1.3.0
	 * @return void
	 */
	protected function add_day() {
		$this->add_year( false );
		$this->add_month( false );

		$this->add_crumb( 'day', [
			'key'     => 'day',
			'ids'     => [ get_the_time('j') ],
			'content' => sprintf( __( '%1$s %2$s Archives', 'max-addons' ), get_the_time('F'), get_the_time('jS') ),
		] );
	}

	/**
	 * Add Month Crumb
	 *
	 * @since  1.3.0
	 * @return void
	 */
	protected function add_month() {
		$this->add_year( false );

		$this->add_crumb( 'month', [
			'key'     => 'month',
			'ids'     => [ get_the_time('m') ],
			'content' => sprintf( __( '%s Archives', 'max-addons' ), get_the_time('F') ),
		] );
	}

	/**
	 * Add Year Crumb
	 *
	 * @since  1.3.0
	 * @return void
	 */
	protected function add_year() {
		$this->add_crumb( 'year', [
			'key'     => 'year',
			'ids'     => [ get_the_time('Y') ],
			'content' => sprintf( __( '%s Archives', 'max-addons' ), get_the_time('Y') ),
		] );
	}

	/**
	 * Add Author Crumb
	 *
	 * @since  1.3.0
	 * @return void
	 */
	protected function add_author() {
		global $author;

		$userdata = get_userdata( $author );

		$this->add_crumb( 'author', [
			'key'     => 'author',
			'ids'     => [ $userdata->user_nicename ],
			'content' => sprintf( __( 'Author: %s', 'max-addons' ), $userdata->display_name ),
		] );
	}

	/**
	 * Add Search Crumb
	 *
	 * @since  1.3.0
	 * @return void
	 */
	protected function add_search() {
		$this->add_crumb( 'search', [
			'current'   => true,
			'separator'	=> false,
			'key'       => 'search',
			'content'   => sprintf( __( 'Search results for: %s', 'max-addons' ), get_search_query() ),
		] );
	}

	/**
	 * Add 404 crumb
	 *
	 * @since  1.3.0
	 * @return void
	 */
	protected function add_404() {
		$this->add_crumb( '404', [
			'current'   => true,
			'separator' => false,
			'key'       => '404',
			'content'   => __( 'Page not found', 'max-addons' ),
		] );
	}

	/**
	 * Add Item Render Attribute
	 * 
	 * Adds the render attributes for a specified item
	 *
	 * @since  1.3.0
	 * @param  key|string 	The render attribute key for the item
	 * @param. index|int 	The index of the item. Defaults to 0 
	 * @return void
	 */
	protected function add_item_render_attribute( $key, $index = 0 ) {
		$settings = $this->settings;

		$this->set_attribute( $key, 'class', 'mab-breadcrumbs-item' );

		if ( isset( $settings['structuredData'] ) ) {
			$this->set_attribute( $key, 'itemprop', 'itemListElement' );
			$this->set_attribute( $key, 'itemscope', '' );
			$this->set_attribute( $key, 'itemtype', 'http://schema.org/ListItem' );
		}
	}

	/**
	 * Add Link Render Attribute
	 * 
	 * Adds the render attributes for the item link
	 *
	 * @since  1.3.0
	 * @param  key|string 	The render attribute key for the item
	 * @return void
	 */
	protected function add_link_render_attribute( $key ) {
		$settings = $this->settings;

		$this->set_attribute( $key, 'class', 'mab-breadcrumbs-crumb' );

		if ( isset( $settings['structuredData'] ) ) {
			$this->set_attribute( $key, 'itemprop', 'item' );
		}
	}

	/**
	 * Set Crumbs
	 * 
	 * Checks current query to determine which
	 * crumbs need to be added
	 *
	 * @access private
	 *
	 * @since  1.3.0
	 * @param  WP_Query 	$query 	The query to check
	 * @return void
	 */
	private function set_crumbs( $query ) {
		$is_custom_archive = $query->is_archive() && ! $query->is_tax() && ! $query->is_category() && ! $query->is_tag() && ! $query->is_date() && ! $query->is_author() && ! $query->is_post_type_archive();

		if ( $is_custom_archive ) {

			$this->add_custom_archive();

		} else if ( $query->is_post_type_archive() ) {

			$this->add_post_type_archive();
			
		} else if ( $query->is_archive() && ( $query->is_tax() || $query->is_category() || $query->is_tag() ) ) {

			$this->add_taxonomy_archive();
			
		} else if ( $query->is_single() || $query->is_page() ) {

			$this->add_single( $query, false );
		
		} else if ( $query->is_day() ) {

			$this->add_day();
			
		} else if ( $query->is_month() ) {

			$this->add_month();
			
		} else if ( $query->is_year() ) {

			$this->add_year();
			
		} else if ( $query->is_author() ) {
			
			$this->add_author();
			
		} else if ( $query->is_search() ) {

			$this->add_search();
		
		} elseif ( $query->is_404() ) {

			$this->add_404();

		}
	}

	/**
	 * Render Item
	 * 
	 * Gets the markup for a breadcrumb item
	 *
	 * @since  1.3.0
	 * @param  name|string
	 * @param  args|array
	 * @return void
	 */
	protected function render_crumb( $key, $index, $args ) {
		global $wp;

		$settings = $this->settings;

		$defaults = [
			'key'     => false,
			'ids'     => [],
			'content' => '',
			'link'    => false,
			'current' => $index === count( $this->crumbs ) - 1,
		];

		$args = wp_parse_args( $args, $defaults );

		if ( $args['current'] && ! isset( $settings['showCurrent'] ) ) {
			return;
		}

		$item_key = $args['key'] . '-item';
		$text_key = $args['key'] . '-text';
		$pos_key  = $args['key'] . '-position';

		$link_key = $args['key'] . '-link';
		$link_tag = ( ! $args['current'] ) ? 'a' : 'link';
		$classes  = [];

		if ( '' !== $args['link'] ) {
			$this->set_attribute( $link_key, 'href', $args['link'] );
		} else {
			$this->set_attribute( $link_key, 'href', home_url( $wp->request ) );
		}

		if ( $args['current'] ) {
			$classes[] = 'mab-breadcrumbs-item-current';
		} else {
			$classes[] = 'mab-breadcrumbs-item-parent';
		}

		if ( ! empty( $args['name'] ) ) {
			$classes[] = 'mab-breadcrumbs-item-' . $args['name'];
		}

		if ( ! empty( $args['ids'] ) && is_array( $args['ids'] ) ) {
			foreach ( $args['ids'] as $id ) {
				if ( ! empty( $args['name'] ) ) {
					$classes[] = 'mab-breadcrumbs-item-' . $args['name'] . '-' . $id;
				} else { $classes[] = 'mab-breadcrumbs-item-' . $id; }
			}
		}

		$this->add_item_render_attribute( $item_key, $index );
		$this->set_attribute( $item_key, 'class', $classes );
		$this->set_attribute( $text_key, 'class', 'mab-breadcrumbs-text' );
		$this->set_attribute( $pos_key, 'content', $index );

		$this->add_link_render_attribute( $link_key );

		if ( isset( $settings['structuredData'] ) ) {
			$this->set_attribute( $text_key, 'itemprop', 'name' );
			$this->set_attribute( $pos_key, 'itemprop', 'position' );
		}
		?><li <?php echo $this->render_attributes( $item_key ); ?>>
			<<?php echo $link_tag; ?> <?php echo $this->render_attributes( $link_key ); ?>>
				<span <?php echo $this->render_attributes( $text_key ); ?>>
					<?php echo $args['content']; ?>
				</span>
			</<?php echo $link_tag; ?>>
			<meta <?php echo $this->render_attributes( $pos_key ); ?>>
		</li><?php

		if ( ! $args['current'] )

		if ( false === $args['current'] ) {
			$this->render_separator();
		}
	}

	/**
	 * Render Crumbs
	 * 
	 * Adds the render attributes for a specified item
	 *
	 * @access private
	 *
	 * @since  1.3.0
	 * @return void
	 */
	private function render_crumbs() {
		$index = 0;

		foreach ( $this->crumbs as $name => $args ) {
			$this->render_crumb( $name, $index, $args );
			$index++;
		}
	}

	/**
	 * Render Breadcrumbs
	 * 
	 * Identifies and outputs all the breadcrumbs
	 *
	 * @access public
	 *
	 * @since  	1.3.0
	 * @param 	WP_Query|bool $query Query used to render the breadcrumbs
	 * @return 	void
	 */
	public function render_breadcrumbs( $query = false ) {
		global $post, $wp_query;

		if ( $query === false ) {

			// Reset post data to parent query
			$wp_query->reset_postdata();

			// Set active query to native query
			$query = $wp_query;
		}

		if ( $query->is_front_page() || $query->is_home() ) {
			return;
		}

		$this->set_crumbs( $query );

		?><ul <?php echo $this->render_attributes( '_root' ); ?>><?php
			$this->render_crumbs();
		?></ul><?php
	}

	public function render_custom_breadcrumbs() {
		$settings = $this->settings;
		$_query   = $this->get_query();

		$this->set_separator();

		if ( isset( $settings['structuredData'] ) ) {
			$this->set_attribute( '_root', 'itemscope', '' );
			$this->set_attribute( '_root', 'itemtype', "http://schema.org/BreadcrumbList" );
		}

		if ( $_query ) {
			if ( $_query->have_posts() ) {

				// Setup post
				$_query->the_post();

				// Render using the new query
				$this->render_breadcrumbs( $_query );

				// Reset post data to original query
				wp_reset_postdata();
				wp_reset_query();

			} else {

				_e( 'Post or page not found', 'max-addons' );

			}
		} else {
			// Render using the original query
			$this->render_breadcrumbs();
		}
	}

	public function render() {
		$settings = $this->settings;
		$type     = isset( $settings['type'] ) ? $settings['type'] : '';

		if ( empty( $type ) ) {
			return $this->render_element_placeholder( [ 'title' => esc_html__( 'Please select breadcrumbs type.', 'max-addons' ) ] );
		}

		$classes = array(
			'mab-breadcrumbs',
			'mab-breadcrumbs-' . $type
		);

		$this->set_attribute( '_root', 'class', $classes );

		if ( 'custom' == $type ) {
			$this->render_custom_breadcrumbs();
		} else {
			?>
			<div <?php echo $this->render_attributes( '_root' ); ?>>
				<?php
				if ( 'yoast' === $type && function_exists( 'yoast_breadcrumb' ) ) {
					yoast_breadcrumb( '<p id="breadcrumbs">','</p>' );
				} elseif ( 'rankmath' === $type && function_exists( 'rank_math_the_breadcrumbs' ) ) {
					rank_math_the_breadcrumbs();
				} elseif ( 'navxt' == $type && function_exists( 'bcn_display' ) ) {
					bcn_display();
				} elseif ( 'seopress' == $type && function_exists( 'seopress_display_breadcrumbs' ) ) {
					seopress_display_breadcrumbs();
				}
				?>
			</div>
			<?php
		}
	}

	public static function render_builder() { ?>
		<script type="text/x-template" id="tmpl-brxe-max-breadcrumbs">
		<component>
			<ul v-if="settings.type === 'custom'" class="mab-breadcrumbs">
				<li v-if="settings.showHome" class="mab-breadcrumbs-item mab-breadcrumbs-item-parent mab-breadcrumbs-item-home">
					<a href="#" class="mab-breadcrumbs-crumb">
						<span class="mab-breadcrumbs-text" v-html="settings.homeText"></span>
					</a>
				</li>
				<li v-if="settings.showHome" class="mab-breadcrumbs-separator">
					<span v-if="settings.separatorType === 'text' && settings.separatorText" class="mab-breadcrumbs-separator-text" v-html="settings.separatorText"></span>
					<span v-if="settings.separatorType === 'icon' && settings.separatorIcon" class="mab-icon">
						<icon-svg :iconSettings="settings.separatorIcon"/>
					</span>
				</li>
				<li class="mab-breadcrumbs-item mab-breadcrumbs-item-parent mab-breadcrumbs-item-post-type-archive mab-breadcrumbs-item-post-type-archive-page">
					<a href="#" class="mab-breadcrumbs-crumb">
						<span class="mab-breadcrumbs-text">Archive</span>
					</a>
				</li>
				<li v-if="settings.showCurrent" class="mab-breadcrumbs-separator">
				<span v-if="settings.separatorType === 'text' && settings.separatorText" class="mab-breadcrumbs-separator-text" v-html="settings.separatorText"></span>
					<span v-if="settings.separatorType === 'icon' && settings.separatorIcon" class="mab-icon">
						<icon-svg :iconSettings="settings.separatorIcon"/>
					</span>
				</li>
				<li v-if="settings.showCurrent" class="mab-breadcrumbs-item mab-breadcrumbs-item-current mab-breadcrumbs-item-single">
					<span class="mab-breadcrumbs-text">Page Title</span>
				</li>
			</ul>
			<div v-if="settings.type === 'yoast'">
				<?php
					if ( function_exists( 'yoast_breadcrumb' ) ) {
						yoast_breadcrumb( '<p id="breadcrumbs">','</p>' );
					}
				?>
			</div>
			<div v-if="settings.type === 'rankmath'">
				<?php
					if ( function_exists( 'rank_math_the_breadcrumbs' ) ) {
						rank_math_the_breadcrumbs();
					}
				?>
			</div>
			<div v-if="settings.type === 'navxt'">
				<?php
					if ( function_exists( 'bcn_display' ) ) {
						bcn_display();
					}
				?>
			</div>
			<div v-if="settings.type === 'seopress'">
				<?php
					if ( function_exists( 'seopress_display_breadcrumbs' ) ) {
						seopress_display_breadcrumbs();
					}
				?>
			</div>
		</component>
		</script>
		<?php
	}
}