<?php
namespace MaxAddonsPro\Elements;

use MaxAddonsPro\Classes\Helper;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Author_List_Element extends \Bricks\Element {
	// Element properties
	public $category     = 'max-addons-elements'; // Use predefined element category 'general'
	public $name         = 'max-author-list'; // Make sure to prefix your elements
	public $icon         = 'ti-user max-element'; // Themify icon font class
	public $css_selector = ''; // Default CSS selector
	public $scripts      = []; // Script(s) run when element is rendered on frontend or updated in builder

	public function get_label() {
		return esc_html__( 'Author List', 'max-addons' );
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		wp_enqueue_style( 'mab-author-list' );
	}

	// Set builder control groups
	public function set_control_groups() {
		$this->control_groups['query'] = [
			'title' => esc_html__( 'Query', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['avatar'] = [
			'title' => esc_html__( 'Author Avatar', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['postsCount'] = [
			'title' => esc_html__( 'Posts Count', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['fields'] = [
			'title' => esc_html__( 'Fields', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['layout'] = [
			'title' => esc_html__( 'Layout', 'max-addons' ),
			'tab'   => 'content',
		];
	}

	public function set_controls() {

		$this->controls['_background']['css'][0]['selector'] = '';
		$this->controls['_gradient']['css'][0]['selector'] = '';

		$this->set_query_controls();
		$this->set_fields_controls();
		$this->set_avatar_controls();
		$this->set_layout_controls();
		$this->set_post_count_controls();
	}

	// Set query controls
	public function set_query_controls() {

		$this->controls['source'] = [
			'tab'       => 'content',
			'group'     => 'query',
			'label'     => esc_html__( 'Source', 'max-addons' ),
			'type'      => 'select',
			'options'   => [
				'custom' => __( 'Custom Query', 'max-addons' ),
				'manual' => __( 'Manual Selection', 'max-addons' ),
			],
			'default'   => 'custom',
			'inline'    => true,
			'clearable' => false,
		];

		$this->controls['includeUserRoles'] = [
			'tab'      => 'content',
			'group'    => 'query',
			'label'    => esc_html__( 'Include Roles', 'max-addons' ),
			'type'     => 'select',
			'options'  => bricks_is_builder() ? Helper::get_user_roles() : [],
			'multiple' => true,
			'inline'   => false,
			'required' => [ 'source', '=', 'custom' ],
		];

		$this->controls['excludeUserRoles'] = [
			'tab'      => 'content',
			'group'    => 'query',
			'label'    => esc_html__( 'Exclude Roles', 'max-addons' ),
			'type'     => 'select',
			'options'  => bricks_is_builder() ? Helper::get_user_roles() : [],
			'inline'   => false,
			'required' => [ 'source', '=', 'custom' ],
		];

		$this->controls['excludeUsers'] = [
			'tab'      => 'content',
			'group'    => 'query',
			'label'    => esc_html__( 'Exclude Users', 'max-addons' ),
			'type'     => 'select',
			'options'  => bricks_is_builder() ? Helper::get_users_options() : [],
			'multiple' => true,
			'inline'   => false,
			'required' => [ 'source', '=', 'custom' ],
		];

		$this->controls['selectUsers'] = [
			'tab'      => 'content',
			'group'    => 'query',
			'label'    => esc_html__( 'Select Users', 'max-addons' ),
			'type'     => 'select',
			'options'  => bricks_is_builder() ? Helper::get_users_options() : [],
			'multiple' => true,
			'inline'   => false,
			'required' => [ 'source', '=', 'manual' ],
		];

		$this->controls['usersCount'] = [
			'tab'      => 'content',
			'group'    => 'query',
			'label'    => esc_html__( 'Number of Users', 'max-addons' ),
			'type'     => 'number',
			'min'      => 1,
			'max'      => 500,
			'inline'   => true,
			'small'    => true,
			'required' => [ 'source', '=', 'custom' ],
		];

		$this->controls['orderby'] = [
			'tab'     => 'content',
			'group'   => 'query',
			'label'   => esc_html__( 'Order By', 'max-addons' ),
			'type'    => 'select',
			'options' => [
				'ID'            => __( 'User ID', 'max-addons' ),
				'display_name'  => __( 'Display Name', 'max-addons' ),
				'user_name'     => __( 'User Name', 'max-addons' ),
				'user_login'    => __( 'User Login', 'max-addons' ),
				'user_nicename' => __( 'Nice Name', 'max-addons' ),
				'user_email'    => __( 'User Email', 'max-addons' ),
				'user_url'      => __( 'User URL', 'max-addons' ),
				'registered'    => __( 'Registered Date', 'max-addons' ),
				'post_count'    => __( 'Post Count', 'max-addons' ),
			],
			'default' => '',
			'inline'  => true,
		];

		$this->controls['order'] = [
			'tab'     => 'content',
			'group'   => 'query',
			'label'   => esc_html__( 'Order', 'max-addons' ),
			'type'    => 'select',
			'options' => [
				'ASC'  => __( 'Ascending', 'max-addons' ),
				'DESC' => __( 'Descending', 'max-addons' ),
			],
			'default' => 'ASC',
			'inline'  => true,
		];
	}

	public function set_fields_controls() {
		$this->controls['fields'] = [
			'tab'           => 'content',
			'group'         => 'fields',
			'placeholder'   => esc_html__( 'Field', 'max-addons' ),
			'type'          => 'repeater',
			'titleProperty' => 'fieldType',
			'fields'        => [
				'fieldType'         => [
					'label'       => esc_html__( 'Field Type', 'max-addons' ),
					'type'        => 'select',
					'options'     => [
						'name'       => 'Name',
						'postsCount' => 'Posts Count',
						'email'      => 'Email',
						'authorBio'  => 'Author Bio',
					],
					'inline'      => true,
					'clearable'   => false,
					'placeholder' => esc_html__( 'Name', 'max-addons' ),
				],

				'nameType'          => [
					'label'       => esc_html__( 'Name Type', 'max-addons' ),
					'type'        => 'select',
					'options'     => [
						'display_name'  => esc_html__( 'Display Name', 'max-addons' ),
						'first_name'    => esc_html__( 'First Name', 'max-addons' ),
						'last_name'     => esc_html__( 'Last Name', 'max-addons' ),
						'nickname'      => esc_html__( 'Nick Name', 'max-addons' ),
						'user_nicename' => esc_html__( 'User Nice Name', 'max-addons' ),
					],
					'inline'      => true,
					'placeholder' => esc_html__( 'Display Name', 'max-addons' ),
					'required'    => [ 'fieldType', '=', 'name' ],
				],

				'linkNameTo'        => [
					'label'       => esc_html__( 'Link Name To', 'max-addons' ),
					'type'        => 'select',
					'options'     => [
						'authorArchive' => esc_html__( 'Author Archive', 'max-addons' ),
						'authorUrl'     => esc_html__( 'Website URL', 'max-addons' ),
					],
					'inline'      => true,
					'placeholder' => esc_html__( 'None', 'max-addons' ),
					'required'    => [ 'fieldType', '=', 'name' ],
				],

				'tag'               => [
					'label'       => esc_html__( 'HTML tag', 'max-addons' ),
					'type'        => 'select',
					'options'     => [
						'div' => 'div',
						'p'   => 'p',
						'h1'  => 'h1',
						'h2'  => 'h2',
						'h3'  => 'h3',
						'h4'  => 'h4',
						'h5'  => 'h5',
						'h6'  => 'h6',
					],
					'inline'      => true,
					'placeholder' => 'div',
				],

				'dynamicMargin'     => [
					'label' => esc_html__( 'Margin', 'max-addons' ),
					'type'  => 'spacing',
					'css'   => [
						[
							'property' => 'margin',
							'selector' => '',
							'nthChild' => false, // NOTE: Undocumented
						],
					],
				],

				'dynamicTypography' => [
					'label'  => esc_html__( 'Typography', 'max-addons' ),
					'type'   => 'typography',
					'css'    => [
						[
							'property' => 'font',
							'selector' => '',
							'nthChild' => false,
						],
					],
					'inline' => true,
					'small'  => true,
				],

			],
			'default'       => [
				[
					'fieldType'     => 'name',
					'nameType'      => 'display_name',
					'id'            => \Bricks\Helpers::generate_random_id( false ),
					'dynamicMargin' => [
						'top'    => 0,
						'right'  => 0,
						'bottom' => 10,
						'left'   => 0,
					],
				],
				[
					'fieldType'     => 'authorBio',
					'id'            => \Bricks\Helpers::generate_random_id( false ),
					'dynamicMargin' => [
						'top'    => 0,
						'right'  => 0,
						'bottom' => 0,
						'left'   => 0,
					],
				],
			],
		];
	}

	// Set layout controls
	public function set_avatar_controls() {

		$this->controls['authorAvatarType'] = [
			'tab'     => 'content',
			'group'   => 'avatar',
			'label'   => esc_html__( 'Avatar', 'max-addons' ),
			'type'    => 'select',
			'options' => [
				'gravatar' => esc_html__( 'Default Avatar', 'max-addons' ),
				'image'    => esc_html__( 'Image', 'max-addons' ),
				'icon'     => esc_html__( 'Icon', 'max-addons' ),
			],
			'inline'  => true,
			'default' => 'gravatar',
		];

		$this->controls['authorImage'] = [
			'tab'      => 'content',
			'group'    => 'avatar',
			'type'     => 'image',
			'label'    => esc_html__( 'Image', 'max-addons' ),
			'required' => [ 'authorAvatarType', '=', [ 'image' ] ],
		];

		$this->controls['authorImageSize'] = [
			'tab'      => 'content',
			'group'    => 'avatar',
			'label'    => esc_html__( 'Image width', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'width',
					'selector' => '.mab-author-list-avatar img',
				],
			],
			'required' => [ 'authorAvatarType', '=', [ 'image' ] ],
		];

		$this->controls['authorAvatarSize'] = [
			'tab'      => 'content',
			'group'    => 'avatar',
			'label'    => esc_html__( 'Avatar Size', 'max-addons' ),
			'type'     => 'number',
			'min'      => 1,
			'max'      => 500,
			'default'  => 45,
			'inline'   => true,
			'small'    => true,
			'required' => [ 'authorAvatarType', '=', [ 'gravatar' ] ],
		];

		$this->controls['authorIcon'] = [
			'tab'      => 'content',
			'group'    => 'avatar',
			'label'    => esc_html__( 'Icon', 'max-addons' ),
			'type'     => 'icon',
			'default'  => [
				'library' => 'themify',
				'icon'    => 'ti-user',
			],
			'required' => [ 'authorAvatarType', '=', [ 'icon' ] ],
		];

		$this->controls['linkAvatarTo'] = [
			'tab'         => 'content',
			'group'       => 'avatar',
			'label'       => esc_html__( 'Link Avatar To', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'authorArchive' => esc_html__( 'Author Archive', 'max-addons' ),
				'authorUrl'     => esc_html__( 'Website URL', 'max-addons' ),
			],
			'inline'      => true,
			'placeholder' => esc_html__( 'None', 'max-addons' ),
			'required'    => [ 'authorAvatarType', '!=', '' ],
		];

		$this->controls['avatarMargin'] = [
			'tab'      => 'content',
			'group'    => 'avatar',
			'label'    => esc_html__( 'Margin', 'max-addons' ),
			'type'     => 'spacing',
			'css'      => [
				[
					'property' => 'margin',
					'selector' => '.mab-author-list-avatar',
				],
			],
			'required' => [ 'authorAvatarType', '!=', '' ],
		];

		$this->controls['avatarBorder'] = [
			'tab'      => 'content',
			'group'    => 'avatar',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'inline'   => true,
			'small'    => true,
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.mab-author-list-avatar img',
				],
			],
			'required' => [ 'authorAvatarType', '=', [ 'gravatar', 'image' ] ],
		];

		$this->controls['avatarBoxShadow'] = [
			'tab'      => 'content',
			'group'    => 'avatar',
			'label'    => esc_html__( 'Box Shadow', 'max-addons' ),
			'type'     => 'box-shadow',
			'css'      => [
				[
					'property' => 'box-shadow',
					'selector' => '.mab-author-list-avatar img',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'authorAvatarType', '=', [ 'gravatar', 'image' ] ],
		];

		$this->controls['iconSize'] = [
			'tab'      => 'content',
			'group'    => 'avatar',
			'label'    => esc_html__( 'Icon size', 'max-addons' ),
			'type'     => 'number',
			'units'    => true,
			'css'      => [
				[
					'property' => 'font-size',
					'selector' => '.mab-author-list-avatar i',
				],
			],
			'required' => [ 'authorAvatarType', '=', [ 'icon' ] ],
		];

		$this->controls['iconColor'] = [
			'tab'      => 'content',
			'group'    => 'avatar',
			'label'    => esc_html__( 'Icon color', 'max-addons' ),
			'type'     => 'color',
			'inline'   => true,
			'css'      => [
				[
					'property' => 'color',
					'selector' => '.mab-author-list-avatar i',
				],
			],
			'required' => [ 'authorAvatarType', '=', [ 'icon' ] ],
		];
	}

	// Set posts count controls
	public function set_post_count_controls() {

		$this->controls['postsCount'] = [
			'tab'     => 'content',
			'group'   => 'postsCount',
			'label'   => esc_html__( 'Posts Count', 'max-addons' ),
			'type'    => 'checkbox',
			'inline'  => true,
			'small'   => true,
			'default' => true,
		];

		$this->controls['postsCountText'] = [
			'tab'           => 'content',
			'group'         => 'postsCount',
			'label'         => esc_html__( 'Posts Count Text', 'max-addons' ),
			'type'          => 'text',
			'spellcheck'    => true,
			'inline'        => true,
			'inlineEditing' => false,
			'default'       => esc_html__( 'Posts: ', 'max-addons' ),
			'required'      => [ 'postsCount', '=', true ],
		];

		$this->controls['postType'] = [
			'tab'      => 'content',
			'group'    => 'postsCount',
			'label'    => esc_html__( 'Post Type', 'max-addons' ),
			'type'     => 'select',
			'options'  => bricks_is_builder() ? Helper::get_post_types() : [],
			'default'  => '',
			'inline'   => true,
			'required' => [ 'postsCount', '=', true ],
		];
		
		$this->controls['postCountTypography'] = [
			'tab'      => 'content',
			'group'    => 'postsCount',
			'type'     => 'typography',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-author-list-post-count',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'postsCount', '=', true ],
		];

		$this->controls['postsCountBackgroundColor'] = [
			'tab'    => 'content',
			'group'  => 'postsCount',
			'label'  => esc_html__( 'Background', 'max-addons' ),
			'type'   => 'color',
			'css'    => [
				[
					'property' => 'background-color',
					'selector' => '.mab-author-list-post-count',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['postsCountBorder'] = [
			'tab'    => 'content',
			'group'  => 'postsCount',
			'label'  => esc_html__( 'Border', 'max-addons' ),
			'type'   => 'border',
			'inline' => true,
			'small'  => true,
			'css'    => [
				[
					'property' => 'border',
					'selector' => '.mab-author-list-post-count',
				],
			],
		];

		$this->controls['postCountMargin'] = [
			'tab'      => 'content',
			'group'    => 'postsCount',
			'label'    => esc_html__( 'Margin', 'max-addons' ),
			'type'     => 'spacing',
			'css'      => [
				[
					'property' => 'margin',
					'selector' => '.mab-author-list-post-count',
				],
			],
			'required' => [ 'postsCount', '=', true ],
		];

		$this->controls['postCountPadding'] = [
			'tab'      => 'content',
			'group'    => 'postsCount',
			'label'    => esc_html__( 'Padding', 'max-addons' ),
			'type'     => 'spacing',
			'css'      => [
				[
					'property' => 'padding',
					'selector' => '.mab-author-list-post-count',
				],
			],
			'required' => [ 'postsCount', '=', true ],
		];
	}

	// Set layout controls
	public function set_layout_controls() {

		$this->controls['layout'] = [
			'tab'         => 'content',
			'group'       => 'layout',
			'label'       => esc_html__( 'Layout', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'list'    => esc_html__( 'List', 'max-addons' ),
				'grid'    => esc_html__( 'Grid', 'max-addons' ),
			],
			'placeholder' => esc_html__( 'List', 'max-addons' ),
			'inline'      => true,
		];

		$this->controls['direction'] = [
			'tab'      => 'content',
			'group'    => 'layout',
			'label'    => esc_html__( 'Direction', 'max-addons' ),
			'type'     => 'direction',
			'css'      => [
				[
					'property' => 'flex-direction',
					'selector' => '.bricks-layout-wrapper[data-layout=list] .bricks-layout-inner',
				],
			],
			'inline'   => true,
			'required' => [ 'layout', '=', 'list' ],
		];

		$this->controls['columns'] = [
			'tab'         => 'content',
			'group'       => 'layout',
			'label'       => esc_html__( 'Columns', 'max-addons' ),
			'type'        => 'number',
			'min'         => 1,
			'css'         => [
				[
					'selector' => '&[data-layout=grid] .bricks-layout-item',
					'property' => 'width',
					'value'    => 'calc(100% / %s)', // NOTE: Undocumented (@since 1.3)
				],
			],
			'rerender'    => true,
			'inline'      => true,
			'small'       => true,
			'placeholder' => 2,
			'required'    => [ 'layout', '=', 'grid' ],
		];

		$this->controls['contentAlign'] = [
			'tab'     => 'content',
			'group'   => 'layout',
			'label'   => esc_html__( 'Alignment', 'max-addons' ),
			'type'    => 'text-align',
			'css'         => [
				[
					'property' => 'text-align',
					'selector' => '.bricks-layout-inner',
				],
			],
			'inline'  => true,
			'default' => 'left',
			'required' => [ 'layout', '=', 'grid' ],
		];

		$this->controls['spacing'] = [
			'tab'     => 'content',
			'group'   => 'layout',
			'label'   => esc_html__( 'Spacing', 'max-addons' ),
			'type'    => 'number',
			'units'   => true,
			'css'     => [
				[
					'property' => 'padding-right',
					'selector' => '.bricks-layout-item',
				],
				[
					'property' => 'padding-bottom',
					'selector' => '.bricks-layout-item',
				],

				// NOTE: Undocumented
				[
					'property' => 'margin-right',
					'selector' => '.bricks-layout-wrapper',
					'invert'   => true,
				],
				[
					'property' => 'margin-bottom',
					'selector' => '.bricks-layout-wrapper',
					'invert'   => true,
				],
			],
			'default' => '30px',
		];

		$this->controls['listPadding'] = [
			'tab'   => 'content',
			'group' => 'layout',
			'label' => esc_html__( 'Padding', 'max-addons' ),
			'type'  => 'spacing',
			'css'   => [
				[
					'property' => 'padding',
					'selector' => '.mab-author-list',
				],
			],
		];

		$this->controls['listBackgroundColor'] = [
			'tab'    => 'content',
			'group'  => 'layout',
			'label'  => esc_html__( 'Background', 'max-addons' ),
			'type'   => 'color',
			'css'    => [
				[
					'property' => 'background-color',
					'selector' => '.mab-author-list',
				],
			],
			'inline' => true,
			'small'  => true,
		];

		$this->controls['listBorder'] = [
			'tab'    => 'content',
			'group'  => 'layout',
			'label'  => esc_html__( 'Border', 'max-addons' ),
			'type'   => 'border',
			'inline' => true,
			'small'  => true,
			'css'    => [
				[
					'property' => 'border',
					'selector' => '.mab-author-list',
				],
			],
		];

		$this->controls['listBoxShadow'] = [
			'tab'    => 'content',
			'group'  => 'layout',
			'label'  => esc_html__( 'Box Shadow', 'max-addons' ),
			'type'   => 'box-shadow',
			'css'    => [
				[
					'property' => 'box-shadow',
					'selector' => '.mab-author-list',
				],
			],
			'inline' => true,
			'small'  => true,
		];
	}

	public function get_normalized_image_settings( $settings ) {
		if ( ! isset( $settings['authorImage'] ) ) {
			$settings['authorImage'] = [
				'id'  => 0,
				'url' => '',
			];
			return $settings;
		}

		$image = $settings['authorImage'];

		if ( isset( $image['useDynamicData']['name'] ) ) {
			$images = $this->render_dynamic_data_tag( $image['useDynamicData']['name'], $image );
			$image['id'] = empty( $images ) ? 0 : $images[0];
		} else {
			$image['id'] = isset( $image['id'] ) ? $image['id'] : 0;
		}

		// Image Size
		$image['size'] = isset( $image['size'] ) ? $settings['authorImage']['size'] : BRICKS_DEFAULT_IMAGE_SIZE;

		// Image URL
		if ( ! isset( $image['url'] ) ) {
			$image['url'] = wp_get_attachment_image_url( $image['id'], $image['size'] );
		}

		$settings['authorImage'] = $image;

		return $settings;
	}

	// Render element HTML
	public function render_image() {
		$settings = $this->settings;
		$settings = $this->get_normalized_image_settings( $settings, 'authorImage' );

		// Dynamic Data is empty
		if ( isset( $settings['authorImage']['useDynamicData']['name'] ) ) {

			if ( empty( $settings['authorImage']['id'] ) ) {

				if ( 'ACF' === $settings['authorImage']['useDynamicData']['group'] && ! class_exists( 'ACF' ) ) {
					$message = esc_html__( 'Can\'t render element, as the selected ACF field is not available. Please activate ACF or edit the element to select different data.', 'max-addons' );
				} elseif ( '{featured_image}' == $settings['authorImage']['useDynamicData']['name'] ) {
					$message = esc_html__( 'No featured image set.', 'max-addons' );
				} else {
					$message = esc_html__( 'Dynamic Data %1$s (%2$s) is empty.', 'max-addons' );
				}

				return $this->render_element_placeholder( [
					'icon-class' => 'ti-image',
					'title'       => sprintf(
						$message,
						$settings['authorImage']['useDynamicData']['label'],
						$settings['authorImage']['useDynamicData']['group']
					),
				] );
			}
		} else {
			// Image id is empty or doesn't exist

			// No image
			if ( empty( $settings['authorImage']['id'] ) ) {
				return $this->render_element_placeholder( [
					'icon-class' => 'ti-image',
					'title'       => esc_html__( 'No image selected.', 'max-addons' ),
				] );
			}

			// Return if image ID does not exist
			if ( ! wp_get_attachment_image_src( $settings['authorImage']['id'] ) ) {
				return $this->render_element_placeholder( [
					'icon-class' => 'ti-image',
					'title'       => sprintf( esc_html__( 'Image ID (%s) no longer exist. Please select another image.', 'max-addons' ), $settings['authorImage']['id'] ),
				] );
			}
		}

		$image_atts = [];
		$image_atts['id'] = 'image-' . $settings['authorImage']['id'];

		$image_wrapper_classes = [ 'image-wrapper' ];
		$img_classes = [ 'post-thumbnail', 'css-filter' ];

		$img_classes[] = 'size-' . $settings['authorImage']['size'];
		$image_atts['class'] = join( ' ', $img_classes );

		$image_html = '';

		if ( isset( $settings['authorImage']['id'] ) ) {
			$image_html = wp_get_attachment_image( $settings['authorImage']['id'], $settings['authorImage']['size'], false, $image_atts );
		} elseif ( ! empty( $settings['authorImage']['url'] ) ) {
			$image_html = '<img src="' . esc_url( $settings['authorImage']['url'] ) . '">';
		}

		return $image_html;
	}

	public function get_user_data() {
		$settings = $this->settings;

		$args = array();

		if ( ! empty( $settings['orderby'] ) ) {
			$args['orderby'] = $settings['orderby'];
		}

		if ( ! empty( $settings['order'] ) ) {
			$args['order'] = $settings['order'];
		}

		if ( 'custom' === $settings['source'] ) {
			$args['number'] = ! empty( $settings['usersCount'] ) ? $settings['usersCount'] : '-1';

			if ( ! empty( $settings['includeUserRoles'] ) ) {
				$args['capability__in'] = $settings['includeUserRoles'];
			}

			if ( ! empty( $settings['excludeUserRoles'] ) ) {
				$args['capability__not_in'] = $settings['excludeUserRoles'];
			}

			if ( ! empty( $settings['excludeUsers'] ) ) {
				$args['exclude'] = $settings['excludeUsers'];
			}
		} else {
			if ( ! empty( $settings['selectUsers'] ) ) {
				$args['include'] = $settings['selectUsers'];
			}
		}

		return get_users( $args );
	}

	public function render() {
		$settings = $this->settings;

		$author_ids = [];
		$layout = isset( $settings['layout'] ) ? $settings['layout'] : 'list';

		$columns                        = ! empty( $settings['columns'] ) ? $settings['columns'] : 2;
		$columns_width_mobile_portrait  = isset( $settings['columns_mobile_portrait'] ) ? $this->calc_column_width( $settings['columns_mobile_portrait'] ) : 100;
		$columns_width_mobile_landscape = isset( $settings['columns_mobile_landscape'] ) ? $this->calc_column_width( $settings['columns_mobile_landscape'] ) : 50;
		$columns_width_tablet_portrait  = isset( $settings['columns_tablet_portrait'] ) ? $this->calc_column_width( $settings['columns_tablet_portrait'] ) : 50;

		$author_ids = [];
		$users      = $this->get_user_data();
		$post_types = empty( $settings['postType'] ) ? 'post' : $settings['postType'];

		foreach ( $users as $user ) {
			$user_post_count = count_user_posts( $user->ID, $post_types, true );

			$author_ids[] = [
				'autor_id'   => $user->ID,
				'post_count' => $user_post_count,
			];
		}

		if ( empty( $author_ids ) ) {
			echo $this->render_element_placeholder( [ 
				'icon-class' => 'ti-user',
				'title' => esc_html__( 'No author found', 'max-addons' ) 
				] 
			);
		}

		// Set element attributes
		$wrapper_classes = [ 'mab-author-list-wrapper', 'bricks-layout-wrapper' ];

		// Set attribute tag for '_root'
		$this->set_attribute( '_root', 'class', $wrapper_classes );
		$this->set_attribute( '_root', 'data-layout', $layout );

		$item_classes = [ 'mab-author-list', 'bricks-layout-item' ];
		?>
		<div <?php echo wp_kses_post( $this->render_attributes( '_root' ) ); ?>>
			<?php foreach ( $author_ids as $index => $author_id ) : ?>
				<?php
					$this->set_attribute( "item-$index", 'class', $item_classes );
				?>
				<div <?php echo $this->render_attributes( "item-$index" ); ?>>
					<div class="bricks-layout-inner">
						<?php
						if ( isset( $settings['authorAvatarType'] ) ) :
							$author_url = '';

							if ( isset( $settings['linkAvatarTo'] ) ) {
								if ( 'authorUrl' === $settings['linkAvatarTo'] ) {
									$author_url = get_the_author_meta( 'user_url', $author_id['autor_id'] );
								} else {
									$author_url = get_author_posts_url( $author_id['autor_id'] );
								}
							}

							if ( $settings['authorAvatarType'] == 'gravatar' ) : ?>
								<div class="mab-author-list-avatar">
									<?php
									if ( isset( $settings['linkAvatarTo'] ) && $author_url ) {
										printf( '<a href="%s">%s</a>',
											esc_url( $author_url ),
											get_avatar( $author_id['autor_id'], $settings['authorAvatarSize'] )
										);
									} else {
										echo get_avatar( $author_id['autor_id'], $settings['authorAvatarSize'] );
									}
									?>
								</div>
								<?php
							elseif ( $settings['authorAvatarType'] == 'image' ) :?>
								<div class="mab-author-list-avatar">
									<?php
									if ( isset( $settings['linkAvatarTo'] ) && $author_url ) {
										printf( '<a href="%s">%s</a>',
											esc_url( $author_url ),
											wp_kses_post( $this->render_image() )
										);
									} else {
										echo wp_kses_post( $this->render_image() );
									}
									?>
								</div>
								<?php
							elseif ( $settings['authorAvatarType'] == 'icon' ) :
								if ( isset( $settings['authorIcon'] ) ) { ?>
									<div class="mab-author-list-avatar">
										<?php
										if ( isset( $settings['linkAvatarTo'] ) && $author_url ) { ?>
											<a href="<?php echo esc_url( $author_url ); ?>">
												<?php echo isset( $settings['authorIcon'] ) ? self::render_icon( $settings['authorIcon'] ) : false; ?>
											</a>
											<?php
										} else {
											echo isset( $settings['authorIcon'] ) ? self::render_icon( $settings['authorIcon'] ) : false;
										}
										?>
									</div>
									<?php
								}
							endif;
						endif;
						?>
						<div class="mab-author-list-meta">
							<?php
							// Content wrapper
							if ( isset( $settings['fields'] ) && is_array( $settings['fields'] ) ) {
								$count = 1;

								foreach ( $settings['fields'] as $index => $field ) {
									$this->set_attribute(
										"content-wrapper-$count",
										'class',
										[
											'content-wrapper',
										]
									);

									$tag      = ! empty( $field['tag'] ) ? esc_attr( $field['tag'] ) : 'div';
									$field_id = isset( $field['id'] ) ? $field['id'] : $index;

									//echo '<div ' . $this->render_attributes( "content-wrapper-$count" ) . '>';
									$field_type = $field['fieldType'];

									if ( 'name' === $field_type ) {
										$name_type = isset( $field['nameType'] ) ? esc_attr( $field['nameType'] ) : 'display_name';

										if ( isset( $field['linkNameTo'] ) ) {
											if ( 'authorUrl' === $field['linkNameTo'] ) {
												$name_url = get_the_author_meta( 'user_url', $author_id['autor_id'] );
											} else {
												$name_url = get_author_posts_url( $author_id['autor_id'] );
											}

											if ( '' !== $name_url ) {
												$content = sprintf( '<a href="%s">%s</a>',
													esc_url( $name_url ),
													esc_html( get_the_author_meta( $name_type, $author_id['autor_id'] ) )
												);
											} else {
												$content = esc_html( get_the_author_meta( $name_type, $author_id['autor_id'] ) );
											}
										} else {
											$content = esc_html( get_the_author_meta( $name_type, $author_id['autor_id'] ) );
										}
									}

									if ( 'email' === $field_type ) {
										$content = esc_html( get_the_author_meta( 'user_email', $author_id['autor_id'] ) );
									}

									if ( 'authorBio' === $field_type ) {
										$content = esc_html( get_the_author_meta( 'description', $author_id['autor_id'] ) );
									}

									echo '<' . $tag . ' class="repeater-item dynamic" data-field-id="' . $field_id . '">' . $content . '</' . $tag . '>';

									//echo '</div>';

									$count++;
								}
							}
							?>
						</div>

						<?php if ( isset( $settings['postsCount'] ) ) : ?>
							<div class="mab-author-list-post-count">
								<?php
								echo isset( $settings['postsCountText'] ) ? esc_html( $settings['postsCountText'] ) : '';
								echo __( $author_id['post_count'], 'max-addons' );
								?>
							</div>
						<?php endif; ?>
					</div>
				</div>
			<?php endforeach; ?>
		</div>

		<?php
	}
}
