<?php
namespace MaxAddonsPro\Elements;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Album_Element extends \Bricks\Element {
	// Element properties
	public $category     = 'max-addons-elements'; // Use predefined element category 'general'
	public $name         = 'max-album'; // Make sure to prefix your elements
	public $icon         = 'ti-agenda max-element'; // Themify icon font class
	public $css_selector = ''; // Default CSS selector
	public $scripts      = [ 'mabAlbum' ]; // Script(s) run when element is rendered on frontend or updated in builder

	public function get_label() {
		return esc_html__( 'Album', 'max-addons' );
	}

	public function get_keywords() {
		return [ 'gallery', 'lightbox' ];
	}

	// Enqueue element styles and scripts
	public function enqueue_scripts() {
		$settings = $this->settings;

		wp_enqueue_style( 'mab-album');

		if ( isset( $settings['lightboxLibrary'] ) && $settings['lightboxLibrary'] === 'fancybox' ) {
			wp_enqueue_script( 'fancybox' );
			wp_enqueue_script( 'mab-album' );
			wp_enqueue_style( 'fancybox' );
		} else {
			wp_enqueue_script( 'bricks-photoswipe' );
			wp_enqueue_script( 'bricks-photoswipe-lightbox' );
			wp_enqueue_style( 'bricks-photoswipe' );
		}
	}

	// Set builder control groups
	public function set_control_groups() {
		$this->control_groups['album'] = [
			'title' => esc_html__( 'Album', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['trigger'] = [
			'title' => esc_html__( 'Trigger', 'max-addons' ),
			'tab'   => 'content',
		];

		$this->control_groups['albumContent'] = [
			'title'    => esc_html__( 'Cover Content', 'max-addons' ),
			'tab'      => 'content',
			'required' => [ 'trigger', '=', 'cover' ],
		];

		$this->control_groups['lightboxSettings'] = [
			'title' => esc_html__( 'Lightbox', 'max-addons' ),
			'tab'   => 'content',
		];
	}

	public function set_controls() {

		// Apply CSS filters only to img tag
		$this->controls['_cssFilters']['css'] = [
			[
				'selector' => 'img',
				'property' => 'filter',
			],
		];

		$this->set_items_controls();
		$this->set_trigger_controls();
		$this->set_cover_content_controls();
		$this->set_settings_controls();
	}

	// Set items controls
	public function set_items_controls() {

		$this->controls['albumImages'] = [
			'tab'  		=> 'content',
			'group'		=> 'album',
			'type' 		=> 'image-gallery',
		];
	}

	// Set trigger controls
	public function set_trigger_controls() {
		$this->controls['trigger'] = [
			'tab'     => 'content',
			'group'   => 'trigger',
			'label'   => esc_html__( 'Trigger', 'max-addons' ),
			'type'    => 'select',
			'options' => [
				'cover'  => __( 'Album Cover', 'max-addons' ),
				'button' => __( 'Button', 'max-addons' ),
			],
			'default' => 'cover',
			'inline'  => true,
		];

		$this->controls['coverHeight'] = [
			'tab'         => 'content',
			'group'       => 'trigger',
			'label'       => esc_html__( 'Cover Image Height', 'max-addons' ),
			'type'        => 'number',
			'unit'        => 'px',
			'min'         => 50,
			'inline'      => true,
			'small'       => true,
			'css'         => [
				[
					'property' => 'height',
					'selector' => 'img',
				],
			],
			'default' 	  => 300,
			'required' => [ 'trigger', '=', 'cover' ],
		];

		$this->controls['objectFit'] = [
			'tab'      => 'content',
			'group'    => 'trigger',
			'label'    => esc_html__( 'Object fit', 'max-addons' ),
			'type'     => 'select',
			'inline'   => true,
			'options'  => $this->control_options['objectFit'],
			'css'      => [
				[
					'property' => 'object-fit',
					'selector' => 'img',
				],
			],
			'required' => [
				[ 'trigger', '=', 'cover' ],
				[ 'coverHeight', '!=', '' ]
			],
		];

		$this->controls['objectPosition'] = [
			'tab'      => 'content',
			'group'    => 'trigger',
			'label'    => esc_html__( 'Object position', 'max-addons' ),
			'type'     => 'text',
			'inline'   => true,
			'dd'       => false,
			'css'      => [
				[
					'property' => 'object-position',
					'selector' => 'img',
				],
			],
			'required' => [
				[ 'trigger', '=', 'cover' ],
				[ 'coverHeight', '!=', '' ]
			],
		];

		$this->controls['coverOverlay'] = [
			'tab'      => 'content',
			'group'    => 'trigger',
			'label'    => esc_html__( 'Image Overlay', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '&.overlay::before',
				],
			],
			'rerender' => true,
			'required' => [ 'trigger', '=', 'cover' ],
		];

		$this->controls['triggerButtonText'] = [
			'tab'            => 'content',
			'group'          => 'trigger',
			'label'          => esc_html__( 'Button Text', 'max-addons' ),
			'type'           => 'text',
			'default'        => esc_html__( 'View Album', 'max-addons' ),
			'hasDynamicData' => 'text',
			'required'       => [ 'trigger', '=', 'button' ],
		];

		$this->controls['triggerButtonSize'] = [
			'tab'         => 'content',
			'group'       => 'trigger',
			'label'       => esc_html__( 'Size', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['buttonSizes'],
			'inline'      => true,
			'reset'       => true,
			'placeholder' => esc_html__( 'Medium', 'max-addons' ),
			'required' => [
				[ 'trigger', '=', 'button' ],
				[ 'triggerButtonText', '!=', '' ]
			],
		];

		$this->controls['triggerButtonStyle'] = [
			'tab'         => 'content',
			'group'       => 'trigger',
			'label'       => esc_html__( 'Style', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['styles'],
			'inline'      => true,
			'reset'       => true,
			'default'     => 'primary',
			'placeholder' => esc_html__( 'None', 'max-addons' ),
			'required' => [
				[ 'trigger', '=', 'button' ],
				[ 'triggerButtonText', '!=', '' ]
			],
		];

		$this->controls['triggerButtonCircle'] = [
			'tab'      => 'content',
			'group'    => 'trigger',
			'label'    => esc_html__( 'Circle', 'max-addons' ),
			'type'     => 'checkbox',
			'reset'    => true,
			'required' => [
				[ 'trigger', '=', 'button' ],
				[ 'triggerButtonText', '!=', '' ]
			],
		];

		$this->controls['triggerButtonBlock'] = [
			'tab'      => 'content',
			'group'    => 'trigger',
			'label'    => esc_html__( 'Stretch', 'max-addons' ),
			'type'     => 'checkbox',
			'reset'    => true,
			'required' => [
				[ 'trigger', '=', 'button' ],
				[ 'triggerButtonText', '!=', '' ]
			],
		];

		$this->controls['triggerButtonOutline'] = [
			'tab'      => 'content',
			'group'    => 'trigger',
			'label'    => esc_html__( 'Outline', 'max-addons' ),
			'type'     => 'checkbox',
			'reset'    => true,
			'required' => [
				[ 'trigger', '=', 'button' ],
				[ 'triggerButtonText', '!=', '' ]
			],
		];

		$this->controls['triggerButtonTypography'] = [
			'tab'      => 'content',
			'group'    => 'trigger',
			'type'     => 'typography',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-album-trigger-button',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [
				[ 'trigger', '=', 'button' ],
				[ 'triggerButtonText', '!=', '' ]
			],
		];

		$this->controls['triggerButtonBackgroundColor'] = [
			'tab'      => 'content',
			'group'    => 'trigger',
			'type'     => 'color',
			'label'    => esc_html__( 'Background', 'max-addons' ),
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.mab-album-trigger-button',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [
				[ 'trigger', '=', 'button' ],
				[ 'triggerButtonText', '!=', '' ]
			],
		];

		$this->controls['triggerButtonBorder'] = [
			'tab'      => 'content',
			'group'    => 'trigger',
			'type'     => 'border',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.mab-album-trigger-button',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [
				[ 'trigger', '=', 'button' ],
				[ 'triggerButtonText', '!=', '' ]
			],
		];

		$this->controls['triggerButtonPadding'] = [
			'tab'      => 'content',
			'group'    => 'trigger',
			'type'     => 'spacing',
			'label'    => esc_html__( 'Padding', 'max-addons' ),
			'css'      => [
				[
					'property' => 'padding',
					'selector' => '.mab-album-trigger-button',
				],
			],
			'required' => [
				[ 'trigger', '=', 'button' ],
				[ 'triggerButtonText', '!=', '' ]
			],
		];
	}

	// Set cover content controls
	public function set_cover_content_controls() {
		// Icon
		$this->controls['iconSep'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'type'     => 'separator',
			'label'    => esc_html__( 'Icon', 'max-addons' ),
			'required' => [ 'trigger', '=', 'cover' ],
		];

		$this->controls['coverIcon'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'label'    => esc_html__( 'Icon', 'max-addons' ),
			'type'     => 'icon',
			'css' 	   => [
				[
					'selector' => '.mab-album-icon',
				],
			],
			'required' => [ 'trigger', '=', 'cover' ],
		];

		$this->controls['coverIconTypography'] = [
			'tab'         => 'content',
			'group'       => 'albumContent',
			'label'       => esc_html__( 'Typography', 'max-addons' ),
			'type'        => 'typography',
			'css'         => [
				[
					'property' => 'font',
					'selector' => '.mab-album-icon',
				],
			],
			'exclude'     => [
				'font-family',
				'font-weight',
				'font-style',
				'text-align',
				'text-decoration',
				'text-transform',
				'line-height',
				'letter-spacing',
			],
			'placeholder' => [
				'font-size' => 60,
			],
			'required'    => [
				[ 'trigger', '=', 'cover' ],
				[ 'coverIcon.icon', '!=', '' ]
			],
		];

		// Caption
		$this->controls['captionSep'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'type'     => 'separator',
			'label'    => esc_html__( 'Caption', 'max-addons' ),
			'required' => [ 'trigger', '=', 'cover' ],
		];

		$this->controls['caption'] = [
			'tab'     => 'content',
			'group'   => 'albumContent',
			'label'   => esc_html__( 'Caption Type', 'max-addons' ),
			'type'    => 'select',
			'options' => [
				'title'       => esc_html__( 'Title', 'max-addons' ),
				'caption'     => esc_html__( 'Caption', 'max-addons' ),
				'description' => esc_html__( 'Description', 'max-addons' ),
				'custom'      => esc_html__( 'Custom', 'max-addons' ),
			],
			'inline'  => true,
		];

		$this->controls['captionCustom'] = [
			'tab'            => 'content',
			'group'          => 'albumContent',
			'label'          => esc_html__( 'Custom Caption', 'max-addons' ),
			'type'           => 'text',
			'hasDynamicData' => 'text',
			'required'       => [
				[ 'trigger', '=', 'cover' ],
				[ 'caption', '=', 'custom' ]
			],
		];

		$this->controls['captionTag'] = [
			'tab'         => 'content',
			'group'       => 'albumContent',
			'label'       => esc_html__( 'HTML Tag', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'div' => 'div',
				'p'   => 'p',
				'h1'  => 'h1',
				'h2'  => 'h2',
				'h3'  => 'h3',
				'h4'  => 'h4',
				'h5'  => 'h5',
				'h6'  => 'h6',
			],
			'inline'      => true,
			'placeholder' => 'div',
			'required'    => [
				[ 'trigger', '=', 'cover' ],
				[ 'caption', '!=', '' ]
			],
		];

		$this->controls['captionTypography'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-album-caption',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required'    => [
				[ 'trigger', '=', 'cover' ],
				[ 'caption', '!=', '' ]
			],
		];

		$this->controls['captionSpacing'] = [
			'tab'         => 'content',
			'group'       => 'albumContent',
			'label'       => esc_html__( 'Spacing', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'css'         => [
				[
					'property' => 'margin-bottom',
					'selector' => '.mab-album-caption',
				],
			],
			'placeholder' => 20,
			'required'    => [
				[ 'trigger', '=', 'cover' ],
				[ 'caption', '!=', '' ]
			],
		];

		// Description
		$this->controls['descriptionSep'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'type'     => 'separator',
			'label'    => esc_html__( 'Description', 'max-addons' ),
			'required' => [ 'trigger', '=', 'cover' ],
		];

		$this->controls['description'] = [
			'tab'     => 'content',
			'group'   => 'albumContent',
			'label'   => esc_html__( 'Description Type', 'max-addons' ),
			'type'    => 'select',
			'options' => [
				'title'       => esc_html__( 'Title', 'max-addons' ),
				'caption'     => esc_html__( 'Caption', 'max-addons' ),
				'description' => esc_html__( 'Description', 'max-addons' ),
				'custom'      => esc_html__( 'Custom', 'max-addons' ),
			],
			'inline'  => true,
		];

		$this->controls['descriptionCustom'] = [
			'tab'            => 'content',
			'group'          => 'albumContent',
			'label'          => esc_html__( 'Custom Description', 'max-addons' ),
			'type'           => 'text',
			'hasDynamicData' => 'text',
			'required'       => [
				[ 'trigger', '=', 'cover' ],
				[ 'description', '=', 'custom' ]
			],
		];

		$this->controls['descriptionTag'] = [
			'tab'         => 'content',
			'group'       => 'albumContent',
			'label'       => esc_html__( 'HTML Tag', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'div' => 'div',
				'p'   => 'p',
				'h1'  => 'h1',
				'h2'  => 'h2',
				'h3'  => 'h3',
				'h4'  => 'h4',
				'h5'  => 'h5',
				'h6'  => 'h6',
			],
			'inline'      => true,
			'placeholder' => 'div',
			'required'    => [
				[ 'trigger', '=', 'cover' ],
				[ 'description', '!=', '' ]
			],
		];

		$this->controls['descriptionTypography'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'type'     => 'typography',
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-album-description',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [
				[ 'trigger', '=', 'cover' ],
				[ 'description', '!=', '' ]
			],
		];

		$this->controls['descriptionSpacing'] = [
			'tab'         => 'content',
			'group'       => 'albumContent',
			'label'       => esc_html__( 'Spacing', 'max-addons' ),
			'type'        => 'number',
			'units'       => true,
			'css'         => [
				[
					'property' => 'margin-bottom',
					'selector' => '.mab-album-description',
				],
			],
			'placeholder' => 20,
			'required'    => [
				[ 'trigger', '=', 'cover' ],
				[ 'description', '!=', '' ]
			],
		];

		// Cover Button
		$this->controls['coverButtonSep'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'type'     => 'separator',
			'label'    => esc_html__( 'Button', 'max-addons' ),
			'required' => [ 'trigger', '=', 'cover' ],
		];

		$this->controls['coverButtonText'] = [
			'tab'            => 'content',
			'group'          => 'albumContent',
			'label'          => esc_html__( 'Button Text', 'max-addons' ),
			'type'           => 'text',
			'hasDynamicData' => 'text',
			'required'       => [ 'trigger', '=', 'cover' ],
		];

		$this->controls['coverButtonSize'] = [
			'tab'         => 'content',
			'group'       => 'albumContent',
			'label'       => esc_html__( 'Size', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['buttonSizes'],
			'inline'      => true,
			'reset'       => true,
			'placeholder' => esc_html__( 'Medium', 'max-addons' ),
			'required'    => [
				[ 'trigger', '=', 'cover' ],
				[ 'coverButtonText', '!=', '' ]
			],
		];

		$this->controls['coverButtonStyle'] = [
			'tab'         => 'content',
			'group'       => 'albumContent',
			'label'       => esc_html__( 'Style', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['styles'],
			'inline'      => true,
			'reset'       => true,
			'default'     => 'primary',
			'placeholder' => esc_html__( 'None', 'max-addons' ),
			'required' => [
				[ 'trigger', '=', 'cover' ],
				[ 'coverButtonText', '!=', '' ]
			],
		];

		$this->controls['coverButtonCircle'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'label'    => esc_html__( 'Circle', 'max-addons' ),
			'type'     => 'checkbox',
			'reset'    => true,
			'required' => [
				[ 'trigger', '=', 'cover' ],
				[ 'coverButtonText', '!=', '' ]
			],
		];

		$this->controls['coverButtonBlock'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'label'    => esc_html__( 'Stretch', 'max-addons' ),
			'type'     => 'checkbox',
			'reset'    => true,
			'required' => [
				[ 'trigger', '=', 'cover' ],
				[ 'coverButtonText', '!=', '' ]
			],
		];

		$this->controls['coverButtonOutline'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'label'    => esc_html__( 'Outline', 'max-addons' ),
			'type'     => 'checkbox',
			'reset'    => true,
			'required' => [
				[ 'trigger', '=', 'cover' ],
				[ 'coverButtonText', '!=', '' ]
			],
		];

		$this->controls['coverButtonTypography'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'type'     => 'typography',
			'label'    => esc_html__( 'Typography', 'max-addons' ),
			'css'      => [
				[
					'property' => 'font',
					'selector' => '.mab-album-cover-button',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [
				[ 'trigger', '=', 'cover' ],
				[ 'coverButtonText', '!=', '' ]
			],
		];

		$this->controls['coverButtonBackgroundColor'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'type'     => 'color',
			'label'    => esc_html__( 'Background', 'max-addons' ),
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.mab-album-cover-button',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [
				[ 'trigger', '=', 'cover' ],
				[ 'coverButtonText', '!=', '' ]
			],
		];

		$this->controls['coverButtonBorder'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'type'     => 'border',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.mab-album-cover-button',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [
				[ 'trigger', '=', 'cover' ],
				[ 'coverButtonText', '!=', '' ]
			],
		];

		$this->controls['coverButtonPadding'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'type'     => 'spacing',
			'label'    => esc_html__( 'Padding', 'max-addons' ),
			'css'      => [
				[
					'property' => 'padding',
					'selector' => '.mab-album-cover-button',
				],
			],
			'required' => [
				[ 'trigger', '=', 'cover' ],
				[ 'coverButtonText', '!=', '' ]
			],
		];

		// Content
		$this->controls['contentSep'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'type'     => 'separator',
			'label'    => esc_html__( 'Content', 'max-addons' ),
			'required' => [ 'trigger', '=', 'cover' ],
		];

		$this->controls['contentVerticalAlign'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'label'    => esc_html__( 'Vertical Align', 'max-addons' ),
			'type'     => 'justify-content',
			'css'      => [
				[
					'property' => 'justify-content',
					'selector' => '.mab-album-content-wrap',
				],
			],
			'default'  => 'flex-end',
			'exclude'  => 'space',
			'inline'   => true,
			'required' => [ 'trigger', '=', 'cover' ],
		];

		$this->controls['contentHorizontalAlign'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'label'    => esc_html__( 'Horizontal Align', 'max-addons' ),
			'type'     => 'align-items',
			'css'      => [
				[
					'property' => 'align-items',
					'selector' => '.mab-album-content-wrap',
				],
			],
			'default'  => 'flex-start',
			'inline'   => true,
			'required' => [ 'trigger', '=', 'cover' ],
		];

		$this->controls['contentAlign'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'label'    => esc_html__( 'Text align', 'max-addons' ),
			'type'     => 'text-align',
			'css'      => [
				[
					'property' => 'text-align',
					'selector' => '.mab-album-content',
				],
			],
			'inline'   => true,
			'required' => [ 'trigger', '=', 'cover' ],
		];

		$this->controls['contentBackground'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'label'    => esc_html__( 'Background', 'max-addons' ),
			'type'     => 'color',
			'css'      => [
				[
					'property' => 'background-color',
					'selector' => '.mab-album-content',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'trigger', '=', 'cover' ],
		];

		$this->controls['contentBorder'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'label'    => esc_html__( 'Border', 'max-addons' ),
			'type'     => 'border',
			'css'      => [
				[
					'property' => 'border',
					'selector' => '.mab-album-content',
				],
			],
			'inline'   => true,
			'small'    => true,
			'required' => [ 'trigger', '=', 'cover' ],
		];

		$this->controls['contentPadding'] = [
			'tab'      => 'content',
			'group'    => 'albumContent',
			'label'    => esc_html__( 'Padding', 'max-addons' ),
			'type'     => 'spacing',
			'css'      => [
				[
					'property' => 'padding',
					'selector' => '.mab-album-content',
				],
			],
			'required' => [ 'trigger', '=', 'cover' ],
		];
	}

	// Set settings controls
	public function set_settings_controls() {

		$this->controls['lightboxLibrary'] = [
			'tab'         => 'content',
			'group'   	  => 'lightboxSettings',
			'label'       => esc_html__( 'Lightbox Library', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'bricks'   => esc_html__( 'Bricks', 'max-addons' ),
				'fancybox' => esc_html__( 'Fancybox', 'max-addons' ),
			],
			'inline'      => true,
			'placeholder' => esc_html__( 'Bricks', 'max-addons' ),
		];

		$this->controls['lightboxImageSize'] = [
			'tab'         => 'content',
			'group'       => 'lightboxSettings',
			'label'       => esc_html__( 'Lightbox image size', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['imageSizes'],
			'placeholder' => esc_html__( 'Full', 'max-addons' ),
		];

		$this->controls['lightboxAnimationType'] = [
			'tab'         => 'content',
			'group'       => 'lightboxSettings',
			'label'       => esc_html__( 'Lightbox animation type', 'max-addons' ),
			'type'        => 'select',
			'options'     => $this->control_options['lightboxAnimationTypes'],
			'placeholder' => esc_html__( 'Zoom', 'max-addons' ),
			'required'    => [ 'lightboxLibrary', '!=', 'fancybox' ],
		];

		$this->controls['lightboxCaptionType'] = [
			'tab'         => 'content',
			'group'       => 'lightboxSettings',
			'label'       => esc_html__( 'Caption', 'max-addons' ),
			'type'        => 'select',
			'options'     => [
				'title'       => esc_html__( 'Title', 'max-addons' ),
				'caption'     => esc_html__( 'Caption', 'max-addons' ),
				'description' => esc_html__( 'Description', 'max-addons' ),
			],
			'default'     => 'caption',
			'placeholder' => esc_html__( 'None', 'max-addons' ),
			'inline'      => true,
			'required'    => [ 'lightboxLibrary', '=', 'fancybox' ],
		];

		$this->controls['lightboxArrows'] = [
			'tab'      => 'content',
			'group'    => 'lightboxSettings',
			'label'    => esc_html__( 'Arrows', 'max-addons' ),
			'type'     => 'checkbox',
			'default'  => true,
			'required' => [ 'lightboxLibrary', '=', 'fancybox' ],
		];

		$this->controls['lightboxToolbar'] = [
			'tab'      => 'content',
			'group'    => 'lightboxSettings',
			'label'    => esc_html__( 'Toolbar', 'max-addons' ),
			'type'     => 'checkbox',
			'default'  => true,
			'required' => [ 'lightboxLibrary', '=', 'fancybox' ],
		];

		$toolbar_buttons = [
			'infobar'    => __( 'Slides Counter', 'max-addons' ),
			'zoomIn'     => __( 'Zoom In', 'max-addons' ),
			'zoomOut'    => __( 'Zoom Out', 'max-addons' ),
			'toggle1to1' => __( 'Toggle 1:1', 'max-addons' ),
			'fullScreen' => __( 'Full Screen', 'max-addons' ),
			'rotateCCW'  => __( 'Rotate Counterclockwise', 'max-addons' ),
			'rotateCW'   => __( 'Rotate Clockwise', 'max-addons' ),
			'flipX'      => __( 'Flip Horizontally', 'max-addons' ),
			'flipY'      => __( 'Flip Vertically', 'max-addons' ),
			'slideshow'  => __( 'Slideshow', 'max-addons' ),
			'thumbs'     => __( 'Thumbs', 'max-addons' ),
			'close'      => __( 'Close', 'max-addons' ),
		];

		$this->controls['lightboxToolbarButtonsLeft'] = [
			'tab'      => 'content',
			'group'    => 'lightboxSettings',
			'label'    => esc_html__( 'Toolbar Buttons Left', 'max-addons' ),
			'type'     => 'select',
			'options'  => $toolbar_buttons,
			'default'  => [ 'infobar' ],
			'inline'   => false,
			'multiple' => true,
			'required' => [
				[ 'lightboxLibrary', '=', 'fancybox' ],
				[ 'lightboxToolbar', '!=', '' ],
			],
		];

		$this->controls['lightboxToolbarButtonsMiddle'] = [
			'tab'      => 'content',
			'group'    => 'lightboxSettings',
			'label'    => esc_html__( 'Toolbar Buttons Middle', 'max-addons' ),
			'type'     => 'select',
			'options'  => $toolbar_buttons,
			'default'  => [ 'zoomIn', 'zoomOut', 'rotateCCW', 'rotateCW' ],
			'inline'   => false,
			'multiple' => true,
			'required' => [
				[ 'lightboxLibrary', '=', 'fancybox' ],
				[ 'lightboxToolbar', '!=', '' ],
			],
		];

		$this->controls['lightboxToolbarButtonsRight'] = [
			'tab'      => 'content',
			'group'    => 'lightboxSettings',
			'label'    => esc_html__( 'Toolbar Buttons Right', 'max-addons' ),
			'type'     => 'select',
			'options'  => $toolbar_buttons,
			'default'  => [ 'slideshow', 'thumbs', 'close' ],
			'inline'   => false,
			'multiple' => true,
			'required' => [
				[ 'lightboxLibrary', '=', 'fancybox' ],
				[ 'lightboxToolbar', '!=', '' ],
			],
		];

		$this->controls['lightboxThumbsAutoStart'] = [
			'tab'         => 'content',
			'group'       => 'lightboxSettings',
			'label'       => esc_html__( 'Thumbs Auto Start', 'max-addons' ),
			'description' => __( 'Display thumbnails on lightbox opening', 'max-addons' ),
			'type'        => 'checkbox',
			'required'    => [ 'lightboxLibrary', '=', 'fancybox' ],
		];

		$this->controls['lightboxThumbstype'] = [
			'tab'      => 'content',
			'group'    => 'lightboxSettings',
			'label'    => esc_html__( 'Thumbs Type', 'max-addons' ),
			'type'     => 'select',
			'options'  => [
				'classic' => __( 'Classic', 'max-addons' ),
				'modern'  => __( 'Modern', 'max-addons' ),
			],
			'inline'   => true,
			'required' => [
				[ 'lightboxLibrary', '=', 'fancybox' ],
				[ 'lightboxToolbar', '!=', '' ],
			],
		];

		/* $this->controls['lightboxAnimation'] = [
			'tab'      => 'content',
			'group'    => 'lightboxSettings',
			'label'    => esc_html__( 'Animation', 'max-addons' ),
			'type'     => 'select',
			'options'  => [
				'fade'        => __( 'Fade', 'powerpack' ),
				'zoom'        => __( 'Zoom', 'powerpack' ),
				'zoom-in-out' => __( 'Zoom in Out', 'powerpack' ),
			],
			'inline'   => true,
			'required' => [
				[ 'link_to', '=', 'lightbox' ],
				[ 'lightboxLibrary', '=', 'fancybox' ],
				[ 'lightboxToolbar', '!=', '' ],
			],
		]; */

		/* $this->controls['lightboxTransitionEffect'] = [
			'tab'      => 'content',
			'group'    => 'lightboxSettings',
			'label'    => esc_html__( 'Transition Effect', 'max-addons' ),
			'type'     => 'select',
			'options'  => [
				'fade'        => __( 'Fade', 'powerpack' ),
				'slide'       => __( 'Slide', 'powerpack' ),
				'circular'    => __( 'Circular', 'powerpack' ),
				'tube'        => __( 'Tube', 'powerpack' ),
				'zoom-in-out' => __( 'Zoom in Out', 'powerpack' ),
				'rotate'      => __( 'Rotate', 'powerpack' ),
			],
			'inline'   => true,
			'required' => [
				[ 'link_to', '=', 'lightbox' ],
				[ 'lightboxLibrary', '=', 'fancybox' ],
				[ 'lightboxToolbar', '!=', '' ],
			],
		]; */
	}

	public function render_album_caption( $image_id ) {
		$settings = $this->settings;

		$title_tag  = ! empty( $settings['captionTag'] ) ? esc_attr( $settings['captionTag'] ) : 'div';

		$this->set_attribute( 'album-title', 'class', 'mab-album-caption' );

		$output = "<{$title_tag} {$this->render_attributes( 'album-title' )}>";
			$output .= $this->render_image_caption( $image_id, 'caption' );
		$output .= "</{$title_tag}>";

		echo wp_kses_post( $output );
	}

	public function render_album_description( $image_id ) {
		$settings = $this->settings;

		$description_tag  = ! empty( $settings['descriptionTag'] ) ? esc_attr( $settings['descriptionTag'] ) : 'div';

		$this->set_attribute( 'album-description', 'class', 'mab-album-description' );

		$output = "<{$description_tag} {$this->render_attributes( 'album-description' )}>";
			$output .= $this->render_image_caption( $image_id, 'description' );
		$output .= "</{$description_tag}>";

		echo wp_kses_post( $output );
	}

	public function render_album_cover_button() {
		$settings = $this->settings;

		$button_classes = array(
			'mab-album-cover-button',
			'bricks-button'
		);

		// Button size
		if ( isset( $settings['coverButtonSize'] ) ) {
			$button_classes[] = $settings['coverButtonSize'];
		}

		// Button outline
		if ( isset( $settings['coverButtonStyle'] ) ) {
			if ( isset( $settings['coverButtonOutline'] ) ) {
				$button_classes[] = 'outline';
				$button_classes[] = 'bricks-color-' . $settings['coverButtonStyle'];
			} else {
				// Fill (default)
				$button_classes[] = 'bricks-background-' . $settings['coverButtonStyle'];
			}
		}

		// Button circle
		if ( isset( $settings['coverButtonCircle'] ) ) {
			$button_classes[] = 'circle';
		}

		// Button stretch
		if ( isset( $settings['coverButtonBlock'] ) ) {
			$button_classes[] = 'block';
		}

		$this->set_attribute( 'cover-button', 'class', $button_classes );
		?>
		<span <?php echo wp_kses_post( $this->render_attributes( 'cover-button' ) ); ?>>
			<?php if ( isset( $settings['coverButtonText'] ) ) { ?>
				<?php echo esc_html( $settings['coverButtonText'] ); ?>
			<?php } ?>
		</span>
		<?php
	}

	public function render_album_content( $image_id ) {
		$settings = $this->settings;

		$has_icon        = isset( $settings['coverIcon'] );
		$has_caption     = isset( $settings['caption'] );
		$has_description = isset( $settings['description'] );
		$has_button      = isset( $settings['coverButtonText'] );

		if ( $has_icon || $has_caption || $has_description || $has_button ) {
			?>
			<div class="mab-album-content-wrap mab-media-content">
				<div class="mab-album-content">
					<?php if ( $has_icon ) { ?>
						<div class="mab-icon mab-album-icon">
							<?php echo self::render_icon( $settings['coverIcon'] ); ?>
						</div>
					<?php } ?>
					<?php
					if ( isset( $settings['caption'] ) ) {
						$this->render_album_caption( $image_id );
					}

					if ( isset( $settings['description'] ) ) {
						$this->render_album_description( $image_id );
					}

					if ( isset( $settings['coverButtonText'] ) ) {
						$this->render_album_cover_button();
					}
					?>
				</div>
			</div>
			<?php
		}
	}

	public function render_album_cover() {
		$settings = $this->settings;

		$album = isset( $settings['albumImages'] ) ? $settings['albumImages'] : [];
		$size  = ! empty( $album['size'] ) ? $album['size'] : BRICKS_DEFAULT_IMAGE_SIZE;

		if ( ! count( $album ) ) {
			return;
		}

		$link_key      = 'album-cover-link';
		$first_item    = $album['images'][0];
		$first_item_id = $first_item['id'];

		$this->get_lightbox_atts( $link_key, $first_item );
		?>
		<a <?php echo wp_kses_post( $this->render_attributes( $link_key ) ); ?>>
			<?php
				$this->render_album_content( $first_item_id );

				echo wp_get_attachment_image( $first_item_id, $size );
			?>
		</a>
		<?php
	}

	public function render_album_trigger_button() {
		$settings = $this->settings;

		$button_classes = array(
			'mab-album-trigger-button',
			'bricks-button'
		);

		// Button size
		if ( isset( $settings['triggerButtonSize'] ) ) {
			$button_classes[] = $settings['triggerButtonSize'];
		}

		// Button outline
		if ( isset( $settings['triggerButtonStyle'] ) ) {
			if ( isset( $settings['triggerButtonOutline'] ) ) {
				$button_classes[] = 'outline';
				$button_classes[] = 'bricks-color-' . $settings['triggerButtonStyle'];
			} else {
				// Fill (default)
				$button_classes[] = 'bricks-background-' . $settings['triggerButtonStyle'];
			}
		}

		// Button circle
		if ( isset( $settings['triggerButtonCircle'] ) ) {
			$button_classes[] = 'circle';
		}

		// Button stretch
		if ( isset( $settings['triggerButtonBlock'] ) ) {
			$button_classes[] = 'block';
		}

		$this->set_attribute( 'trigger-button', 'class', $button_classes );

		$album = isset( $settings['albumImages'] ) ? $settings['albumImages'] : [];

		if ( ! count( $album ) ) {
			return;
		}

		$album_first_item = $album['images'][0];

		$this->get_lightbox_atts( 'trigger-button', $album_first_item );

		$this->set_attribute( 'trigger-button-content', 'class', 'mab-album-button-content' );
		?>
		<a <?php echo wp_kses_post( $this->render_attributes( 'trigger-button' ) ); ?>>
			<span <?php echo wp_kses_post( $this->render_attributes( 'trigger-button-content' ) ); ?>>
				<?php if ( isset( $settings['triggerButtonText'] ) ) { ?>
					<span>
						<?php echo esc_html( $settings['triggerButtonText'] ); ?>
					</span>
				<?php } ?>
			</span>
		</a>
		<?php
	}

	public function render() {
		$settings = $this->settings;

		if ( isset( $settings['lightboxLibrary'] ) && $settings['lightboxLibrary'] === 'fancybox' ) {
			$fancybox_settings = $this->fancybox_settings();

			$this->set_attribute( '_root', 'data-fancybox-settings', wp_json_encode( $fancybox_settings ) );
		}

		$has_overlay = isset( $settings['coverOverlay'] );

		if ( $has_overlay ) {
			$this->set_attribute( '_root', 'class', 'overlay' );
		}
		?>
		<div <?php echo wp_kses_post( $this->render_attributes( '_root' ) ); ?>>
			<?php
			if ( isset( $settings['albumImages'] ) ) {
				if ( 'cover' === $settings['trigger'] ) {
					// Album Cover
					$this->render_album_cover();
				} else {
					// Album Trigger Button
					$this->render_album_trigger_button();
				}
				?>
				<div class="mab-album-gallery">
					<?php $this->render_album_images(); ?>
				</div>
				<?php
			} else {
				echo wp_kses_post( $this->render_element_placeholder(
					array(
						'icon-class' => 'ti-gallery',
						'title' => __( 'Album is empty!', 'max-addons' ),
					)
				) );
			}
			?>
		</div>
		<?php
	}

	/**
	 * Render album items
	 */
	protected function render_album_images() {
		$settings = $this->settings;

		$gallery  = isset( $settings['albumImages'] ) ? $settings['albumImages']['images'] : [];
		$is_first = true;

		if ( ! count( $gallery ) ) {
			return;
		}

		foreach ( $gallery as $index => $item ) {
			if ( $is_first ) {
				$is_first = false;
				continue;
			}

			$image_key = 'album_image_' . $index;

			$this->set_attribute( $image_key, 'class', 'mab-album-image' );
			$this->get_lightbox_atts( $image_key, $item );

			echo '<a ' . wp_kses_post( $this->render_attributes( $image_key ) ) . '></a>';
		}
	}

	/**
	 * Lightbox attributes.
	 *
	 * @return array
	 */
	protected function get_lightbox_atts( $key = '', $item = '' ) {
		$settings = $this->settings;

		if ( isset( $settings['lightboxLibrary'] ) && 'fancybox' === $settings['lightboxLibrary'] ) {
			$fancybox_options = array();

			if ( ! isset( $settings['lightboxArrows'] ) ) {
				$fancybox_options['Carousel'] = array(
					'Navigation' => ( isset( $settings['lightboxArrows'] ) ),
				);
			}

			if ( isset( $settings['lightboxToolbar'] ) ) {
				if ( isset( $settings['lightboxToolbarButtonsLeft'] ) ) {
					$toolbar_buttons_left = $settings['lightboxToolbarButtonsLeft'];
				} else {
					$toolbar_buttons_left = array();
				}

				if ( isset( $settings['lightboxToolbarButtonsMiddle'] ) ) {
					$toolbar_buttons_middle = $settings['lightboxToolbarButtonsMiddle'];
				} else {
					$toolbar_buttons_middle = array();
				}

				if ( isset( $settings['lightboxToolbarButtonsRight'] ) ) {
					$toolbar_buttons_right = $settings['lightboxToolbarButtonsRight'];
				} else {
					$toolbar_buttons_right = array();
				}

				$fancybox_options['Toolbar'] = array(
					'display' => array(
						'left'   => $toolbar_buttons_left,
						'middle' => $toolbar_buttons_middle,
						'right'  => $toolbar_buttons_right,
					),
				);
			} else {
				$fancybox_options['Toolbar'] = array(
					'enabled' => false
				);
			}

			$fancybox_options['Thumbs'] = array(
				'showOnStart' => ( isset( $settings['lightboxThumbsAutoStart'] ) ),
				'type'        => ( isset( $settings['lightboxThumbstype'] ) ? $settings['lightboxThumbstype'] : 'modern' ),
			);

			$caption_type  = isset( $settings['lightboxCaptionType'] ) ? $settings['lightboxCaptionType'] : '';
			$caption = $this->get_image_caption( $item['id'], $caption_type );

			$lightbox_image_size = ! empty( $settings['lightboxImageSize'] ) ? $settings['lightboxImageSize'] : 'full';
			$lightbox_image      = wp_get_attachment_image_src( $item['id'], $lightbox_image_size );
			$lightbox_image      = ! empty( $lightbox_image ) && is_array( $lightbox_image ) ? $lightbox_image : '';
			$lightbox_thumb      = wp_get_attachment_image_src( $item['id'], 'thumbnail' );
			$lightbox_thumb      = ! empty( $lightbox_thumb ) && is_array( $lightbox_thumb ) ? $lightbox_thumb : '';

			$this->set_attribute( $key, 'data-fancybox', 'mab-album-' . $this->id );
			$this->set_attribute( $key, 'data-caption', $caption );
			$this->set_attribute( $key, 'data-src', $lightbox_image[0] );
			$this->set_attribute( $key, 'data-thumb', $lightbox_thumb[0] );
		} else {
			$lightbox_image_size = ! empty( $settings['lightboxImageSize'] ) ? $settings['lightboxImageSize'] : 'full';
			$lightbox_image      = wp_get_attachment_image_src( $item['id'], $lightbox_image_size );
			$lightbox_image      = ! empty( $lightbox_image ) && is_array( $lightbox_image ) ? $lightbox_image : '';

			$this->set_attribute( $key, 'class', 'bricks-lightbox' );
			$this->set_attribute( $key, 'data-pswp-src', $lightbox_image[0] );
			$this->set_attribute( $key, 'data-pswp-width', $lightbox_image[1] );
			$this->set_attribute( $key, 'data-pswp-height', $lightbox_image[2] );
			$this->set_attribute( $key, 'data-pswp-id', $this->id );

			if ( ! empty( $settings['lightboxAnimationType'] ) ) {
				$this->set_attribute( $key, 'data-animation-type', esc_attr( $settings['lightboxAnimationType'] ) );
			}
		}
	}

	/**
	 * Fancybox Settings.
	 *
	 * @return array
	 */
	protected function fancybox_settings() {
		$settings = $this->settings;

		$base_class = 'mab-gallery-fancybox mab-gallery-fancybox-' . esc_attr( $this->id ) . ' ';

		$fancybox_options = array();

		if ( ! isset( $settings['lightboxArrows'] ) ) {
			$fancybox_options['Carousel'] = array(
				'Navigation' => ( isset( $settings['lightboxArrows'] ) ),
			);
		}

		if ( isset( $settings['lightboxToolbar'] ) ) {
			if ( isset( $settings['lightboxToolbarButtonsLeft'] ) ) {
				$toolbar_buttons_left = $settings['lightboxToolbarButtonsLeft'];
			} else {
				$toolbar_buttons_left = array();
			}

			if ( isset( $settings['lightboxToolbarButtonsMiddle'] ) ) {
				$toolbar_buttons_middle = $settings['lightboxToolbarButtonsMiddle'];
			} else {
				$toolbar_buttons_middle = array();
			}

			if ( isset( $settings['lightboxToolbarButtonsRight'] ) ) {
				$toolbar_buttons_right = $settings['lightboxToolbarButtonsRight'];
			} else {
				$toolbar_buttons_right = array();
			}

			$fancybox_options['Toolbar'] = array(
				'display' => array(
					'left'   => $toolbar_buttons_left,
					'middle' => $toolbar_buttons_middle,
					'right'  => $toolbar_buttons_right,
				),
			);
		} else {
			$fancybox_options['Toolbar'] = array(
				'enabled' => false
			);
		}

		$fancybox_options['Thumbs'] = array(
			'showOnStart' => ( isset( $settings['lightboxThumbsAutoStart'] ) ),
			'type'        => ( isset( $settings['lightboxThumbstype'] ) ? $settings['lightboxThumbstype'] : 'modern' ),
		);

		return $fancybox_options;
	}

	/**
	 * Render image caption
	 *
	 * @param  int $id image ID.
	 * @return $html
	 */
	protected function render_image_caption( $id, $type = 'caption' ) {
		$settings = $this->settings;

		if ( ! isset( $settings[ $type ] ) ) {
			return '';
		}

		$caption_type = isset( $settings[ $type ] ) ? $settings[ $type ] : '';

		$caption = $this->get_image_caption( $id, $caption_type );

		if ( '' === $caption ) {
			return '';
		}

		ob_start();

		echo wp_kses_post( $caption );

		$html = ob_get_contents();
		ob_end_clean();
		return $html;
	}

	/**
	 * Get Image Caption.
	 *
	 * @since 1.0.0
     *
	 * @access public
	 *
	 * @return string image caption.
	 */
    public function get_image_caption( $id, $caption_type = 'caption' ) {
		$settings   = $this->settings;
        $attachment = get_post( $id );
        
        $attachment_caption = '';

		switch ( $caption_type ) {
			case 'caption':
				$attachment_caption = wp_get_attachment_caption( $id );
				break;

			case 'title':
				$attachment_caption = $attachment->post_title;
				break;

			case 'description':
				$attachment_caption = $attachment->post_content;
				break;

			case 'custom':
				$attachment_caption = isset( $settings['captionCustom'] ) ? $settings['captionCustom'] : '';
				break;

			default:
				$attachment_caption = '';
				break;
		}

        return $attachment_caption;
	}
        
}
