<?php
namespace MaxAddonsPro;

use MaxAddonsPro\Classes\Helper;

/**
 * Class for Max Addons conditions
 *
 * @package MaxAddons
 * @since 1.5.0
 */

/**
 * Exit if accessed directly.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * MAB_Conditions.
 */
class MAB_Conditions {

	/**
	 * Conditions info
	 *
	 * @var conditions
	 */
	public static $conditions = null;

	/**
	 * Initializing conditions.
	 *
	 * @since 1.5.0
	 */
	public static function init() {
		$enabled_conditions = self::get_enabled_conditions();

		if ( 'disabled' != $enabled_conditions ) {
			add_filter( 'bricks/conditions/groups', array( __CLASS__, 'set_groups' ) );
			add_filter( 'bricks/conditions/options', array( __CLASS__, 'set_options' ) );
			add_filter( 'bricks/conditions/result', array( __CLASS__, 'set_conditions' ), 10, 3 );
		}
	}

	/**
	 * Returns an array of all Max Addons conditions that are enabled.
	 *
	 * @since 1.6.1
	 *
	 * @return array
	 */
	private static function get_enabled_conditions() {
		$enabled_conditions = \MaxAddonsPro\Classes\Admin_Settings::get_enabled_conditions();

		return $enabled_conditions;
	}

	/**
	 * Check if option is enabled from Max Addons settings
	 *
	 * @since 1.6.1
	 *
	 * @return bool
	 */
	private static function is_option_enabled( $option_name = '' ) {
		$enabled_conditions = self::get_enabled_conditions();

		if ( $option_name && in_array( $option_name, $enabled_conditions ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Set Max Addons conditions group
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function set_groups( $groups ) {
		$groups[] = [
			'name'  => 'mab_conditions',
			'label' => esc_html__( 'Max Addons', 'max-addons' ),
		];

		if ( class_exists( 'WooCommerce' ) ) {
			$groups[] = [
				'name'  => 'mab_woo_conditions',
				'label' => esc_html__( 'WooCommerce - Max Addons', 'max-addons' ),
			];
		}

        return $groups;
	}

	/**
	 * Set Max Addons conditions options
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function set_options( $options ) {
		// OPTIONS
		$math_options = [
			'==' => '==',
			'!=' => '!=',
			'>=' => '>=',
			'<=' => '<=',
			'>'  => '>',
			'<'  => '<',
		];

		$is_not_options = [
			'==' => esc_html__( 'is', 'max-addons' ),
			'!=' => esc_html__( 'is not', 'max-addons' ),
		];

		if ( self::is_option_enabled( 'post_category' ) ) {
			$options[] = [
				'key'     => 'mab_post_category', 
				'label'   => esc_html__( 'Post category', 'max-addons' ),
				'group'   => 'mab_conditions',
				'compare' => [
					'type'        => 'select',
					'options'     => $is_not_options,
					'placeholder' => esc_html__( 'is', 'max-addons' ),
				],
				'value'   => [
					'type'        => 'select',
					'options'     => bricks_is_builder() ? Helper::get_terms_options( 'category', 'id', false ) : [],
					'multiple'    => true,
					'placeholder' => esc_html__( 'Select', 'max-addons' ),
				],
			];
		}

		if ( self::is_option_enabled( 'post_tag' ) ) {
			$options[] = [
				'key'     => 'mab_post_tag', 
				'label'   => esc_html__( 'Post tag', 'max-addons' ),
				'group'   => 'mab_conditions',
				'compare' => [
					'type'        => 'select',
					'options'     => $is_not_options,
					'placeholder' => esc_html__( 'is', 'max-addons' ),
				],
				'value'   => [
					'type'        => 'select',
					'options'     => bricks_is_builder() ? Helper::get_terms_options( 'post_tag', 'id', false ) : [],
					'multiple'    => true,
					'placeholder' => esc_html__( 'Select', 'max-addons' ),
				],
			];
		}

		if ( self::is_option_enabled( 'post_type' ) ) {
			$options[] = [
				'key'     => 'mab_post_type', 
				'label'   => esc_html__( 'Post type', 'max-addons' ),
				'group'   => 'mab_conditions',
				'compare' => [
					'type'        => 'select',
					'options'     => $is_not_options,
					'placeholder' => esc_html__( 'is', 'max-addons' ),
				],
				'value'   => [
					'type'        => 'select',
					'options'     => bricks_is_builder() ? Helper::get_post_types() : [],
					'multiple'    => true,
					'placeholder' => esc_html__( 'Select', 'max-addons' ),
				],
			];
		}

		if ( self::is_option_enabled( 'post_comments_count' ) ) {
			$options[] = [
				'key'     => 'mab_post_comments_count', 
				'group'   => 'mab_conditions',
				'label'   => esc_html__( 'Post comments count', 'max-addons' ),
				'compare' => [
					'type'        => 'select',
					'options'     => $math_options,
					'placeholder' => esc_html__( '==', 'max-addons' )
				],
				'value'   => [
					'type'        => 'text',
					'placeholder' => esc_html__( '0', 'max-addons' ),
				],
			];
		}

		if ( self::is_option_enabled( 'post_comments_status' ) ) {
			$options[] = [
				'key'     => 'mab_post_comments_status', 
				'group'   => 'mab_conditions',
				'label'   => esc_html__( 'Post comments status', 'max-addons' ),
				'compare' => [
					'type'        => 'select',
					'options'     => $is_not_options,
					'placeholder' => esc_html__( 'is', 'max-addons' )
				],
				'value'   => [
					'type'        => 'select',
					'options'     => [
						'open'   => esc_html__( 'Open', 'max-addons' ),
						'closed' => esc_html__( 'Closed', 'max-addons' ),
					],
					'placeholder' => esc_html__( 'Select', 'max-addons' )
				],
			];
		}

		if ( self::is_option_enabled( 'author_archive' ) ) {
			$options[] = [
				'key'     => 'mab_author_archive', 
				'group'   => 'mab_conditions',
				'label'   => esc_html__( 'Author archive', 'max-addons' ),
				'compare' => [
					'type'        => 'select',
					'options'     => $is_not_options,
					'placeholder' => esc_html__( 'is', 'max-addons' )
				],
				'value'   => [
					'type'        => 'select',
					'options'     => bricks_is_builder() ? Helper::get_users_options() : [],
					'placeholder' => esc_html__( 'Select', 'max-addons' ),
				],
			];
		}

		if ( self::is_option_enabled( 'category_archive' ) ) {
			$options[] = [
				'key'     => 'mab_category_archive', 
				'group'   => 'mab_conditions',
				'label'   => esc_html__( 'Category archive', 'max-addons' ),
				'compare' => [
					'type'        => 'select',
					'options'     => $is_not_options,
					'placeholder' => esc_html__( 'is', 'max-addons' )
				],
				'value'   => [
					'type'        => 'select',
					'options'     => bricks_is_builder() ? Helper::get_terms_options( 'category', 'id', false ) : [],
					'placeholder' => esc_html__( 'Select', 'max-addons' ),
				],
			];
		}

		if ( self::is_option_enabled( 'tag_archive' ) ) {
			$options[] = [
				'key'     => 'mab_tag_archive', 
				'group'   => 'mab_conditions',
				'label'   => esc_html__( 'Tag archive', 'max-addons' ),
				'compare' => [
					'type'        => 'select',
					'options'     => $is_not_options,
					'placeholder' => esc_html__( 'is', 'max-addons' )
				],
				'value'   => [
					'type'        => 'select',
					'options'     => bricks_is_builder() ? Helper::get_terms_options( 'post_tag', 'id', false ) : [],
					'placeholder' => esc_html__( 'Select', 'max-addons' ),
				],
			];
		}

		if ( self::is_option_enabled( 'search_results' ) ) {
			$options[] = [
				'key'     => 'mab_search_results', 
				'group'   => 'mab_conditions',
				'label'   => esc_html__( 'Search', 'max-addons' ),
				'compare' => [
					'type'        => 'select',
					'options'     => $is_not_options,
					'placeholder' => esc_html__( 'is', 'max-addons' )
				],
				'value'   => [
					'type'        => 'text',
					'placeholder' => esc_html__( 'Keywords', 'max-addons' ),
					'description' => esc_html__( 'Enter keywords, separated by commas, to condition the display on specific keywords and leave blank for any.', 'max-addons' )
				],
			];
		}

		/* $options[] = [
			'key'     => 'mab_visitor_type', 
			'group'   => 'mab_conditions',
			'label'   => esc_html__( 'Visitor type', 'bricks' ),
			'compare' => [
				'type'        => 'select',
				'options'     => $is_not_options,
				'placeholder' => esc_html__( 'is', 'bricks' )
			],
			'value'   => [
				'type'        => 'select',
				'options'     => [
					'new'       => esc_html__( 'First Time Visitor', 'bricks' ),
					'returning' => esc_html__( 'Returning Visitor', 'bricks' ),
				],
				'placeholder' => esc_html__( 'Select', 'bricks' )
			],
		]; */

		if ( self::is_option_enabled( 'request_parameter' ) ) {
			$options[] = [
				'key'     => 'mab_request_parameter', 
				'group'   => 'mab_conditions',
				'label'   => esc_html__( 'Request parameter', 'max-addons' ),
				'compare' => [
					'type'        => 'select',
					'options'     => $is_not_options,
					'placeholder' => esc_html__( 'is', 'max-addons' )
				],
				'value'   => [
					'type'        => 'textarea',
					'placeholder' => '',
					'description' => esc_html__( 'Enter each request parameter on a new line as pairs of param=value or param1=value1&amp;param2=value2.', 'max-addons' )
				],
			];
		}

		if ( class_exists( 'WooCommerce' ) ) {
			if ( self::is_option_enabled( 'woo_product_category' ) ) {
				$options[] = [
					'key'     => 'mab_woo_product_category', 
					'group'   => 'mab_woo_conditions',
					'label'   => esc_html__( 'Product category', 'max-addons' ),
					'compare' => [
						'type'        => 'select',
						'options'     => $is_not_options,
						'placeholder' => esc_html__( 'is', 'max-addons' )
					],
					'value'   => [
						'type'        => 'select',
						'options'     => bricks_is_builder() ? Helper::get_terms_options( 'product_cat', 'id', false ) : [],
						'multiple'    => true,
						'placeholder' => esc_html__( 'Select', 'max-addons' ),
					],
				];
			}

			if ( self::is_option_enabled( 'woo_product_price' ) ) {
				$options[] = [
					'key'     => 'mab_woo_product_price', 
					'group'   => 'mab_woo_conditions',
					'label'   => esc_html__( 'Product price', 'max-addons' ),
					'compare' => [
						'type'        => 'select',
						'options'     => $math_options,
						'placeholder' => esc_html__( '==', 'max-addons' )
					],
					'value'   => [
						'type'        => 'text',
						'placeholder' => ''
					],
				];
			}

			if ( self::is_option_enabled( 'woo_product_rating' ) ) {
				$options[] = [
					'key'     => 'mab_woo_product_rating', 
					'group'   => 'mab_woo_conditions',
					'label'   => esc_html__( 'Product rating', 'max-addons' ),
					'compare' => [
						'type'        => 'select',
						'options'     => $math_options,
						'placeholder' => esc_html__( '==', 'max-addons' )
					],
					'value'   => [
						'type'        => 'text',
						'placeholder' => ''
					],
				];
			}

			if ( self::is_option_enabled( 'woo_product_stock' ) ) {
				$options[] = [
					'key'     => 'mab_woo_product_stock', 
					'group'   => 'mab_woo_conditions',
					'label'   => esc_html__( 'Product stock', 'max-addons' ),
					'compare' => [
						'type'        => 'select',
						'options'     => $math_options,
						'placeholder' => esc_html__( '==', 'max-addons' )
					],
					'value'   => [
						'type'        => 'text',
						'placeholder' => ''
					],
				];
			}

			if ( self::is_option_enabled( 'woo_product_type' ) ) {
				$options[] = [
					'key'     => 'mab_woo_product_type', 
					'group'   => 'mab_woo_conditions',
					'label'   => esc_html__( 'Product type', 'max-addons' ),
					'compare' => [
						'type'        => 'select',
						'options'     => $is_not_options,
						'placeholder' => esc_html__( 'is', 'max-addons' )
					],
					'value'   => [
						'type'        => 'select',
						'options'     =>  [
							'simple'   => esc_html__( 'Simple', 'max-addons' ),
							'grouped'  => esc_html__( 'Grouped', 'max-addons' ),
							'variable' => esc_html__( 'Variable', 'max-addons' ),
							'external' => esc_html__( 'External / Affiliate', 'max-addons' ),
						],
						'placeholder' => esc_html__( 'Select', 'max-addons' ),
					],
				];
			}

			if ( self::is_option_enabled( 'woo_orders_placed' ) ) {
				$options[] = [
					'key'     => 'mab_woo_orders_placed', 
					'group'   => 'mab_woo_conditions',
					'label'   => esc_html__( 'Orders placed', 'max-addons' ),
					'compare' => [
						'type'        => 'select',
						'options'     => $math_options,
						'placeholder' => esc_html__( '==', 'max-addons' )
					],
					'value'   => [
						'type'        => 'text',
						'placeholder' => ''
					],
				];
			}

			if ( self::is_option_enabled( 'woo_customer_total_spent' ) ) {
				$options[] = [
					'key'     => 'mab_woo_customer_total_spent', 
					'group'   => 'mab_woo_conditions',
					'label'   => esc_html__( 'Customer total spent', 'max-addons' ),
					'compare' => [
						'type'        => 'select',
						'options'     => $math_options,
						'placeholder' => esc_html__( '==', 'max-addons' )
					],
					'value'   => [
						'type'        => 'text',
						'placeholder' => esc_html__( '0', 'max-addons' ),
					],
				];
			}

			if ( self::is_option_enabled( 'woo_cart_products_count' ) ) {
				$options[] = [
					'key'     => 'mab_woo_cart_products_count', 
					'group'   => 'mab_woo_conditions',
					'label'   => esc_html__( 'Cart products count', 'max-addons' ),
					'compare' => [
						'type'        => 'select',
						'options'     => $math_options,
						'placeholder' => esc_html__( '==', 'max-addons' )
					],
					'value'   => [
						'type'        => 'text',
						'placeholder' => esc_html__( '0', 'max-addons' ),
					],
				];
			}

			if ( self::is_option_enabled( 'woo_cart_products' ) ) {
				$options[] = [
					'key'     => 'mab_woo_cart_products', 
					'group'   => 'mab_woo_conditions',
					'label'   => esc_html__( 'Cart products', 'max-addons' ),
					'compare' => [
						'type'        => 'select',
						'options'     => $is_not_options,
						'placeholder' => esc_html__( '==', 'max-addons' )
					],
					'value'   => [
						'type'        => 'select',
						'options'     => bricks_is_builder() ? Helper::get_posts_options( 'product' ) : [],
						'multiple'    => true,
						'placeholder' => esc_html__( 'Select', 'max-addons' ),
					],
				];
			}

			if ( self::is_option_enabled( 'woo_cart_total' ) ) {
				$options[] = [
					'key'     => 'mab_woo_cart_total', 
					'group'   => 'mab_woo_conditions',
					'label'   => esc_html__( 'Cart total', 'max-addons' ),
					'compare' => [
						'type'        => 'select',
						'options'     => $math_options,
						'placeholder' => esc_html__( '==', 'max-addons' )
					],
					'value'   => [
						'type'        => 'text',
						'placeholder' => ''
					],
				];
			}
		}

        return $options;
	}

	/**
	 * Set Max Addons conditions
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function set_conditions( $result, $key, $condition ) {
		$show = false;

		switch ( $key ) {
			case 'mab_post_category':
			case 'mab_post_tag':
				$show = self::check_post_term( $condition );
				break;

			case 'mab_post_type':
				$show = self::check_post_type( $condition );
				break;

			case 'mab_post_comments_count':
				$show = self::check_post_comments_count( $condition );
				break;

			case 'mab_post_comments_status':
				$show = self::check_post_comments_status( $condition );
				break;

			case 'mab_author_archive':
				$show = self::check_author_archive( $condition );
				break;

			case 'mab_category_archive':
				$show = self::check_category_archive( $condition );
				break;

			case 'mab_tag_archive':
				$show = self::check_tag_archive( $condition );
				break;

			case 'mab_search_results':
				$show = self::check_search_results( $condition );
				break;

			case 'mab_visitor_type':
				$show = self::check_visitor_type( $condition );
				break;

			case 'mab_request_parameter':
				$show = self::check_request_parameter( $condition );
				break;

			case 'mab_woo_product_category':
				$show = self::check_woo_product_category( $condition );
				break;

			case 'mab_woo_product_price':
				$show = self::check_woo_product_price( $condition );
				break;

			case 'mab_woo_product_rating':
				$show = self::check_woo_product_rating( $condition );
				break;

			case 'mab_woo_product_stock':
				$show = self::check_woo_product_stock( $condition );
				break;

			case 'mab_woo_product_type':
				$show = self::check_woo_product_type( $condition );
				break;

			case 'mab_woo_orders_placed':
				$show = self::check_woo_orders_placed( $condition );
				break;

			case 'mab_woo_customer_total_spent':
				$show = self::check_woo_customer_total_spent( $condition );
				break;

			case 'mab_woo_cart_products_count':
				$show = self::check_woo_cart_products_count( $condition );
				break;

			case 'mab_woo_cart_products':
				$show = self::check_woo_cart_products( $condition );
				break;

			case 'mab_woo_cart_total':
				$show = self::check_woo_cart_total( $condition );
				break;

			default:
				$show = $result;
				break;
		}

		return $show;
	}
	

	/**
	 * Compare conditions.
	 *
	 * Calls compare method
	 *
	 * @since 1.5.0
	 *
	 * @param mixed  $left_value  First value to compare.
	 * @param mixed  $right_value Second value to compare.
	 * @param string $operator    Comparison operator.
	 *
	 * @return bool
	 */
	public static function compare( $left_value, $right_value, $operator ) {
		switch ( $operator ) {
			case '==':
				return $left_value == $right_value;
			case '!=':
				return $left_value != $right_value;
			case '>=':
				return $left_value >= $right_value;
			case '<=':
				return $left_value <= $right_value;
			case '>':
				return $left_value > $right_value;
			case '<':
				return $left_value < $right_value;
			default:
				return $left_value === $right_value;
		}
	}

	/**
	 * Check if keyword exists in phrase
	 *
	 * @since 1.5.0
	 *
	 * @access protected
	 *
	 * @param string    $keyword    Keyword to search
	 * @param string    $phrase     Phrase to search in
	 */
	public static function keyword_exists( $keyword, $phrase ) {
		return strpos( $phrase, trim( $keyword ) ) !== false;
	}

	/**
	 * Check post term condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_post_term( $condition ) {
		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		$value = (array) $value;

		if ( ! $value || empty( $value ) ) {
			return self::compare( true, true, $compare );
		}

		$show = false;

		foreach ( $value as $term_id ) {
			$term = get_term( $term_id );

			if ( has_term( $term->name, $term->taxonomy ) ) {
				$show = true;
				break;
			}
		}

		return self::compare( $show, true, $compare );
	}

	/**
	 * Check post type condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_post_type( $condition ) {
		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		$show = false;

		if ( is_array( $value ) && ! empty( $value ) ) {
			foreach ( $value as $_key => $_value ) {
				if ( is_singular( $_value ) ) {
					$show = true;
					break;
				}
			}
		} else {
			$show = is_singular( $value );
		}

		return self::compare( $show, true, $compare );
	}

	/**
	 * Check post comments count condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_post_comments_count( $condition ) {
		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		$comments_count = get_comments_number( get_the_ID() );

		return self::compare( $comments_count, intval( $value ), $compare );
	}

	/**
	 * Check post comments status condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_post_comments_status( $condition ) {
		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		$show  = comments_open( get_the_ID() );
		$value = ( $value == 'open' ) ? true : false;

		return self::compare( $show, $value, $compare );
	}

	/**
	 * Check author archive condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_author_archive( $condition ) {
		$show = false;

		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		if ( is_array( $value ) && ! empty( $value ) ) {
			foreach ( $value as $_key => $_value ) {
				if ( is_author( $_value ) ) {
					$show = true;
					break;
				}
			}
		} else {
			$show = is_author( $value );
		}

		return self::compare( $show, true, $compare );
	}

	/**
	 * Check category archive condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_category_archive( $condition ) {
		$show = false;

		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		if ( is_array( $value ) && ! empty( $value ) ) {
			foreach ( $value as $_key => $_value ) {
				$show = $this->is_category( $_value );
				if ( $show ) {
					break;
				}
			}
		} else {
			$show = $this->is_category( $value );
		}

		return self::compare( $show, true, $compare );
	}

	/**
	 * Check tag archive condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_tag_archive( $condition ) {
		$show = false;

		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		if ( is_array( $value ) && ! empty( $value ) ) {
			foreach ( $value as $_key => $_value ) {
				$show = $this->is_tag( $_value );
				if ( $show ) {
					break;
				}
			}
		} else {
			$show = $this->is_tag( $value );
		}

		return self::compare( $show, true, $compare );
	}

	/**
	 * Check search results condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_search_results( $condition ) {
		$show = false;

		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		if ( is_search() ) {

			if ( empty( $value ) ) { // We're showing on all search pages

				$show = true;

			} else { // We're showing on specific keywords

				$phrase = get_search_query(); // The user search query

				if ( '' !== $phrase && ! empty( $phrase ) ) { // Only proceed if there is a query

					$keywords = explode( ',', $value ); // Separate keywords

					foreach ( $keywords as $index => $keyword ) {
						if ( self::keyword_exists( trim( $keyword ), $phrase ) ) {
							$show = true;
							break;
						}
					}
				}
			}
		}

		return self::compare( $show, true, $compare );
	}

	/**
	 * Check visitor type condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_visitor_type( $condition ) {
		$show = 'new';

		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : 'new';

		if ( ! isset( $_COOKIE['MABVisitorData'] ) && ! isset( $_COOKIE['MABNewVisitor'] ) ) {
			wp_add_inline_script(
				'bricks-builder',
				'jQuery( window ).on( "elementor/frontend/init", function() {
					var current_time = new Date().getTime();
	
					var mab_secure = ( document.location.protocol === "https:" ) ? "secure" : "";
					var visit_date = new Date( current_time + 1000 * 86400 * 365 ).toGMTString();
					var visit_date_expire = new Date( current_time + 86400 * 1000 ).toGMTString();
	
					document.cookie = "MABVisitorData=enabled;expires=" + visit_date + "SameSite=Strict;" + mab_secure;
					document.cookie = "MABNewVisitor=enabled;expires=" + visit_date_expire + "SameSite=Strict;" + mab_secure;
				}); '
			);
		}

		if ( isset( $_COOKIE['MABVisitorData'] ) && ! isset( $_COOKIE['MABNewVisitor'] ) ) {
			$show = 'returning';
		}

		return self::compare( $show, $value, $compare );
	}

	/**
	 * Check request parameter condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_request_parameter( $condition ) {
		$show = false;

		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		if ( ! isset( $_SERVER['REQUEST_URI'] ) || empty( $_SERVER['REQUEST_URI'] ) ) {
			$show = false;
		}

		$url = wp_parse_url( filter_var( wp_unslash( $_SERVER['REQUEST_URI'] ), FILTER_SANITIZE_STRING ) );

		if ( $url && isset( $url['query'] ) && ! empty( $url['query'] ) ) {
			$query_params = explode( '&', $url['query'] );

			$value = str_replace( '&', "\n", $value );
			$value = explode( "\n", sanitize_textarea_field( $value ) );

			if ( ! empty( $value ) ) {
				foreach ( $value as $index => $param ) {
					if ( ! empty( $param ) ) {
						$is_strict = strpos( $param, '=' );
						if ( ! $is_strict ) {
							$value[ $index ] = $value[ $index ] . '=' . rawurlencode( $_GET[ $param ] );
						}
					}
				}
				$show = ! empty( array_intersect( $value, $query_params ) ) ? true : false;
			}
		}

		return $show;
	}

	/**
	 * Check woo product category condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_woo_product_category( $condition ) {
		if ( ! class_exists( 'WooCommerce' ) ) {
			return true;
		}

		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		$show = false;

		$product_id = get_queried_object_id();

		$type = get_post_type();

		if ( 'product' !== $type ) {
			$show = false;
		}

		$product = wc_get_product( $product_id );

		$product_cats = $product->get_category_ids();

		$show = ! empty( array_intersect( (array) $value, $product_cats ) ) ? true : false;

		return self::compare( $show, true, $compare );
	}

	/**
	 * Check woo product price condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_woo_product_price( $condition ) {
		if ( ! class_exists( 'WooCommerce' ) ) {
			return true;
		}

		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		$type = get_post_type();

		if ( 'product' === $type ) {
			$product_id    = get_queried_object_id();
			$product       = wc_get_product( $product_id );
			$product_price = $product->get_price();

			return self::compare( $product_price, $value, $compare );
		}

		return false;
	}

	/**
	 * Check woo product rating condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_woo_product_rating( $condition ) {
		if ( ! class_exists( 'WooCommerce' ) ) {
			return true;
		}

		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		if ( 'product' !== get_post_type( get_the_ID() ) ) {
			return false;
		}

		$type = get_post_type();

		if ( 'product' === $type ) {
			$product_id     = get_queried_object_id();
			$product        = wc_get_product( $product_id );
			$product_rating = $product->get_average_rating();

			return self::compare( $product_rating, $value, $compare );
		}

		return false;
	}

	/**
	 * Check woo product stock condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_woo_product_stock( $condition ) {
		if ( ! class_exists( 'WooCommerce' ) ) {
			return true;
		}

		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		$show = false;

		$type = get_post_type();

		if ( 'product' === $type ) {
			$product_id       = get_queried_object_id();
			$product          = wc_get_product( $product_id );
			$product_quantity = $product->get_stock_quantity();

			if ( 0 === $value ) {
				// Check if product is in stock or backorder is allowed.
				$product_quantity = $product->is_in_stock() || $product->backorders_allowed();

				$show = ( $value == $product_quantity ) ? true : false;
			} else {
				$show = self::compare( $product_quantity, $value, $compare );
			}
		}

		return self::compare( $show, true, $compare );
	}

	/**
	 * Check woo product type condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_woo_product_type( $condition ) {
		if ( ! class_exists( 'WooCommerce' ) ) {
			return true;
		}

		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		$show = false;

		if ( 'product' !== get_post_type( get_the_ID() ) ) {
			return false;
		}

		$product_id = get_queried_object_id();

		$type = get_post_type();
		
		if ( 'product' === $type ) {
			$product = wc_get_product( $product_id );

			$product_type = $product->get_type();

			$show = ( $value == $product_type ) ? true : false;
		}

		return self::compare( $show, true, $compare );
	}

	/**
	 * Check woo cart products count condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_woo_cart_products_count( $condition ) {
		if ( ! class_exists( 'WooCommerce' ) ) {
			return true;
		}

		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		if ( '' !== $value ) {
			$items_count = WC()->cart->get_cart_contents_count();

			return self::compare( $items_count, $value, $compare );
		}

		return false;
	}

	/**
	 * Check woo cart products condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_woo_cart_products( $condition ) {
		if ( ! class_exists( 'WooCommerce' ) ) {
			return true;
		}

		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		$show = false;

		$products_in_cart = array_column( WC()->cart->get_cart(), 'product_id' );

		foreach ( $value as $product_id ) {
			if ( in_array( $product_id, $products_in_cart ) ) {
				$show = true;
				break;
			}
		}

		return self::compare( $show, true, $compare );
	}

	/**
	 * Check woo cart total condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_woo_cart_total( $condition ) {
		if ( ! class_exists( 'WooCommerce' ) ) {
			return true;
		}

		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		$cart = WC()->cart;

		if ( ! $cart->is_empty() ) {
			$cart_total = $cart->total;

			return self::compare( $cart_total, $value, $compare );
		}

		return false;
	}

	/**
	 * Check woo orders placed condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_woo_orders_placed( $condition ) {
		if ( ! class_exists( 'WooCommerce' ) ) {
			return true;
		}

		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		if ( '' !== $value ) {
			$args = array(
				'customer_id' => get_current_user_id(),
				'status'      => array( 'wc-completed' ),
			);

			$items_count = count( wc_get_orders( $args ) );

			return self::compare( $items_count, $value, $compare );
		}

		return false;
	}

	/**
	 * Check woo user total spend condition
	 *
	 * @since 1.5.0
	 *
	 * @return array
	 */
	public static function check_woo_customer_total_spent( $condition ) {
		if ( ! class_exists( 'WooCommerce' ) ) {
			return true;
		}

		$compare = isset( $condition['compare'] ) ? $condition['compare'] : '==';
		$value   = isset( $condition['value'] ) ? $condition['value'] : '';

		$user = wp_get_current_user();

		$total_spent = ( $user->ID ) ? wc_get_customer_total_spent( $user->ID ) : 0;

		return self::compare( $total_spent, floatval( $value ), $compare );
	}
}
