<?php
/**
 * Plugin loader class.
 *
 * Loads the plugin and all the required classes and functions when the
 * plugin is activate.
 *
 * @package MaxAddonsPro\Classes
 * @since 1.0.0
 */

namespace MaxAddonsPro\Classes;

use MaxAddonsPro\Classes\Helper;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * MaxAddons Plugin.
 *
 * Main plugin class responsible for initiazling MaxAddons Plugin. The class
 * registers all the components required to run the plugin.
 */
class Plugin {

	/**
	 * Instance.
	 *
	 * Holds the plugin instance.
	 *
	 * @since 1.0.0
	 * @access public
	 * @static
	 *
	 * @var $instance
	 */
	private static $instance = null;

	/**
	 * Instance.
	 *
	 * Ensures only one instance of the plugin class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 * @access public
	 * @static
	 *
	 * @return Plugin An instance of the class.
	 */
	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Constructor
	 */
	public function __construct() {

		$this->includes();

		add_action( 'plugins_loaded', array( __CLASS__, 'conditions_init' ), 1 );

		add_action( 'init', array( $this, 'init' ), 11 );

		if ( Helper::is_woocommerce_active() ) {
			new \MaxAddonsPro\MaxWoo();
		}
	}

	/**
	 * Shows an admin notice if you're not using the Bricks Builder.
	 */
	public function max_addons_admin_notice_message() {
		if ( ! is_admin() ) {
			return;
		} elseif ( ! is_user_logged_in() ) {
			return;
		} elseif ( ! current_user_can( 'update_core' ) ) {
			return;
		}
	
		$message = sprintf(
			/* translators: 1: Max Addons 2: Bricks Builder */
			esc_html__( '%1$s requires %2$s to be installed and activated.', 'max-addons' ),
			'<strong>Max Addons</strong>',
			'<strong>Bricks Builder</strong>'
		);

		$html = sprintf( '<div class="notice notice-warning">%s</div>', wpautop( $message ) );

		echo wp_kses_post( $html );
	}

	/**
	 * AutoLoad
	 *
	 * @since 1.0.0
	 * @param string $class class.
	 */
	public function autoload( $class ) {

		if ( 0 !== strpos( $class, __NAMESPACE__ ) ) {
			return;
		}

		$class_to_load = $class;

		if ( ! class_exists( $class_to_load ) ) {
			$filename = strtolower(
				preg_replace(
					array( '/^' . __NAMESPACE__ . '\\\/', '/([a-z])([A-Z])/', '/_/', '/\\\/' ),
					array( '', '$1-$2', '-', DIRECTORY_SEPARATOR ),
					$class_to_load
				)
			);
			$filename = MAB_PRO_DIR . $filename . '.php';

			if ( is_readable( $filename ) ) {
				include $filename;
			}
		}
	}

	/**
	 * Includes.
	 *
	 * @since 1.0.0
	 */
	private function includes() {

		require MAB_PRO_DIR . 'classes/class-mab-admin-settings.php';
		require MAB_PRO_DIR . 'classes/class-mab-maintenance-mode.php';
		require MAB_PRO_DIR . 'classes/class-mab-helper.php';
		require MAB_PRO_DIR . 'classes/class-mab-posts-helper.php';
		require MAB_PRO_DIR . 'includes/max-query.php';
		require MAB_PRO_DIR . 'includes/max-interactions.php';

		if ( Helper::is_woocommerce_active() ) {
			require MAB_PRO_DIR . 'classes/class-mab-woocommerce.php';
		}
	}

	/**
	 * Init.
	 *
	 * Initialize plugin components.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function init() {
		if ( ! defined( 'BRICKS_VERSION' ) ) {
			add_action( 'admin_notices', array( $this, 'max_addons_admin_notice_message' ) );
			add_action( 'network_admin_notices', array( $this, 'max_addons_admin_notice_message' ) );

			return;
		}

		new \MaxAddonsPro\MaxQuery();
		\MaxAddonsPro\MaxInteractions::init();

		/* if ( Helper::is_woocommerce_active() ) {
			new \MaxAddonsPro\MaxWoo();
		} */

		$this->load_textdomain();
		$this->init_actions_filters();
		$this->load_elements();
		$this->load_features();
		$this->load_providers();
	}

	/**
	 * Loads Max Addons text domain.
	 *
	 * @since 1.2.0
	 * @access public
	 *
	 * @return void
	 */
	public function load_textdomain() {
		load_plugin_textdomain( 'max-addons', false, basename( MAB_PRO_DIR ) . '/languages' );
	}

	/**
	 * Init action and filters.
	 *
	 * @since 1.0.0
	 * @access private
	 */
	private function init_actions_filters() {
		$enabled_elements = \MaxAddonsPro\Classes\Admin_Settings::get_enabled_elements();

		if ( 'disabled' !== $enabled_elements ) {
			// Provide translatable string for element category 'Max Addons'
			add_filter( 'bricks/builder/i18n', array( $this, 'elements_category' ) );
			add_filter( 'bricks/element/set_root_attributes', array( $this, 'filter_element_root_attrs' ), 10, 2 );
			add_filter( "bricks/elements/max-modal-popup/controls", array( '\MaxAddonsPro\Elements\Modal_Popup_Element', 'inject_custom_controls' ) );
		}

		// Action to include script.
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );

		if ( bricks_is_frontend() ) {
			add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_interactions_script' ), 1 );
		}
	}

	/**
	 * Provide translatable string for element category 'max addons'
	 *
	 * @since 1.0.0
	 * @param array $i18n.
	 */
	public function elements_category( $i18n ) {
		$i18n['max-addons-elements'] = esc_html__( 'Max Addons', 'max-addons' );

		if ( Helper::is_woocommerce_active() ) {
			$i18n['max-addons-woo-elements'] = esc_html__( 'Max Addons - WooCommerce', 'max-addons' );
			$i18n['max-addons-woo-thankyou-elements'] = esc_html__( 'Max Addons - Woo Thank You Page', 'max-addons' );
		}

		return $i18n;
	}

	public function filter_element_root_attrs( $attrs, $element ) {
		if ( false !== strpos( $element->name, 'max-' ) ) {
			// Sometimes bricksIsFrontend in JS is not enough to check.
			// So adding mab-backend class to the element's root.
			if ( isset( $element->is_frontend ) && ! $element->is_frontend ) {
				$attrs['class'][] = 'mab-backend';
			}

			if ( ! isset( $attrs['data-mab-id'] ) && isset( $element->id ) ) {
				$attrs['data-mab-id'] = $element->id;
			}
		}

		return $attrs;
	}

	public function get_script_url( $filename ) {
		$script_route  = defined( 'MAB_DEVELOPMENT' ) ? 'src/' : '';
		$script_suffix = defined( 'MAB_DEVELOPMENT' ) ? '' : '.min';

		if ( 'frontend' === $filename ) {
			return MAB_PRO_URL . 'assets/js/' . $filename . $script_suffix . '.js';
		}

		return MAB_PRO_URL . 'assets/js/elements/' . $script_route . $filename . $script_suffix . '.js';
	}

	/**
	 * Enqueue scripts.
	 *
	 * @since 1.0.0
	 */
	public function enqueue_scripts() {
		$debug_suffix = ( Helper::is_script_debug() ) ? '' : '.min';

		/**
		 * Styles.
		 */
		if ( bricks_is_builder() ) {
			wp_enqueue_style( 'mab-editor', MAB_PRO_URL . 'assets/css/editor.css', array(), MAB_PRO_VER );
		}

		wp_register_style(
			'mab-frontend-filters',
			MAB_PRO_URL . 'assets/css/frontend-filters' . $debug_suffix . '.css',
			array(),
			MAB_PRO_VER,
			'all'
		);

		wp_register_style(
			'mab-floating-effects',
			MAB_PRO_URL . 'assets/css/floating-effects' . $debug_suffix . '.css',
			array(),
			MAB_PRO_VER,
			'all'
		);

		wp_register_style(
			'mab-tooltip',
			MAB_PRO_URL . 'assets/css/tooltip' . $debug_suffix . '.css',
			array(),
			MAB_PRO_VER,
			'all'
		);

		wp_register_style(
			'tablesaw',
			MAB_PRO_URL . 'assets/lib/tablesaw/tablesaw' . $debug_suffix . '.css',
			array(),
			'2.0.0',
			'all'
		);

		wp_register_style(
			'fancybox',
			MAB_PRO_URL . 'assets/lib/fancybox/fancybox.css',
			array(),
			'5.0.33',
			'all'
		);

		foreach ( $this->get_css_files() as $file_name ) {
			wp_register_style(
				'mab-' . $file_name,
				MAB_PRO_URL . 'assets/css/elements/' . $file_name . $debug_suffix . '.css',
				array(),
				MAB_PRO_VER,
				'all'
			);
		}

		/**
		 * Scripts.
		 */
		
		// Vendor scripts.
		wp_register_script(
			'image-compare',
			MAB_PRO_URL . 'assets/lib/image-compare/image-compare-viewer' . $debug_suffix . '.js',
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'tippy',
			MAB_PRO_URL . 'assets/lib/tippy/tippy.js',
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'tablesaw',
			MAB_PRO_URL . 'assets/lib/tablesaw/tablesaw' . $debug_suffix . '.js',
			'',
			'3.0.3',
			true
		);

		wp_register_script(
			'tablesaw-init',
			MAB_PRO_URL . 'assets/lib/tablesaw/tablesaw-init.js',
			'',
			'3.0.3',
			true
		);

		wp_register_script(
			'gridjs',
			MAB_PRO_URL . 'assets/lib/gridjs/gridjs.umd.js',
			'',
			'6.2.0',
			true
		);

		wp_register_script(
			'vivus',
			MAB_PRO_URL . 'assets/lib/vivus/vivus.min.js',
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'tilt',
			MAB_PRO_URL . 'assets/lib/tilt/vanilla-tilt.min.js',
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'lottie-player',
			MAB_PRO_URL . 'assets/lib/lottie/lottie-player.js',
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'lottie-interactivity',
			MAB_PRO_URL . 'assets/lib/lottie/lottie-interactivity.min.js',
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'fancybox',
			MAB_PRO_URL . 'assets/lib/fancybox/fancybox.js',
			'',
			'5.0.33',
			true
		);

		// Internal scripts.
		wp_enqueue_script(
			'mab-frontend',
			$this->get_script_url( 'frontend' ),
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-album',
			$this->get_script_url( 'album' ),
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-hotspots',
			$this->get_script_url( 'hotspots' ),
			[ 'tippy' ],
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-tooltip',
			$this->get_script_url( 'tooltip' ),
			[ 'tippy' ],
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-image-accordion',
			$this->get_script_url( 'image-accordion' ),
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-image-gallery',
			$this->get_script_url( 'image-gallery' ),
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-toggle',
			$this->get_script_url( 'toggle' ),
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-slide-menu',
			$this->get_script_url( 'slide-menu' ),
			'',
			MAB_PRO_VER,
			true
		);		

		wp_register_script(
			'mab-lottie',
			$this->get_script_url( 'lottie' ),
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-ticker',
			$this->get_script_url( 'content-ticker' ),
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-offcanvas',
			$this->get_script_url( 'offcanvas' ),
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-svg-animation',
			$this->get_script_url( 'svg-animation' ),
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-dynamic-table',
			$this->get_script_url( 'dynamic-table' ),
			[ 'gridjs' ],
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-nested-table',
			$this->get_script_url( 'table-nested' ),
			[ 'gridjs' ],
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-timeline',
			$this->get_script_url( 'timeline' ),
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-unfold',
			$this->get_script_url( 'unfold' ),
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-video',
			$this->get_script_url( 'video' ),
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-video-gallery',
			$this->get_script_url( 'video-gallery' ),
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-modal-popup',
			$this->get_script_url( 'modal-popup' ),
			[ 'bricks-scripts' ],
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-wrapper-link',
			$this->get_script_url( 'wrapper-link' ),
			'',
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-toc',
			$this->get_script_url( 'toc' ),
			[ 'bricks-scripts' ],
			MAB_PRO_VER,
			true
		);

		wp_register_script(
			'mab-thumbnail-slider',
			$this->get_script_url( 'thumbnail-slider' ),
			[ 'bricks-splide' ],
			MAB_PRO_VER,
			true
		);

		if ( Helper::is_woocommerce_active() ) {
			wp_register_style(
				'mab-woocommerce',
				MAB_PRO_URL . 'assets/css/woocommerce' . $debug_suffix . '.css',
				array(),
				MAB_PRO_VER,
				'all'
			);

			wp_register_script(
				'mab-woo-stock-progress-bar',
				$this->get_script_url( 'woo-stock-progress-bar' ),
				array( 'jquery' ),
				MAB_PRO_VER,
				true
			);

			wp_register_script(
				'mab-woo-free-shipping-bar',
				$this->get_script_url( 'woo-free-shipping-bar' ),
				array( 'wc-cart-fragments' ),
				MAB_PRO_VER,
				true
			);

			wp_register_script(
				'mab-woo-product-gallery',
				$this->get_script_url( 'woo-product-gallery' ),
				[ 'bricks-splide' ],
				MAB_PRO_VER,
				true
			);

			wp_register_script(
				'mab-empty-cart-button',
				$this->get_script_url( 'woo-empty-cart-button' ),
				array( 'jquery' ),
				MAB_PRO_VER,
				true
			);

			wp_register_script(
				'mab-add-to-cart',
				$this->get_script_url( 'add-to-cart' ),
				array( 'jquery' ),
				MAB_PRO_VER,
				true
			);

			$mab_woo_localize = apply_filters(
				'mab_woo_add_to_cart_js_localize',
				array(
					'is_cart'  => is_cart(),
					'cart_url' => apply_filters( 'mab_woo_add_to_cart_redirect', wc_get_cart_url() ),
				)
			);
			wp_localize_script( 'mab-add-to-cart', 'mab_woo_add_to_cart_script', $mab_woo_localize );
		}

		if ( bricks_is_builder_iframe() ) {
			wp_enqueue_script( 'image-compare' );
			wp_enqueue_style( 'tablesaw' );
			wp_enqueue_script( 'tablesaw' );
			wp_enqueue_script( 'tablesaw-init' );
			wp_enqueue_script( 'mab-slide-menu' );
			wp_enqueue_script( 'vivus' );
			wp_enqueue_script( 'mab-timeline' );
			wp_enqueue_script( 'lottie-player' );
			wp_enqueue_script( 'lottie-interactivity' );
			wp_enqueue_script( 'mab-lottie' );
			wp_enqueue_script( 'mab-offcanvas' );
			wp_enqueue_script( 'tilt' );
		}
	}

	/**
	 * Get CSS files for all the elements
	 */
	public function get_css_files() {
		$paths = glob( MAB_PRO_DIR . 'assets/css/elements/*' );

		// Make sure we have an array.
		if ( ! is_array( $paths ) ) {
			return;
		}

		$file_names = array();

		foreach ( $paths as $path ) {
			// Get the CSS file slug.
			$slug = basename( $path, '.css' );

			if ( file_exists( $path ) ) {
				if ( ! str_contains( $slug, '.min' ) ) {
					$file_names[] = $slug;
				}
			}
		}

		return $file_names;
	}

	public function enqueue_interactions_script() {
		wp_enqueue_script( 'mab-interactions', MAB_PRO_URL . 'assets/js/interactions.js', ['bricks-scripts'], MAB_PRO_VER, true );
	}

	/**
	 * Loading the elements
	 */
	public function load_elements() {
		$deactivated_elements             = \MaxAddonsPro\Classes\Admin_Settings::get_enabled_elements();
		$deactivated_woo_builder_elements = \MaxAddonsPro\Classes\Admin_Settings::get_enabled_woo_builder_elements();

		if ( 'disabled' === $deactivated_elements ) {
			$deactivated_elements = [];
		}

		if ( 'disabled' === $deactivated_woo_builder_elements ) {
			$deactivated_woo_builder_elements = [];
		}

		$all_elements = array_unique( array_merge( $deactivated_elements, $deactivated_woo_builder_elements ) );
		
		if ( empty( $all_elements ) ) {
			return;
		}

		$paths = glob( MAB_PRO_DIR . 'includes/elements/*' );

		// Make sure we have an array.
		if ( ! is_array( $paths ) ) {
			return;
		}

		// Load all found modules.
		foreach ( $paths as $path ) {

			// Get the module slug.
			$slug = basename( $path, '.php' );

			if ( ! in_array( $slug, $all_elements ) ) {
				continue;
			}

			if ( file_exists( $path ) ) {
				\Bricks\Elements::register_element( $path );
			}
		}
	}

	/**
	 * Loading the features
	 */
	public function load_features() {
		$deactivated_features = \MaxAddonsPro\Classes\Admin_Settings::get_enabled_features();

		if ( 'disabled' === $deactivated_features ) {
			return;
		}

		require MAB_PRO_DIR . 'includes/features/class-mab-features-base.php';
	}

	/**
	 * Loading the providers
	 */
	public function load_providers() {
		require MAB_PRO_DIR . 'includes/providers/max-providers.php';
		require MAB_PRO_DIR . 'includes/providers/provider-max-general.php';
		require MAB_PRO_DIR . 'includes/providers/provider-max-woo.php';

		$providers = [
			'max_general',
			'max_woo'
		];

		\MaxAddonsPro\MaxProviders::register($providers);
	}

	/**
	 * Includes Display Conditions.
	 *
	 * @since 1.5.0
	 */
	public static function conditions_init() {

		require MAB_PRO_DIR . 'includes/conditions.php';
		(new \MaxAddonsPro\MAB_Conditions())->init();
	}
}

/**
 *  Prepare if class 'Plugin' exist.
 *  Kicking this off by calling 'get_instance()' method
 */
Plugin::get_instance();
