<?php
namespace MaxAddonsPro\Classes;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class Helper
 */
class Helper {

	/**
	 * Script debug
	 *
	 * @var script_debug
	 */
	private static $script_debug = null;

	/**
	 * Users options
	 *
	 * @var users_options
	 */
	private static $users_options = null;

	/**
	 * Post types
	 *
	 * @var post_types
	 */
	private static $post_types = null;

	/**
	 * Convert Comma Separated List into Array
	 *
	 * @param string $list Comma separated list.
	 * @return array
	 * @since 1.0.0
	 */
	public static function comma_list_to_array( $list = '' ) {
		$list_array = explode( ',', $list );

		return $list_array;
	}

	/**
	 * Check if script debug is enabled.
	 *
	 * @since 1.0.0
	 *
	 * @return string The CSS suffix.
	 */
	public static function is_script_debug() {
		if ( null === self::$script_debug ) {

			self::$script_debug = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG;
		}

		return self::$script_debug;
	}

	/**
	 * Get contact forms of supported forms plugins
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public static function get_contact_forms( $plugin = '' ) {
		$options       = array();
		$contact_forms = array();

		// Contact Form 7
		if ( 'Contact_Form_7' == $plugin && class_exists( 'WPCF7_ContactForm' ) ) {
			$args = array(
				'post_type'      => 'wpcf7_contact_form',
				'posts_per_page' => -1,
			);

			$cf7_forms = get_posts( $args );

			if ( ! empty( $cf7_forms ) && ! is_wp_error( $cf7_forms ) ) {
				foreach ( $cf7_forms as $form ) {
					$contact_forms[ $form->ID ] = $form->post_title;
				}
			}
		}

		// Fluent Forms
		if ( 'Fluent_Forms' == $plugin && function_exists( 'wpFluentForm' ) ) {
			global $wpdb;

			$fluent_forms = $wpdb->get_results( "SELECT * FROM {$wpdb->prefix}fluentform_forms" );

			if ( $fluent_forms ) {
				foreach ( $fluent_forms as $form ) {
					$contact_forms[ $form->id ] = $form->title;
				}
			}
		}

		// Formidable Forms
		if ( 'Formidable_Forms' == $plugin && class_exists( 'FrmForm' ) ) {
			$formidable_forms = \FrmForm::get_published_forms( array(), 999, 'exclude' );
			if ( count( $formidable_forms ) ) {
				foreach ( $formidable_forms as $form ) {
					$contact_forms[ $form->id ] = $form->name;
				}
			}
		}

		// Gravity Forms
		if ( 'Gravity_Forms' == $plugin && class_exists( 'GFCommon' ) ) {
			$gravity_forms = \RGFormsModel::get_forms( null, 'title' );

			if ( ! empty( $gravity_forms ) && ! is_wp_error( $gravity_forms ) ) {
				foreach ( $gravity_forms as $form ) {
					$contact_forms[ $form->id ] = $form->title;
				}
			}
		}

		// Ninja Forms
		if ( 'Ninja_Forms' == $plugin && class_exists( 'Ninja_Forms' ) ) {
			$ninja_forms = Ninja_Forms()->form()->get_forms();

			if ( ! empty( $ninja_forms ) && ! is_wp_error( $ninja_forms ) ) {
				foreach ( $ninja_forms as $form ) {
					$contact_forms[ $form->get_id() ] = $form->get_setting( 'title' );
				}
			}
		}

		// WPforms
		if ( 'WP_Forms' == $plugin && function_exists( 'wpforms' ) ) {
			$args = array(
				'post_type'      => 'wpforms',
				'posts_per_page' => -1,
			);

			$wpf_forms = get_posts( $args );

			if ( ! empty( $wpf_forms ) && ! is_wp_error( $wpf_forms ) ) {
				foreach ( $wpf_forms as $form ) {
					$contact_forms[ $form->ID ] = $form->post_title;
				}
			}
		}

		// Contact Forms List
		if ( ! empty( $contact_forms ) ) {
			//$options[0] = esc_html__( 'Select a Contact Form', 'max-addons' );
			foreach ( $contact_forms as $form_id => $form_title ) {
				$options[ $form_id ] = $form_title;
			}
		}

		if ( empty( $options ) ) {
			$options[0] = esc_html__( 'No contact forms found!', 'max-addons' );
		}

		return $options;
	}

	/**
	 * Get Templates List
	 *
	 * @return array
	 * @since 1.0.0
	 */
	public static function get_templates_list( $exclude_template_id = '' ) {
		$args = array(
			'post_type'      => BRICKS_DB_TEMPLATE_SLUG,
			'posts_per_page' => -1,
			'meta_query'     => array(
				array(
					'key'   => BRICKS_DB_TEMPLATE_TYPE,
					'value' => array( 'section', 'content' ),
				),
			),
			'fields'         => 'ids',
			'post_status'    => 'publish',
		);

		$templates = get_posts( $args );

		$list = array();

		foreach ( $templates as $template_id ) {
			if ( $exclude_template_id == $template_id ) {
				continue;
			}

			$list[ $template_id ] = get_the_title( $template_id );
		}

		return $list;
	}

	/**
	 * Get Limited Text
	 *
	 * @return string
	 * @since 1.0.0
	 */
	public static function get_limited_text( $text, $length ) {
		if ( $length ) {
			if ( strlen( $text ) > $length ) {
				$title = substr( $text, 0, $length ) . '&hellip;';
			}
		}

		return $text;
	}
	
	/**
	 * Get User Roles
	 *
	 * @since 1.0.0
	 * @access public
	 */
    public static function get_user_roles() {
		global $wp_roles;
	
		$roles = array();
		$wproles = $wp_roles->roles;

		foreach ( $wproles as $key => $role ) {
			$roles[$key] = $role['name'];
		}
		
		return $roles;
	}

	/**
	 * Get available users
	 *
	 * @since 1.0.0
	 */
	public static function get_users_options( $args = [] ) {
		/* $options = array();

		$users = get_users(
			array(
				'fields' => array( 'ID', 'display_name' ),
			)
		);

		if ( empty( $users ) ) {
			$options[''] = __( 'No users found', 'max-addons' );
			return $options;
		}

		foreach ( $users as $user ) {
			$options[ $user->ID ] = $user->display_name;
		} */

		$args = array( 'role__not_in' => array( 'customer', 'subscriber' ) );

		if ( null === self::$users_options ) {
			self::$users_options = \Bricks\Helpers::get_users_options( $args, true );
		}

		return self::$users_options;
	}

	/**
	 * Return a list of taxonomies
	 */
	public static function get_taxonomies_options( $post_type = false ) {

		$taxonomies_options = [];

		if ( ! $post_type ) {
			// Get all available taxonomies
			$taxonomies = get_taxonomies( array(
				'show_in_nav_menus' => true
			), 'objects' );
		} else {
			$taxonomies = get_object_taxonomies( $post_type, 'objects' );
		}

		foreach ( $taxonomies as $taxonomy ) {
			if ( ! $taxonomy->publicly_queryable ) {
				continue;
			}

			$taxonomies_options[ $taxonomy->name ] = $taxonomy->label;
		}

		if ( empty( $taxonomies_options ) ) {
			$taxonomies_options[0] = __( 'No taxonomies found', 'max-addons' );
			return $taxonomies_options;
		}

		return $taxonomies_options;
	}

	/**
	 * Get Terms Options
	 *
	 * Retrieve the terms options array for a control
	 *
	 * @since  1.5.0
	 * @param  taxonomy     The taxonomy for the terms
	 * @param  key|string   The key to use when building the options. Can be 'slug' or 'id'
	 * @param  all|bool     The string to use for the first option. Can be false to disable. Default: true
	 * @return array
	 */
	public static function get_terms_options( $taxonomy, $key = 'slug', $all = true ) {
		if ( false !== $all ) {
			$all     = ( true === $all ) ? __( 'All', 'max-addons' ) : $all;
			$options = array( '' => $all );
		}

		$terms = get_terms(
			array(
				'taxonomy' => $taxonomy,
			)
		);
	
		if ( empty( $terms ) ) {
			$options[''] = sprintf( __( 'No terms found', 'max-addons' ), $taxonomy );
			return $options;
		}
	
		foreach ( $terms as $term ) {
			$term_key             = ( 'id' === $key ) ? $term->term_id : $term->slug;
			$options[ $term_key ] = $term->name;
		}
	
		return $options;
	}

	/**
	 * Recursively sort an array of taxonomy terms hierarchically. Child categories will be
	 * placed under a 'children' member of their parent term.
	 *
	 * @since 1.3.0
	 *
	 * @param Array   $cats     taxonomy term objects to sort
	 * @param Array   $into     result array to put them in
	 * @param integer $parentId the current parent ID to put them in
	 */
	public static function sort_terms_hierarchicaly( Array &$cats, Array &$into, $parentId = 0 ) {
	    foreach ( $cats as $i => $cat ) {
	        if ( $cat->parent == $parentId ) {
	            $into[ $cat->term_id ] = $cat;
	            unset( $cats[ $i ] );
	        }
	    }

	    foreach ( $into as $topCat ) {
	        $topCat->children = [];
	        self::sort_terms_hierarchicaly( $cats, $topCat->children, $topCat->term_id );
	    }
	}

	/**
	 * Get Post Types.
	 *
	 * @since 1.5.0
	 * @access public
	 */
	public static function get_post_types() {
		if ( null === self::$post_types ) {
			self::$post_types = \Bricks\Helpers::get_registered_post_types();
		}

		return self::$post_types;
	}

	/**
	 * Return a list of posts
	 */
	public static function get_posts_options( $post_type = 'post' ) {
		$posts = \Bricks\Helpers::get_posts_by_post_id( [ 'post_type' => $post_type ] );

		return $posts;
	}

	/**
	 * Is WooCommerce active
	 *
	 * @return bool
	 */
	public static function is_woocommerce_active() {
		if ( ! function_exists( 'is_plugin_active' ) ) {
			include_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		if ( class_exists( 'WooCommerce' ) || is_plugin_active( 'woocommerce/woocommerce.php' ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Populate query vars if the element is not using query type controls
	 * Currently used in thumbnail-slider element and query_results_count when targetting thumbnail-slider element
	 *
	 * @return array
	 * @since 1.8.0
	 */
	public static function populate_query_vars_for_element( $element_data, $post_id ) {
		$element_name = $element_data['name'] ?? '';

		if ( empty( $element_name ) || is_null( $post_id ) ) {
			return [];
		}

		$settings = $element_data['settings'] ?? [];

		$args = [];

		if ( $element_name === 'max-thumbnail-slider' ) {
			$source = $settings['source'] ?? 'media';

			// Only populate query vars if source is 'media'
			if ( $source === 'media' ) {
				// STEP: Media type carousel might use dynamic data, so we need to normalize the settings (logic originally inside carousel element)
				$gallery_class_name = \Bricks\Elements::$elements['image-gallery']['class'] ?? false;

				if ( $gallery_class_name && ! empty( $settings['items']['useDynamicData'] ) ) {
					$gallery = new $gallery_class_name();

					$gallery->set_post_id( $post_id );

					$settings = $gallery->get_normalized_image_settings( $settings );
				}

				// STEP: Populate query vars based on the 'items' settings, if no items are set, no query vars are populated
				if ( ! empty( $settings['items']['images'] ) ) {
					$args = [
						'post_status'            => 'any',
						'post_type'              => 'attachment',
						'orderby'                => 'post__in',
						'objectType'             => 'post', // @since 1.9.3
						'bricks_skip_query_vars' => true, // Skip Query::prepare_query_vars_from_settings() @since 1.9.3
						'no_found_rows'          => true, // No pagination
					];

					$images = $settings['items']['images'] ?? $settings['items'];

					foreach ( $images as $image ) {
						if ( isset( $image['id'] ) ) {
							$args['post__in'][] = $image['id'];
						}
					}

					if ( isset( $args['post__in'] ) ) {
						$args['posts_per_page'] = count( $args['post__in'] );
					}
				}
			}
		}

		return $args;
	}
}
