<?php
namespace MaxAddonsPro\Classes;

use MaxAddonsPro\Classes\Helper;

/**
 * Handles logic for the admin settings page.
 *
 * @since 1.0.0
 */
final class Admin_Settings {
	/**
	 * Holds any errors that may arise from
	 * saving admin settings.
	 *
	 * @since 1.0.0
	 * @var array $errors
	 */
	public static $errors = array();

	public static $settings = array();

	/**
	 * Initializes the admin settings.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public static function init() {
		self::migrate_settings();

		add_action( 'plugins_loaded', __CLASS__ . '::init_hooks' );

		add_action( 'plugins_loaded', __CLASS__ . '::save_elements' );

		add_action( 'plugins_loaded', __CLASS__ . '::save_woo_builder_elements' );

		add_action( 'plugins_loaded', __CLASS__ . '::save_conditions' );

		add_action( 'plugins_loaded', __CLASS__ . '::save_features' );
	}

	/**
	 * Adds the admin menu and enqueues CSS/JS if we are on
	 * the plugin's admin settings page.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public static function init_hooks() {
		if ( ! is_admin() ) {
			return;
		}

		add_action( 'admin_menu', __CLASS__ . '::add_menu_page', 601 );
		add_filter( 'all_plugins', __CLASS__ . '::update_branding' );

		if ( isset( $_REQUEST['page'] ) && 'mab-settings' == $_REQUEST['page'] ) {
			//add_action( 'admin_enqueue_scripts', __CLASS__ . '::styles_scripts' );
			self::save();
		}

		add_action( 'admin_init', __CLASS__ . '::refresh_instagram_access_token' );
	}

	/**
	 * Enqueues the needed CSS/JS for the builder's admin settings page.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public static function styles_scripts() {
		// Styles
		//wp_enqueue_style( 'mab-admin-settings', MAB_PRO_URL . 'assets/css/admin-settings.css', array(), MAB_PRO_VER );
	}

	/**
	 * Get elements.
	 *
	 * @since 1.0.0
	 * @return array
	 */
	public static function get_elements() {
		$elements = array(
			'album'            => __( 'Album', 'max-addons' ),
			'author-list'      => __( 'Author List', 'max-addons' ),
			'breadcrumbs'      => __( 'Breadcrumbs', 'max-addons' ),
			'cta-button'       => __( 'CTA Button', 'max-addons' ),
			'content-ticker'   => __( 'Content Ticker', 'max-addons' ),
			'dynamic-table'    => __( 'Dynamic Table', 'max-addons' ),
			'flip-box'         => __( 'Flip Box', 'max-addons' ),
			'icon-list'        => __( 'Icon List', 'max-addons' ),
			'image-accordion'  => __( 'Image Accordion', 'max-addons' ),
			'image-gallery'    => __( 'Image Gallery', 'max-addons' ),
			'hotspots'         => __( 'Hotspots', 'max-addons' ),
			'image-comparison' => __( 'Image Comparison', 'max-addons' ),
			'instagram-feed'   => __( 'Instagram Feed', 'max-addons' ),
			'lottie'           => __( 'Lottie', 'max-addons' ),
			'multi-heading'    => __( 'Multi Heading', 'max-addons' ),
			'offcanvas'        => __( 'Off Canvas', 'max-addons' ),
			'post-tiles'       => __( 'Post Tiles', 'max-addons' ),
			'random-image'     => __( 'Random Image', 'max-addons' ),
			'reviews'          => __( 'Reviews', 'max-addons' ),
			'table'            => __( 'Table', 'max-addons' ),
			'table-nested'     => __( 'Table (Nestable)', 'max-addons' ),
			'unfold'           => __( 'Unfold', 'max-addons' ),
			'slide-menu'       => __( 'Slide Menu', 'max-addons' ),
			'rating'           => __( 'Rating', 'max-addons' ),
			'svg-animation'    => __( 'SVG Animation', 'max-addons' ),
			'toggle'           => __( 'Toggle', 'max-addons' ),
			'thumbnail-slider' => __( 'Thumbnail Slider', 'max-addons' ),
			'timeline'         => __( 'Timeline', 'max-addons' ),
			'video'            => __( 'Video', 'max-addons' ),
			'video-gallery'    => __( 'Video Gallery', 'max-addons' ),
			'modal-popup'      => __( 'Modal Popup', 'max-addons' ),
			'reusable-block'   => __( 'Reusable Block', 'max-addons' ),
			'toc'              => __( 'Table of Contents', 'max-addons' ),
		);

		// Contact Form 7
		if ( function_exists( 'wpcf7' ) ) {
			$elements['cf7-styler'] = __( 'Contact Form 7 Styler', 'max-addons' );
		}

		// Gravity Forms
		if ( class_exists( 'GFForms' ) ) {
			$elements['gravity-forms-styler'] = __( 'Gravity Forms Styler', 'max-addons' );
		}

		// Fluent Forms
		if ( function_exists( 'wpFluentForm' ) ) {
			$elements['fluent-forms-styler'] = __( 'Fluent Forms Styler', 'max-addons' );
		}

		ksort( $elements );

		return $elements;
	}

	/**
	 * Get woo elements.
	 *
	 * @since 1.7.1
	 * @return array
	 */
	public static function get_woo_elements() {
		$elements = array();

		if ( Helper::is_woocommerce_active() ) {
			$elements['woo-add-to-cart']        = __( 'Woo - Add to Cart', 'max-addons' );
			$elements['woo-buy-now-button']     = __( 'Woo - Buy Now Button', 'max-addons' );
			$elements['woo-free-shipping-bar']  = __( 'Woo - Free Shipping Bar', 'max-addons' );
			$elements['woo-product-badge']      = __( 'Woo - Product Badges', 'max-addons' );
			$elements['woo-product-gallery']    = __( 'Woo - Product Gallery', 'max-addons' );
			$elements['woo-product-image']      = __( 'Woo - Product Image', 'max-addons' );
			$elements['woo-product-reviews']    = __( 'Woo - Product Reviews', 'max-addons' );
			$elements['woo-empty-cart-button']  = __( 'Woo - Empty Cart Button', 'max-addons' );
			$elements['woo-stock-progress-bar'] = __( 'Woo - Stock Progress Bar', 'max-addons' );

			ksort( $elements );
		}

		return $elements;
	}

	/**
	 * Get woo builder elements.
	 *
	 * @since 1.11.0
	 * @return array
	 */
	public static function get_woo_builder_elements() {
		$elements = array();

		if ( Helper::is_woocommerce_active() ) {
			$elements['woo-order-details']     = [
				'category' => __( 'Thank You Page Builder', 'max-addons' ),
				'title'    => __( 'Woo - Order Details', 'max-addons' )
			];
			$elements['woo-order-overview']    = [
				'category' => __( 'Thank You Page Builder', 'max-addons' ),
				'title'    => __( 'Woo - Order Overview', 'max-addons' )
			];
			$elements['woo-order-billing-address'] = [
				'category' => __( 'Thank You Page Builder', 'max-addons' ),
				'title'    => __( 'Woo - Order Billing Address', 'max-addons' )
			];
			$elements['woo-order-shipping-address'] = [
				'category' => __( 'Thank You Page Builder', 'max-addons' ),
				'title'    => __( 'Woo - Order Shipping Address', 'max-addons' )
			];

			// ksort( $elements );
		}

		return $elements;
	}

	/**
	 * Get conditions.
	 *
	 * @since 1.2.0
	 * @return array
	 */
	public static function get_conditions() {
		$conditions = array(
			'post_category'        => __( 'Post category', 'max-addons' ),
			'post_tag'             => __( 'Post tag', 'max-addons' ),
			'post_type'            => __( 'Post type', 'max-addons' ),
			'post_comments_count'  => __( 'Post comments count', 'max-addons' ),
			'post_comments_status' => __( 'Post comments status', 'max-addons' ),
			'author_archive'       => __( 'Author archive', 'max-addons' ),
			'category_archive'     => __( 'Category archive', 'max-addons' ),
			'tag_archive'          => __( 'Tag archive', 'max-addons' ),
			'search_results'       => __( 'Search', 'max-addons' ),
			'request_parameter'    => __( 'Request parameter', 'max-addons' ),
		);

		ksort( $conditions );

		return $conditions;
	}

	/**
	 * Get conditions.
	 *
	 * @since 1.2.0
	 * @return array
	 */
	public static function get_woo_conditions() {
		$conditions = array();

		if ( Helper::is_woocommerce_active() ) {
			$conditions = array(
				'woo_product_category'     => __( 'Product category', 'max-addons' ),
				'woo_product_price'        => __( 'Product price', 'max-addons' ),
				'woo_product_rating'       => __( 'Product rating', 'max-addons' ),
				'woo_product_stock'        => __( 'Product stock', 'max-addons' ),
				'woo_product_type'         => __( 'Product type', 'max-addons' ),
				'woo_orders_placed'        => __( 'Orders placed', 'max-addons' ),
				'woo_customer_total_spent' => __( 'Customer total spent', 'max-addons' ),
				'woo_cart_products_count'  => __( 'Cart products count', 'max-addons' ),
				'woo_cart_products'        => __( 'Cart products', 'max-addons' ),
				'woo_cart_total'           => __( 'Cart total', 'max-addons' ),
			);

			ksort( $conditions );
		}

		return $conditions;
	}

	/**
	 * Get features.
	 *
	 * @since 1.2.0
	 * @return array
	 */
	public static function get_features() {
		$features = array(
			'mask'             => __( 'Mask', 'max-addons' ),
			'tooltip'          => __( 'Tooltip', 'max-addons' ),
			'wrapper-link'     => __( 'Wrapper Link', 'max-addons' ),
		);

		ksort( $features );

		return $features;
	}

	/**
	 * Get enabled elements.
	 *
	 * @since 1.0.0
	 * @return array
	 */
	public static function get_enabled_elements() {
		$enabled_modules = get_option( 'max_bricks_elements', 'disabled' );

		if ( is_array( $enabled_modules ) ) {
			return $enabled_modules;
		}

		if ( 'disabled' == $enabled_modules ) {
			return $enabled_modules;
		}

		return self::get_elements();
	}

	/**
	 * Get enabled woo builder elements.
	 *
	 * @since 1.11.0
	 * @return array
	 */
	public static function get_enabled_woo_builder_elements() {
		$enabled_modules = get_option( 'max_bricks_woo_builder_elements', 'disabled' );

		if ( is_array( $enabled_modules ) ) {
			return $enabled_modules;
		}

		if ( 'disabled' == $enabled_modules ) {
			return $enabled_modules;
		}

		return self::get_woo_builder_elements();
	}

	/**
	 * Get enabled elements.
	 *
	 * @since 1.0.0
	 * @return array
	 */
	public static function get_enabled_conditions() {
		$enabled_conditions = get_option( 'max_bricks_conditions', 'disabled' );

		if ( is_array( $enabled_conditions ) ) {
			return $enabled_conditions;
		}

		if ( 'disabled' == $enabled_conditions ) {
			return $enabled_conditions;
		}

		return self::get_conditions();
	}

	/**
	 * Get enabled elements.
	 *
	 * @since 1.0.0
	 * @return array
	 */
	public static function get_enabled_features() {
		$enabled_features = get_option( 'max_bricks_features', 'disabled' );

		if ( is_array( $enabled_features ) ) {
			return $enabled_features;
		}

		if ( 'disabled' == $enabled_features ) {
			return $enabled_features;
		}

		return self::get_features();
	}

	/**
	 * Get settings.
	 *
	 * @since 1.0.0
	 * @return array
	 */
	public static function get_settings() {
		$default_settings = array(
			'plugin_name'          => '',
			'plugin_short_name'    => '',
			'plugin_desc'          => '',
			'plugin_author'        => '',
			'plugin_uri'           => '',
			'admin_label'          => '',
			'support_link'         => '',
			'hide_support'         => 'off',
			'hide_wl_settings'     => 'off',
			'hide_plugin'          => 'off',
		);

		$settings = get_option( 'mab_elements_settings' );

		if ( ! is_array( $settings ) || empty( $settings ) ) {
			$settings = $default_settings;
		}

		if ( is_array( $settings ) && ! empty( $settings ) ) {
			$settings = array_merge( $default_settings, $settings );
		}

		return apply_filters( 'mab_elements_admin_settings', $settings );
	}

	/**
	 * Get admin label from settings.
	 *
	 * @since 1.0.0
	 * @return string
	 */
	public static function get_admin_label() {
		$settings = self::get_settings();

		$admin_label = $settings['admin_label'];

		return trim( $admin_label ) == '' ? 'Max Addons' : trim( $admin_label );
	}

	/**
	 * Renders the update message.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public static function render_update_message() {
		if ( ! empty( self::$errors ) ) {
			foreach ( self::$errors as $message ) {
				echo '<div class="error"><p>' . wp_kses_post( $message ) . '</p></div>';
			}
		} elseif ( ! empty( $_POST ) && ! isset( $_POST['email'] ) ) {
			echo '<div class="updated"><p>' . esc_html__( 'Settings updated!', 'max-addons' ) . '</p></div>';
		}
	}

	/**
	 * Adds an error message to be rendered.
	 *
	 * @since 1.0.0
	 * @param string $message The error message to add.
	 * @return void
	 */
	public static function add_error( $message ) {
		self::$errors[] = $message;
	}

	/**
	 * Renders the admin settings menu.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public static function add_menu_page() {
		$admin_label = self::get_admin_label();

		$title = $admin_label;
		$cap   = 'manage_options';
		$slug  = 'mab-settings';
		$func  = __CLASS__ . '::render';

		add_submenu_page( 'bricks', $title, $title, $cap, $slug, $func );
	}

	public static function render() {
		include MAB_PRO_DIR . 'includes/admin/admin-settings.php';
		//include MAB_PRO_DIR . 'includes/modules-manager.php';
	}

	public static function get_tabs() {
		$settings = self::get_settings();

		return apply_filters( 'mab_elements_admin_settings_tabs', array(
			'general'     => array(
				'title'    => esc_html__( 'General', 'max-addons' ),
				'show'     => true,
				'cap'      => 'manage_options',
				'file'     => MAB_PRO_DIR . 'includes/admin/admin-settings-license.php',
				'priority' => 50,
			),
			'elements'    => array(
				'title'    => esc_html__( 'Elements', 'max-addons' ),
				'show'     => true,
				'cap'      => 'edit_posts',
				'file'     => MAB_PRO_DIR . 'includes/admin/admin-settings-elements.php',
				'priority' => 100,
			),
			'conditions'    => array(
				'title'    => esc_html__( 'Conditions', 'max-addons' ),
				'show'     => true,
				'cap'      => 'edit_posts',
				'file'     => MAB_PRO_DIR . 'includes/admin/admin-settings-conditions.php',
				'priority' => 150,
			),
			'features'     => array(
				'title'    => esc_html__( 'Features', 'max-addons' ),
				'show'     => true,
				'cap'      => 'edit_posts',
				'file'     => MAB_PRO_DIR . 'includes/admin/admin-settings-features.php',
				'priority' => 200,
			),
			'woo-builder'     => array(
				'title'    => esc_html__( 'Woo Builder', 'max-addons' ),
				'show'     => true,
				'cap'      => 'edit_posts',
				'file'     => MAB_PRO_DIR . 'includes/admin/admin-settings-woo-builder.php',
				'priority' => 210,
			),
			'integration' => array(
				'title'    => esc_html__( 'Integration', 'max-addons' ),
				'show'     => ! self::get_option( 'mab_hide_integration_tab' ),
				'cap'      => ! is_network_admin() ? 'manage_options' : 'manage_network_plugins',
				'file'     => MAB_PRO_DIR . 'includes/admin/admin-settings-integration.php',
				'priority' => 250,
			),
			'white-label'   => array(
				'title'     => esc_html__( 'White Label', 'max-addons' ),
				'show'      => ( is_network_admin() || ! is_multisite() ) && ( ! self::get_option( 'mab_hide_wl_settings' ) || self::get_option( 'mab_hide_wl_settings' ) == 0 ),
				'cap'		=> ! is_network_admin() ? 'manage_options' : 'manage_network_plugins',
				'file'      => MAB_PRO_DIR . 'includes/admin/admin-settings-wl.php',
				'priority'  => 300,
			),
		) );
	}

	public static function render_tabs( $current_tab ) {
		$tabs = self::get_tabs();
		$sorted_data = array();

		foreach ( $tabs as $key => $data ) {
			$data['key'] = $key;
			$sorted_data[ $data['priority'] ] = $data;
		}

		ksort( $sorted_data );

		foreach ( $sorted_data as $data ) {
			if ( $data['show'] ) {
				if ( isset( $data['cap'] ) && ! current_user_can( $data['cap'] ) ) {
					continue;
				}
				?>
				<a href="<?php echo self::get_form_action( '&tab=' . $data['key'] ); ?>" class="nav-tab<?php echo ( $current_tab == $data['key'] ? ' nav-tab-active' : '' ); ?>"><span><?php echo $data['title']; ?></span></a>
				<?php
			}
		}
	}

	public static function render_setting_page() {
		$tabs = self::get_tabs();
		$current_tab = self::get_current_tab();

		if ( isset( $tabs[ $current_tab ] ) ) {
			$no_setting_file_msg = esc_html__( 'Setting page file could not be located.', 'max-addons' );

			if ( ! isset( $tabs[ $current_tab ]['file'] ) || empty( $tabs[ $current_tab ]['file'] ) ) {
				echo $no_setting_file_msg;
				return;
			}

			if ( ! file_exists( $tabs[ $current_tab ]['file'] ) ) {
				echo $no_setting_file_msg;
				return;
			}

			$render = ! isset( $tabs[ $current_tab ]['show'] ) ? true : $tabs[ $current_tab ]['show'];
			$cap = 'manage_options';

			if ( isset( $tabs[ $current_tab ]['cap'] ) && ! empty( $tabs[ $current_tab ]['cap'] ) ) {
				$cap = $tabs[ $current_tab ]['cap'];
			} else {
				$cap = ! is_network_admin() ? 'manage_options' : 'manage_network_plugins';
			}

			if ( ! $render || ! current_user_can( $cap ) ) {
				esc_html_e( 'You do not have permission to view this setting.', 'max-addons' );
				return;
			}

			include $tabs[ $current_tab ]['file'];
		}
	}

	/**
	 * Get current tab.
	 */
	public static function get_current_tab() {
		$current_tab = isset( $_GET['tab'] ) ? $_GET['tab'] : 'general';

		// if ( ! isset( $_GET['tab'] ) ) {
		// 	if ( is_multisite() && ! is_network_admin() ) {
		// 		$current_tab = 'elements';
		// 	}
		// }

		return $current_tab;
	}

	/**
	 * Renders the action for a form.
	 *
	 * @since 1.0.0
	 * @param string $type The type of form being rendered.
	 * @return void
	 */
	public static function get_form_action( $type = '' ) {
		if ( is_network_admin() ) {
			return esc_url( network_admin_url( '/admin.php?page=mab-settings' . $type ) );
		} else {
			return esc_url( admin_url( '/admin.php?page=mab-settings' . $type ) );
		}
	}

	public static function get_user_roles() {
		global $wp_roles;

		return $wp_roles->get_names();
	}

	/**
	 * Returns an option from the database for
	 * the admin settings page.
	 *
	 * @since 1.0.0
	 * @param string $key The option key.
	 * @return mixed
	 */
	public static function get_option( $key, $network_override = true ) {
		if ( is_network_admin() ) {
			$value = get_site_option( $key );
		} elseif ( ! $network_override && is_multisite() ) {
			$value = get_site_option( $key );
		} elseif ( $network_override && is_multisite() ) {
			$value = get_option( $key );
			$value = ( false === $value || ( is_array( $value ) && in_array( 'disabled', $value ) && get_option( 'mab_override_ms' ) != 1 ) ) ? get_site_option( $key ) : $value;
		} else {
			$value = get_option( $key );
		}

		return $value;
	}

	/**
	 * Updates an option from the admin settings page.
	 *
	 * @since 1.0.0
	 * @param string $key The option key.
	 * @param mixed $value The value to update.
	 * @return mixed
	 */
	public static function update_option( $key, $value, $network_override = true ) {
		if ( is_network_admin() ) {
			update_site_option( $key, $value );
		}
		// Delete the option if network overrides are allowed and the override checkbox isn't checked.
		elseif ( $network_override && is_multisite() && ! isset( $_POST['mab_override_ms'] ) ) {
			delete_option( $key );
		} else {
			update_option( $key, $value );
		}
	}

	/**
	 * Delete an option from the admin settings page.
	 *
	 * @since 1.0.0
	 * @param string $key The option key.
	 * @param mixed $value The value to delete.
	 * @return mixed
	 */
	public static function delete_option( $key ) {
		if ( is_network_admin() ) {
			delete_site_option( $key );
		} else {
			delete_option( $key );
		}
	}

	/**
	 * Set the branding data to plugin.
	 *
	 * @since 1.0.0
	 * @return array
	 */
	public static function update_branding( $all_plugins ) {
		if ( ! is_array( $all_plugins ) || empty( $all_plugins ) || ! isset( $all_plugins[ MAB_PRO_BASE ] ) ) {
			return $all_plugins;
		}

		$settings = self::get_settings();

		$all_plugins[ MAB_PRO_BASE ]['Name']           = ! empty( self::get_option('mab_plugin_name') ) ? self::get_option('mab_plugin_name') : $all_plugins[ MAB_PRO_BASE ]['Name'];
		$all_plugins[ MAB_PRO_BASE ]['PluginURI']      = ! empty( self::get_option('mab_plugin_uri') ) ? self::get_option('mab_plugin_uri') : $all_plugins[ MAB_PRO_BASE ]['PluginURI'];
		$all_plugins[ MAB_PRO_BASE ]['Description']    = ! empty( self::get_option('mab_plugin_desc') ) ? self::get_option('mab_plugin_desc') : $all_plugins[ MAB_PRO_BASE ]['Description'];
		$all_plugins[ MAB_PRO_BASE ]['Author']         = ! empty( self::get_option('mab_plugin_author') ) ? self::get_option('mab_plugin_author') : $all_plugins[ MAB_PRO_BASE ]['Author'];
		$all_plugins[ MAB_PRO_BASE ]['AuthorURI']      = ! empty( self::get_option('mab_plugin_uri') ) ? self::get_option('mab_plugin_uri') : $all_plugins[ MAB_PRO_BASE ]['AuthorURI'];
		$all_plugins[ MAB_PRO_BASE ]['Title']          = ! empty( self::get_option('mab_plugin_name') ) ? self::get_option('mab_plugin_name') : $all_plugins[ MAB_PRO_BASE ]['Title'];
		$all_plugins[ MAB_PRO_BASE ]['AuthorName']     = ! empty( self::get_option('mab_plugin_author') ) ? self::get_option('mab_plugin_author') : $all_plugins[ MAB_PRO_BASE ]['AuthorName'];

		if ( self::get_option('hide_plugin') == 'on' ) {
			unset( $all_plugins[ MAB_PRO_BASE ] );
		}

		return $all_plugins;
	}

	public static function save() {
		// Only admins can save settings.
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		self::save_license();
		self::save_elements();
		self::save_integration();
		self::save_white_label();

		do_action( 'mab_elements_admin_settings_save' );
	}

	/**
	 * Saves the license.
	 *
	 * @since 1.0.0
	 * @access private
	 * @return void
	 */
	private static function save_license() {
		if ( isset( $_POST['mab_license_key'] ) ) {

			$old = get_option( 'mab_license_key' );
			$new = $_POST['mab_license_key'];

			if ( $old && $old != $new ) {
				delete_option( 'mab_license_status' ); // new license has been entered, so must reactivate
			}

			update_option( 'mab_license_key', $new );
		}
	}

	/**
	 * Saves integrations.
	 *
	 * @since 1.0.0
	 * @access private
	 * @return void
	 */
	private static function save_integration() {
		if ( ! isset( $_POST['mab_license_deactivate'] ) && ! isset( $_POST['mab_license_activate'] ) ) {
			if ( isset( $_POST['mab_google_places_api_key'] ) ) {

				// Validate Google Places API.
				if ( ! empty( $_POST['mab_google_places_api_key'] ) ) {
					$place_id = 'ChIJjZwurGTlZzkRLVOKFH8rKYc';
					$api_key  = $_POST['mab_google_places_api_key'];

					$parameters = "key=$api_key&placeid=$place_id";

					$url = "https://maps.googleapis.com/maps/api/place/details/json?$parameters";

					$response = wp_remote_post(
						$url,
						array(
							'method'      => 'POST',
							'timeout'     => 60,
							'httpversion' => '1.0',
						)
					);

					$google_status = '';
					$error_message = '';
					if ( ! is_wp_error( $response ) || wp_remote_retrieve_response_code( $response ) === 200 ) {
						$final_result  = json_decode( wp_remote_retrieve_body( $response ) );
						$result_status = $final_result->status;
						$error_message = $final_result->status;

						if ( 'OVER_QUERY_LIMIT' === $result_status ) {
							$google_status = 'exceeded';
						} elseif ( 'OK' === $result_status ) {
							$google_status = 'yes';
						} else {
							$google_status = 'no';
						}
					} else {
						$google_status = 'no';
					}

					if ( 'no' === $google_status ) {
						//$error = sprintf( __( 'Google Places API Key is invalid. Error: %s', 'max-addons' ), wp_remote_retrieve_response_message( $response ) );
						$error = sprintf( __( 'Google Places API Key is invalid. Error: %s', 'max-addons' ), $error_message );

						if ( ! empty( $error ) ) {
							wp_die( wp_kses_post( $error ), esc_html__( 'Google Places API', 'max-addons' ), array(
								'back_link' => true,
							) );
						}
					}
				}

				self::update_option( 'mab_google_places_api_key', trim( $_POST['mab_google_places_api_key'] ), false );
			}

			if ( isset( $_POST['mab_yelp_api_key'] ) ) {
				self::update_option( 'mab_yelp_api_key', trim( $_POST['mab_yelp_api_key'] ), false );
			}

			if ( isset( $_POST['mab_instagram_access_token'] ) ) {
				self::update_option( 'mab_instagram_access_token', trim( $_POST['mab_instagram_access_token'] ), false );
			}

			if ( isset( $_POST['mab_youtube_api_key'] ) ) {
				self::update_option( 'mab_youtube_api_key', trim( $_POST['mab_youtube_api_key'] ), false );
			}
		}
	}

	public static function save_elements() {
		if ( ! isset( $_POST['mab-elements-settings-nonce'] ) || ! wp_verify_nonce( $_POST['mab-elements-settings-nonce'], 'mab-elements-settings' ) ) {
			return;
		}

		if ( ! empty( $_POST['mab_enabled_elements'] ) ) {
			update_option( 'max_bricks_elements', $_POST['mab_enabled_elements'] );
		} else {
			update_option( 'max_bricks_elements', 'disabled' );
		}
	}

	public static function save_woo_builder_elements() {
		if ( ! isset( $_POST['mab-woo-builder-elements-settings-nonce'] ) || ! wp_verify_nonce( $_POST['mab-woo-builder-elements-settings-nonce'], 'mab-woo-builder-elements-settings' ) ) {
			return;
		}

		if ( ! empty( $_POST['mab_enabled_woo_builder_elements'] ) ) {
			update_option( 'max_bricks_woo_builder_elements', $_POST['mab_enabled_woo_builder_elements'] );
		} else {
			update_option( 'max_bricks_woo_builder_elements', 'disabled' );
		}
	}

	public static function save_conditions() {
		if ( ! isset( $_POST['mab-conditions-settings-nonce'] ) || ! wp_verify_nonce( $_POST['mab-conditions-settings-nonce'], 'mab-conditions-settings' ) ) {
			return;
		}

		if ( ! empty( $_POST['mab_enabled_conditions'] ) ) {
			update_option( 'max_bricks_conditions', $_POST['mab_enabled_conditions'] );
		} else {
			update_option( 'max_bricks_conditions', 'disabled' );
		}
	}

	public static function save_features() {
		if ( ! isset( $_POST['mab-features-settings-nonce'] ) || ! wp_verify_nonce( $_POST['mab-features-settings-nonce'], 'mab-features-settings' ) ) {
			return;
		}

		if ( ! empty( $_POST['mab_enabled_features'] ) ) {
			update_option( 'max_bricks_features', $_POST['mab_enabled_features'] );
		} else {
			update_option( 'max_bricks_features', 'disabled' );
		}
	}

	/**
	 * Saves the white label settings.
	 *
	 * @since 1.0.0
	 * @access private
	 * @return void
	 */
	private static function save_white_label() {
		if ( ! isset( $_POST['mab-wl-settings-nonce'] ) || ! wp_verify_nonce( $_POST['mab-wl-settings-nonce'], 'mab-wl-settings' ) ) {
			return;
		}

		$settings = self::get_settings();

		if ( isset( $_POST['mab_plugin_name'] ) ) {
			self::update_option( 'mab_plugin_name', sanitize_text_field( $_POST['mab_plugin_name'] ) );
			self::update_option( 'mab_plugin_desc', esc_textarea( $_POST['mab_plugin_desc'] ) );
			self::update_option( 'mab_plugin_author', sanitize_text_field( $_POST['mab_plugin_author'] ) );
			self::update_option( 'mab_plugin_uri', esc_url( $_POST['mab_plugin_uri'] ) );
		}
		$plugin_short_name      = isset( $_POST['mab_plugin_short_name'] ) ? sanitize_text_field( $_POST['mab_plugin_short_name'] ) : '';
		$admin_label            = isset( $_POST['mab_admin_label'] ) ? sanitize_text_field( $_POST['mab_admin_label'] ) : 'Max Addons';
		$support_link           = isset( $_POST['mab_support_link'] ) ? esc_url_raw( $_POST['mab_support_link'] ) : 'httsp://wpbricksaddons.com/contact/';
		$hide_integration_tab   = isset( $_POST['mab_hide_integration_tab'] ) ? absint( $_POST['mab_hide_integration_tab'] ) : 0;
		$hide_maintenance_tab   = isset( $_POST['mab_hide_maintenance_tab'] ) ? absint( $_POST['mab_hide_maintenance_tab'] ) : 0;
		$hide_wl_form           = isset( $_POST['mab_hide_wl_settings'] ) ? absint( $_POST['mab_hide_wl_settings'] ) : 0;
		$hide_plugin            = isset( $_POST['mab_hide_plugin'] ) ? absint( $_POST['mab_hide_plugin'] ) : 0;

		self::update_option( 'mab_plugin_short_name', $plugin_short_name );
		self::update_option( 'mab_admin_label', $admin_label );
		self::update_option( 'mab_support_link', $support_link );
		self::update_option( 'mab_hide_integration_tab', $hide_integration_tab );
		self::update_option( 'mab_hide_maintenance_tab', $hide_maintenance_tab );
		self::update_option( 'mab_hide_wl_settings', $hide_wl_form );
		self::update_option( 'mab_hide_plugin', $hide_plugin );
	}

	public static function migrate_settings() {
		if ( ! is_multisite() ) {
			return;
		}

		if ( 'yes' === get_option( 'mab_multisite_settings_migrated' ) ) {
			return;
		}

		$fields = array(
			'mab_license_status',
			'mab_license_key',
			'mab_elements_settings',
			'max_bricks_elements',
			'max_bricks_woo_builder_elements',
			'max_bricks_conditions',
			'max_bricks_features',
		);

		foreach ( $fields as $field ) {
			$value = get_site_option( $field );
			if ( $value ) {
				update_option( $field, $value );
			}
		}

		update_option( 'mab_multisite_settings_migrated', 'yes' );
	}

	/**
	* Refresh instagram token after 30 days.
	*
	* @since 2.5.2
	*/
	public static function refresh_instagram_access_token( $access_token = '', $widget_id = '' ) {
		if ( empty( $access_token ) ) {
			$access_token = trim( \MaxAddonsPro\Classes\Admin_Settings::get_option( 'mab_instagram_access_token' ) );
		}

		$updated_access_token = "mab_updated_instagram_access_token";
		
		if ( ! empty( $widget_id ) ) {
			$updated_access_token = "mab_updated_instagram_access_token_widget_$widget_id";
		}

		if ( empty( $access_token ) ) {
			return;
		}
	
		$updated = get_transient( $updated_access_token );

		if ( ! empty( $updated ) ) {
			return;
		}
	
		$endpoint_url = add_query_arg(
			[
				'access_token' => $access_token,
				'grant_type'   => 'ig_refresh_token',
			],
			'https://graph.instagram.com/refresh_access_token'
		);
	
		$response = wp_remote_get( $endpoint_url );
	
		if ( ! $response || 200 !== wp_remote_retrieve_response_code( $response ) || is_wp_error( $response ) ) {
			set_transient( $updated_access_token, 'error', DAY_IN_SECONDS );
			return;
		}
	
		$body = wp_remote_retrieve_body( $response );
	
		if ( ! $body ) {
			set_transient( $updated_access_token, 'error', DAY_IN_SECONDS );
			return;
		}
	
		$body = json_decode( $body, true );
	
		if ( empty( $body['access_token'] ) || empty( $body['expires_in'] ) ) {
			set_transient( $updated_access_token, 'error', DAY_IN_SECONDS );
			return;
		}
	
		set_transient( $updated_access_token, 'updated', 30 * DAY_IN_SECONDS );
	}
}

Admin_Settings::init();
